CREATE TABLE "Categories" (
  "CategoryID" NUMBER(10) NOT NULL,
  "CategoryName" VARCHAR2(15) NOT NULL,
  "Description" VARCHAR2(4000),
  "Picture" BLOB,
  CONSTRAINT "PK_Categories" PRIMARY KEY ("CategoryID"))
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
)
LOGGING;

CREATE INDEX "CategoryName" ON "Categories" ("CategoryName")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE TABLE "Customers" (
  "CustomerID" CHAR(5) NOT NULL,
  "CompanyName" VARCHAR2(40) NOT NULL,
  "ContactName" VARCHAR2(30),
  "ContactTitle" VARCHAR2(30),
  "Address" VARCHAR2(60),
  "City" VARCHAR2(15),
  "Region" CHAR(15),
  "PostalCode" VARCHAR2(10),
  "Country" VARCHAR2(15),
  "Phone" VARCHAR2(24),
  "Fax" VARCHAR2(24),
  CONSTRAINT "PK_Customers" PRIMARY KEY ("CustomerID"))
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
)
LOGGING;

CREATE INDEX "City" ON "Customers" ("City")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "CompanyName" ON "Customers" ("CompanyName")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "PostalCode" ON "Customers" ("PostalCode")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "Region" ON "Customers" ("Region")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE TABLE "Regions" (
  "RegionID" NUMBER(10) NOT NULL,
  "RegionDescription" CHAR(50) NOT NULL,
  CONSTRAINT "PK_Region" PRIMARY KEY ("RegionID"))
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
)
LOGGING;

CREATE TABLE "Suppliers" (
  "SupplierID" NUMBER(10) NOT NULL,
  "CompanyName" VARCHAR2(40) NOT NULL,
  "ContactName" VARCHAR2(30),
  "ContactTitle" VARCHAR2(30),
  "Address" VARCHAR2(60),
  "City" VARCHAR2(15),
  "Region" VARCHAR2(15),
  "PostalCode" VARCHAR2(10),
  "Country" VARCHAR2(15),
  "Phone" VARCHAR2(24),
  "Fax" VARCHAR2(24),
  "HomePage" CLOB,
  CONSTRAINT "PK_Suppliers" PRIMARY KEY ("SupplierID"))
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
)
LOGGING;

CREATE INDEX "CompanyName00000" ON "Suppliers" ("CompanyName")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "PostalCode00001" ON "Suppliers" ("PostalCode")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE TABLE "Products" (
  "ProductID" NUMBER(10) NOT NULL,
  "ProductName" VARCHAR2(40) NOT NULL,
  "SupplierID" NUMBER(10),
  "CategoryID" NUMBER(10),
  "QuantityPerUnit" VARCHAR2(20),
  "UnitPrice" NUMBER(23, 4) DEFAULT 0,
  "UnitsInStock" NUMBER(5) DEFAULT 0,
  "UnitsOnOrder" NUMBER(5) DEFAULT 0,
  "ReorderLevel" NUMBER(5) DEFAULT 0,
  "Discontinued" NUMBER(1) DEFAULT 0 NOT NULL  ,
  "DiscontinuedDate" TIMESTAMP(6) DEFAULT NULL,
  CONSTRAINT "CK_Products_UnitPrice" CHECK (  "UnitPrice">=0  ),
  CONSTRAINT "CK_ReorderLevel" CHECK (  "ReorderLevel" >=0  ),
  CONSTRAINT "CK_UnitsInStock" CHECK (  "UnitsInStock">=0  ),
  CONSTRAINT "CK_UnitsOnOrder" CHECK (  "UnitsOnOrder">=0  ),
  CONSTRAINT "FK_Products_Categories" FOREIGN KEY ("CategoryID")
    REFERENCES "Categories"("CategoryID"),
  CONSTRAINT "FK_Products_Suppliers" FOREIGN KEY ("SupplierID")
    REFERENCES "Suppliers"("SupplierID"),
  CONSTRAINT "PK_Products" PRIMARY KEY ("ProductID"))
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
)
LOGGING;

CREATE INDEX "CategoriesProducts" ON "Products" ("CategoryID")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "ProductName" ON "Products" ("ProductName")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "SupplierID" ON "Products" ("SupplierID")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE TABLE "Territories" (
  "TerritoryID" NUMBER(10) NOT NULL,
  "TerritoryDescription" CHAR(50) NOT NULL,
  "RegionID" NUMBER(10) NOT NULL,
  CONSTRAINT "FK_Territories_Region" FOREIGN KEY ("RegionID")
    REFERENCES "Regions"("RegionID"),
  CONSTRAINT "PK_Territories" PRIMARY KEY ("TerritoryID")
  )
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
)
LOGGING;

CREATE TABLE "Employees" (
  "EmployeeID" NUMBER(10) NOT NULL,
  "LastName" VARCHAR2(20) NOT NULL,
  "FirstName" VARCHAR2(10) NOT NULL,
  "Title" VARCHAR2(30),
  "TitleOfCourtesy" VARCHAR2(25),
  "BirthDate" TIMESTAMP(6),
  "HireDate" TIMESTAMP(6),
  "Address" VARCHAR2(60),
  "City" VARCHAR2(15),
  "Region" VARCHAR2(15),
  "PostalCode" VARCHAR2(10),
  "Country" VARCHAR2(15),
  "HomePhone" VARCHAR2(24),
  "Extension" VARCHAR2(4),
  "Photo" BLOB,
  "Notes" VARCHAR2(4000),
  "PhotoPath" VARCHAR2(255),
  CONSTRAINT "CK_Birthday" CHECK (  "BirthDate"<''  ),
  CONSTRAINT "PK_Employees" PRIMARY KEY ("EmployeeID"))
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
)
LOGGING;

CREATE INDEX "LastName" ON "Employees" ("LastName")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "PostalCode00000" ON "Employees" ("PostalCode")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE TABLE "EmployeesTerritories" (
  "EmployeeID" NUMBER(10) NOT NULL,
  "TerritoryID" NUMBER(10) NOT NULL,
  CONSTRAINT "PK_EmployeeTerritories" PRIMARY KEY ("EmployeeID", "TerritoryID"))
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
)
LOGGING;

CREATE TABLE "Orders" (
  "OrderID" NUMBER(10) NOT NULL,
  "CustomerID" CHAR(5),
  "EmployeeID" NUMBER(10),
  "OrderDate" TIMESTAMP(6),
  "RequiredDate" TIMESTAMP(6),
  "ShippedDate" TIMESTAMP(6),
  "ShipVia" NUMBER(10),
  "Freight" NUMBER(23, 4) DEFAULT 0,
  "ShipName" VARCHAR2(40),
  "ShipAddress" VARCHAR2(60),
  "ShipCity" VARCHAR2(15),
  "ShipRegion" VARCHAR2(15),
  "ShipPostalCode" VARCHAR2(10),
  "ShipCountry" VARCHAR2(15),
  CONSTRAINT "PK_Orders" PRIMARY KEY ("OrderID"))
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
)
LOGGING;

CREATE INDEX "CustomerID" ON "Orders" ("CustomerID")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "EmployeeID" ON "Orders" ("EmployeeID")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "OrderDate" ON "Orders" ("OrderDate")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "ShipPostalCode" ON "Orders" ("ShipPostalCode")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "ShippedDate" ON "Orders" ("ShippedDate")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE TABLE "OrderDetails" (
  "OrderID" NUMBER(10) NOT NULL,
  "ProductID" NUMBER(10) NOT NULL,
  "UnitPrice" NUMBER(23, 4)  DEFAULT 0  NOT NULL,
  "Quantity" NUMBER(5)  DEFAULT 1 NOT NULL ,
  "Discount" NUMBER  DEFAULT 0 NOT NULL  ,
  CONSTRAINT "CK_Discount" CHECK (  "Discount">=0 AND "Discount"<=1  ),
  CONSTRAINT "CK_Quantity" CHECK (  "Quantity">0  ),
  CONSTRAINT "CK_UnitPrice" CHECK (  "UnitPrice">=0  ),
  CONSTRAINT "PK_Order_Details" PRIMARY KEY ("OrderID", "ProductID")
  )
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
)
LOGGING;

CREATE INDEX "OrderID" ON "OrderDetails" ("OrderID")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

CREATE INDEX "ProductID" ON "OrderDetails" ("ProductID")
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
);

ALTER TABLE "EmployeesTerritories" 
  ADD CONSTRAINT "FK_EmployeeTerritories_Empl" FOREIGN KEY ("EmployeeID")
    REFERENCES "Employees"("EmployeeID");

ALTER TABLE "EmployeesTerritories" 
  ADD CONSTRAINT "FK_EmployeeTerritories_Terr" FOREIGN KEY ("TerritoryID")
    REFERENCES "Territories"("TerritoryID");

ALTER TABLE "Orders" 
  ADD CONSTRAINT "FK_Orders_Customers" FOREIGN KEY ("CustomerID")
    REFERENCES "Customers"("CustomerID");

ALTER TABLE "OrderDetails" 
  ADD CONSTRAINT "FK_Order_Details_Orders" FOREIGN KEY ("OrderID")
    REFERENCES "Orders"("OrderID");

ALTER TABLE "OrderDetails" 
  ADD CONSTRAINT "FK_Order_Details_Products" FOREIGN KEY ("ProductID")
    REFERENCES "Products"("ProductID");

CREATE SEQUENCE "Categories_0"
  START WITH 1
  INCREMENT BY 1
  NOCACHE
  NOCYCLE
  ORDER;

CREATE SEQUENCE "Employees_0"
  START WITH 1
  INCREMENT BY 1
  NOCACHE
  NOCYCLE
  ORDER;

CREATE SEQUENCE "Orders_0"
  START WITH 1
  INCREMENT BY 1
  NOCACHE
  NOCYCLE
  ORDER;

CREATE SEQUENCE "Products_0"
  START WITH 1
  INCREMENT BY 1
  NOCACHE
  NOCYCLE
  ORDER;

CREATE SEQUENCE "Suppliers_0"
  START WITH 1
  INCREMENT BY 1
  NOCACHE
  NOCYCLE
  ORDER;

CREATE OR REPLACE TRIGGER "Categories"
  BEFORE INSERT ON "Categories" FOR EACH ROW
DECLARE last_Sequence NUMBER; last_InsertID NUMBER; BEGIN IF (:NEW."CategoryID" IS NULL) THEN SELECT "Categories_0".NEXTVAL INTO :NEW."CategoryID" FROM DUAL; ELSE SELECT Last_Number-1 INTO last_Sequence FROM User_Sequences WHERE UPPER(Sequence_Name) = UPPER('Categories_0'); SELECT :NEW."CategoryID" INTO last_InsertID FROM DUAL; WHILE (last_InsertID > last_Sequence) LOOP SELECT "Categories_0".NEXTVAL INTO last_Sequence FROM DUAL; END LOOP; END IF; END;
/

CREATE OR REPLACE TRIGGER "Employees"
  BEFORE INSERT ON "Employees" FOR EACH ROW
DECLARE last_Sequence NUMBER; last_InsertID NUMBER; BEGIN IF (:NEW."EmployeeID" IS NULL) THEN SELECT "Employees_0".NEXTVAL INTO :NEW."EmployeeID" FROM DUAL; ELSE SELECT Last_Number-1 INTO last_Sequence FROM User_Sequences WHERE UPPER(Sequence_Name) = UPPER('Employees_0'); SELECT :NEW."EmployeeID" INTO last_InsertID FROM DUAL; WHILE (last_InsertID > last_Sequence) LOOP SELECT "Employees_0".NEXTVAL INTO last_Sequence FROM DUAL; END LOOP; END IF; END;
/

CREATE OR REPLACE TRIGGER "Orders"
  BEFORE INSERT ON "Orders" FOR EACH ROW
DECLARE last_Sequence NUMBER; last_InsertID NUMBER; BEGIN IF (:NEW."OrderID" IS NULL) THEN SELECT "Orders_0".NEXTVAL INTO :NEW."OrderID" FROM DUAL; ELSE SELECT Last_Number-1 INTO last_Sequence FROM User_Sequences WHERE UPPER(Sequence_Name) = UPPER('Orders_0'); SELECT :NEW."OrderID" INTO last_InsertID FROM DUAL; WHILE (last_InsertID > last_Sequence) LOOP SELECT "Orders_0".NEXTVAL INTO last_Sequence FROM DUAL; END LOOP; END IF; END;
/

CREATE OR REPLACE TRIGGER "Products"
  BEFORE INSERT ON "Products" FOR EACH ROW
DECLARE last_Sequence NUMBER; last_InsertID NUMBER; BEGIN IF (:NEW."ProductID" IS NULL) THEN SELECT "Products_0".NEXTVAL INTO :NEW."ProductID" FROM DUAL; ELSE SELECT Last_Number-1 INTO last_Sequence FROM User_Sequences WHERE UPPER(Sequence_Name) = UPPER('Products_0'); SELECT :NEW."ProductID" INTO last_InsertID FROM DUAL; WHILE (last_InsertID > last_Sequence) LOOP SELECT "Products_0".NEXTVAL INTO last_Sequence FROM DUAL; END LOOP; END IF; END;
/

CREATE OR REPLACE TRIGGER "Suppliers"
  BEFORE INSERT ON "Suppliers" FOR EACH ROW
DECLARE last_Sequence NUMBER; last_InsertID NUMBER; BEGIN IF (:NEW."SupplierID" IS NULL) THEN SELECT "Suppliers_0".NEXTVAL INTO :NEW."SupplierID" FROM DUAL; ELSE SELECT Last_Number-1 INTO last_Sequence FROM User_Sequences WHERE UPPER(Sequence_Name) = UPPER('Suppliers_0'); SELECT :NEW."SupplierID" INTO last_InsertID FROM DUAL; WHILE (last_InsertID > last_Sequence) LOOP SELECT "Suppliers_0".NEXTVAL INTO last_Sequence FROM DUAL; END LOOP; END IF; END;
/

CREATE OR REPLACE VIEW "Alphabetical list of products" (
  "ProductID",
  "ProductName",
  "SupplierID",
  "CategoryID",
  "QuantityPerUnit",
  "UnitPrice",
  "UnitsInStock",
  "UnitsOnOrder",
  "ReorderLevel",
  "Discontinued",
  "DiscontinuedDate",
  "CategoryName"
)
AS
  SELECT "Products".*, "Categories"."CategoryName"
FROM "Categories" INNER JOIN "Products" ON "Categories"."CategoryID" = "Products"."CategoryID"
WHERE ((("Products"."Discontinued")=0));

CREATE OR REPLACE VIEW "Current Product List" (
  "ProductID",
  "ProductName"
)
AS
  SELECT Product_List."ProductID", Product_List."ProductName"
FROM "Products" Product_List
WHERE (((Product_List."Discontinued")=0));

CREATE OR REPLACE VIEW "Customer and Suppliers by City" (
  "City",
  "CompanyName",
  "ContactName",
  "Relationship"
)
AS
  SELECT "City", "CompanyName", "ContactName", 'Customers'  "Relationship" 
FROM "Customers"
UNION SELECT "City", "CompanyName", "ContactName", 'Suppliers'
FROM "Suppliers";

CREATE OR REPLACE VIEW "Order Details Extended" (
  "OrderID",
  "ProductID",
  "ProductName",
  "UnitPrice",
  "Quantity",
  "Discount",
  "ExtendedPrice"
)
AS
  SELECT "OrderDetails"."OrderID", "OrderDetails"."ProductID", "Products"."ProductName", 
	"OrderDetails"."UnitPrice", "OrderDetails"."Quantity", "OrderDetails"."Discount", 
	TO_NUMBER((("OrderDetails"."UnitPrice"*"OrderDetails"."Quantity"*(1-"OrderDetails"."Discount")/100)*100)) AS "ExtendedPrice"
FROM "Products" INNER JOIN "OrderDetails" ON "Products"."ProductID" = "OrderDetails"."ProductID";

CREATE OR REPLACE VIEW "Order Subtotals" (
  "OrderID",
  "Subtotal"
)
AS
  SELECT "OrderDetails"."OrderID", 
       SUM(("OrderDetails"."UnitPrice"*"OrderDetails"."Quantity"*(1-"OrderDetails"."Discount")/100)*100)  "Subtotal"
FROM "OrderDetails"
GROUP BY "OrderDetails"."OrderID";

CREATE OR REPLACE VIEW "Orders Qry" (
  "OrderID",
  "CustomerID",
  "EmployeeID",
  "OrderDate",
  "RequiredDate",
  "ShippedDate",
  "ShipVia",
  "Freight",
  "ShipName",
  "ShipAddress",
  "ShipCity",
  "ShipRegion",
  "ShipPostalCode",
  "ShipCountry",
  "CompanyName",
  "Address",
  "City",
  "Region",
  "PostalCode",
  "Country"
)
AS
  SELECT "Orders"."OrderID", "Orders"."CustomerID", "Orders"."EmployeeID", "Orders"."OrderDate", "Orders"."RequiredDate", 
	"Orders"."ShippedDate", "Orders"."ShipVia", "Orders"."Freight", "Orders"."ShipName", "Orders"."ShipAddress", "Orders"."ShipCity", 
	"Orders"."ShipRegion", "Orders"."ShipPostalCode", "Orders"."ShipCountry", 
	"Customers"."CompanyName", "Customers"."Address", "Customers"."City", "Customers"."Region", "Customers"."PostalCode", 
  "Customers"."Country"
FROM "Customers" INNER JOIN "Orders" ON "Customers"."CustomerID" = "Orders"."CustomerID";

CREATE OR REPLACE VIEW "Product Sales for 1997" (
  "CategoryName",
  "ProductName",
  "ProductSales"
)
AS
  SELECT "Categories"."CategoryName", "Products"."ProductName", 
SUM(("OrderDetails"."UnitPrice"*"OrderDetails"."Quantity"*(1-"OrderDetails"."Discount")/100)*100) AS "ProductSales"
FROM ("Categories" INNER JOIN "Products" ON "Categories"."CategoryID" = "Products"."CategoryID") 
	INNER JOIN ("Orders" 
		INNER JOIN "OrderDetails" ON "Orders"."OrderID" = "OrderDetails"."OrderID") 
	ON "Products"."ProductID" = "OrderDetails"."ProductID"
WHERE ("Orders"."ShippedDate" BETWEEN TO_DATE('01-01-1997','dd-mm-yyyy') AND TO_DATE('31-12-1997','dd-mm-yyyy'))
GROUP BY "Categories"."CategoryName", "Products"."ProductName";

CREATE OR REPLACE VIEW "Products Above Average Price" (
  "ProductName",
  "UnitPrice"
)
AS
  SELECT "Products"."ProductName", "Products"."UnitPrice"
FROM "Products"
WHERE "Products"."UnitPrice">(SELECT AVG("UnitPrice") FROM "Products");

CREATE OR REPLACE VIEW "Products by Category" (
  "CategoryName",
  "ProductName",
  "QuantityPerUnit",
  "UnitsInStock",
  "Discontinued"
)
AS
  SELECT "Categories"."CategoryName", "Products"."ProductName",
 "Products"."QuantityPerUnit", "Products"."UnitsInStock", "Products"."Discontinued"
FROM "Categories" INNER JOIN "Products" ON "Categories"."CategoryID" = "Products"."CategoryID"
WHERE "Products"."Discontinued" <> 1;

CREATE OR REPLACE VIEW "Quarterly Orders" (
  "CustomerID",
  "CompanyName",
  "City",
  "Country"
)
AS
  SELECT DISTINCT "Customers"."CustomerID", "Customers"."CompanyName", "Customers"."City", 
"Customers"."Country"
FROM "Customers" RIGHT JOIN "Orders" ON "Customers"."CustomerID" = "Orders"."CustomerID"
WHERE "Orders"."OrderDate" BETWEEN TO_DATE('01-01-1997','dd-mm-yyyy') AND TO_DATE('31-12-1997','dd-mm-yyyy');

CREATE OR REPLACE VIEW "Category Sales for 1997" (
  "CategoryName",
  "CategorySales"
)
AS
  SELECT "Product Sales for 1997"."CategoryName",
 SUM("Product Sales for 1997"."ProductSales") AS "CategorySales"
FROM "Product Sales for 1997"
GROUP BY "Product Sales for 1997"."CategoryName";

CREATE OR REPLACE VIEW "Sales by Category" (
  "CategoryID",
  "CategoryName",
  "ProductName",
  PRODUCTSALES
)
AS
  SELECT "Categories"."CategoryID", "Categories"."CategoryName", "Products"."ProductName", 
	SUM("Order Details Extended"."ExtendedPrice") ProductSales
FROM 	"Categories" INNER JOIN 
		("Products" INNER JOIN 
			("Orders" INNER JOIN "Order Details Extended" ON "Orders"."OrderID" = "Order Details Extended"."OrderID") 
		ON "Products"."ProductID" = "Order Details Extended"."ProductID") 
	ON "Categories"."CategoryID" = "Products"."CategoryID"
WHERE "Orders"."OrderDate" BETWEEN TO_DATE('01-01-1997','dd-mm-yyyy') AND TO_DATE('31-12-1997','dd-mm-yyyy')
GROUP BY "Categories"."CategoryID", "Categories"."CategoryName", "Products"."ProductName";

CREATE OR REPLACE VIEW "Sales Totals by Amount" (
  SALEAMOUNT,
  "OrderID",
  "CompanyName",
  "ShippedDate"
)
AS
  SELECT "Order Subtotals"."Subtotal"  SaleAmount,
   "Orders"."OrderID",
 "Customers"."CompanyName", "Orders"."ShippedDate"
FROM 	"Customers" INNER JOIN 
		("Orders" INNER JOIN "Order Subtotals" ON "Orders"."OrderID" = "Order Subtotals"."OrderID") 
	ON "Customers"."CustomerID" = "Orders"."CustomerID"
WHERE ("Order Subtotals"."Subtotal" >2500) AND 
     ("Orders"."ShippedDate" BETWEEN TO_DATE('01-01-1997','dd-mm-yyyy') AND TO_DATE('31-12-1997','dd-mm-yyyy'));

CREATE OR REPLACE VIEW "Summary of Sales by Quarter" (
  "ShippedDate",
  "OrderID",
  "Subtotal"
)
AS
  SELECT "Orders"."ShippedDate", "Orders"."OrderID", "Order Subtotals"."Subtotal"
FROM "Orders" INNER JOIN "Order Subtotals" ON "Orders"."OrderID" = "Order Subtotals"."OrderID"
WHERE "Orders"."ShippedDate" IS NOT NULL;

CREATE OR REPLACE VIEW "Summary of Sales by Year" (
  "ShippedDate",
  "OrderID",
  "Subtotal"
)
AS
  SELECT "Orders"."ShippedDate", "Orders"."OrderID", "Order Subtotals"."Subtotal"
FROM "Orders" INNER JOIN "Order Subtotals" ON "Orders"."OrderID" = "Order Subtotals"."OrderID"
WHERE "Orders"."ShippedDate" IS NOT NULL;

CREATE OR REPLACE FUNCTION "CustOrderHist" ("CustomerID_" CHAR) RETURN SYS_REFCURSOR IS 
 cur SYS_REFCURSOR;
BEGIN
   OPEN cur FOR 
      SELECT "ProductName", SUM("Quantity") "Total"
         FROM "Products" P, "OrderDetails" OD, "Orders" O, "Customers" C
         WHERE C."CustomerID" = "CustomerID_"
               AND C."CustomerID" = O."CustomerID" AND 
               O."OrderID" = OD."OrderID" AND OD."ProductID" = P."ProductID"
         GROUP BY "ProductName";
   RETURN CUR;
END;

/

CREATE OR REPLACE FUNCTION "CustOrdersDetail" ("OrderID_" NUMBER) RETURN SYS_REFCURSOR IS
 cur SYS_REFCURSOR;
BEGIN
 OPEN cur FOR
   SELECT "ProductName",
          ROUND(Od."UnitPrice", 2) "UnitPrice",
          "Quantity",
          TO_NUMBER("Discount" * 100) "Discount", 
          ROUND(TO_NUMBER("Quantity" * (1 - "Discount") * Od."UnitPrice"), 2) "ExtendedPrice"
      FROM "Products" P, "OrderDetails" Od
      WHERE Od."ProductID" = P."ProductID" AND Od."OrderID" = "OrderID_";
   RETURN CUR;
END;



/

CREATE OR REPLACE FUNCTION "CustOrdersOrders" ("CustomerID_" CHAR) RETURN SYS_REFCURSOR IS
 cur SYS_REFCURSOR;
BEGIN
  OPEN cur FOR SELECT "OrderID", 
      	"OrderDate",
	      "RequiredDate",
	      "ShippedDate"
      FROM "Orders"
      WHERE "CustomerID" ="CustomerID_"
      ORDER BY "OrderID";
   RETURN CUR;
END;


/

CREATE OR REPLACE FUNCTION "Employee Sales by Country" ("Beginning_Date_" TIMESTAMP, "Ending_Date_" TIMESTAMP) RETURN SYS_REFCURSOR
IS
 cur SYS_REFCURSOR;
BEGIN  
  OPEN cur FOR SELECT "Employees"."Country", "Employees"."LastName",
          "Employees"."FirstName", "Orders"."ShippedDate",
          "Orders"."OrderID", "Order Subtotals"."Subtotal" AS "SaleAmount"
     FROM "Employees" INNER JOIN 
	        ("Orders" INNER JOIN "Order Subtotals" ON "Orders"."OrderID" = "Order Subtotals"."OrderID") 
	         ON "Employees"."EmployeeID" = "Orders"."EmployeeID"
     WHERE "Orders"."ShippedDate" BETWEEN "Beginning_Date_" AND "Ending_Date_";
   RETURN CUR;
END;



/

CREATE OR REPLACE FUNCTION "Sales by Year" ("Beginning_Date_" TIMESTAMP, "Ending_Date_" TIMESTAMP) RETURN SYS_REFCURSOR IS
 cur SYS_REFCURSOR;
BEGIN
  OPEN cur FOR 
     SELECT "Orders"."ShippedDate", "Orders"."OrderID", "Order Subtotals"."Subtotal",
            TO_CHAR(TRUNC("ShippedDate", 'YYYY'), 'YYYY') AS "Year"
          FROM "Orders" INNER JOIN "Order Subtotals" ON 
               "Orders"."OrderID" = "Order Subtotals"."OrderID"
         WHERE "Orders"."ShippedDate" BETWEEN "Beginning_Date_" AND "Ending_Date_";
  RETURN cur;
END;


/

CREATE OR REPLACE FUNCTION "SalesByCategory"("CategoryName_" VARCHAR2, "OrdYear_" VARCHAR2 DEFAULT'1998')  RETURN SYS_REFCURSOR
IS
 cur SYS_REFCURSOR;
 par VARCHAR2(100);
BEGIN
par:="OrdYear_" ;
IF "OrdYear_" != '1996' AND "OrdYear_" != '1997' AND "OrdYear_" != '1998' THEN
	par:= '1998';
END IF;

OPEN cur FOR
  SELECT "ProductName",
	     ROUND(SUM( OD."Quantity" * (1-OD."Discount") * OD."UnitPrice"), 0) "TotalPurchase"
    FROM "OrderDetails" OD, "Orders" O, "Products" P, "Categories" C
        WHERE OD."OrderID" = O."OrderID" 
	        AND OD."ProductID" = P."ProductID" 
	        AND P."CategoryID" = C."CategoryID"
	        AND C."CategoryName" = "CategoryName_"
	        AND TO_CHAR(O."OrderDate",'YYYY') = par
          GROUP BY "ProductName"
      ORDER BY "ProductName";
    RETURN cur;
END;



/

CREATE OR REPLACE FUNCTION "Ten Most Expensive Products"  RETURN SYS_REFCURSOR
IS
 cur SYS_REFCURSOR;
BEGIN
    OPEN cur FOR 
       SELECT * FROM
       (SELECT "Products"."ProductName" AS "TenMostExpensiveProducts", "Products"."UnitPrice"
        FROM "Products"
        ORDER BY "Products"."UnitPrice" DESC)
        WHERE ROWNUM <=10;
   RETURN CUR;
 END;


/

INSERT INTO "Regions" Values (1,'Eastern');
INSERT INTO "Regions" Values (2,'Western');
INSERT INTO "Regions" Values (3,'Northern');
INSERT INTO "Regions" Values (4,'Southern');
INSERT INTO "Territories" Values ('01581', 'Westboro',1);
INSERT INTO "Territories" Values ('01730', 'Bedford',1);
INSERT INTO "Territories" Values ('01833', 'Georgetow',1);
INSERT INTO "Territories" Values ('02116', 'Boston',1);
INSERT INTO "Territories" Values ('02139', 'Cambridge',1);
INSERT INTO "Territories" Values ('02184', 'Braintree',1);
INSERT INTO "Territories" Values ('02903', 'Providence',1);
INSERT INTO "Territories" Values ('03049', 'Hollis',3);
INSERT INTO "Territories" Values ('03801', 'Portsmouth',3);
INSERT INTO "Territories" Values ('06897', 'Wilton',1);
INSERT INTO "Territories" Values ('07960', 'Morristown',1);
INSERT INTO "Territories" Values ('08837', 'Edison',1);
INSERT INTO "Territories" Values ('10019', 'New York',1);
INSERT INTO "Territories" Values ('10038', 'New York',1);
INSERT INTO "Territories" Values ('11747', 'Mellvile',1);
INSERT INTO "Territories" Values ('14450', 'Fairport',1);
INSERT INTO "Territories" Values ('19428', 'Philadelphia',3);
INSERT INTO "Territories" Values ('19713', 'Neward',1);
INSERT INTO "Territories" Values ('20852', 'Rockville',1);
INSERT INTO "Territories" Values ('27403', 'Greensboro',1);
INSERT INTO "Territories" Values ('27511', 'Cary',1);
INSERT INTO "Territories" Values ('29202', 'Columbia',4);
INSERT INTO "Territories" Values ('30346', 'Atlanta',4);
INSERT INTO "Territories" Values ('31406', 'Savannah',4);
INSERT INTO "Territories" Values ('32859', 'Orlando',4);
INSERT INTO "Territories" Values ('33607', 'Tampa',4);
INSERT INTO "Territories" Values ('40222', 'Louisville',1);
INSERT INTO "Territories" Values ('44122', 'Beachwood',3);
INSERT INTO "Territories" Values ('45839', 'Findlay',3);
INSERT INTO "Territories" Values ('48075', 'Southfield',3);
INSERT INTO "Territories" Values ('48084', 'Troy',3);
INSERT INTO "Territories" Values ('48304', 'Bloomfield Hills',3);
INSERT INTO "Territories" Values ('53404', 'Racine',3);
INSERT INTO "Territories" Values ('55113', 'Roseville',3);
INSERT INTO "Territories" Values ('55439', 'Minneapolis',3);
INSERT INTO "Territories" Values ('60179', 'Hoffman Estates',2);
INSERT INTO "Territories" Values ('60601', 'Chicago',2);
INSERT INTO "Territories" Values ('72716', 'Bentonville',4);
INSERT INTO "Territories" Values ('75234', 'Dallas',4);
INSERT INTO "Territories" Values ('78759', 'Austin',4);
INSERT INTO "Territories" Values ('80202', 'Denver',2);
INSERT INTO "Territories" Values ('80909', 'Colorado Springs',2);
INSERT INTO "Territories" Values ('85014', 'Phoenix',2);
INSERT INTO "Territories" Values ('85251', 'Scottsdale',2);
INSERT INTO "Territories" Values ('90405', 'Santa Monica',2);
INSERT INTO "Territories" Values ('94025', 'Menlo Park',2);
INSERT INTO "Territories" Values ('94105', 'San Francisco',2);
INSERT INTO "Territories" Values ('95008', 'Campbell',2);
INSERT INTO "Territories" Values ('95054', 'Santa Clara',2);
INSERT INTO "Territories" Values ('95060', 'Santa Cruz',2);
INSERT INTO "Territories" Values ('98004', 'Bellevue',2);
INSERT INTO "Territories" Values ('98052', 'Redmond',2);
INSERT INTO "Territories" Values ('98104', 'Seattle',2);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(1,'Exotic Liquids', 'Charlotte Cooper', 'Purchasing Manager', '49 Gilbert St.', 'London',NULL,'EC1 4SD', 'UK', '(171) 555-2222',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(2,'New Orleans Cajun Delights', 'Shelley Burke', 'Order Administrator', 'P.O. Box 78934', 'New Orleans', 'LA', '70117', 'USA', '(100) 555-4822',NULL,'#CAJUN.HTM#');
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(3,'Grandma Kelly''s Homestead', 'Regina Murphy', 'Sales Representative', '707 Oxford Rd.', 'Ann Arbor', 'MI', '48104', 'USA', '(313) 555-5735', '(313) 555-3349',NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(4,'Tokyo Traders', 'Yoshi Nagase', 'Marketing Manager', '9-8 Sekimai Musashino-shi', 'Tokyo',NULL,'100', 'Japan', '(03) 3555-5011',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(5,'Cooperativa de Quesos ''Las Cabras''', 'Antonio del Valle Saavedra', 'Export Administrator', 'Calle del Rosal 4', 'Oviedo', 'Asturias', '33007', 'Spain', '(98) 598 76 54',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(6,'Mayumi''s', 'Mayumi Ohno', 'Marketing Representative', '92 Setsuko Chuo-ku', 'Osaka',NULL,'545', 'Japan', '(06) 431-7877',NULL,'Mayumi''s (on the World Wide Web)#http://www.microsoft.com/accessdev/sampleapps/mayumi.htm#');
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(7,'Pavlova,Ltd.', 'Ian Devling', 'Marketing Manager', '74 Rose St. Moonie Ponds', 'Melbourne', 'Victoria', '3058', 'Australia', '(03) 444-2343', '(03) 444-6588',NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(8,'Specialty Biscuits,Ltd.', 'Peter Wilson', 'Sales Representative', '29 King''s Way', 'Manchester',NULL,'M14 GSD', 'UK', '(161) 555-4448',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(9,'PB Knackebrod AB', 'Lars Peterson', 'Sales Agent', 'Kaloadagatan 13', 'Goteborg',NULL,'S-345 67', 'Sweden', '031-987 65 43', '031-987 65 91',NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(10,'Refrescos Americanas LTDA', 'Carlos Diaz', 'Marketing Manager', 'Av. das Americanas 12.890', 'Sao Paulo',NULL,'5442', 'Brazil', '(11) 555 4640',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(11,'Heli Su?waren GmbH & Co. KG', 'Petra Winkler', 'Sales Manager', 'Tiergartenstra?e 5', 'Berlin',NULL,'10785', 'Germany', '(010) 9984510',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(12,'Plutzer Lebensmittelgro?markte AG', 'Martin Bein', 'International Marketing Mgr.', 'Bogenallee 51', 'Frankfurt',NULL,'60439', 'Germany', '(069) 992755',NULL,'Plutzer (on the World Wide Web)#http://www.microsoft.com/accessdev/sampleapps/plutzer.htm#');
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(13,'Nord-Ost-Fisch Handelsgesellschaft mbH', 'Sven Petersen', 'Coordinator Foreign Markets', 'Frahmredder 112a', 'Cuxhaven',NULL,'27478', 'Germany', '(04721) 8713', '(04721) 8714',NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(14,'Formaggi Fortini s.r.l.', 'Elio Rossi', 'Sales Representative', 'Viale Dante,75', 'Ravenna',NULL,'48100', 'Italy', '(0544) 60323', '(0544) 60603', '#FORMAGGI.HTM#');
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(15,'Norske Meierier', 'Beate Vileid', 'Marketing Manager', 'Hatlevegen 5', 'Sandvika',NULL,'1320', 'Norway', '(0)2-953010',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(16,'Bigfoot Breweries', 'Cheryl Saylor', 'Regional Account Rep.', '3400 - 8th Avenue Suite 210', 'Bend', 'OR', '97101', 'USA', '(503) 555-9931',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(17,'Svensk Sjofoda AB', 'Michael Bjorn', 'Sales Representative', 'Brovallavagen 231', 'Stockholm',NULL,'S-123 45', 'Sweden', '08-123 45 67',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(18,'Aux joyeux ecclesiastiques', 'Guylene Nodier', 'Sales Manager', '203,Rue des Francs-Bourgeois', 'Paris',NULL,'75004', 'France', '(1) 03.83.00.68', '(1) 03.83.00.62',NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(19,'New England Seafood Cannery', 'Robb Merchant', 'Wholesale Account Agent', 'Order Processing Dept. 2100 Paul Revere Blvd.', 'Boston', 'MA', '02134', 'USA', '(617) 555-3267', '(617) 555-3389',NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(20,'Leka Trading', 'Chandra Leka', 'Owner', '471 Serangoon Loop,Suite #402', 'Singapore',NULL,'0512', 'Singapore', '555-8787',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(21,'Lyngbysild', 'Niels Petersen', 'Sales Manager', 'Lyngbysild Fiskebakken 10', 'Lyngby',NULL,'2800', 'Denmark', '43844108', '43844115',NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(22,'Zaanse Snoepfabriek', 'Dirk Luchte', 'Accounting Manager', 'Verkoop Rijnweg 22', 'Zaandam',NULL,'9999 ZZ', 'Netherlands', '(12345) 1212', '(12345) 1210',NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(23,'Karkki Oy', 'Anne Heikkonen', 'Product Manager', 'Valtakatu 12', 'Lappeenranta',NULL,'53120', 'Finland', '(953) 10956',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(24,'G''day,Mate', 'Wendy Mackenzie', 'Sales Representative', '170 Prince Edward Parade Hunter''s Hill', 'Sydney', 'NSW', '2042', 'Australia', '(02) 555-5914', '(02) 555-4873', 'G''day Mate (on the World Wide Web)#http://www.microsoft.com/accessdev/sampleapps/gdaymate.htm#');
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(25,'Ma Maison', 'Jean-Guy Lauzon', 'Marketing Manager', '2960 Rue St. Laurent', 'Montreal', 'Quebec', 'H1J 1C3', 'Canada', '(514) 555-9022',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(26,'Pasta Buttini s.r.l.', 'Giovanni Giudici', 'Order Administrator', 'Via dei Gelsomini,153', 'Salerno',NULL,'84100', 'Italy', '(089) 6547665', '(089) 6547667',NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(27,'Escargots Nouveaux', 'Marie Delamare', 'Sales Manager', '22,rue H. Voiron', 'Montceau',NULL,'71300', 'France', '85.57.00.07',NULL,NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(28,'Gai paturage', 'Eliane Noz', 'Sales Representative', 'Bat. B 3,rue des Alpes', 'Annecy',NULL,'74000', 'France', '38.76.98.06', '38.76.98.58',NULL);
INSERT INTO "Suppliers"("SupplierID","CompanyName","ContactName","ContactTitle","Address","City","Region","PostalCode","Country","Phone","Fax","HomePage") VALUES(29,'Forets d''erables', 'Chantal Goulet', 'Accounting Manager', '148 rue Chasseur', 'Ste-Hyacinthe', 'Quebec', 'J2S 7S8', 'Canada', '(514) 555-2955', '(514) 555-2921',NULL);
INSERT INTO "Categories"("CategoryID","CategoryName","Description","Picture") VALUES(1,'Beverages', 'Soft drinks,coffees,teas,beers,and ales', NULL);
INSERT INTO "Categories"("CategoryID","CategoryName","Description","Picture") VALUES(2,'Condiments', 'Sweet and savory sauces,relishes,spreads,and seasonings', NULL);
INSERT INTO "Categories"("CategoryID","CategoryName","Description","Picture") VALUES(3,'Confections', 'Desserts,candies,and sweet breads', NULL);
INSERT INTO "Categories"("CategoryID","CategoryName","Description","Picture") VALUES(4,'Dairy Products', 'Cheeses', NULL);
INSERT INTO "Categories"("CategoryID","CategoryName","Description","Picture") VALUES(5,'Grains/Cereals', 'Breads,crackers,pasta,and cereal', NULL);
INSERT INTO "Categories"("CategoryID","CategoryName","Description","Picture") VALUES(6,'Meat/Poultry', 'Prepared meats', NULL);
INSERT INTO "Categories"("CategoryID","CategoryName","Description","Picture") VALUES(7,'Produce', 'Dried fruit and bean curd', NULL);
INSERT INTO "Categories"("CategoryID","CategoryName","Description","Picture") VALUES(8,'Seafood', 'Seaweed and fish', NULL);
INSERT INTO "Customers" VALUES('ALFKI', 'Alfreds Futterkiste', 'Maria Anders', 'Sales Representative', 'Obere Str. 57', 'Berlin',NULL,'12209', 'Germany', '030-0074321', '030-0076545');
INSERT INTO "Customers" VALUES('ANATR', 'Ana Trujillo Emparedados y helados', 'Ana Trujillo', 'Owner', 'Avda. de la Constitucion 2222', 'Mexico D.F.',NULL,'05021', 'Mexico', '(5) 555-4729', '(5) 555-3745');
INSERT INTO "Customers" VALUES('ANTON', 'Antonio Moreno Taqueria', 'Antonio Moreno', 'Owner', 'Mataderos 2312', 'Mexico D.F.',NULL,'05023', 'Mexico', '(5) 555-3932',NULL);
INSERT INTO "Customers" VALUES('AROUT', 'Around the Horn', 'Thomas Hardy', 'Sales Representative', '120 Hanover Sq.', 'London',NULL,'WA1 1DP', 'UK', '(171) 555-7788', '(171) 555-6750');
INSERT INTO "Customers" VALUES('BERGS', 'Berglunds snabbkop', 'Christina Berglund', 'Order Administrator', 'Berguvsvagen 8', 'Lulea',NULL,'S-958 22', 'Sweden', '0921-12 34 65', '0921-12 34 67');
INSERT INTO "Customers" VALUES('BLAUS', 'Blauer See Delikatessen', 'Hanna Moos', 'Sales Representative', 'Forsterstr. 57', 'Mannheim',NULL,'68306', 'Germany', '0621-08460', '0621-08924');
INSERT INTO "Customers" VALUES('BLONP', 'Blondesddsl pere et fils', 'Frederique Citeaux', 'Marketing Manager', '24,place Kleber', 'Strasbourg',NULL,'67000', 'France', '88.60.15.31', '88.60.15.32');
INSERT INTO "Customers" VALUES('BOLID', 'Bolido Comidas preparadas', 'Martin Sommer', 'Owner', 'C/ Araquil,67', 'Madrid',NULL,'28023', 'Spain', '(91) 555 22 82', '(91) 555 91 99');
INSERT INTO "Customers" VALUES('BONAP', 'Bon app''', 'Laurence Lebihan', 'Owner', '12,rue des Bouchers', 'Marseille',NULL,'13008', 'France', '91.24.45.40', '91.24.45.41');
INSERT INTO "Customers" VALUES('BOTTM', 'Bottom-Dollar Markets', 'Elizabeth Lincoln', 'Accounting Manager', '23 Tsawassen Blvd.', 'Tsawassen', 'BC', 'T2F 8M4', 'Canada', '(604) 555-4729', '(604) 555-3745');
INSERT INTO "Customers" VALUES('BSBEV', 'B''s Beverages', 'Victoria Ashworth', 'Sales Representative', 'Fauntleroy Circus', 'London',NULL,'EC2 5NT', 'UK', '(171) 555-1212',NULL);
INSERT INTO "Customers" VALUES('CACTU', 'Cactus Comidas para llevar', 'Patricio Simpson', 'Sales Agent', 'Cerrito 333', 'Buenos Aires',NULL,'1010', 'Argentina', '(1) 135-5555', '(1) 135-4892');
INSERT INTO "Customers" VALUES('CENTC', 'Centro comercial Moctezuma', 'Francisco Chang', 'Marketing Manager', 'Sierras de Granada 9993', 'Mexico D.F.',NULL,'05022', 'Mexico', '(5) 555-3392', '(5) 555-7293');
INSERT INTO "Customers" VALUES('CHOPS', 'Chop-suey Chinese', 'Yang Wang', 'Owner', 'Hauptstr. 29', 'Bern',NULL,'3012', 'Switzerland', '0452-076545',NULL);
INSERT INTO "Customers" VALUES('COMMI', 'Comercio Mineiro', 'Pedro Afonso', 'Sales Associate', 'Av. dos Lusiadas,23', 'Sao Paulo', 'SP', '05432-043', 'Brazil', '(11) 555-7647',NULL);
INSERT INTO "Customers" VALUES('CONSH', 'Consolidated Holdings', 'Elizabeth Brown', 'Sales Representative', 'Berkeley Gardens 12 Brewery', 'London',NULL,'WX1 6LT', 'UK', '(171) 555-2282', '(171) 555-9199');
INSERT INTO "Customers" VALUES('DRACD', 'Drachenblut Delikatessen', 'Sven Ottlieb', 'Order Administrator', 'Walserweg 21', 'Aachen',NULL,'52066', 'Germany', '0241-039123', '0241-059428');
INSERT INTO "Customers" VALUES('DUMON', 'Du monde entier', 'Janine Labrune', 'Owner', '67,rue des Cinquante Otages', 'Nantes',NULL,'44000', 'France', '40.67.88.88', '40.67.89.89');
INSERT INTO "Customers" VALUES('EASTC', 'Eastern Connection', 'Ann Devon', 'Sales Agent', '35 King George', 'London',NULL,'WX3 6FW', 'UK', '(171) 555-0297', '(171) 555-3373');
INSERT INTO "Customers" VALUES('ERNSH', 'Ernst Handel', 'Roland Mendel', 'Sales Manager', 'Kirchgasse 6', 'Graz',NULL,'8010', 'Austria', '7675-3425', '7675-3426');
INSERT INTO "Customers" VALUES('FAMIA', 'Familia Arquibaldo', 'Aria Cruz', 'Marketing Assistant', 'Rua Oros,92', 'Sao Paulo', 'SP', '05442-030', 'Brazil', '(11) 555-9857',NULL);
INSERT INTO "Customers" VALUES('FISSA', 'FISSA Fabrica Inter. Salchichas S.A.', 'Diego Roel', 'Accounting Manager', 'C/ Moralzarzal,86', 'Madrid',NULL,'28034', 'Spain', '(91) 555 94 44', '(91) 555 55 93');
INSERT INTO "Customers" VALUES('FOLIG', 'Folies gourmandes', 'Martine Rance', 'Assistant Sales Agent', '184,chaussee de Tournai', 'Lille',NULL,'59000', 'France', '20.16.10.16', '20.16.10.17');
INSERT INTO "Customers" VALUES('FOLKO', 'Folk och fa HB', 'Maria Larsson', 'Owner', 'Akergatan 24', 'Bracke',NULL,'S-844 67', 'Sweden', '0695-34 67 21',NULL);
INSERT INTO "Customers" VALUES('FRANK', 'Frankenversand', 'Peter Franken', 'Marketing Manager', 'Berliner Platz 43', 'Munchen',NULL,'80805', 'Germany', '089-0877310', '089-0877451');
INSERT INTO "Customers" VALUES('FRANR', 'France restauration', 'Carine Schmitt', 'Marketing Manager', '54,rue Royale', 'Nantes',NULL,'44000', 'France', '40.32.21.21', '40.32.21.20');
INSERT INTO "Customers" VALUES('FRANS', 'Franchi S.p.A.', 'Paolo Accorti', 'Sales Representative', 'Via Monte Bianco 34', 'Torino',NULL,'10100', 'Italy', '011-4988260', '011-4988261');
INSERT INTO "Customers" VALUES('FURIB', 'Furia Bacalhau e Frutos do Mar', 'Lino Rodriguez', 'Sales Manager', 'Jardim das rosas n. 32', 'Lisboa',NULL,'1675', 'Portugal', '(1) 354-2534', '(1) 354-2535');
INSERT INTO "Customers" VALUES('GALED', 'Galeria del gastronomo', 'Eduardo Saavedra', 'Marketing Manager', 'Rambla de Cataluna,23', 'Barcelona',NULL,'08022', 'Spain', '(93) 203 4560', '(93) 203 4561');
INSERT INTO "Customers" VALUES('GODOS', 'Godos Cocina Tipica', 'Jose Pedro Freyre', 'Sales Manager', 'C/ Romero,33', 'Sevilla',NULL,'41101', 'Spain', '(95) 555 82 82',NULL);
INSERT INTO "Customers" VALUES('GOURL', 'Gourmet Lanchonetes', 'Andre Fonseca', 'Sales Associate', 'Av. Brasil,442', 'Campinas', 'SP', '04876-786', 'Brazil', '(11) 555-9482',NULL);
INSERT INTO "Customers" VALUES('GREAL', 'Great Lakes Food Market', 'Howard Snyder', 'Marketing Manager', '2732 Baker Blvd.', 'Eugene', 'OR', '97403', 'USA', '(503) 555-7555',NULL);
INSERT INTO "Customers" VALUES('GROSR', 'GROSELLA-Restaurante', 'Manuel Pereira', 'Owner', '5? Ave. Los Palos Grandes', 'Caracas', 'DF', '1081', 'Venezuela', '(2) 283-2951', '(2) 283-3397');
INSERT INTO "Customers" VALUES('HANAR', 'Hanari Carnes', 'Mario Pontes', 'Accounting Manager', 'Rua do Paco,67', 'Rio de Janeiro', 'RJ', '05454-876', 'Brazil', '(21) 555-0091', '(21) 555-8765');
INSERT INTO "Customers" VALUES('HILAA', 'HILARION-Abastos', 'Carlos Hernandez', 'Sales Representative', 'Carrera 22 con Ave. Carlos Soublette #8-35', 'San Cristobal', 'Tachira', '5022', 'Venezuela', '(5) 555-1340', '(5) 555-1948');
INSERT INTO "Customers" VALUES('HUNGC', 'Hungry Coyote Import Store', 'Yoshi Latimer', 'Sales Representative', 'City Center Plaza 516 Main St.', 'Elgin', 'OR', '97827', 'USA', '(503) 555-6874', '(503) 555-2376');
INSERT INTO "Customers" VALUES('HUNGO', 'Hungry Owl All-Night Grocers', 'Patricia McKenna', 'Sales Associate', '8 Johnstown Road', 'Cork', 'Co. Cork',NULL,'Ireland', '2967 542', '2967 3333');
INSERT INTO "Customers" VALUES('ISLAT', 'Island Trading', 'Helen Bennett', 'Marketing Manager', 'Garden House Crowther Way', 'Cowes', 'Isle of Wight', 'PO31 7PJ', 'UK', '(198) 555-8888',NULL);
INSERT INTO "Customers" VALUES('KOENE', 'Koniglich Essen', 'Philip Cramer', 'Sales Associate', 'Maubelstr. 90', 'Brandenburg',NULL,'14776', 'Germany', '0555-09876',NULL);
INSERT INTO "Customers" VALUES('LACOR', 'La corne d''abondance', 'Daniel Tonini', 'Sales Representative', '67,avenue de l''Europe', 'Versailles',NULL,'78000', 'France', '30.59.84.10', '30.59.85.11');
INSERT INTO "Customers" VALUES('LAMAI', 'La maison d''Asie', 'Annette Roulet', 'Sales Manager', '1 rue Alsace-Lorraine', 'Toulouse',NULL,'31000', 'France', '61.77.61.10', '61.77.61.11');
INSERT INTO "Customers" VALUES('LAUGB', 'Laughing Bacchus Wine Cellars', 'Yoshi Tannamuri', 'Marketing Assistant', '1900 Oak St.', 'Vancouver', 'BC', 'V3F 2K1', 'Canada', '(604) 555-3392', '(604) 555-7293');
INSERT INTO "Customers" VALUES('LAZYK', 'Lazy K Kountry Store', 'John Steel', 'Marketing Manager', '12 Orchestra Terrace', 'Walla Walla', 'WA', '99362', 'USA', '(509) 555-7969', '(509) 555-6221');
INSERT INTO "Customers" VALUES('LEHMS', 'Lehmanns Marktstand', 'Renate Messner', 'Sales Representative', 'Magazinweg 7', 'Frankfurt a.M.',NULL,'60528', 'Germany', '069-0245984', '069-0245874');
INSERT INTO "Customers" VALUES('LETSS', 'Let''s Stop N Shop', 'Jaime Yorres', 'Owner', '87 Polk St. Suite 5', 'San Francisco', 'CA', '94117', 'USA', '(415) 555-5938',NULL);
INSERT INTO "Customers" VALUES('LILAS', 'LILA-Supermercado', 'Carlos Gonzalez', 'Accounting Manager', 'Carrera 52 con Ave. Bolivar #65-98 Llano Largo', 'Barquisimeto', 'Lara', '3508', 'Venezuela', '(9) 331-6954', '(9) 331-7256');
INSERT INTO "Customers" VALUES('LINOD', 'LINO-Delicateses', 'Felipe Izquierdo', 'Owner', 'Ave. 5 de Mayo Porlamar', 'I. de Margarita', 'Nueva Esparta', '4980', 'Venezuela', '(8) 34-56-12', '(8) 34-93-93');
INSERT INTO "Customers" VALUES('LONEP', 'Lonesome Pine Restaurant', 'Fran Wilson', 'Sales Manager', '89 Chiaroscuro Rd.', 'Portland', 'OR', '97219', 'USA', '(503) 555-9573', '(503) 555-9646');
INSERT INTO "Customers" VALUES('MAGAA', 'Magazzini Alimentari Riuniti', 'Giovanni Rovelli', 'Marketing Manager', 'Via Ludovico il Moro 22', 'Bergamo',NULL,'24100', 'Italy', '035-640230', '035-640231');
INSERT INTO "Customers" VALUES('MAISD', 'Maison Dewey', 'Catherine Dewey', 'Sales Agent', 'Rue Joseph-Bens 532', 'Bruxelles',NULL,'B-1180', 'Belgium', '(02) 201 24 67', '(02) 201 24 68');
INSERT INTO "Customers" VALUES('MEREP', 'Mere Paillarde', 'Jean Fresniere', 'Marketing Assistant', '43 rue St. Laurent', 'Montreal', 'Quebec', 'H1J 1C3', 'Canada', '(514) 555-8054', '(514) 555-8055');
INSERT INTO "Customers" VALUES('MORGK', 'Morgenstern Gesundkost', 'Alexander Feuer', 'Marketing Assistant', 'Heerstr. 22', 'Leipzig',NULL,'04179', 'Germany', '0342-023176',NULL);
INSERT INTO "Customers" VALUES('NORTS', 'North/South', 'Simon Crowther', 'Sales Associate', 'South House 300 Queensbridge', 'London',NULL,'SW7 1RZ', 'UK', '(171) 555-7733', '(171) 555-2530');
INSERT INTO "Customers" VALUES('OCEAN', 'Oceano Atlantico Ltda.', 'Yvonne Moncada', 'Sales Agent', 'Ing. Gustavo Moncada 8585 Piso 20-A', 'Buenos Aires',NULL,'1010', 'Argentina', '(1) 135-5333', '(1) 135-5535');
INSERT INTO "Customers" VALUES('OLDWO', 'Old World Delicatessen', 'Rene Phillips', 'Sales Representative', '2743 Bering St.', 'Anchorage', 'AK', '99508', 'USA', '(907) 555-7584', '(907) 555-2880');
INSERT INTO "Customers" VALUES('OTTIK', 'Ottilies Kaseladen', 'Henriette Pfalzheim', 'Owner', 'Mehrheimerstr. 369', 'Koln',NULL,'50739', 'Germany', '0221-0644327', '0221-0765721');
INSERT INTO "Customers" VALUES('PARIS', 'Paris specialites', 'Marie Bertrand', 'Owner', '265,boulevard Charonne', 'Paris',NULL,'75012', 'France', '(1) 42.34.22.66', '(1) 42.34.22.77');
INSERT INTO "Customers" VALUES('PERIC', 'Pericles Comidas clasicas', 'Guillermo Fernandez', 'Sales Representative', 'Calle Dr. Jorge Cash 321', 'Mexico D.F.',NULL,'05033', 'Mexico', '(5) 552-3745', '(5) 545-3745');
INSERT INTO "Customers" VALUES('PICCO', 'Piccolo und mehr', 'Georg Pipps', 'Sales Manager', 'Geislweg 14', 'Salzburg',NULL,'5020', 'Austria', '6562-9722', '6562-9723');
INSERT INTO "Customers" VALUES('PRINI', 'Princesa Isabel Vinhos', 'Isabel de Castro', 'Sales Representative', 'Estrada da saude n. 58', 'Lisboa',NULL,'1756', 'Portugal', '(1) 356-5634',NULL);
INSERT INTO "Customers" VALUES('QUEDE', 'Que Delicia', 'Bernardo Batista', 'Accounting Manager', 'Rua da Panificadora,12', 'Rio de Janeiro', 'RJ', '02389-673', 'Brazil', '(21) 555-4252', '(21) 555-4545');
INSERT INTO "Customers" VALUES('QUEEN', 'Queen Cozinha', 'Lucia Carvalho', 'Marketing Assistant', 'Alameda dos Canarios,891', 'Sao Paulo', 'SP', '05487-020', 'Brazil', '(11) 555-1189',NULL);
INSERT INTO "Customers" VALUES('QUICK', 'QUICK-Stop', 'Horst Kloss', 'Accounting Manager', 'Taucherstra?e 10', 'Cunewalde',NULL,'01307', 'Germany', '0372-035188',NULL);
INSERT INTO "Customers" VALUES('RANCH', 'Rancho grande', 'Sergio Gutierrez', 'Sales Representative', 'Av. del Libertador 900', 'Buenos Aires',NULL,'1010', 'Argentina', '(1) 123-5555', '(1) 123-5556');
INSERT INTO "Customers" VALUES('RATTC', 'Rattlesnake Canyon Grocery', 'Paula Wilson', 'Assistant Sales Representative', '2817 Milton Dr.', 'Albuquerque', 'NM', '87110', 'USA', '(505) 555-5939', '(505) 555-3620');
INSERT INTO "Customers" VALUES('REGGC', 'Reggiani Caseifici', 'Maurizio Moroni', 'Sales Associate', 'Strada Provinciale 124', 'Reggio Emilia',NULL,'42100', 'Italy', '0522-556721', '0522-556722');
INSERT INTO "Customers" VALUES('RICAR', 'Ricardo Adocicados', 'Janete Limeira', 'Assistant Sales Agent', 'Av. Copacabana,267', 'Rio de Janeiro', 'RJ', '02389-890', 'Brazil', '(21) 555-3412',NULL);
INSERT INTO "Customers" VALUES('RICSU', 'Richter Supermarkt', 'Michael Holz', 'Sales Manager', 'Grenzacherweg 237', 'Geneve',NULL,'1203', 'Switzerland', '0897-034214',NULL);
INSERT INTO "Customers" VALUES('ROMEY', 'Romero y tomillo', 'Alejandra Camino', 'Accounting Manager', 'Gran Via,1', 'Madrid',NULL,'28001', 'Spain', '(91) 745 6200', '(91) 745 6210');
INSERT INTO "Customers" VALUES('SANTG', 'Sante Gourmet', 'Jonas Bergulfsen', 'Owner', 'Erling Skakkes gate 78', 'Stavern',NULL,'4110', 'Norway', '07-98 92 35', '07-98 92 47');
INSERT INTO "Customers" VALUES('SAVEA', 'Save-a-lot Markets', 'Jose Pavarotti', 'Sales Representative', '187 Suffolk Ln.', 'Boise', 'ID', '83720', 'USA', '(208) 555-8097',NULL);
INSERT INTO "Customers" VALUES('SEVES', 'Seven Seas Imports', 'Hari Kumar', 'Sales Manager', '90 Wadhurst Rd.', 'London',NULL,'OX15 4NB', 'UK', '(171) 555-1717', '(171) 555-5646');
INSERT INTO "Customers" VALUES('SIMOB', 'Simons bistro', 'Jytte Petersen', 'Owner', 'Vinb?ltet 34', 'Kobenhavn',NULL,'1734', 'Denmark', '31 12 34 56', '31 13 35 57');
INSERT INTO "Customers" VALUES('SPECD', 'Specialites du monde', 'Dominique Perrier', 'Marketing Manager', '25,rue Lauriston', 'Paris',NULL,'75016', 'France', '(1) 47.55.60.10', '(1) 47.55.60.20');
INSERT INTO "Customers" VALUES('SPLIR', 'Split Rail Beer & Ale', 'Art Braunschweiger', 'Sales Manager', 'P.O. Box 555', 'Lander', 'WY', '82520', 'USA', '(307) 555-4680', '(307) 555-6525');
INSERT INTO "Customers" VALUES('SUPRD', 'Supremes delices', 'Pascale Cartrain', 'Accounting Manager', 'Boulevard Tirou,255', 'Charleroi',NULL,'B-6000', 'Belgium', '(071) 23 67 22 20', '(071) 23 67 22 21');
INSERT INTO "Customers" VALUES('THEBI', 'The Big Cheese', 'Liz Nixon', 'Marketing Manager', '89 Jefferson Way Suite 2', 'Portland', 'OR', '97201', 'USA', '(503) 555-3612',NULL);
INSERT INTO "Customers" VALUES('THECR', 'The Cracker Box', 'Liu Wong', 'Marketing Assistant', '55 Grizzly Peak Rd.', 'Butte', 'MT', '59801', 'USA', '(406) 555-5834', '(406) 555-8083');
INSERT INTO "Customers" VALUES('TOMSP', 'Toms Spezialitaten', 'Karin Josephs', 'Marketing Manager', 'Luisenstr. 48', 'Munster',NULL,'44087', 'Germany', '0251-031259', '0251-035695');
INSERT INTO "Customers" VALUES('TORTU', 'Tortuga Restaurante', 'Miguel Angel Paolino', 'Owner', 'Avda. Azteca 123', 'Mexico D.F.',NULL,'05033', 'Mexico', '(5) 555-2933',NULL);
INSERT INTO "Customers" VALUES('TRADH', 'Tradicao Hipermercados', 'Anabela Domingues', 'Sales Representative', 'Av. Ines de Castro,414', 'Sao Paulo', 'SP', '05634-030', 'Brazil', '(11) 555-2167', '(11) 555-2168');
INSERT INTO "Customers" VALUES('TRAIH', 'Trail''s Head Gourmet Provisioners', 'Helvetius Nagy', 'Sales Associate', '722 DaVinci Blvd.', 'Kirkland', 'WA', '98034', 'USA', '(206) 555-8257', '(206) 555-2174');
INSERT INTO "Customers" VALUES('VAFFE', 'Vaffeljernet', 'Palle Ibsen', 'Sales Manager', 'Smagsloget 45', 'Arhus',NULL,'8200', 'Denmark', '86 21 32 43', '86 22 33 44');
INSERT INTO "Customers" VALUES('VICTE', 'Victuailles en stock', 'Mary Saveley', 'Sales Agent', '2,rue du Commerce', 'Lyon',NULL,'69004', 'France', '78.32.54.86', '78.32.54.87');
INSERT INTO "Customers" VALUES('VINET', 'Vins et alcools Chevalier', 'Paul Henriot', 'Accounting Manager', '59 rue de l''Abbaye', 'Reims',NULL,'51100', 'France', '26.47.15.10', '26.47.15.11');
INSERT INTO "Customers" VALUES('WANDK', 'Die Wandernde Kuh', 'Rita Muller', 'Sales Representative', 'Adenauerallee 900', 'Stuttgart',NULL,'70563', 'Germany', '0711-020361', '0711-035428');
INSERT INTO "Customers" VALUES('WARTH', 'Wartian Herkku', 'Pirkko Koskitalo', 'Accounting Manager', 'Torikatu 38', 'Oulu',NULL,'90110', 'Finland', '981-443655', '981-443655');
INSERT INTO "Customers" VALUES('WELLI', 'Wellington Importadora', 'Paula Parente', 'Sales Manager', 'Rua do Mercado,12', 'Resende', 'SP', '08737-363', 'Brazil', '(14) 555-8122',NULL);
INSERT INTO "Customers" VALUES('WHITC', 'White Clover Markets', 'Karl Jablonski', 'Owner', '305 - 14th Ave. S. Suite 3B', 'Seattle', 'WA', '98128', 'USA', '(206) 555-4112', '(206) 555-4115');
INSERT INTO "Customers" VALUES('WILMK', 'Wilman Kala', 'Matti Karttunen', 'Owner/Marketing Assistant', 'Keskuskatu 45', 'Helsinki',NULL,'21240', 'Finland', '90-224 8858', '90-224 8858');
INSERT INTO "Customers" VALUES('WOLZA', 'Wolski Zajazd', 'Zbyszek Piestrzeniewicz', 'Owner', 'ul. Filtrowa 68', 'Warszawa',NULL,'01-012', 'Poland', '(26) 642-7012', '(26) 642-7012');
INSERT INTO "Employees"("EmployeeID","LastName","FirstName","Title","TitleOfCourtesy","BirthDate","HireDate","Address","City","Region","PostalCode","Country","HomePhone","Extension","Photo","Notes","PhotoPath") VALUES(2,'Fuller', 'Andrew', 'Vice President,Sales', 'Dr.',DATE '1952-02-19',DATE '1992-08-14', '908 W. Capital Way', 'Tacoma', 'WA', '98401', 'USA', '(206) 555-9482', '3457', NULL, 'Andrew received his BTS commercial in 1974 and a Ph.D. in international marketing from the University of Dallas in 1981. He is fluent in French and Italian and reads German. He joined the company as a sales representative,was promoted to sales manager in January 1992 and to vice president of sales in March 1993. Andrew is a member of the Sales Management Roundtable,the Seattle Chamber of Commerce,and the Pacific Rim Importers Association.','http://accweb/emmployees/fuller.bmp');
INSERT INTO "Employees"("EmployeeID","LastName","FirstName","Title","TitleOfCourtesy","BirthDate","HireDate","Address","City","Region","PostalCode","Country","HomePhone","Extension","Photo","Notes","PhotoPath") VALUES(1,'Davolio', 'Nancy', 'Sales Representative', 'Ms.',DATE '1948-12-08',DATE '1992-05-01', '507 - 20th Ave. E.Apt. 2A', 'Seattle', 'WA', '98122', 'USA', '(206) 555-9857', '5467', NULL, 'Education includes a BA in psychology from Colorado State University in 1970.  She also completed "The Art of the Cold Call."  Nancy is a member of Toastmasters International.','http://accweb/emmployees/davolio.bmp');
INSERT INTO "Employees"("EmployeeID","LastName","FirstName","Title","TitleOfCourtesy","BirthDate","HireDate","Address","City","Region","PostalCode","Country","HomePhone","Extension","Photo","Notes","PhotoPath") VALUES(3,'Leverling', 'Janet', 'Sales Representative', 'Ms.',DATE '1963-08-30',DATE '1992-04-01', '722 Moss Bay Blvd.', 'Kirkland', 'WA', '98033', 'USA', '(206) 555-3412', '3355', NULL, 'Janet has a BS degree in chemistry from Boston College (1984). She has also completed a certificate program in food retailing management. Janet was hired as a sales associate in 1991 and promoted to sales representative in February 1992.','http://accweb/emmployees/leverling.bmp');
INSERT INTO "Employees"("EmployeeID","LastName","FirstName","Title","TitleOfCourtesy","BirthDate","HireDate","Address","City","Region","PostalCode","Country","HomePhone","Extension","Photo","Notes","PhotoPath") VALUES(4,'Peacock', 'Margaret', 'Sales Representative', 'Mrs.',DATE '1937-09-19',DATE '1993-05-03', '4110 Old Redmond Rd.', 'Redmond', 'WA', '98052', 'USA', '(206) 555-8122', '5176', NULL, 'Margaret holds a BA in English literature from Concordia College (1958) and an MA from the American Institute of Culinary Arts (1966). She was assigned to the London office temporarily from July through November 1992.','http://accweb/emmployees/peacock.bmp');
INSERT INTO "Employees"("EmployeeID","LastName","FirstName","Title","TitleOfCourtesy","BirthDate","HireDate","Address","City","Region","PostalCode","Country","HomePhone","Extension","Photo","Notes","PhotoPath") VALUES(5,'Buchanan', 'Steven', 'Sales Manager', 'Mr.',DATE '1955-03-04',DATE '1993-10-17', '14 Garrett Hill', 'London',NULL,'SW1 8JR', 'UK', '(71) 555-4848', '3453', NULL, 'Steven Buchanan graduated from St. Andrews University,Scotland,with a BSC degree in 1976. Upon joining the company as a sales representative in 1992,he spent 6 months in an orientation program at the Seattle office and then returned to his permanent post in London. He was promoted to sales manager in March 1993. Mr. Buchanan has completed the courses "Successful Telemarketing" and "International Sales Management." He is fluent in French.','http://accweb/emmployees/buchanan.bmp');
INSERT INTO "Employees"("EmployeeID","LastName","FirstName","Title","TitleOfCourtesy","BirthDate","HireDate","Address","City","Region","PostalCode","Country","HomePhone","Extension","Photo","Notes","PhotoPath") VALUES(6,'Suyama', 'Michael', 'Sales Representative', 'Mr.',DATE '1963-07-02',DATE '1993-10-17', 'Coventry House Miner Rd.', 'London',NULL,'EC2 7JR', 'UK', '(71) 555-7773', '428', NULL, 'Michael is a graduate of Sussex University (MA, economics, 1983) and the University of California at Los Angeles (MBA, marketing, 1986).  He has also taken the courses "Multi-Cultural Selling" and "Time Management for the Sales Professional."  He is fluent in Japanese and can read and write French, Portuguese, and Spanish.','http://accweb/emmployees/davolio.bmp');
INSERT INTO "Employees"("EmployeeID","LastName","FirstName","Title","TitleOfCourtesy","BirthDate","HireDate","Address","City","Region","PostalCode","Country","HomePhone","Extension","Photo","Notes","PhotoPath") VALUES(8,'Callahan', 'Laura', 'Inside Sales Coordinator', 'Ms.',DATE '1958-01-09',DATE '1994-03-05', '4726 - 11th Ave. N.E.', 'Seattle', 'WA', '98105', 'USA', '(206) 555-1189', '2344', NULL, 'Laura received a BA in psychology from the University of Washington. She has also completed a course in business French. She reads and writes French.','http://accweb/emmployees/davolio.bmp');
INSERT INTO "Employees"("EmployeeID","LastName","FirstName","Title","TitleOfCourtesy","BirthDate","HireDate","Address","City","Region","PostalCode","Country","HomePhone","Extension","Photo","Notes","PhotoPath") VALUES(9,'Dodsworth', 'Anne', 'Sales Representative', 'Ms.',DATE '1966-01-27',DATE '1994-11-15', '7 Houndstooth Rd.', 'London',NULL,'WG2 7LT', 'UK', '(71) 555-4444', '452', NULL, 'Anne has a BA degree in English from St. Lawrence College. She is fluent in French and German.','http://accweb/emmployees/davolio.bmp');

INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10248,N'VINET',5,DATE '1996-07-04',DATE '1996-08-01',DATE '1996-07-16',32.38,
	N'Vins et alcools Chevalier',N'59 rue de l''Abbaye',N'Reims',
	NULL,N'51100',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10249,N'TOMSP',6,DATE '1996-07-05',DATE '1996-08-16',DATE '1996-07-10',11.61,
	N'Toms Spezialitaten',N'Luisenstr. 48',N'Munster',
	NULL,N'44087',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10250,N'HANAR',4,DATE '1996-07-08',DATE '1996-08-05',DATE '1996-07-12',65.83,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10251,N'VICTE',3,DATE '1996-07-08',DATE '1996-08-05',DATE '1996-07-15',41.34,
	N'Victuailles en stock',N'2, rue du Commerce',N'Lyon',
	NULL,N'69004',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10252,N'SUPRD',4,DATE '1996-07-09',DATE '1996-08-06',DATE '1996-07-11',51.30,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10253,N'HANAR',3,DATE '1996-07-10',DATE '1996-07-24',DATE '1996-07-16',58.17,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10254,N'CHOPS',5,DATE '1996-07-11',DATE '1996-08-08',DATE '1996-07-23',22.98,
	N'Chop-suey Chinese',N'Hauptstr. 31',N'Bern',
	NULL,N'3012',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10255,N'RICSU',9,DATE '1996-07-12',DATE '1996-08-09',DATE '1996-07-15',148.33,
	N'Richter Supermarkt',N'Starenweg 5',N'Geneve',
	NULL,N'1204',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10256,N'WELLI',3,DATE '1996-07-15',DATE '1996-08-12',DATE '1996-07-17',13.97,
	N'Wellington Importadora',N'Rua do Mercado, 12',N'Resende',
	N'SP',N'08737-363',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10257,N'HILAA',4,DATE '1996-07-16',DATE '1996-08-13',DATE '1996-07-22',81.91,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10258,N'ERNSH',1,DATE '1996-07-17',DATE '1996-08-14',DATE '1996-07-23',140.51,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10259,N'CENTC',4,DATE '1996-07-18',DATE '1996-08-15',DATE '1996-07-25',3.25,
	N'Centro comercial Moctezuma',N'Sierras de Granada 9993',N'Mexico D.F.',
	NULL,N'05022',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10260,N'OTTIK',4,DATE '1996-07-19',DATE '1996-08-16',DATE '1996-07-29',55.09,
	N'Ottilies Kaseladen',N'Mehrheimerstr. 369',N'Koln',
	NULL,N'50739',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10261,N'QUEDE',4,DATE '1996-07-19',DATE '1996-08-16',DATE '1996-07-30',3.05,
	N'Que Delicia',N'Rua da Panificadora, 12',N'Rio de Janeiro',
	N'RJ',N'02389-673',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10262,N'RATTC',8,DATE '1996-07-22',DATE '1996-08-19',DATE '1996-07-25',48.29,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10263,N'ERNSH',9,DATE '1996-07-23',DATE '1996-08-20',DATE '1996-07-31',146.06,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10264,N'FOLKO',6,DATE '1996-07-24',DATE '1996-08-21',DATE '1996-08-23',3.67,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10265,N'BLONP',2,DATE '1996-07-25',DATE '1996-08-22',DATE '1996-08-12',55.28,
	N'Blondel pere et fils',N'24, place Kleber',N'Strasbourg',
	NULL,N'67000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10266,N'WARTH',3,DATE '1996-07-26',DATE '1996-09-06',DATE '1996-07-31',25.73,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10267,N'FRANK',4,DATE '1996-07-29',DATE '1996-08-26',DATE '1996-08-06',208.58,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10268,N'GROSR',8,DATE '1996-07-30',DATE '1996-08-27',DATE '1996-08-02',66.29,
	N'GROSELLA-Restaurante',N'5? Ave. Los Palos Grandes',N'Caracas',
	N'DF',N'1081',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10269,N'WHITC',5,DATE '1996-07-31',DATE '1996-08-14',DATE '1996-08-09',4.56,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10270,N'WARTH',1,DATE '1996-08-01',DATE '1996-08-29',DATE '1996-08-02',136.54,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10271,N'SPLIR',6,DATE '1996-08-01',DATE '1996-08-29',DATE '1996-08-30',4.54,
	N'Split Rail Beer & Ale',N'P.O. Box 555',N'Lander',
	N'WY',N'82520',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10272,N'RATTC',6,DATE '1996-08-02',DATE '1996-08-30',DATE '1996-08-06',98.03,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10273,N'QUICK',3,DATE '1996-08-05',DATE '1996-09-02',DATE '1996-08-12',76.07,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10274,N'VINET',6,DATE '1996-08-06',DATE '1996-09-03',DATE '1996-08-16',6.01,
	N'Vins et alcools Chevalier',N'59 rue de l''Abbaye',N'Reims',
	NULL,N'51100',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10275,N'MAGAA',1,DATE '1996-08-07',DATE '1996-09-04',DATE '1996-08-09',26.93,
	N'Magazzini Alimentari Riuniti',N'Via Ludovico il Moro 22',N'Bergamo',
	NULL,N'24100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10276,N'TORTU',8,DATE '1996-08-08',DATE '1996-08-22',DATE '1996-08-14',13.84,
	N'Tortuga Restaurante',N'Avda. Azteca 123',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10277,N'MORGK',2,DATE '1996-08-09',DATE '1996-09-06',DATE '1996-08-13',125.77,
	N'Morgenstern Gesundkost',N'Heerstr. 22',N'Leipzig',
	NULL,N'04179',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10278,N'BERGS',8,DATE '1996-08-12',DATE '1996-09-09',DATE '1996-08-16',92.69,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10279,N'LEHMS',8,DATE '1996-08-13',DATE '1996-09-10',DATE '1996-08-16',25.83,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10280,N'BERGS',2,DATE '1996-08-14',DATE '1996-09-11',DATE '1996-09-12',8.98,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10281,N'ROMEY',4,DATE '1996-08-14',DATE '1996-08-28',DATE '1996-08-21',2.94,
	N'Romero y tomillo',N'Gran Via, 1',N'Madrid',
	NULL,N'28001',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10282,N'ROMEY',4,DATE '1996-08-15',DATE '1996-09-12',DATE '1996-08-21',12.69,
	N'Romero y tomillo',N'Gran Via, 1',N'Madrid',
	NULL,N'28001',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10283,N'LILAS',3,DATE '1996-08-16',DATE '1996-09-13',DATE '1996-08-23',84.81,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10284,N'LEHMS',4,DATE '1996-08-19',DATE '1996-09-16',DATE '1996-08-27',76.56,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10285,N'QUICK',1,DATE '1996-08-20',DATE '1996-09-17',DATE '1996-08-26',76.83,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10286,N'QUICK',8,DATE '1996-08-21',DATE '1996-09-18',DATE '1996-08-30',229.24,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10287,N'RICAR',8,DATE '1996-08-22',DATE '1996-09-19',DATE '1996-08-28',12.76,
	N'Ricardo Adocicados',N'Av. Copacabana, 267',N'Rio de Janeiro',
	N'RJ',N'02389-890',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10288,N'REGGC',4,DATE '1996-08-23',DATE '1996-09-20',DATE '1996-09-03',7.45,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10289,N'BSBEV',7,DATE '1996-08-26',DATE '1996-09-23',DATE '1996-08-28',22.77,
	N'B''s Beverages',N'Fauntleroy Circus',N'London',
	NULL,N'EC2 5NT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10290,N'COMMI',8,DATE '1996-08-27',DATE '1996-09-24',DATE '1996-09-03',79.70,
	N'Comercio Mineiro',N'Av. dos Lusiadas, 23',N'Sao Paulo',
	N'SP',N'05432-043',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10291,N'QUEDE',6,DATE '1996-08-27',DATE '1996-09-24',DATE '1996-09-04',6.40,
	N'Que Delicia',N'Rua da Panificadora, 12',N'Rio de Janeiro',
	N'RJ',N'02389-673',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10292,N'TRADH',1,DATE '1996-08-28',DATE '1996-09-25',DATE '1996-09-02',1.35,
	N'Tradicao Hipermercados',N'Av. Ines de Castro, 414',N'Sao Paulo',
	N'SP',N'05634-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10293,N'TORTU',1,DATE '1996-08-29',DATE '1996-09-26',DATE '1996-09-11',21.18,
	N'Tortuga Restaurante',N'Avda. Azteca 123',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10294,N'RATTC',4,DATE '1996-08-30',DATE '1996-09-27',DATE '1996-09-05',147.26,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10295,N'VINET',2,DATE '1996-09-02',DATE '1996-09-30',DATE '1996-09-10',1.15,
	N'Vins et alcools Chevalier',N'59 rue de l''Abbaye',N'Reims',
	NULL,N'51100',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10296,N'LILAS',6,DATE '1996-09-03',DATE '1996-10-01',DATE '1996-09-11',0.12,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10297,N'BLONP',5,DATE '1996-09-04',DATE '1996-10-16',DATE '1996-09-10',5.74,
	N'Blondel pere et fils',N'24, place Kleber',N'Strasbourg',
	NULL,N'67000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10298,N'HUNGO',6,DATE '1996-09-05',DATE '1996-10-03',DATE '1996-09-11',168.22,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10299,N'RICAR',4,DATE '1996-09-06',DATE '1996-10-04',DATE '1996-09-13',29.76,
	N'Ricardo Adocicados',N'Av. Copacabana, 267',N'Rio de Janeiro',
	N'RJ',N'02389-890',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10300,N'MAGAA',2,DATE '1996-09-09',DATE '1996-10-07',DATE '1996-09-18',17.68,
	N'Magazzini Alimentari Riuniti',N'Via Ludovico il Moro 22',N'Bergamo',
	NULL,N'24100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10301,N'WANDK',8,DATE '1996-09-09',DATE '1996-10-07',DATE '1996-09-17',45.08,
	N'Die Wandernde Kuh',N'Adenauerallee 900',N'Stuttgart',
	NULL,N'70563',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10302,N'SUPRD',4,DATE '1996-09-10',DATE '1996-10-08',DATE '1996-10-09',6.27,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10303,N'GODOS',7,DATE '1996-09-11',DATE '1996-10-09',DATE '1996-09-18',107.83,
	N'Godos Cocina Tipica',N'C/ Romero, 33',N'Sevilla',
	NULL,N'41101',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10304,N'TORTU',1,DATE '1996-09-12',DATE '1996-10-10',DATE '1996-09-17',63.79,
	N'Tortuga Restaurante',N'Avda. Azteca 123',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10305,N'OLDWO',8,DATE '1996-09-13',DATE '1996-10-11',DATE '1996-10-09',257.62,
	N'Old World Delicatessen',N'2743 Bering St.',N'Anchorage',
	N'AK',N'99508',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10306,N'ROMEY',1,DATE '1996-09-16',DATE '1996-10-14',DATE '1996-09-23',7.56,
	N'Romero y tomillo',N'Gran Via, 1',N'Madrid',
	NULL,N'28001',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10307,N'LONEP',2,DATE '1996-09-17',DATE '1996-10-15',DATE '1996-09-25',0.56,
	N'Lonesome Pine Restaurant',N'89 Chiaroscuro Rd.',N'Portland',
	N'OR',N'97219',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10308,N'ANATR',7,DATE '1996-09-18',DATE '1996-10-16',DATE '1996-09-24',1.61,
	N'Ana Trujillo Emparedados y helados',N'Avda. de la Constitucion 2222',N'Mexico D.F.',
	NULL,N'05021',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10309,N'HUNGO',3,DATE '1996-09-19',DATE '1996-10-17',DATE '1996-10-23',47.30,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10310,N'THEBI',8,DATE '1996-09-20',DATE '1996-10-18',DATE '1996-09-27',17.52,
	N'The Big Cheese',N'89 Jefferson Way Suite 2',N'Portland',
	N'OR',N'97201',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10311,N'DUMON',1,DATE '1996-09-20',DATE '1996-10-04',DATE '1996-09-26',24.69,
	N'Du monde entier',N'67, rue des Cinquante Otages',N'Nantes',
	NULL,N'44000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10312,N'WANDK',2,DATE '1996-09-23',DATE '1996-10-21',DATE '1996-10-03',40.26,
	N'Die Wandernde Kuh',N'Adenauerallee 900',N'Stuttgart',
	NULL,N'70563',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10313,N'QUICK',2,DATE '1996-09-24',DATE '1996-10-22',DATE '1996-10-04',1.96,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10314,N'RATTC',1,DATE '1996-09-25',DATE '1996-10-23',DATE '1996-10-04',74.16,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10315,N'ISLAT',4,DATE '1996-09-26',DATE '1996-10-24',DATE '1996-10-03',41.76,
	N'Island Trading',N'Garden House Crowther Way',N'Cowes',
	N'Isle of Wight',N'PO31 7PJ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10316,N'RATTC',1,DATE '1996-09-27',DATE '1996-10-25',DATE '1996-10-08',150.15,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10317,N'LONEP',6,DATE '1996-09-30',DATE '1996-10-28',DATE '1996-10-10',12.69,
	N'Lonesome Pine Restaurant',N'89 Chiaroscuro Rd.',N'Portland',
	N'OR',N'97219',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10318,N'ISLAT',8,DATE '1996-10-01',DATE '1996-10-29',DATE '1996-10-04',4.73,
	N'Island Trading',N'Garden House Crowther Way',N'Cowes',
	N'Isle of Wight',N'PO31 7PJ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10319,N'TORTU',7,DATE '1996-10-02',DATE '1996-10-30',DATE '1996-10-11',64.50,
	N'Tortuga Restaurante',N'Avda. Azteca 123',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10320,N'WARTH',5,DATE '1996-10-03',DATE '1996-10-17',DATE '1996-10-18',34.57,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10321,N'ISLAT',3,DATE '1996-10-03',DATE '1996-10-31',DATE '1996-10-11',3.43,
	N'Island Trading',N'Garden House Crowther Way',N'Cowes',
	N'Isle of Wight',N'PO31 7PJ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10322,N'PERIC',7,DATE '1996-10-04',DATE '1996-11-01',DATE '1996-10-23',0.40,
	N'Pericles Comidas clasicas',N'Calle Dr. Jorge Cash 321',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10323,N'KOENE',4,DATE '1996-10-07',DATE '1996-11-04',DATE '1996-10-14',4.88,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10324,N'SAVEA',9,DATE '1996-10-08',DATE '1996-11-05',DATE '1996-10-10',214.27,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10325,N'KOENE',1,DATE '1996-10-09',DATE '1996-10-23',DATE '1996-10-14',64.86,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10326,N'BOLID',4,DATE '1996-10-10',DATE '1996-11-07',DATE '1996-10-14',77.92,
	N'Bolido Comidas preparadas',N'C/ Araquil, 67',N'Madrid',
	NULL,N'28023',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10327,N'FOLKO',2,DATE '1996-10-11',DATE '1996-11-08',DATE '1996-10-14',63.36,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10328,N'FURIB',4,DATE '1996-10-14',DATE '1996-11-11',DATE '1996-10-17',87.03,
	N'Furia Bacalhau e Frutos do Mar',N'Jardim das rosas n. 32',N'Lisboa',
	NULL,N'1675',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10329,N'SPLIR',4,DATE '1996-10-15',DATE '1996-11-26',DATE '1996-10-23',191.67,
	N'Split Rail Beer & Ale',N'P.O. Box 555',N'Lander',
	N'WY',N'82520',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10330,N'LILAS',3,DATE '1996-10-16',DATE '1996-11-13',DATE '1996-10-28',12.75,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10331,N'BONAP',9,DATE '1996-10-16',DATE '1996-11-27',DATE '1996-10-21',10.19,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10332,N'MEREP',3,DATE '1996-10-17',DATE '1996-11-28',DATE '1996-10-21',52.84,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10333,N'WARTH',5,DATE '1996-10-18',DATE '1996-11-15',DATE '1996-10-25',0.59,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10334,N'VICTE',8,DATE '1996-10-21',DATE '1996-11-18',DATE '1996-10-28',8.56,
	N'Victuailles en stock',N'2, rue du Commerce',N'Lyon',
	NULL,N'69004',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10335,N'HUNGO',7,DATE '1996-10-22',DATE '1996-11-19',DATE '1996-10-24',42.11,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10336,N'PRINI',7,DATE '1996-10-23',DATE '1996-11-20',DATE '1996-10-25',15.51,
	N'Princesa Isabel Vinhos',N'Estrada da saude n. 58',N'Lisboa',
	NULL,N'1756',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10337,N'FRANK',4,DATE '1996-10-24',DATE '1996-11-21',DATE '1996-10-29',108.26,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10338,N'OLDWO',4,DATE '1996-10-25',DATE '1996-11-22',DATE '1996-10-29',84.21,
	N'Old World Delicatessen',N'2743 Bering St.',N'Anchorage',
	N'AK',N'99508',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10339,N'MEREP',2,DATE '1996-10-28',DATE '1996-11-25',DATE '1996-11-04',15.66,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10340,N'BONAP',1,DATE '1996-10-29',DATE '1996-11-26',DATE '1996-11-08',166.31,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10341,N'SIMOB',7,DATE '1996-10-29',DATE '1996-11-26',DATE '1996-11-05',26.78,
	N'Simons bistro',N'Vinb?ltet 34',N'Kobenhavn',
	NULL,N'1734',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10342,N'FRANK',4,DATE '1996-10-30',DATE '1996-11-13',DATE '1996-11-04',54.83,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10343,N'LEHMS',4,DATE '1996-10-31',DATE '1996-11-28',DATE '1996-11-06',110.37,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10344,N'WHITC',4,DATE '1996-11-01',DATE '1996-11-29',DATE '1996-11-05',23.29,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10345,N'QUICK',2,DATE '1996-11-04',DATE '1996-12-02',DATE '1996-11-11',249.06,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10346,N'RATTC',3,DATE '1996-11-05',DATE '1996-12-17',DATE '1996-11-08',142.08,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10347,N'FAMIA',4,DATE '1996-11-06',DATE '1996-12-04',DATE '1996-11-08',3.10,
	N'Familia Arquibaldo',N'Rua Oros, 92',N'Sao Paulo',
	N'SP',N'05442-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10348,N'WANDK',4,DATE '1996-11-07',DATE '1996-12-05',DATE '1996-11-15',0.78,
	N'Die Wandernde Kuh',N'Adenauerallee 900',N'Stuttgart',
	NULL,N'70563',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10349,N'SPLIR',7,DATE '1996-11-08',DATE '1996-12-06',DATE '1996-11-15',8.63,
	N'Split Rail Beer & Ale',N'P.O. Box 555',N'Lander',
	N'WY',N'82520',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10350,N'LAMAI',6,DATE '1996-11-11',DATE '1996-12-09',DATE '1996-12-03',64.19,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10351,N'ERNSH',1,DATE '1996-11-11',DATE '1996-12-09',DATE '1996-11-20',162.33,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10352,N'FURIB',3,DATE '1996-11-12',DATE '1996-11-26',DATE '1996-11-18',1.30,
	N'Furia Bacalhau e Frutos do Mar',N'Jardim das rosas n. 32',N'Lisboa',
	NULL,N'1675',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10353,N'PICCO',7,DATE '1996-11-13',DATE '1996-12-11',DATE '1996-11-25',360.63,
	N'Piccolo und mehr',N'Geislweg 14',N'Salzburg',
	NULL,N'5020',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10354,N'PERIC',8,DATE '1996-11-14',DATE '1996-12-12',DATE '1996-11-20',53.80,
	N'Pericles Comidas clasicas',N'Calle Dr. Jorge Cash 321',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10355,N'AROUT',6,DATE '1996-11-15',DATE '1996-12-13',DATE '1996-11-20',41.95,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10356,N'WANDK',6,DATE '1996-11-18',DATE '1996-12-16',DATE '1996-11-27',36.71,
	N'Die Wandernde Kuh',N'Adenauerallee 900',N'Stuttgart',
	NULL,N'70563',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10357,N'LILAS',1,DATE '1996-11-19',DATE '1996-12-17',DATE '1996-12-02',34.88,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10358,N'LAMAI',5,DATE '1996-11-20',DATE '1996-12-18',DATE '1996-11-27',19.64,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10359,N'SEVES',5,DATE '1996-11-21',DATE '1996-12-19',DATE '1996-11-26',288.43,
	N'Seven Seas Imports',N'90 Wadhurst Rd.',N'London',
	NULL,N'OX15 4NB',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10360,N'BLONP',4,DATE '1996-11-22',DATE '1996-12-20',DATE '1996-12-02',131.70,
	N'Blondel pere et fils',N'24, place Kleber',N'Strasbourg',
	NULL,N'67000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10361,N'QUICK',1,DATE '1996-11-22',DATE '1996-12-20',DATE '1996-12-03',183.17,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10362,N'BONAP',3,DATE '1996-11-25',DATE '1996-12-23',DATE '1996-11-28',96.04,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10363,N'DRACD',4,DATE '1996-11-26',DATE '1996-12-24',DATE '1996-12-04',30.54,
	N'Drachenblut Delikatessen',N'Walserweg 21',N'Aachen',
	NULL,N'52066',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10364,N'EASTC',1,DATE '1996-11-26',DATE '1997-01-07',DATE '1996-12-04',71.97,
	N'Eastern Connection',N'35 King George',N'London',
	NULL,N'WX3 6FW',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10365,N'ANTON',3,DATE '1996-11-27',DATE '1996-12-25',DATE '1996-12-02',22.00,
	N'Antonio Moreno Taqueria',N'Mataderos  2312',N'Mexico D.F.',
	NULL,N'05023',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10366,N'GALED',8,DATE '1996-11-28',DATE '1997-01-09',DATE '1996-12-30',10.14,
	N'Galeria del gastronomo',N'Rambla de Cataluna, 23',N'Barcelona',
	NULL,N'8022',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10367,N'VAFFE',7,DATE '1996-11-28',DATE '1996-12-26',DATE '1996-12-02',13.55,
	N'Vaffeljernet',N'Smagsloget 45',N'Arhus',
	NULL,N'8200',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10368,N'ERNSH',2,DATE '1996-11-29',DATE '1996-12-27',DATE '1996-12-02',101.95,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10369,N'SPLIR',8,DATE '1996-12-02',DATE '1996-12-30',DATE '1996-12-09',195.68,
	N'Split Rail Beer & Ale',N'P.O. Box 555',N'Lander',
	N'WY',N'82520',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10370,N'CHOPS',6,DATE '1996-12-03',DATE '1996-12-31',DATE '1996-12-27',1.17,
	N'Chop-suey Chinese',N'Hauptstr. 31',N'Bern',
	NULL,N'3012',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10371,N'LAMAI',1,DATE '1996-12-03',DATE '1996-12-31',DATE '1996-12-24',0.45,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10372,N'QUEEN',5,DATE '1996-12-04',DATE '1997-01-01',DATE '1996-12-09',890.78,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10373,N'HUNGO',4,DATE '1996-12-05',DATE '1997-01-02',DATE '1996-12-11',124.12,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10374,N'WOLZA',1,DATE '1996-12-05',DATE '1997-01-02',DATE '1996-12-09',3.94,
	N'Wolski Zajazd',N'ul. Filtrowa 68',N'Warszawa',
	NULL,N'01-012',N'Poland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10375,N'HUNGC',3,DATE '1996-12-06',DATE '1997-01-03',DATE '1996-12-09',20.12,
	N'Hungry Coyote Import Store',N'City Center Plaza 516 Main St.',N'Elgin',
	N'OR',N'97827',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10376,N'MEREP',1,DATE '1996-12-09',DATE '1997-01-06',DATE '1996-12-13',20.39,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10377,N'SEVES',1,DATE '1996-12-09',DATE '1997-01-06',DATE '1996-12-13',22.21,
	N'Seven Seas Imports',N'90 Wadhurst Rd.',N'London',
	NULL,N'OX15 4NB',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10378,N'FOLKO',5,DATE '1996-12-10',DATE '1997-01-07',DATE '1996-12-19',5.44,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10379,N'QUEDE',2,DATE '1996-12-11',DATE '1997-01-08',DATE '1996-12-13',45.03,
	N'Que Delicia',N'Rua da Panificadora, 12',N'Rio de Janeiro',
	N'RJ',N'02389-673',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10380,N'HUNGO',8,DATE '1996-12-12',DATE '1997-01-09',DATE '1997-01-16',35.03,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10381,N'LILAS',3,DATE '1996-12-12',DATE '1997-01-09',DATE '1996-12-13',7.99,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10382,N'ERNSH',4,DATE '1996-12-13',DATE '1997-01-10',DATE '1996-12-16',94.77,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10383,N'AROUT',8,DATE '1996-12-16',DATE '1997-01-13',DATE '1996-12-18',34.24,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10384,N'BERGS',3,DATE '1996-12-16',DATE '1997-01-13',DATE '1996-12-20',168.64,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10385,N'SPLIR',1,DATE '1996-12-17',DATE '1997-01-14',DATE '1996-12-23',30.96,
	N'Split Rail Beer & Ale',N'P.O. Box 555',N'Lander',
	N'WY',N'82520',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10386,N'FAMIA',9,DATE '1996-12-18',DATE '1997-01-01',DATE '1996-12-25',13.99,
	N'Familia Arquibaldo',N'Rua Oros, 92',N'Sao Paulo',
	N'SP',N'05442-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10387,N'SANTG',1,DATE '1996-12-18',DATE '1997-01-15',DATE '1996-12-20',93.63,
	N'Sante Gourmet',N'Erling Skakkes gate 78',N'Stavern',
	NULL,N'4110',N'Norway');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10388,N'SEVES',2,DATE '1996-12-19',DATE '1997-01-16',DATE '1996-12-20',34.86,
	N'Seven Seas Imports',N'90 Wadhurst Rd.',N'London',
	NULL,N'OX15 4NB',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10389,N'BOTTM',4,DATE '1996-12-20',DATE '1997-01-17',DATE '1996-12-24',47.42,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10390,N'ERNSH',6,DATE '1996-12-23',DATE '1997-01-20',DATE '1996-12-26',126.38,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10391,N'DRACD',3,DATE '1996-12-23',DATE '1997-01-20',DATE '1996-12-31',5.45,
	N'Drachenblut Delikatessen',N'Walserweg 21',N'Aachen',
	NULL,N'52066',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10392,N'PICCO',2,DATE '1996-12-24',DATE '1997-01-21',DATE '1997-01-01',122.46,
	N'Piccolo und mehr',N'Geislweg 14',N'Salzburg',
	NULL,N'5020',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10393,N'SAVEA',1,DATE '1996-12-25',DATE '1997-01-22',DATE '1997-01-03',126.56,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10394,N'HUNGC',1,DATE '1996-12-25',DATE '1997-01-22',DATE '1997-01-03',30.34,
	N'Hungry Coyote Import Store',N'City Center Plaza 516 Main St.',N'Elgin',
	N'OR',N'97827',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10395,N'HILAA',6,DATE '1996-12-26',DATE '1997-01-23',DATE '1997-01-03',184.41,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10396,N'FRANK',1,DATE '1996-12-27',DATE '1997-01-10',DATE '1997-01-06',135.35,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10397,N'PRINI',5,DATE '1996-12-27',DATE '1997-01-24',DATE '1997-01-02',60.26,
	N'Princesa Isabel Vinhos',N'Estrada da saude n. 58',N'Lisboa',
	NULL,N'1756',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10398,N'SAVEA',2,DATE '1996-12-30',DATE '1997-01-27',DATE '1997-01-09',89.16,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10399,N'VAFFE',8,DATE '1996-12-31',DATE '1997-01-14',DATE '1997-01-08',27.36,
	N'Vaffeljernet',N'Smagsloget 45',N'Arhus',
	NULL,N'8200',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10400,N'EASTC',1,DATE '1997-01-01',DATE '1997-01-29',DATE '1997-01-16',83.93,
	N'Eastern Connection',N'35 King George',N'London',
	NULL,N'WX3 6FW',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10401,N'RATTC',1,DATE '1997-01-01',DATE '1997-01-29',DATE '1997-01-10',12.51,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10402,N'ERNSH',8,DATE '1997-01-02',DATE '1997-02-13',DATE '1997-01-10',67.88,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10403,N'ERNSH',4,DATE '1997-01-03',DATE '1997-01-31',DATE '1997-01-09',73.79,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10404,N'MAGAA',2,DATE '1997-01-03',DATE '1997-01-31',DATE '1997-01-08',155.97,
	N'Magazzini Alimentari Riuniti',N'Via Ludovico il Moro 22',N'Bergamo',
	NULL,N'24100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10405,N'LINOD',1,DATE '1997-01-06',DATE '1997-02-03',DATE '1997-01-22',34.82,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10406,N'QUEEN',7,DATE '1997-01-07',DATE '1997-02-18',DATE '1997-01-13',108.04,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10407,N'OTTIK',2,DATE '1997-01-07',DATE '1997-02-04',DATE '1997-01-30',91.48,
	N'Ottilies Kaseladen',N'Mehrheimerstr. 369',N'Koln',
	NULL,N'50739',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10408,N'FOLIG',8,DATE '1997-01-08',DATE '1997-02-05',DATE '1997-01-14',11.26,
	N'Folies gourmandes',N'184, chaussee de Tournai',N'Lille',
	NULL,N'59000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10409,N'OCEAN',3,DATE '1997-01-09',DATE '1997-02-06',DATE '1997-01-14',29.83,
	N'Oceano Atlantico Ltda.',N'Ing. Gustavo Moncada 8585 Piso 20-A',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10410,N'BOTTM',3,DATE '1997-01-10',DATE '1997-02-07',DATE '1997-01-15',2.40,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10411,N'BOTTM',9,DATE '1997-01-10',DATE '1997-02-07',DATE '1997-01-21',23.65,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10412,N'WARTH',8,DATE '1997-01-13',DATE '1997-02-10',DATE '1997-01-15',3.77,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10413,N'LAMAI',3,DATE '1997-01-14',DATE '1997-02-11',DATE '1997-01-16',95.66,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10414,N'FAMIA',2,DATE '1997-01-14',DATE '1997-02-11',DATE '1997-01-17',21.48,
	N'Familia Arquibaldo',N'Rua Oros, 92',N'Sao Paulo',
	N'SP',N'05442-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10415,N'HUNGC',3,DATE '1997-01-15',DATE '1997-02-12',DATE '1997-01-24',0.20,
	N'Hungry Coyote Import Store',N'City Center Plaza 516 Main St.',N'Elgin',
	N'OR',N'97827',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10416,N'WARTH',8,DATE '1997-01-16',DATE '1997-02-13',DATE '1997-01-27',22.72,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10417,N'SIMOB',4,DATE '1997-01-16',DATE '1997-02-13',DATE '1997-01-28',70.29,
	N'Simons bistro',N'Vinb?ltet 34',N'Kobenhavn',
	NULL,N'1734',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10418,N'QUICK',4,DATE '1997-01-17',DATE '1997-02-14',DATE '1997-01-24',17.55,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10419,N'RICSU',4,DATE '1997-01-20',DATE '1997-02-17',DATE '1997-01-30',137.35,
	N'Richter Supermarkt',N'Starenweg 5',N'Geneve',
	NULL,N'1204',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10420,N'WELLI',3,DATE '1997-01-21',DATE '1997-02-18',DATE '1997-01-27',44.12,
	N'Wellington Importadora',N'Rua do Mercado, 12',N'Resende',
	N'SP',N'08737-363',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10421,N'QUEDE',8,DATE '1997-01-21',DATE '1997-03-04',DATE '1997-01-27',99.23,
	N'Que Delicia',N'Rua da Panificadora, 12',N'Rio de Janeiro',
	N'RJ',N'02389-673',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10422,N'FRANS',2,DATE '1997-01-22',DATE '1997-02-19',DATE '1997-01-31',3.02,
	N'Franchi S.p.A.',N'Via Monte Bianco 34',N'Torino',
	NULL,N'10100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10423,N'GOURL',6,DATE '1997-01-23',DATE '1997-02-06',DATE '1997-02-24',24.50,
	N'Gourmet Lanchonetes',N'Av. Brasil, 442',N'Campinas',
	N'SP',N'04876-786',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10424,N'MEREP',7,DATE '1997-01-23',DATE '1997-02-20',DATE '1997-01-27',370.61,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10425,N'LAMAI',6,DATE '1997-01-24',DATE '1997-02-21',DATE '1997-02-14',7.93,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10426,N'GALED',4,DATE '1997-01-27',DATE '1997-02-24',DATE '1997-02-06',18.69,
	N'Galeria del gastronomo',N'Rambla de Cataluna, 23',N'Barcelona',
	NULL,N'8022',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10427,N'PICCO',4,DATE '1997-01-27',DATE '1997-02-24',DATE '1997-03-03',31.29,
	N'Piccolo und mehr',N'Geislweg 14',N'Salzburg',
	NULL,N'5020',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10428,N'REGGC',7,DATE '1997-01-28',DATE '1997-02-25',DATE '1997-02-04',11.09,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10429,N'HUNGO',3,DATE '1997-01-29',DATE '1997-03-12',DATE '1997-02-07',56.63,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10430,N'ERNSH',4,DATE '1997-01-30',DATE '1997-02-13',DATE '1997-02-03',458.78,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10431,N'BOTTM',4,DATE '1997-01-30',DATE '1997-02-13',DATE '1997-02-07',44.17,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10432,N'SPLIR',3,DATE '1997-01-31',DATE '1997-02-14',DATE '1997-02-07',4.34,
	N'Split Rail Beer & Ale',N'P.O. Box 555',N'Lander',
	N'WY',N'82520',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10433,N'PRINI',3,DATE '1997-02-03',DATE '1997-03-03',DATE '1997-03-04',73.83,
	N'Princesa Isabel Vinhos',N'Estrada da saude n. 58',N'Lisboa',
	NULL,N'1756',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10434,N'FOLKO',3,DATE '1997-02-03',DATE '1997-03-03',DATE '1997-02-13',17.92,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10435,N'CONSH',8,DATE '1997-02-04',DATE '1997-03-18',DATE '1997-02-07',9.21,
	N'Consolidated Holdings',N'Berkeley Gardens 12  Brewery',N'London',
	NULL,N'WX1 6LT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10436,N'BLONP',3,DATE '1997-02-05',DATE '1997-03-05',DATE '1997-02-11',156.66,
	N'Blondel pere et fils',N'24, place Kleber',N'Strasbourg',
	NULL,N'67000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10437,N'WARTH',8,DATE '1997-02-05',DATE '1997-03-05',DATE '1997-02-12',19.97,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10438,N'TOMSP',3,DATE '1997-02-06',DATE '1997-03-06',DATE '1997-02-14',8.24,
	N'Toms Spezialitaten',N'Luisenstr. 48',N'Munster',
	NULL,N'44087',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10439,N'MEREP',6,DATE '1997-02-07',DATE '1997-03-07',DATE '1997-02-10',4.07,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10440,N'SAVEA',4,DATE '1997-02-10',DATE '1997-03-10',DATE '1997-02-28',86.53,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10441,N'OLDWO',3,DATE '1997-02-10',DATE '1997-03-24',DATE '1997-03-14',73.02,
	N'Old World Delicatessen',N'2743 Bering St.',N'Anchorage',
	N'AK',N'99508',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10442,N'ERNSH',3,DATE '1997-02-11',DATE '1997-03-11',DATE '1997-02-18',47.94,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10443,N'REGGC',8,DATE '1997-02-12',DATE '1997-03-12',DATE '1997-02-14',13.95,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10444,N'BERGS',3,DATE '1997-02-12',DATE '1997-03-12',DATE '1997-02-21',3.50,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10445,N'BERGS',3,DATE '1997-02-13',DATE '1997-03-13',DATE '1997-02-20',9.30,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10446,N'TOMSP',6,DATE '1997-02-14',DATE '1997-03-14',DATE '1997-02-19',14.68,
	N'Toms Spezialitaten',N'Luisenstr. 48',N'Munster',
	NULL,N'44087',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10447,N'RICAR',4,DATE '1997-02-14',DATE '1997-03-14',DATE '1997-03-07',68.66,
	N'Ricardo Adocicados',N'Av. Copacabana, 267',N'Rio de Janeiro',
	N'RJ',N'02389-890',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10448,N'RANCH',4,DATE '1997-02-17',DATE '1997-03-17',DATE '1997-02-24',38.82,
	N'Rancho grande',N'Av. del Libertador 900',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10449,N'BLONP',3,DATE '1997-02-18',DATE '1997-03-18',DATE '1997-02-27',53.30,
	N'Blondel pere et fils',N'24, place Kleber',N'Strasbourg',
	NULL,N'67000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10450,N'VICTE',8,DATE '1997-02-19',DATE '1997-03-19',DATE '1997-03-11',7.23,
	N'Victuailles en stock',N'2, rue du Commerce',N'Lyon',
	NULL,N'69004',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10451,N'QUICK',4,DATE '1997-02-19',DATE '1997-03-05',DATE '1997-03-12',189.09,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10452,N'SAVEA',8,DATE '1997-02-20',DATE '1997-03-20',DATE '1997-02-26',140.26,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10453,N'AROUT',1,DATE '1997-02-21',DATE '1997-03-21',DATE '1997-02-26',25.36,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10454,N'LAMAI',4,DATE '1997-02-21',DATE '1997-03-21',DATE '1997-02-25',2.74,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10455,N'WARTH',8,DATE '1997-02-24',DATE '1997-04-07',DATE '1997-03-03',180.45,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10456,N'KOENE',8,DATE '1997-02-25',DATE '1997-04-08',DATE '1997-02-28',8.12,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10457,N'KOENE',2,DATE '1997-02-25',DATE '1997-03-25',DATE '1997-03-03',11.57,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10458,N'SUPRD',7,DATE '1997-02-26',DATE '1997-03-26',DATE '1997-03-04',147.06,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10459,N'VICTE',4,DATE '1997-02-27',DATE '1997-03-27',DATE '1997-02-28',25.09,
	N'Victuailles en stock',N'2, rue du Commerce',N'Lyon',
	NULL,N'69004',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10460,N'FOLKO',8,DATE '1997-02-28',DATE '1997-03-28',DATE '1997-03-03',16.27,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10461,N'LILAS',1,DATE '1997-02-28',DATE '1997-03-28',DATE '1997-03-05',148.61,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10462,N'CONSH',2,DATE '1997-03-03',DATE '1997-03-31',DATE '1997-03-18',6.17,
	N'Consolidated Holdings',N'Berkeley Gardens 12  Brewery',N'London',
	NULL,N'WX1 6LT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10463,N'SUPRD',5,DATE '1997-03-04',DATE '1997-04-01',DATE '1997-03-06',14.78,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10464,N'FURIB',4,DATE '1997-03-04',DATE '1997-04-01',DATE '1997-03-14',89.00,
	N'Furia Bacalhau e Frutos do Mar',N'Jardim das rosas n. 32',N'Lisboa',
	NULL,N'1675',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10465,N'VAFFE',1,DATE '1997-03-05',DATE '1997-04-02',DATE '1997-03-14',145.04,
	N'Vaffeljernet',N'Smagsloget 45',N'Arhus',
	NULL,N'8200',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10466,N'COMMI',4,DATE '1997-03-06',DATE '1997-04-03',DATE '1997-03-13',11.93,
	N'Comercio Mineiro',N'Av. dos Lusiadas, 23',N'Sao Paulo',
	N'SP',N'05432-043',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10467,N'MAGAA',8,DATE '1997-03-06',DATE '1997-04-03',DATE '1997-03-11',4.93,
	N'Magazzini Alimentari Riuniti',N'Via Ludovico il Moro 22',N'Bergamo',
	NULL,N'24100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10468,N'KOENE',3,DATE '1997-03-07',DATE '1997-04-04',DATE '1997-03-12',44.12,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10469,N'WHITC',1,DATE '1997-03-10',DATE '1997-04-07',DATE '1997-03-14',60.18,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10470,N'BONAP',4,DATE '1997-03-11',DATE '1997-04-08',DATE '1997-03-14',64.56,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10471,N'BSBEV',2,DATE '1997-03-11',DATE '1997-04-08',DATE '1997-03-18',45.59,
	N'B''s Beverages',N'Fauntleroy Circus',N'London',
	NULL,N'EC2 5NT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10472,N'SEVES',8,DATE '1997-03-12',DATE '1997-04-09',DATE '1997-03-19',4.20,
	N'Seven Seas Imports',N'90 Wadhurst Rd.',N'London',
	NULL,N'OX15 4NB',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10473,N'ISLAT',1,DATE '1997-03-13',DATE '1997-03-27',DATE '1997-03-21',16.37,
	N'Island Trading',N'Garden House Crowther Way',N'Cowes',
	N'Isle of Wight',N'PO31 7PJ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10474,N'PERIC',5,DATE '1997-03-13',DATE '1997-04-10',DATE '1997-03-21',83.49,
	N'Pericles Comidas clasicas',N'Calle Dr. Jorge Cash 321',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10475,N'SUPRD',9,DATE '1997-03-14',DATE '1997-04-11',DATE '1997-04-04',68.52,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10476,N'HILAA',8,DATE '1997-03-17',DATE '1997-04-14',DATE '1997-03-24',4.41,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10477,N'PRINI',5,DATE '1997-03-17',DATE '1997-04-14',DATE '1997-03-25',13.02,
	N'Princesa Isabel Vinhos',N'Estrada da saude n. 58',N'Lisboa',
	NULL,N'1756',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10478,N'VICTE',2,DATE '1997-03-18',DATE '1997-04-01',DATE '1997-03-26',4.81,
	N'Victuailles en stock',N'2, rue du Commerce',N'Lyon',
	NULL,N'69004',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10479,N'RATTC',3,DATE '1997-03-19',DATE '1997-04-16',DATE '1997-03-21',708.95,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10480,N'FOLIG',6,DATE '1997-03-20',DATE '1997-04-17',DATE '1997-03-24',1.35,
	N'Folies gourmandes',N'184, chaussee de Tournai',N'Lille',
	NULL,N'59000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10481,N'RICAR',8,DATE '1997-03-20',DATE '1997-04-17',DATE '1997-03-25',64.33,
	N'Ricardo Adocicados',N'Av. Copacabana, 267',N'Rio de Janeiro',
	N'RJ',N'02389-890',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10482,N'LAZYK',1,DATE '1997-03-21',DATE '1997-04-18',DATE '1997-04-10',7.48,
	N'Lazy K Kountry Store',N'12 Orchestra Terrace',N'Walla Walla',
	N'WA',N'99362',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10483,N'WHITC',7,DATE '1997-03-24',DATE '1997-04-21',DATE '1997-04-25',15.28,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10484,N'BSBEV',3,DATE '1997-03-24',DATE '1997-04-21',DATE '1997-04-01',6.88,
	N'B''s Beverages',N'Fauntleroy Circus',N'London',
	NULL,N'EC2 5NT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10485,N'LINOD',4,DATE '1997-03-25',DATE '1997-04-08',DATE '1997-03-31',64.45,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10486,N'HILAA',1,DATE '1997-03-26',DATE '1997-04-23',DATE '1997-04-02',30.53,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10487,N'QUEEN',2,DATE '1997-03-26',DATE '1997-04-23',DATE '1997-03-28',71.07,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10488,N'FRANK',8,DATE '1997-03-27',DATE '1997-04-24',DATE '1997-04-02',4.93,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10489,N'PICCO',6,DATE '1997-03-28',DATE '1997-04-25',DATE '1997-04-09',5.29,
	N'Piccolo und mehr',N'Geislweg 14',N'Salzburg',
	NULL,N'5020',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10490,N'HILAA',7,DATE '1997-03-31',DATE '1997-04-28',DATE '1997-04-03',210.19,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10491,N'FURIB',8,DATE '1997-03-31',DATE '1997-04-28',DATE '1997-04-08',16.96,
	N'Furia Bacalhau e Frutos do Mar',N'Jardim das rosas n. 32',N'Lisboa',
	NULL,N'1675',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10492,N'BOTTM',3,DATE '1997-04-01',DATE '1997-04-29',DATE '1997-04-11',62.89,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10493,N'LAMAI',4,DATE '1997-04-02',DATE '1997-04-30',DATE '1997-04-10',10.64,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10494,N'COMMI',4,DATE '1997-04-02',DATE '1997-04-30',DATE '1997-04-09',65.99,
	N'Comercio Mineiro',N'Av. dos Lusiadas, 23',N'Sao Paulo',
	N'SP',N'05432-043',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10495,N'LAUGB',3,DATE '1997-04-03',DATE '1997-05-01',DATE '1997-04-11',4.65,
	N'Laughing Bacchus Wine Cellars',N'2319 Elm St.',N'Vancouver',
	N'BC',N'V3F 2K1',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10496,N'TRADH',7,DATE '1997-04-04',DATE '1997-05-02',DATE '1997-04-07',46.77,
	N'Tradicao Hipermercados',N'Av. Ines de Castro, 414',N'Sao Paulo',
	N'SP',N'05634-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10497,N'LEHMS',7,DATE '1997-04-04',DATE '1997-05-02',DATE '1997-04-07',36.21,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10498,N'HILAA',8,DATE '1997-04-07',DATE '1997-05-05',DATE '1997-04-11',29.75,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10499,N'LILAS',4,DATE '1997-04-08',DATE '1997-05-06',DATE '1997-04-16',102.02,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10500,N'LAMAI',6,DATE '1997-04-09',DATE '1997-05-07',DATE '1997-04-17',42.68,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10501,N'BLAUS',9,DATE '1997-04-09',DATE '1997-05-07',DATE '1997-04-16',8.85,
	N'Blauer See Delikatessen',N'Forsterstr. 57',N'Mannheim',
	NULL,N'68306',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10502,N'PERIC',2,DATE '1997-04-10',DATE '1997-05-08',DATE '1997-04-29',69.32,
	N'Pericles Comidas clasicas',N'Calle Dr. Jorge Cash 321',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10503,N'HUNGO',6,DATE '1997-04-11',DATE '1997-05-09',DATE '1997-04-16',16.74,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10504,N'WHITC',4,DATE '1997-04-11',DATE '1997-05-09',DATE '1997-04-18',59.13,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10505,N'MEREP',3,DATE '1997-04-14',DATE '1997-05-12',DATE '1997-04-21',7.13,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10506,N'KOENE',9,DATE '1997-04-15',DATE '1997-05-13',DATE '1997-05-02',21.19,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10507,N'ANTON',7,DATE '1997-04-15',DATE '1997-05-13',DATE '1997-04-22',47.45,
	N'Antonio Moreno Taqueria',N'Mataderos  2312',N'Mexico D.F.',
	NULL,N'05023',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10508,N'OTTIK',1,DATE '1997-04-16',DATE '1997-05-14',DATE '1997-05-13',4.99,
	N'Ottilies Kaseladen',N'Mehrheimerstr. 369',N'Koln',
	NULL,N'50739',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10509,N'BLAUS',4,DATE '1997-04-17',DATE '1997-05-15',DATE '1997-04-29',0.15,
	N'Blauer See Delikatessen',N'Forsterstr. 57',N'Mannheim',
	NULL,N'68306',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10510,N'SAVEA',6,DATE '1997-04-18',DATE '1997-05-16',DATE '1997-04-28',367.63,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10511,N'BONAP',4,DATE '1997-04-18',DATE '1997-05-16',DATE '1997-04-21',350.64,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10512,N'FAMIA',7,DATE '1997-04-21',DATE '1997-05-19',DATE '1997-04-24',3.53,
	N'Familia Arquibaldo',N'Rua Oros, 92',N'Sao Paulo',
	N'SP',N'05442-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10513,N'WANDK',7,DATE '1997-04-22',DATE '1997-06-03',DATE '1997-04-28',105.65,
	N'Die Wandernde Kuh',N'Adenauerallee 900',N'Stuttgart',
	NULL,N'70563',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10514,N'ERNSH',3,DATE '1997-04-22',DATE '1997-05-20',DATE '1997-05-16',789.95,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10515,N'QUICK',2,DATE '1997-04-23',DATE '1997-05-07',DATE '1997-05-23',204.47,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10516,N'HUNGO',2,DATE '1997-04-24',DATE '1997-05-22',DATE '1997-05-01',62.78,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10517,N'NORTS',3,DATE '1997-04-24',DATE '1997-05-22',DATE '1997-04-29',32.07,
	N'North/South',N'South House 300 Queensbridge',N'London',
	NULL,N'SW7 1RZ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10518,N'TORTU',4,DATE '1997-04-25',DATE '1997-05-09',DATE '1997-05-05',218.15,
	N'Tortuga Restaurante',N'Avda. Azteca 123',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10519,N'CHOPS',6,DATE '1997-04-28',DATE '1997-05-26',DATE '1997-05-01',91.76,
	N'Chop-suey Chinese',N'Hauptstr. 31',N'Bern',
	NULL,N'3012',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10520,N'SANTG',7,DATE '1997-04-29',DATE '1997-05-27',DATE '1997-05-01',13.37,
	N'Sante Gourmet',N'Erling Skakkes gate 78',N'Stavern',
	NULL,N'4110',N'Norway');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10521,N'CACTU',8,DATE '1997-04-29',DATE '1997-05-27',DATE '1997-05-02',17.22,
	N'Cactus Comidas para llevar',N'Cerrito 333',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10522,N'LEHMS',4,DATE '1997-04-30',DATE '1997-05-28',DATE '1997-05-06',45.33,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10523,N'SEVES',7,DATE '1997-05-01',DATE '1997-05-29',DATE '1997-05-30',77.63,
	N'Seven Seas Imports',N'90 Wadhurst Rd.',N'London',
	NULL,N'OX15 4NB',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10524,N'BERGS',1,DATE '1997-05-01',DATE '1997-05-29',DATE '1997-05-07',244.79,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10525,N'BONAP',1,DATE '1997-05-02',DATE '1997-05-30',DATE '1997-05-23',11.06,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10526,N'WARTH',4,DATE '1997-05-05',DATE '1997-06-02',DATE '1997-05-15',58.59,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10527,N'QUICK',7,DATE '1997-05-05',DATE '1997-06-02',DATE '1997-05-07',41.90,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10528,N'GREAL',6,DATE '1997-05-06',DATE '1997-05-20',DATE '1997-05-09',3.35,
	N'Great Lakes Food Market',N'2732 Baker Blvd.',N'Eugene',
	N'OR',N'97403',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10529,N'MAISD',5,DATE '1997-05-07',DATE '1997-06-04',DATE '1997-05-09',66.69,
	N'Maison Dewey',N'Rue Joseph-Bens 532',N'Bruxelles',
	NULL,N'B-1180',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10530,N'PICCO',3,DATE '1997-05-08',DATE '1997-06-05',DATE '1997-05-12',339.22,
	N'Piccolo und mehr',N'Geislweg 14',N'Salzburg',
	NULL,N'5020',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10531,N'OCEAN',7,DATE '1997-05-08',DATE '1997-06-05',DATE '1997-05-19',8.12,
	N'Oceano Atlantico Ltda.',N'Ing. Gustavo Moncada 8585 Piso 20-A',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10532,N'EASTC',7,DATE '1997-05-09',DATE '1997-06-06',DATE '1997-05-12',74.46,
	N'Eastern Connection',N'35 King George',N'London',
	NULL,N'WX3 6FW',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10533,N'FOLKO',8,DATE '1997-05-12',DATE '1997-06-09',DATE '1997-05-22',188.04,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10534,N'LEHMS',8,DATE '1997-05-12',DATE '1997-06-09',DATE '1997-05-14',27.94,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10535,N'ANTON',4,DATE '1997-05-13',DATE '1997-06-10',DATE '1997-05-21',15.64,
	N'Antonio Moreno Taqueria',N'Mataderos  2312',N'Mexico D.F.',
	NULL,N'05023',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10536,N'LEHMS',3,DATE '1997-05-14',DATE '1997-06-11',DATE '1997-06-06',58.88,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10537,N'RICSU',1,DATE '1997-05-14',DATE '1997-05-28',DATE '1997-05-19',78.85,
	N'Richter Supermarkt',N'Starenweg 5',N'Geneve',
	NULL,N'1204',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10538,N'BSBEV',9,DATE '1997-05-15',DATE '1997-06-12',DATE '1997-05-16',4.87,
	N'B''s Beverages',N'Fauntleroy Circus',N'London',
	NULL,N'EC2 5NT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10539,N'BSBEV',6,DATE '1997-05-16',DATE '1997-06-13',DATE '1997-05-23',12.36,
	N'B''s Beverages',N'Fauntleroy Circus',N'London',
	NULL,N'EC2 5NT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10540,N'QUICK',3,DATE '1997-05-19',DATE '1997-06-16',DATE '1997-06-13',1007.64,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10541,N'HANAR',2,DATE '1997-05-19',DATE '1997-06-16',DATE '1997-05-29',68.65,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10542,N'KOENE',1,DATE '1997-05-20',DATE '1997-06-17',DATE '1997-05-26',10.95,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10543,N'LILAS',8,DATE '1997-05-21',DATE '1997-06-18',DATE '1997-05-23',48.17,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10544,N'LONEP',4,DATE '1997-05-21',DATE '1997-06-18',DATE '1997-05-30',24.91,
	N'Lonesome Pine Restaurant',N'89 Chiaroscuro Rd.',N'Portland',
	N'OR',N'97219',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10545,N'LAZYK',8,DATE '1997-05-22',DATE '1997-06-19',DATE '1997-06-26',11.92,
	N'Lazy K Kountry Store',N'12 Orchestra Terrace',N'Walla Walla',
	N'WA',N'99362',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10546,N'VICTE',1,DATE '1997-05-23',DATE '1997-06-20',DATE '1997-05-27',194.72,
	N'Victuailles en stock',N'2, rue du Commerce',N'Lyon',
	NULL,N'69004',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10547,N'SEVES',3,DATE '1997-05-23',DATE '1997-06-20',DATE '1997-06-02',178.43,
	N'Seven Seas Imports',N'90 Wadhurst Rd.',N'London',
	NULL,N'OX15 4NB',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10548,N'TOMSP',3,DATE '1997-05-26',DATE '1997-06-23',DATE '1997-06-02',1.43,
	N'Toms Spezialitaten',N'Luisenstr. 48',N'Munster',
	NULL,N'44087',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10549,N'QUICK',5,DATE '1997-05-27',DATE '1997-06-10',DATE '1997-05-30',171.24,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10550,N'GODOS',7,DATE '1997-05-28',DATE '1997-06-25',DATE '1997-06-06',4.32,
	N'Godos Cocina Tipica',N'C/ Romero, 33',N'Sevilla',
	NULL,N'41101',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10551,N'FURIB',4,DATE '1997-05-28',DATE '1997-07-09',DATE '1997-06-06',72.95,
	N'Furia Bacalhau e Frutos do Mar',N'Jardim das rosas n. 32',N'Lisboa',
	NULL,N'1675',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10552,N'HILAA',2,DATE '1997-05-29',DATE '1997-06-26',DATE '1997-06-05',83.22,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10553,N'WARTH',2,DATE '1997-05-30',DATE '1997-06-27',DATE '1997-06-03',149.49,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10554,N'OTTIK',4,DATE '1997-05-30',DATE '1997-06-27',DATE '1997-06-05',120.97,
	N'Ottilies Kaseladen',N'Mehrheimerstr. 369',N'Koln',
	NULL,N'50739',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10555,N'SAVEA',6,DATE '1997-06-02',DATE '1997-06-30',DATE '1997-06-04',252.49,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10556,N'SIMOB',2,DATE '1997-06-03',DATE '1997-07-15',DATE '1997-06-13',9.80,
	N'Simons bistro',N'Vinb?ltet 34',N'Kobenhavn',
	NULL,N'1734',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10557,N'LEHMS',9,DATE '1997-06-03',DATE '1997-06-17',DATE '1997-06-06',96.72,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10558,N'AROUT',1,DATE '1997-06-04',DATE '1997-07-02',DATE '1997-06-10',72.97,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10559,N'BLONP',6,DATE '1997-06-05',DATE '1997-07-03',DATE '1997-06-13',8.05,
	N'Blondel pere et fils',N'24, place Kleber',N'Strasbourg',
	NULL,N'67000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10560,N'FRANK',8,DATE '1997-06-06',DATE '1997-07-04',DATE '1997-06-09',36.65,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10561,N'FOLKO',2,DATE '1997-06-06',DATE '1997-07-04',DATE '1997-06-09',242.21,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10562,N'REGGC',1,DATE '1997-06-09',DATE '1997-07-07',DATE '1997-06-12',22.95,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10563,N'RICAR',2,DATE '1997-06-10',DATE '1997-07-22',DATE '1997-06-24',60.43,
	N'Ricardo Adocicados',N'Av. Copacabana, 267',N'Rio de Janeiro',
	N'RJ',N'02389-890',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10564,N'RATTC',4,DATE '1997-06-10',DATE '1997-07-08',DATE '1997-06-16',13.75,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10565,N'MEREP',8,DATE '1997-06-11',DATE '1997-07-09',DATE '1997-06-18',7.15,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10566,N'BLONP',9,DATE '1997-06-12',DATE '1997-07-10',DATE '1997-06-18',88.40,
	N'Blondel pere et fils',N'24, place Kleber',N'Strasbourg',
	NULL,N'67000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10567,N'HUNGO',1,DATE '1997-06-12',DATE '1997-07-10',DATE '1997-06-17',33.97,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10568,N'GALED',3,DATE '1997-06-13',DATE '1997-07-11',DATE '1997-07-09',6.54,
	N'Galeria del gastronomo',N'Rambla de Cataluna, 23',N'Barcelona',
	NULL,N'8022',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10569,N'RATTC',5,DATE '1997-06-16',DATE '1997-07-14',DATE '1997-07-11',58.98,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10570,N'MEREP',3,DATE '1997-06-17',DATE '1997-07-15',DATE '1997-06-19',188.99,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10571,N'ERNSH',8,DATE '1997-06-17',DATE '1997-07-29',DATE '1997-07-04',26.06,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10572,N'BERGS',3,DATE '1997-06-18',DATE '1997-07-16',DATE '1997-06-25',116.43,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10573,N'ANTON',7,DATE '1997-06-19',DATE '1997-07-17',DATE '1997-06-20',84.84,
	N'Antonio Moreno Taqueria',N'Mataderos  2312',N'Mexico D.F.',
	NULL,N'05023',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10574,N'TRAIH',4,DATE '1997-06-19',DATE '1997-07-17',DATE '1997-06-30',37.60,
	N'Trail''s Head Gourmet Provisioners',N'722 DaVinci Blvd.',N'Kirkland',
	N'WA',N'98034',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10575,N'MORGK',5,DATE '1997-06-20',DATE '1997-07-04',DATE '1997-06-30',127.34,
	N'Morgenstern Gesundkost',N'Heerstr. 22',N'Leipzig',
	NULL,N'04179',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10576,N'TORTU',3,DATE '1997-06-23',DATE '1997-07-07',DATE '1997-06-30',18.56,
	N'Tortuga Restaurante',N'Avda. Azteca 123',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10577,N'TRAIH',9,DATE '1997-06-23',DATE '1997-08-04',DATE '1997-06-30',25.41,
	N'Trail''s Head Gourmet Provisioners',N'722 DaVinci Blvd.',N'Kirkland',
	N'WA',N'98034',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10578,N'BSBEV',4,DATE '1997-06-24',DATE '1997-07-22',DATE '1997-07-25',29.60,
	N'B''s Beverages',N'Fauntleroy Circus',N'London',
	NULL,N'EC2 5NT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10579,N'LETSS',1,DATE '1997-06-25',DATE '1997-07-23',DATE '1997-07-04',13.73,
	N'Let''s Stop N Shop',N'87 Polk St. Suite 5',N'San Francisco',
	N'CA',N'94117',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10580,N'OTTIK',4,DATE '1997-06-26',DATE '1997-07-24',DATE '1997-07-01',75.89,
	N'Ottilies Kaseladen',N'Mehrheimerstr. 369',N'Koln',
	NULL,N'50739',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10581,N'FAMIA',3,DATE '1997-06-26',DATE '1997-07-24',DATE '1997-07-02',3.01,
	N'Familia Arquibaldo',N'Rua Oros, 92',N'Sao Paulo',
	N'SP',N'05442-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10582,N'BLAUS',3,DATE '1997-06-27',DATE '1997-07-25',DATE '1997-07-14',27.71,
	N'Blauer See Delikatessen',N'Forsterstr. 57',N'Mannheim',
	NULL,N'68306',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10583,N'WARTH',2,DATE '1997-06-30',DATE '1997-07-28',DATE '1997-07-04',7.28,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10584,N'BLONP',4,DATE '1997-06-30',DATE '1997-07-28',DATE '1997-07-04',59.14,
	N'Blondel pere et fils',N'24, place Kleber',N'Strasbourg',
	NULL,N'67000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10585,N'WELLI',7,DATE '1997-07-01',DATE '1997-07-29',DATE '1997-07-10',13.41,
	N'Wellington Importadora',N'Rua do Mercado, 12',N'Resende',
	N'SP',N'08737-363',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10586,N'REGGC',9,DATE '1997-07-02',DATE '1997-07-30',DATE '1997-07-09',0.48,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10587,N'QUEDE',1,DATE '1997-07-02',DATE '1997-07-30',DATE '1997-07-09',62.52,
	N'Que Delicia',N'Rua da Panificadora, 12',N'Rio de Janeiro',
	N'RJ',N'02389-673',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10588,N'QUICK',2,DATE '1997-07-03',DATE '1997-07-31',DATE '1997-07-10',194.67,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10589,N'GREAL',8,DATE '1997-07-04',DATE '1997-08-01',DATE '1997-07-14',4.42,
	N'Great Lakes Food Market',N'2732 Baker Blvd.',N'Eugene',
	N'OR',N'97403',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10590,N'MEREP',4,DATE '1997-07-07',DATE '1997-08-04',DATE '1997-07-14',44.77,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10591,N'VAFFE',1,DATE '1997-07-07',DATE '1997-07-21',DATE '1997-07-16',55.92,
	N'Vaffeljernet',N'Smagsloget 45',N'Arhus',
	NULL,N'8200',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10592,N'LEHMS',3,DATE '1997-07-08',DATE '1997-08-05',DATE '1997-07-16',32.10,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10593,N'LEHMS',7,DATE '1997-07-09',DATE '1997-08-06',DATE '1997-08-13',174.20,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10594,N'OLDWO',3,DATE '1997-07-09',DATE '1997-08-06',DATE '1997-07-16',5.24,
	N'Old World Delicatessen',N'2743 Bering St.',N'Anchorage',
	N'AK',N'99508',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10595,N'ERNSH',2,DATE '1997-07-10',DATE '1997-08-07',DATE '1997-07-14',96.78,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10596,N'WHITC',8,DATE '1997-07-11',DATE '1997-08-08',DATE '1997-08-12',16.34,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10597,N'PICCO',7,DATE '1997-07-11',DATE '1997-08-08',DATE '1997-07-18',35.12,
	N'Piccolo und mehr',N'Geislweg 14',N'Salzburg',
	NULL,N'5020',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10598,N'RATTC',1,DATE '1997-07-14',DATE '1997-08-11',DATE '1997-07-18',44.42,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10599,N'BSBEV',6,DATE '1997-07-15',DATE '1997-08-26',DATE '1997-07-21',29.98,
	N'B''s Beverages',N'Fauntleroy Circus',N'London',
	NULL,N'EC2 5NT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10600,N'HUNGC',4,DATE '1997-07-16',DATE '1997-08-13',DATE '1997-07-21',45.13,
	N'Hungry Coyote Import Store',N'City Center Plaza 516 Main St.',N'Elgin',
	N'OR',N'97827',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10601,N'HILAA',7,DATE '1997-07-16',DATE '1997-08-27',DATE '1997-07-22',58.30,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10602,N'VAFFE',8,DATE '1997-07-17',DATE '1997-08-14',DATE '1997-07-22',2.92,
	N'Vaffeljernet',N'Smagsloget 45',N'Arhus',
	NULL,N'8200',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10603,N'SAVEA',8,DATE '1997-07-18',DATE '1997-08-15',DATE '1997-08-08',48.77,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10604,N'FURIB',1,DATE '1997-07-18',DATE '1997-08-15',DATE '1997-07-29',7.46,
	N'Furia Bacalhau e Frutos do Mar',N'Jardim das rosas n. 32',N'Lisboa',
	NULL,N'1675',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10605,N'MEREP',1,DATE '1997-07-21',DATE '1997-08-18',DATE '1997-07-29',379.13,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10606,N'TRADH',4,DATE '1997-07-22',DATE '1997-08-19',DATE '1997-07-31',79.40,
	N'Tradicao Hipermercados',N'Av. Ines de Castro, 414',N'Sao Paulo',
	N'SP',N'05634-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10607,N'SAVEA',5,DATE '1997-07-22',DATE '1997-08-19',DATE '1997-07-25',200.24,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10608,N'TOMSP',4,DATE '1997-07-23',DATE '1997-08-20',DATE '1997-08-01',27.79,
	N'Toms Spezialitaten',N'Luisenstr. 48',N'Munster',
	NULL,N'44087',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10609,N'DUMON',7,DATE '1997-07-24',DATE '1997-08-21',DATE '1997-07-30',1.85,
	N'Du monde entier',N'67, rue des Cinquante Otages',N'Nantes',
	NULL,N'44000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10610,N'LAMAI',8,DATE '1997-07-25',DATE '1997-08-22',DATE '1997-08-06',26.78,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10611,N'WOLZA',6,DATE '1997-07-25',DATE '1997-08-22',DATE '1997-08-01',80.65,
	N'Wolski Zajazd',N'ul. Filtrowa 68',N'Warszawa',
	NULL,N'01-012',N'Poland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10612,N'SAVEA',1,DATE '1997-07-28',DATE '1997-08-25',DATE '1997-08-01',544.08,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10613,N'HILAA',4,DATE '1997-07-29',DATE '1997-08-26',DATE '1997-08-01',8.11,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10614,N'BLAUS',8,DATE '1997-07-29',DATE '1997-08-26',DATE '1997-08-01',1.93,
	N'Blauer See Delikatessen',N'Forsterstr. 57',N'Mannheim',
	NULL,N'68306',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10615,N'WILMK',2,DATE '1997-07-30',DATE '1997-08-27',DATE '1997-08-06',0.75,
	N'Wilman Kala',N'Keskuskatu 45',N'Helsinki',
	NULL,N'21240',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10616,N'GREAL',1,DATE '1997-07-31',DATE '1997-08-28',DATE '1997-08-05',116.53,
	N'Great Lakes Food Market',N'2732 Baker Blvd.',N'Eugene',
	N'OR',N'97403',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10617,N'GREAL',4,DATE '1997-07-31',DATE '1997-08-28',DATE '1997-08-04',18.53,
	N'Great Lakes Food Market',N'2732 Baker Blvd.',N'Eugene',
	N'OR',N'97403',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10618,N'MEREP',1,DATE '1997-08-01',DATE '1997-09-12',DATE '1997-08-08',154.68,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10619,N'MEREP',3,DATE '1997-08-04',DATE '1997-09-01',DATE '1997-08-07',91.05,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10620,N'LAUGB',2,DATE '1997-08-05',DATE '1997-09-02',DATE '1997-08-14',0.94,
	N'Laughing Bacchus Wine Cellars',N'2319 Elm St.',N'Vancouver',
	N'BC',N'V3F 2K1',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10621,N'ISLAT',4,DATE '1997-08-05',DATE '1997-09-02',DATE '1997-08-11',23.73,
	N'Island Trading',N'Garden House Crowther Way',N'Cowes',
	N'Isle of Wight',N'PO31 7PJ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10622,N'RICAR',4,DATE '1997-08-06',DATE '1997-09-03',DATE '1997-08-11',50.97,
	N'Ricardo Adocicados',N'Av. Copacabana, 267',N'Rio de Janeiro',
	N'RJ',N'02389-890',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10623,N'FRANK',8,DATE '1997-08-07',DATE '1997-09-04',DATE '1997-08-12',97.18,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10624,N'THECR',4,DATE '1997-08-07',DATE '1997-09-04',DATE '1997-08-19',94.80,
	N'The Cracker Box',N'55 Grizzly Peak Rd.',N'Butte',
	N'MT',N'59801',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10625,N'ANATR',3,DATE '1997-08-08',DATE '1997-09-05',DATE '1997-08-14',43.90,
	N'Ana Trujillo Emparedados y helados',N'Avda. de la Constitucion 2222',N'Mexico D.F.',
	NULL,N'05021',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10626,N'BERGS',1,DATE '1997-08-11',DATE '1997-09-08',DATE '1997-08-20',138.69,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10627,N'SAVEA',8,DATE '1997-08-11',DATE '1997-09-22',DATE '1997-08-21',107.46,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10628,N'BLONP',4,DATE '1997-08-12',DATE '1997-09-09',DATE '1997-08-20',30.36,
	N'Blondel pere et fils',N'24, place Kleber',N'Strasbourg',
	NULL,N'67000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10629,N'GODOS',4,DATE '1997-08-12',DATE '1997-09-09',DATE '1997-08-20',85.46,
	N'Godos Cocina Tipica',N'C/ Romero, 33',N'Sevilla',
	NULL,N'41101',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10630,N'KOENE',1,DATE '1997-08-13',DATE '1997-09-10',DATE '1997-08-19',32.35,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10631,N'LAMAI',8,DATE '1997-08-14',DATE '1997-09-11',DATE '1997-08-15',0.87,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10632,N'WANDK',8,DATE '1997-08-14',DATE '1997-09-11',DATE '1997-08-19',41.38,
	N'Die Wandernde Kuh',N'Adenauerallee 900',N'Stuttgart',
	NULL,N'70563',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10633,N'ERNSH',7,DATE '1997-08-15',DATE '1997-09-12',DATE '1997-08-18',477.90,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10634,N'FOLIG',4,DATE '1997-08-15',DATE '1997-09-12',DATE '1997-08-21',487.38,
	N'Folies gourmandes',N'184, chaussee de Tournai',N'Lille',
	NULL,N'59000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10635,N'MAGAA',8,DATE '1997-08-18',DATE '1997-09-15',DATE '1997-08-21',47.46,
	N'Magazzini Alimentari Riuniti',N'Via Ludovico il Moro 22',N'Bergamo',
	NULL,N'24100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10636,N'WARTH',4,DATE '1997-08-19',DATE '1997-09-16',DATE '1997-08-26',1.15,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10637,N'QUEEN',6,DATE '1997-08-19',DATE '1997-09-16',DATE '1997-08-26',201.29,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10638,N'LINOD',3,DATE '1997-08-20',DATE '1997-09-17',DATE '1997-09-01',158.44,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10639,N'SANTG',7,DATE '1997-08-20',DATE '1997-09-17',DATE '1997-08-27',38.64,
	N'Sante Gourmet',N'Erling Skakkes gate 78',N'Stavern',
	NULL,N'4110',N'Norway');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10640,N'WANDK',4,DATE '1997-08-21',DATE '1997-09-18',DATE '1997-08-28',23.55,
	N'Die Wandernde Kuh',N'Adenauerallee 900',N'Stuttgart',
	NULL,N'70563',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10641,N'HILAA',4,DATE '1997-08-22',DATE '1997-09-19',DATE '1997-08-26',179.61,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10642,N'SIMOB',7,DATE '1997-08-22',DATE '1997-09-19',DATE '1997-09-05',41.89,
	N'Simons bistro',N'Vinb?ltet 34',N'Kobenhavn',
	NULL,N'1734',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10643,N'ALFKI',6,DATE '1997-08-25',DATE '1997-09-22',DATE '1997-09-02',29.46,
	N'Alfreds Futterkiste',N'Obere Str. 57',N'Berlin',
	NULL,N'12209',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10644,N'WELLI',3,DATE '1997-08-25',DATE '1997-09-22',DATE '1997-09-01',0.14,
	N'Wellington Importadora',N'Rua do Mercado, 12',N'Resende',
	N'SP',N'08737-363',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10645,N'HANAR',4,DATE '1997-08-26',DATE '1997-09-23',DATE '1997-09-02',12.41,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10646,N'HUNGO',9,DATE '1997-08-27',DATE '1997-10-08',DATE '1997-09-03',142.33,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10647,N'QUEDE',4,DATE '1997-08-27',DATE '1997-09-10',DATE '1997-09-03',45.54,
	N'Que Delicia',N'Rua da Panificadora, 12',N'Rio de Janeiro',
	N'RJ',N'02389-673',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10648,N'RICAR',5,DATE '1997-08-28',DATE '1997-10-09',DATE '1997-09-09',14.25,
	N'Ricardo Adocicados',N'Av. Copacabana, 267',N'Rio de Janeiro',
	N'RJ',N'02389-890',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10649,N'MAISD',5,DATE '1997-08-28',DATE '1997-09-25',DATE '1997-08-29',6.20,
	N'Maison Dewey',N'Rue Joseph-Bens 532',N'Bruxelles',
	NULL,N'B-1180',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10650,N'FAMIA',5,DATE '1997-08-29',DATE '1997-09-26',DATE '1997-09-03',176.81,
	N'Familia Arquibaldo',N'Rua Oros, 92',N'Sao Paulo',
	N'SP',N'05442-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10651,N'WANDK',8,DATE '1997-09-01',DATE '1997-09-29',DATE '1997-09-11',20.60,
	N'Die Wandernde Kuh',N'Adenauerallee 900',N'Stuttgart',
	NULL,N'70563',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10652,N'GOURL',4,DATE '1997-09-01',DATE '1997-09-29',DATE '1997-09-08',7.14,
	N'Gourmet Lanchonetes',N'Av. Brasil, 442',N'Campinas',
	N'SP',N'04876-786',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10653,N'FRANK',1,DATE '1997-09-02',DATE '1997-09-30',DATE '1997-09-19',93.25,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10654,N'BERGS',5,DATE '1997-09-02',DATE '1997-09-30',DATE '1997-09-11',55.26,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10655,N'REGGC',1,DATE '1997-09-03',DATE '1997-10-01',DATE '1997-09-11',4.41,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10656,N'GREAL',6,DATE '1997-09-04',DATE '1997-10-02',DATE '1997-09-10',57.15,
	N'Great Lakes Food Market',N'2732 Baker Blvd.',N'Eugene',
	N'OR',N'97403',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10657,N'SAVEA',2,DATE '1997-09-04',DATE '1997-10-02',DATE '1997-09-15',352.69,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10658,N'QUICK',4,DATE '1997-09-05',DATE '1997-10-03',DATE '1997-09-08',364.15,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10659,N'QUEEN',7,DATE '1997-09-05',DATE '1997-10-03',DATE '1997-09-10',105.81,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10660,N'HUNGC',8,DATE '1997-09-08',DATE '1997-10-06',DATE '1997-10-15',111.29,
	N'Hungry Coyote Import Store',N'City Center Plaza 516 Main St.',N'Elgin',
	N'OR',N'97827',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10661,N'HUNGO',7,DATE '1997-09-09',DATE '1997-10-07',DATE '1997-09-15',17.55,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10662,N'LONEP',3,DATE '1997-09-09',DATE '1997-10-07',DATE '1997-09-18',1.28,
	N'Lonesome Pine Restaurant',N'89 Chiaroscuro Rd.',N'Portland',
	N'OR',N'97219',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10663,N'BONAP',2,DATE '1997-09-10',DATE '1997-09-24',DATE '1997-10-03',113.15,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10664,N'FURIB',1,DATE '1997-09-10',DATE '1997-10-08',DATE '1997-09-19',1.27,
	N'Furia Bacalhau e Frutos do Mar',N'Jardim das rosas n. 32',N'Lisboa',
	NULL,N'1675',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10665,N'LONEP',1,DATE '1997-09-11',DATE '1997-10-09',DATE '1997-09-17',26.31,
	N'Lonesome Pine Restaurant',N'89 Chiaroscuro Rd.',N'Portland',
	N'OR',N'97219',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10666,N'RICSU',7,DATE '1997-09-12',DATE '1997-10-10',DATE '1997-09-22',232.42,
	N'Richter Supermarkt',N'Starenweg 5',N'Geneve',
	NULL,N'1204',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10667,N'ERNSH',7,DATE '1997-09-12',DATE '1997-10-10',DATE '1997-09-19',78.09,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10668,N'WANDK',1,DATE '1997-09-15',DATE '1997-10-13',DATE '1997-09-23',47.22,
	N'Die Wandernde Kuh',N'Adenauerallee 900',N'Stuttgart',
	NULL,N'70563',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10669,N'SIMOB',2,DATE '1997-09-15',DATE '1997-10-13',DATE '1997-09-22',24.39,
	N'Simons bistro',N'Vinb?ltet 34',N'Kobenhavn',
	NULL,N'1734',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10670,N'FRANK',4,DATE '1997-09-16',DATE '1997-10-14',DATE '1997-09-18',203.48,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10671,N'FRANR',1,DATE '1997-09-17',DATE '1997-10-15',DATE '1997-09-24',30.34,
	N'France restauration',N'54, rue Royale',N'Nantes',
	NULL,N'44000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10672,N'BERGS',9,DATE '1997-09-17',DATE '1997-10-01',DATE '1997-09-26',95.75,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10673,N'WILMK',2,DATE '1997-09-18',DATE '1997-10-16',DATE '1997-09-19',22.76,
	N'Wilman Kala',N'Keskuskatu 45',N'Helsinki',
	NULL,N'21240',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10674,N'ISLAT',4,DATE '1997-09-18',DATE '1997-10-16',DATE '1997-09-30',0.90,
	N'Island Trading',N'Garden House Crowther Way',N'Cowes',
	N'Isle of Wight',N'PO31 7PJ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10675,N'FRANK',5,DATE '1997-09-19',DATE '1997-10-17',DATE '1997-09-23',31.85,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10676,N'TORTU',2,DATE '1997-09-22',DATE '1997-10-20',DATE '1997-09-29',2.01,
	N'Tortuga Restaurante',N'Avda. Azteca 123',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10677,N'ANTON',1,DATE '1997-09-22',DATE '1997-10-20',DATE '1997-09-26',4.03,
	N'Antonio Moreno Taqueria',N'Mataderos  2312',N'Mexico D.F.',
	NULL,N'05023',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10678,N'SAVEA',7,DATE '1997-09-23',DATE '1997-10-21',DATE '1997-10-16',388.98,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10679,N'BLONP',8,DATE '1997-09-23',DATE '1997-10-21',DATE '1997-09-30',27.94,
	N'Blondel pere et fils',N'24, place Kleber',N'Strasbourg',
	NULL,N'67000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10680,N'OLDWO',1,DATE '1997-09-24',DATE '1997-10-22',DATE '1997-09-26',26.61,
	N'Old World Delicatessen',N'2743 Bering St.',N'Anchorage',
	N'AK',N'99508',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10681,N'GREAL',3,DATE '1997-09-25',DATE '1997-10-23',DATE '1997-09-30',76.13,
	N'Great Lakes Food Market',N'2732 Baker Blvd.',N'Eugene',
	N'OR',N'97403',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10682,N'ANTON',3,DATE '1997-09-25',DATE '1997-10-23',DATE '1997-10-01',36.13,
	N'Antonio Moreno Taqueria',N'Mataderos  2312',N'Mexico D.F.',
	NULL,N'05023',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10683,N'DUMON',2,DATE '1997-09-26',DATE '1997-10-24',DATE '1997-10-01',4.40,
	N'Du monde entier',N'67, rue des Cinquante Otages',N'Nantes',
	NULL,N'44000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10684,N'OTTIK',3,DATE '1997-09-26',DATE '1997-10-24',DATE '1997-09-30',145.63,
	N'Ottilies Kaseladen',N'Mehrheimerstr. 369',N'Koln',
	NULL,N'50739',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10685,N'GOURL',4,DATE '1997-09-29',DATE '1997-10-13',DATE '1997-10-03',33.75,
	N'Gourmet Lanchonetes',N'Av. Brasil, 442',N'Campinas',
	N'SP',N'04876-786',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10686,N'PICCO',2,DATE '1997-09-30',DATE '1997-10-28',DATE '1997-10-08',96.50,
	N'Piccolo und mehr',N'Geislweg 14',N'Salzburg',
	NULL,N'5020',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10687,N'HUNGO',9,DATE '1997-09-30',DATE '1997-10-28',DATE '1997-10-30',296.43,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10688,N'VAFFE',4,DATE '1997-10-01',DATE '1997-10-15',DATE '1997-10-07',299.09,
	N'Vaffeljernet',N'Smagsloget 45',N'Arhus',
	NULL,N'8200',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10689,N'BERGS',1,DATE '1997-10-01',DATE '1997-10-29',DATE '1997-10-07',13.42,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10690,N'HANAR',1,DATE '1997-10-02',DATE '1997-10-30',DATE '1997-10-03',15.80,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10691,N'QUICK',2,DATE '1997-10-03',DATE '1997-11-14',DATE '1997-10-22',810.05,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10692,N'ALFKI',4,DATE '1997-10-03',DATE '1997-10-31',DATE '1997-10-13',61.02,
	N'Alfred''s Futterkiste',N'Obere Str. 57',N'Berlin',
	NULL,N'12209',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10693,N'WHITC',3,DATE '1997-10-06',DATE '1997-10-20',DATE '1997-10-10',139.34,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10694,N'QUICK',8,DATE '1997-10-06',DATE '1997-11-03',DATE '1997-10-09',398.36,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10695,N'WILMK',7,DATE '1997-10-07',DATE '1997-11-18',DATE '1997-10-14',16.72,
	N'Wilman Kala',N'Keskuskatu 45',N'Helsinki',
	NULL,N'21240',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10696,N'WHITC',8,DATE '1997-10-08',DATE '1997-11-19',DATE '1997-10-14',102.55,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10697,N'LINOD',3,DATE '1997-10-08',DATE '1997-11-05',DATE '1997-10-14',45.52,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10698,N'ERNSH',4,DATE '1997-10-09',DATE '1997-11-06',DATE '1997-10-17',272.47,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10699,N'MORGK',3,DATE '1997-10-09',DATE '1997-11-06',DATE '1997-10-13',0.58,
	N'Morgenstern Gesundkost',N'Heerstr. 22',N'Leipzig',
	NULL,N'04179',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10700,N'SAVEA',3,DATE '1997-10-10',DATE '1997-11-07',DATE '1997-10-16',65.10,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10701,N'HUNGO',6,DATE '1997-10-13',DATE '1997-10-27',DATE '1997-10-15',220.31,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10702,N'ALFKI',4,DATE '1997-10-13',DATE '1997-11-24',DATE '1997-10-21',23.94,
	N'Alfred''s Futterkiste',N'Obere Str. 57',N'Berlin',
	NULL,N'12209',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10703,N'FOLKO',6,DATE '1997-10-14',DATE '1997-11-11',DATE '1997-10-20',152.30,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10704,N'QUEEN',6,DATE '1997-10-14',DATE '1997-11-11',DATE '1997-11-07',4.78,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10705,N'HILAA',9,DATE '1997-10-15',DATE '1997-11-12',DATE '1997-11-18',3.52,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10706,N'OLDWO',8,DATE '1997-10-16',DATE '1997-11-13',DATE '1997-10-21',135.63,
	N'Old World Delicatessen',N'2743 Bering St.',N'Anchorage',
	N'AK',N'99508',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10707,N'AROUT',4,DATE '1997-10-16',DATE '1997-10-30',DATE '1997-10-23',21.74,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10708,N'THEBI',6,DATE '1997-10-17',DATE '1997-11-28',DATE '1997-11-05',2.96,
	N'The Big Cheese',N'89 Jefferson Way Suite 2',N'Portland',
	N'OR',N'97201',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10709,N'GOURL',1,DATE '1997-10-17',DATE '1997-11-14',DATE '1997-11-20',210.80,
	N'Gourmet Lanchonetes',N'Av. Brasil, 442',N'Campinas',
	N'SP',N'04876-786',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10710,N'FRANS',1,DATE '1997-10-20',DATE '1997-11-17',DATE '1997-10-23',4.98,
	N'Franchi S.p.A.',N'Via Monte Bianco 34',N'Torino',
	NULL,N'10100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10711,N'SAVEA',5,DATE '1997-10-21',DATE '1997-12-02',DATE '1997-10-29',52.41,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10712,N'HUNGO',3,DATE '1997-10-21',DATE '1997-11-18',DATE '1997-10-31',89.93,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10713,N'SAVEA',1,DATE '1997-10-22',DATE '1997-11-19',DATE '1997-10-24',167.05,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10714,N'SAVEA',5,DATE '1997-10-22',DATE '1997-11-19',DATE '1997-10-27',24.49,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10715,N'BONAP',3,DATE '1997-10-23',DATE '1997-11-06',DATE '1997-10-29',63.20,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10716,N'RANCH',4,DATE '1997-10-24',DATE '1997-11-21',DATE '1997-10-27',22.57,
	N'Rancho grande',N'Av. del Libertador 900',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10717,N'FRANK',1,DATE '1997-10-24',DATE '1997-11-21',DATE '1997-10-29',59.25,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10718,N'KOENE',1,DATE '1997-10-27',DATE '1997-11-24',DATE '1997-10-29',170.88,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10719,N'LETSS',8,DATE '1997-10-27',DATE '1997-11-24',DATE '1997-11-05',51.44,
	N'Let''s Stop N Shop',N'87 Polk St. Suite 5',N'San Francisco',
	N'CA',N'94117',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10720,N'QUEDE',8,DATE '1997-10-28',DATE '1997-11-11',DATE '1997-11-05',9.53,
	N'Que Delicia',N'Rua da Panificadora, 12',N'Rio de Janeiro',
	N'RJ',N'02389-673',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10721,N'QUICK',5,DATE '1997-10-29',DATE '1997-11-26',DATE '1997-10-31',48.92,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10722,N'SAVEA',8,DATE '1997-10-29',DATE '1997-12-10',DATE '1997-11-04',74.58,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10723,N'WHITC',3,DATE '1997-10-30',DATE '1997-11-27',DATE '1997-11-25',21.72,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10724,N'MEREP',8,DATE '1997-10-30',DATE '1997-12-11',DATE '1997-11-05',57.75,
	N'Mere Paillarde',N'43 rue St. Laurent',N'Montreal',
	N'Quebec',N'H1J 1C3',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10725,N'FAMIA',4,DATE '1997-10-31',DATE '1997-11-28',DATE '1997-11-05',10.83,
	N'Familia Arquibaldo',N'Rua Oros, 92',N'Sao Paulo',
	N'SP',N'05442-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10726,N'EASTC',4,DATE '1997-11-03',DATE '1997-11-17',DATE '1997-12-05',16.56,
	N'Eastern Connection',N'35 King George',N'London',
	NULL,N'WX3 6FW',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10727,N'REGGC',2,DATE '1997-11-03',DATE '1997-12-01',DATE '1997-12-05',89.90,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10728,N'QUEEN',4,DATE '1997-11-04',DATE '1997-12-02',DATE '1997-11-11',58.33,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10729,N'LINOD',8,DATE '1997-11-04',DATE '1997-12-16',DATE '1997-11-14',141.06,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10730,N'BONAP',5,DATE '1997-11-05',DATE '1997-12-03',DATE '1997-11-14',20.12,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10731,N'CHOPS',7,DATE '1997-11-06',DATE '1997-12-04',DATE '1997-11-14',96.65,
	N'Chop-suey Chinese',N'Hauptstr. 31',N'Bern',
	NULL,N'3012',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10732,N'BONAP',3,DATE '1997-11-06',DATE '1997-12-04',DATE '1997-11-07',16.97,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10733,N'BERGS',1,DATE '1997-11-07',DATE '1997-12-05',DATE '1997-11-10',110.11,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10734,N'GOURL',2,DATE '1997-11-07',DATE '1997-12-05',DATE '1997-11-12',1.63,
	N'Gourmet Lanchonetes',N'Av. Brasil, 442',N'Campinas',
	N'SP',N'04876-786',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10735,N'LETSS',6,DATE '1997-11-10',DATE '1997-12-08',DATE '1997-11-21',45.97,
	N'Let''s Stop N Shop',N'87 Polk St. Suite 5',N'San Francisco',
	N'CA',N'94117',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10736,N'HUNGO',9,DATE '1997-11-11',DATE '1997-12-09',DATE '1997-11-21',44.10,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10737,N'VINET',2,DATE '1997-11-11',DATE '1997-12-09',DATE '1997-11-18',7.79,
	N'Vins et alcools Chevalier',N'59 rue de l''Abbaye',N'Reims',
	NULL,N'51100',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10738,N'SPECD',2,DATE '1997-11-12',DATE '1997-12-10',DATE '1997-11-18',2.91,
	N'Specialites du monde',N'25, rue Lauriston',N'Paris',
	NULL,N'75016',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10739,N'VINET',3,DATE '1997-11-12',DATE '1997-12-10',DATE '1997-11-17',11.08,
	N'Vins et alcools Chevalier',N'59 rue de l''Abbaye',N'Reims',
	NULL,N'51100',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10740,N'WHITC',4,DATE '1997-11-13',DATE '1997-12-11',DATE '1997-11-25',81.88,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10741,N'AROUT',4,DATE '1997-11-14',DATE '1997-11-28',DATE '1997-11-18',10.96,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10742,N'BOTTM',3,DATE '1997-11-14',DATE '1997-12-12',DATE '1997-11-18',243.73,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10743,N'AROUT',1,DATE '1997-11-17',DATE '1997-12-15',DATE '1997-11-21',23.72,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10744,N'VAFFE',6,DATE '1997-11-17',DATE '1997-12-15',DATE '1997-11-24',69.19,
	N'Vaffeljernet',N'Smagsloget 45',N'Arhus',
	NULL,N'8200',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10745,N'QUICK',9,DATE '1997-11-18',DATE '1997-12-16',DATE '1997-11-27',3.52,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10746,N'CHOPS',1,DATE '1997-11-19',DATE '1997-12-17',DATE '1997-11-21',31.43,
	N'Chop-suey Chinese',N'Hauptstr. 31',N'Bern',
	NULL,N'3012',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10747,N'PICCO',6,DATE '1997-11-19',DATE '1997-12-17',DATE '1997-11-26',117.33,
	N'Piccolo und mehr',N'Geislweg 14',N'Salzburg',
	NULL,N'5020',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10748,N'SAVEA',3,DATE '1997-11-20',DATE '1997-12-18',DATE '1997-11-28',232.55,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10749,N'ISLAT',4,DATE '1997-11-20',DATE '1997-12-18',DATE '1997-12-19',61.53,
	N'Island Trading',N'Garden House Crowther Way',N'Cowes',
	N'Isle of Wight',N'PO31 7PJ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10750,N'WARTH',9,DATE '1997-11-21',DATE '1997-12-19',DATE '1997-11-24',79.30,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10751,N'RICSU',3,DATE '1997-11-24',DATE '1997-12-22',DATE '1997-12-03',130.79,
	N'Richter Supermarkt',N'Starenweg 5',N'Geneve',
	NULL,N'1204',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10752,N'NORTS',2,DATE '1997-11-24',DATE '1997-12-22',DATE '1997-11-28',1.39,
	N'North/South',N'South House 300 Queensbridge',N'London',
	NULL,N'SW7 1RZ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10753,N'FRANS',3,DATE '1997-11-25',DATE '1997-12-23',DATE '1997-11-27',7.70,
	N'Franchi S.p.A.',N'Via Monte Bianco 34',N'Torino',
	NULL,N'10100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10754,N'MAGAA',6,DATE '1997-11-25',DATE '1997-12-23',DATE '1997-11-27',2.38,
	N'Magazzini Alimentari Riuniti',N'Via Ludovico il Moro 22',N'Bergamo',
	NULL,N'24100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10755,N'BONAP',4,DATE '1997-11-26',DATE '1997-12-24',DATE '1997-11-28',16.71,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10756,N'SPLIR',8,DATE '1997-11-27',DATE '1997-12-25',DATE '1997-12-02',73.21,
	N'Split Rail Beer & Ale',N'P.O. Box 555',N'Lander',
	N'WY',N'82520',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10757,N'SAVEA',6,DATE '1997-11-27',DATE '1997-12-25',DATE '1997-12-15',8.19,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10758,N'RICSU',3,DATE '1997-11-28',DATE '1997-12-26',DATE '1997-12-04',138.17,
	N'Richter Supermarkt',N'Starenweg 5',N'Geneve',
	NULL,N'1204',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10759,N'ANATR',3,DATE '1997-11-28',DATE '1997-12-26',DATE '1997-12-12',11.99,
	N'Ana Trujillo Emparedados y helados',N'Avda. de la Constitucion 2222',N'Mexico D.F.',
	NULL,N'05021',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10760,N'MAISD',4,DATE '1997-12-01',DATE '1997-12-29',DATE '1997-12-10',155.64,
	N'Maison Dewey',N'Rue Joseph-Bens 532',N'Bruxelles',
	NULL,N'B-1180',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10761,N'RATTC',5,DATE '1997-12-02',DATE '1997-12-30',DATE '1997-12-08',18.66,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10762,N'FOLKO',3,DATE '1997-12-02',DATE '1997-12-30',DATE '1997-12-09',328.74,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10763,N'FOLIG',3,DATE '1997-12-03',DATE '1997-12-31',DATE '1997-12-08',37.35,
	N'Folies gourmandes',N'184, chaussee de Tournai',N'Lille',
	NULL,N'59000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10764,N'ERNSH',6,DATE '1997-12-03',DATE '1997-12-31',DATE '1997-12-08',145.45,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10765,N'QUICK',3,DATE '1997-12-04',DATE '1998-01-01',DATE '1997-12-09',42.74,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10766,N'OTTIK',4,DATE '1997-12-05',DATE '1998-01-02',DATE '1997-12-09',157.55,
	N'Ottilies Kaseladen',N'Mehrheimerstr. 369',N'Koln',
	NULL,N'50739',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10767,N'SUPRD',4,DATE '1997-12-05',DATE '1998-01-02',DATE '1997-12-15',1.59,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10768,N'AROUT',3,DATE '1997-12-08',DATE '1998-01-05',DATE '1997-12-15',146.32,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10769,N'VAFFE',3,DATE '1997-12-08',DATE '1998-01-05',DATE '1997-12-12',65.06,
	N'Vaffeljernet',N'Smagsloget 45',N'Arhus',
	NULL,N'8200',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10770,N'HANAR',8,DATE '1997-12-09',DATE '1998-01-06',DATE '1997-12-17',5.32,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10771,N'ERNSH',9,DATE '1997-12-10',DATE '1998-01-07',DATE '1998-01-02',11.19,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10772,N'LEHMS',3,DATE '1997-12-10',DATE '1998-01-07',DATE '1997-12-19',91.28,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10773,N'ERNSH',1,DATE '1997-12-11',DATE '1998-01-08',DATE '1997-12-16',96.43,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10774,N'FOLKO',4,DATE '1997-12-11',DATE '1997-12-25',DATE '1997-12-12',48.20,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10775,N'THECR',7,DATE '1997-12-12',DATE '1998-01-09',DATE '1997-12-26',20.25,
	N'The Cracker Box',N'55 Grizzly Peak Rd.',N'Butte',
	N'MT',N'59801',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10776,N'ERNSH',1,DATE '1997-12-15',DATE '1998-01-12',DATE '1997-12-18',351.53,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10777,N'GOURL',7,DATE '1997-12-15',DATE '1997-12-29',DATE '1998-01-21',3.01,
	N'Gourmet Lanchonetes',N'Av. Brasil, 442',N'Campinas',
	N'SP',N'04876-786',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10778,N'BERGS',3,DATE '1997-12-16',DATE '1998-01-13',DATE '1997-12-24',6.79,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10779,N'MORGK',3,DATE '1997-12-16',DATE '1998-01-13',DATE '1998-01-14',58.13,
	N'Morgenstern Gesundkost',N'Heerstr. 22',N'Leipzig',
	NULL,N'04179',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10780,N'LILAS',2,DATE '1997-12-16',DATE '1997-12-30',DATE '1997-12-25',42.13,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10781,N'WARTH',2,DATE '1997-12-17',DATE '1998-01-14',DATE '1997-12-19',73.16,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10782,N'CACTU',9,DATE '1997-12-17',DATE '1998-01-14',DATE '1997-12-22',1.10,
	N'Cactus Comidas para llevar',N'Cerrito 333',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10783,N'HANAR',4,DATE '1997-12-18',DATE '1998-01-15',DATE '1997-12-19',124.98,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10784,N'MAGAA',4,DATE '1997-12-18',DATE '1998-01-15',DATE '1997-12-22',70.09,
	N'Magazzini Alimentari Riuniti',N'Via Ludovico il Moro 22',N'Bergamo',
	NULL,N'24100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10785,N'GROSR',1,DATE '1997-12-18',DATE '1998-01-15',DATE '1997-12-24',1.51,
	N'GROSELLA-Restaurante',N'5? Ave. Los Palos Grandes',N'Caracas',
	N'DF',N'1081',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10786,N'QUEEN',8,DATE '1997-12-19',DATE '1998-01-16',DATE '1997-12-23',110.87,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10787,N'LAMAI',2,DATE '1997-12-19',DATE '1998-01-02',DATE '1997-12-26',249.93,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10788,N'QUICK',1,DATE '1997-12-22',DATE '1998-01-19',DATE '1998-01-19',42.70,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10789,N'FOLIG',1,DATE '1997-12-22',DATE '1998-01-19',DATE '1997-12-31',100.60,
	N'Folies gourmandes',N'184, chaussee de Tournai',N'Lille',
	NULL,N'59000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10790,N'GOURL',6,DATE '1997-12-22',DATE '1998-01-19',DATE '1997-12-26',28.23,
	N'Gourmet Lanchonetes',N'Av. Brasil, 442',N'Campinas',
	N'SP',N'04876-786',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10791,N'FRANK',6,DATE '1997-12-23',DATE '1998-01-20',DATE '1998-01-01',16.85,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10792,N'WOLZA',1,DATE '1997-12-23',DATE '1998-01-20',DATE '1997-12-31',23.79,
	N'Wolski Zajazd',N'ul. Filtrowa 68',N'Warszawa',
	NULL,N'01-012',N'Poland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10793,N'AROUT',3,DATE '1997-12-24',DATE '1998-01-21',DATE '1998-01-08',4.52,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10794,N'QUEDE',6,DATE '1997-12-24',DATE '1998-01-21',DATE '1998-01-02',21.49,
	N'Que Delicia',N'Rua da Panificadora, 12',N'Rio de Janeiro',
	N'RJ',N'02389-673',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10795,N'ERNSH',8,DATE '1997-12-24',DATE '1998-01-21',DATE '1998-01-20',126.66,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10796,N'HILAA',3,DATE '1997-12-25',DATE '1998-01-22',DATE '1998-01-14',26.52,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10797,N'DRACD',7,DATE '1997-12-25',DATE '1998-01-22',DATE '1998-01-05',33.35,
	N'Drachenblut Delikatessen',N'Walserweg 21',N'Aachen',
	NULL,N'52066',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10798,N'ISLAT',2,DATE '1997-12-26',DATE '1998-01-23',DATE '1998-01-05',2.33,
	N'Island Trading',N'Garden House Crowther Way',N'Cowes',
	N'Isle of Wight',N'PO31 7PJ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10799,N'KOENE',9,DATE '1997-12-26',DATE '1998-02-06',DATE '1998-01-05',30.76,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10800,N'SEVES',1,DATE '1997-12-26',DATE '1998-01-23',DATE '1998-01-05',137.44,
	N'Seven Seas Imports',N'90 Wadhurst Rd.',N'London',
	NULL,N'OX15 4NB',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10801,N'BOLID',4,DATE '1997-12-29',DATE '1998-01-26',DATE '1997-12-31',97.09,
	N'Bolido Comidas preparadas',N'C/ Araquil, 67',N'Madrid',
	NULL,N'28023',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10802,N'SIMOB',4,DATE '1997-12-29',DATE '1998-01-26',DATE '1998-01-02',257.26,
	N'Simons bistro',N'Vinb?ltet 34',N'Kobenhavn',
	NULL,N'1734',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10803,N'WELLI',4,DATE '1997-12-30',DATE '1998-01-27',DATE '1998-01-06',55.23,
	N'Wellington Importadora',N'Rua do Mercado, 12',N'Resende',
	N'SP',N'08737-363',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10804,N'SEVES',6,DATE '1997-12-30',DATE '1998-01-27',DATE '1998-01-07',27.33,
	N'Seven Seas Imports',N'90 Wadhurst Rd.',N'London',
	NULL,N'OX15 4NB',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10805,N'THEBI',2,DATE '1997-12-30',DATE '1998-01-27',DATE '1998-01-09',237.34,
	N'The Big Cheese',N'89 Jefferson Way Suite 2',N'Portland',
	N'OR',N'97201',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10806,N'VICTE',3,DATE '1997-12-31',DATE '1998-01-28',DATE '1998-01-05',22.11,
	N'Victuailles en stock',N'2, rue du Commerce',N'Lyon',
	NULL,N'69004',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10807,N'FRANS',4,DATE '1997-12-31',DATE '1998-01-28',DATE '1998-01-30',1.36,
	N'Franchi S.p.A.',N'Via Monte Bianco 34',N'Torino',
	NULL,N'10100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10808,N'OLDWO',2,DATE '1998-01-01',DATE '1998-01-29',DATE '1998-01-09',45.53,
	N'Old World Delicatessen',N'2743 Bering St.',N'Anchorage',
	N'AK',N'99508',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10809,N'WELLI',7,DATE '1998-01-01',DATE '1998-01-29',DATE '1998-01-07',4.87,
	N'Wellington Importadora',N'Rua do Mercado, 12',N'Resende',
	N'SP',N'08737-363',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10810,N'LAUGB',2,DATE '1998-01-01',DATE '1998-01-29',DATE '1998-01-07',4.33,
	N'Laughing Bacchus Wine Cellars',N'2319 Elm St.',N'Vancouver',
	N'BC',N'V3F 2K1',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10811,N'LINOD',8,DATE '1998-01-02',DATE '1998-01-30',DATE '1998-01-08',31.22,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10812,N'REGGC',5,DATE '1998-01-02',DATE '1998-01-30',DATE '1998-01-12',59.78,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10813,N'RICAR',1,DATE '1998-01-05',DATE '1998-02-02',DATE '1998-01-09',47.38,
	N'Ricardo Adocicados',N'Av. Copacabana, 267',N'Rio de Janeiro',
	N'RJ',N'02389-890',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10814,N'VICTE',3,DATE '1998-01-05',DATE '1998-02-02',DATE '1998-01-14',130.94,
	N'Victuailles en stock',N'2, rue du Commerce',N'Lyon',
	NULL,N'69004',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10815,N'SAVEA',2,DATE '1998-01-05',DATE '1998-02-02',DATE '1998-01-14',14.62,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10816,N'GREAL',4,DATE '1998-01-06',DATE '1998-02-03',DATE '1998-02-04',719.78,
	N'Great Lakes Food Market',N'2732 Baker Blvd.',N'Eugene',
	N'OR',N'97403',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10817,N'KOENE',3,DATE '1998-01-06',DATE '1998-01-20',DATE '1998-01-13',306.07,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10818,N'MAGAA',7,DATE '1998-01-07',DATE '1998-02-04',DATE '1998-01-12',65.48,
	N'Magazzini Alimentari Riuniti',N'Via Ludovico il Moro 22',N'Bergamo',
	NULL,N'24100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10819,N'CACTU',2,DATE '1998-01-07',DATE '1998-02-04',DATE '1998-01-16',19.76,
	N'Cactus Comidas para llevar',N'Cerrito 333',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10820,N'RATTC',3,DATE '1998-01-07',DATE '1998-02-04',DATE '1998-01-13',37.52,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10821,N'SPLIR',1,DATE '1998-01-08',DATE '1998-02-05',DATE '1998-01-15',36.68,
	N'Split Rail Beer & Ale',N'P.O. Box 555',N'Lander',
	N'WY',N'82520',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10822,N'TRAIH',6,DATE '1998-01-08',DATE '1998-02-05',DATE '1998-01-16',7.00,
	N'Trail''s Head Gourmet Provisioners',N'722 DaVinci Blvd.',N'Kirkland',
	N'WA',N'98034',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10823,N'LILAS',5,DATE '1998-01-09',DATE '1998-02-06',DATE '1998-01-13',163.97,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10824,N'FOLKO',8,DATE '1998-01-09',DATE '1998-02-06',DATE '1998-01-30',1.23,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10825,N'DRACD',1,DATE '1998-01-09',DATE '1998-02-06',DATE '1998-01-14',79.25,
	N'Drachenblut Delikatessen',N'Walserweg 21',N'Aachen',
	NULL,N'52066',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10826,N'BLONP',6,DATE '1998-01-12',DATE '1998-02-09',DATE '1998-02-06',7.09,
	N'Blondel pere et fils',N'24, place Kleber',N'Strasbourg',
	NULL,N'67000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10827,N'BONAP',1,DATE '1998-01-12',DATE '1998-01-26',DATE '1998-02-06',63.54,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10828,N'RANCH',9,DATE '1998-01-13',DATE '1998-01-27',DATE '1998-02-04',90.85,
	N'Rancho grande',N'Av. del Libertador 900',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10829,N'ISLAT',9,DATE '1998-01-13',DATE '1998-02-10',DATE '1998-01-23',154.72,
	N'Island Trading',N'Garden House Crowther Way',N'Cowes',
	N'Isle of Wight',N'PO31 7PJ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10830,N'TRADH',4,DATE '1998-01-13',DATE '1998-02-24',DATE '1998-01-21',81.83,
	N'Tradicao Hipermercados',N'Av. Ines de Castro, 414',N'Sao Paulo',
	N'SP',N'05634-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10831,N'SANTG',3,DATE '1998-01-14',DATE '1998-02-11',DATE '1998-01-23',72.19,
	N'Sante Gourmet',N'Erling Skakkes gate 78',N'Stavern',
	NULL,N'4110',N'Norway');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10832,N'LAMAI',2,DATE '1998-01-14',DATE '1998-02-11',DATE '1998-01-19',43.26,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10833,N'OTTIK',6,DATE '1998-01-15',DATE '1998-02-12',DATE '1998-01-23',71.49,
	N'Ottilies Kaseladen',N'Mehrheimerstr. 369',N'Koln',
	NULL,N'50739',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10834,N'TRADH',1,DATE '1998-01-15',DATE '1998-02-12',DATE '1998-01-19',29.78,
	N'Tradicao Hipermercados',N'Av. Ines de Castro, 414',N'Sao Paulo',
	N'SP',N'05634-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10835,N'ALFKI',1,DATE '1998-01-15',DATE '1998-02-12',DATE '1998-01-21',69.53,
	N'Alfred''s Futterkiste',N'Obere Str. 57',N'Berlin',
	NULL,N'12209',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10836,N'ERNSH',7,DATE '1998-01-16',DATE '1998-02-13',DATE '1998-01-21',411.88,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10837,N'BERGS',9,DATE '1998-01-16',DATE '1998-02-13',DATE '1998-01-23',13.32,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10838,N'LINOD',3,DATE '1998-01-19',DATE '1998-02-16',DATE '1998-01-23',59.28,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10839,N'TRADH',3,DATE '1998-01-19',DATE '1998-02-16',DATE '1998-01-22',35.43,
	N'Tradicao Hipermercados',N'Av. Ines de Castro, 414',N'Sao Paulo',
	N'SP',N'05634-030',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10840,N'LINOD',4,DATE '1998-01-19',DATE '1998-03-02',DATE '1998-02-16',2.71,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10841,N'SUPRD',5,DATE '1998-01-20',DATE '1998-02-17',DATE '1998-01-29',424.30,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10842,N'TORTU',1,DATE '1998-01-20',DATE '1998-02-17',DATE '1998-01-29',54.42,
	N'Tortuga Restaurante',N'Avda. Azteca 123',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10843,N'VICTE',4,DATE '1998-01-21',DATE '1998-02-18',DATE '1998-01-26',9.26,
	N'Victuailles en stock',N'2, rue du Commerce',N'Lyon',
	NULL,N'69004',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10844,N'PICCO',8,DATE '1998-01-21',DATE '1998-02-18',DATE '1998-01-26',25.22,
	N'Piccolo und mehr',N'Geislweg 14',N'Salzburg',
	NULL,N'5020',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10845,N'QUICK',8,DATE '1998-01-21',DATE '1998-02-04',DATE '1998-01-30',212.98,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10846,N'SUPRD',2,DATE '1998-01-22',DATE '1998-03-05',DATE '1998-01-23',56.46,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10847,N'SAVEA',4,DATE '1998-01-22',DATE '1998-02-05',DATE '1998-02-10',487.57,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10848,N'CONSH',7,DATE '1998-01-23',DATE '1998-02-20',DATE '1998-01-29',38.24,
	N'Consolidated Holdings',N'Berkeley Gardens 12  Brewery',N'London',
	NULL,N'WX1 6LT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10849,N'KOENE',9,DATE '1998-01-23',DATE '1998-02-20',DATE '1998-01-30',0.56,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10850,N'VICTE',1,DATE '1998-01-23',DATE '1998-03-06',DATE '1998-01-30',49.19,
	N'Victuailles en stock',N'2, rue du Commerce',N'Lyon',
	NULL,N'69004',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10851,N'RICAR',5,DATE '1998-01-26',DATE '1998-02-23',DATE '1998-02-02',160.55,
	N'Ricardo Adocicados',N'Av. Copacabana, 267',N'Rio de Janeiro',
	N'RJ',N'02389-890',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10852,N'RATTC',8,DATE '1998-01-26',DATE '1998-02-09',DATE '1998-01-30',174.05,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10853,N'BLAUS',9,DATE '1998-01-27',DATE '1998-02-24',DATE '1998-02-03',53.83,
	N'Blauer See Delikatessen',N'Forsterstr. 57',N'Mannheim',
	NULL,N'68306',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10854,N'ERNSH',3,DATE '1998-01-27',DATE '1998-02-24',DATE '1998-02-05',100.22,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10855,N'OLDWO',3,DATE '1998-01-27',DATE '1998-02-24',DATE '1998-02-04',170.97,
	N'Old World Delicatessen',N'2743 Bering St.',N'Anchorage',
	N'AK',N'99508',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10856,N'ANTON',3,DATE '1998-01-28',DATE '1998-02-25',DATE '1998-02-10',58.43,
	N'Antonio Moreno Taqueria',N'Mataderos  2312',N'Mexico D.F.',
	NULL,N'05023',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10857,N'BERGS',8,DATE '1998-01-28',DATE '1998-02-25',DATE '1998-02-06',188.85,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10858,N'LACOR',2,DATE '1998-01-29',DATE '1998-02-26',DATE '1998-02-03',52.51,
	N'La corne d''abondance',N'67, avenue de l''Europe',N'Versailles',
	NULL,N'78000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10859,N'FRANK',1,DATE '1998-01-29',DATE '1998-02-26',DATE '1998-02-02',76.10,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10860,N'FRANR',3,DATE '1998-01-29',DATE '1998-02-26',DATE '1998-02-04',19.26,
	N'France restauration',N'54, rue Royale',N'Nantes',
	NULL,N'44000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10861,N'WHITC',4,DATE '1998-01-30',DATE '1998-02-27',DATE '1998-02-17',14.93,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10862,N'LEHMS',8,DATE '1998-01-30',DATE '1998-03-13',DATE '1998-02-02',53.23,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10863,N'HILAA',4,DATE '1998-02-02',DATE '1998-03-02',DATE '1998-02-17',30.26,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10864,N'AROUT',4,DATE '1998-02-02',DATE '1998-03-02',DATE '1998-02-09',3.04,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10865,N'QUICK',2,DATE '1998-02-02',DATE '1998-02-16',DATE '1998-02-12',348.14,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10866,N'BERGS',5,DATE '1998-02-03',DATE '1998-03-03',DATE '1998-02-12',109.11,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10867,N'LONEP',6,DATE '1998-02-03',DATE '1998-03-17',DATE '1998-02-11',1.93,
	N'Lonesome Pine Restaurant',N'89 Chiaroscuro Rd.',N'Portland',
	N'OR',N'97219',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10868,N'QUEEN',7,DATE '1998-02-04',DATE '1998-03-04',DATE '1998-02-23',191.27,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10869,N'SEVES',5,DATE '1998-02-04',DATE '1998-03-04',DATE '1998-02-09',143.28,
	N'Seven Seas Imports',N'90 Wadhurst Rd.',N'London',
	NULL,N'OX15 4NB',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10870,N'WOLZA',5,DATE '1998-02-04',DATE '1998-03-04',DATE '1998-02-13',12.04,
	N'Wolski Zajazd',N'ul. Filtrowa 68',N'Warszawa',
	NULL,N'01-012',N'Poland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10871,N'BONAP',9,DATE '1998-02-05',DATE '1998-03-05',DATE '1998-02-10',112.27,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10872,N'GODOS',5,DATE '1998-02-05',DATE '1998-03-05',DATE '1998-02-09',175.32,
	N'Godos Cocina Tipica',N'C/ Romero, 33',N'Sevilla',
	NULL,N'41101',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10873,N'WILMK',4,DATE '1998-02-06',DATE '1998-03-06',DATE '1998-02-09',0.82,
	N'Wilman Kala',N'Keskuskatu 45',N'Helsinki',
	NULL,N'21240',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10874,N'GODOS',5,DATE '1998-02-06',DATE '1998-03-06',DATE '1998-02-11',19.58,
	N'Godos Cocina Tipica',N'C/ Romero, 33',N'Sevilla',
	NULL,N'41101',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10875,N'BERGS',4,DATE '1998-02-06',DATE '1998-03-06',DATE '1998-03-03',32.37,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10876,N'BONAP',7,DATE '1998-02-09',DATE '1998-03-09',DATE '1998-02-12',60.42,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10877,N'RICAR',1,DATE '1998-02-09',DATE '1998-03-09',DATE '1998-02-19',38.06,
	N'Ricardo Adocicados',N'Av. Copacabana, 267',N'Rio de Janeiro',
	N'RJ',N'02389-890',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10878,N'QUICK',4,DATE '1998-02-10',DATE '1998-03-10',DATE '1998-02-12',46.69,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10879,N'WILMK',3,DATE '1998-02-10',DATE '1998-03-10',DATE '1998-02-12',8.50,
	N'Wilman Kala',N'Keskuskatu 45',N'Helsinki',
	NULL,N'21240',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10880,N'FOLKO',7,DATE '1998-02-10',DATE '1998-03-24',DATE '1998-02-18',88.01,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10881,N'CACTU',4,DATE '1998-02-11',DATE '1998-03-11',DATE '1998-02-18',2.84,
	N'Cactus Comidas para llevar',N'Cerrito 333',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10882,N'SAVEA',4,DATE '1998-02-11',DATE '1998-03-11',DATE '1998-02-20',23.10,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10883,N'LONEP',8,DATE '1998-02-12',DATE '1998-03-12',DATE '1998-02-20',0.53,
	N'Lonesome Pine Restaurant',N'89 Chiaroscuro Rd.',N'Portland',
	N'OR',N'97219',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10884,N'LETSS',4,DATE '1998-02-12',DATE '1998-03-12',DATE '1998-02-13',90.97,
	N'Let''s Stop N Shop',N'87 Polk St. Suite 5',N'San Francisco',
	N'CA',N'94117',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10885,N'SUPRD',6,DATE '1998-02-12',DATE '1998-03-12',DATE '1998-02-18',5.64,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10886,N'HANAR',1,DATE '1998-02-13',DATE '1998-03-13',DATE '1998-03-02',4.99,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10887,N'GALED',8,DATE '1998-02-13',DATE '1998-03-13',DATE '1998-02-16',1.25,
	N'Galeria del gastronomo',N'Rambla de Cataluna, 23',N'Barcelona',
	NULL,N'8022',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10888,N'GODOS',1,DATE '1998-02-16',DATE '1998-03-16',DATE '1998-02-23',51.87,
	N'Godos Cocina Tipica',N'C/ Romero, 33',N'Sevilla',
	NULL,N'41101',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10889,N'RATTC',9,DATE '1998-02-16',DATE '1998-03-16',DATE '1998-02-23',280.61,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10890,N'DUMON',7,DATE '1998-02-16',DATE '1998-03-16',DATE '1998-02-18',32.76,
	N'Du monde entier',N'67, rue des Cinquante Otages',N'Nantes',
	NULL,N'44000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10891,N'LEHMS',7,DATE '1998-02-17',DATE '1998-03-17',DATE '1998-02-19',20.37,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10892,N'MAISD',4,DATE '1998-02-17',DATE '1998-03-17',DATE '1998-02-19',120.27,
	N'Maison Dewey',N'Rue Joseph-Bens 532',N'Bruxelles',
	NULL,N'B-1180',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10893,N'KOENE',9,DATE '1998-02-18',DATE '1998-03-18',DATE '1998-02-20',77.78,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10894,N'SAVEA',1,DATE '1998-02-18',DATE '1998-03-18',DATE '1998-02-20',116.13,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10895,N'ERNSH',3,DATE '1998-02-18',DATE '1998-03-18',DATE '1998-02-23',162.75,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10896,N'MAISD',7,DATE '1998-02-19',DATE '1998-03-19',DATE '1998-02-27',32.45,
	N'Maison Dewey',N'Rue Joseph-Bens 532',N'Bruxelles',
	NULL,N'B-1180',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10897,N'HUNGO',3,DATE '1998-02-19',DATE '1998-03-19',DATE '1998-02-25',603.54,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10898,N'OCEAN',4,DATE '1998-02-20',DATE '1998-03-20',DATE '1998-03-06',1.27,
	N'Oceano Atlantico Ltda.',N'Ing. Gustavo Moncada 8585 Piso 20-A',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10899,N'LILAS',5,DATE '1998-02-20',DATE '1998-03-20',DATE '1998-02-26',1.21,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10900,N'WELLI',1,DATE '1998-02-20',DATE '1998-03-20',DATE '1998-03-04',1.66,
	N'Wellington Importadora',N'Rua do Mercado, 12',N'Resende',
	N'SP',N'08737-363',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10901,N'HILAA',4,DATE '1998-02-23',DATE '1998-03-23',DATE '1998-02-26',62.09,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10902,N'FOLKO',1,DATE '1998-02-23',DATE '1998-03-23',DATE '1998-03-03',44.15,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10903,N'HANAR',3,DATE '1998-02-24',DATE '1998-03-24',DATE '1998-03-04',36.71,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10904,N'WHITC',3,DATE '1998-02-24',DATE '1998-03-24',DATE '1998-02-27',162.95,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10905,N'WELLI',9,DATE '1998-02-24',DATE '1998-03-24',DATE '1998-03-06',13.72,
	N'Wellington Importadora',N'Rua do Mercado, 12',N'Resende',
	N'SP',N'08737-363',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10906,N'WOLZA',4,DATE '1998-02-25',DATE '1998-03-11',DATE '1998-03-03',26.29,
	N'Wolski Zajazd',N'ul. Filtrowa 68',N'Warszawa',
	NULL,N'01-012',N'Poland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10907,N'SPECD',6,DATE '1998-02-25',DATE '1998-03-25',DATE '1998-02-27',9.19,
	N'Specialites du monde',N'25, rue Lauriston',N'Paris',
	NULL,N'75016',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10908,N'REGGC',4,DATE '1998-02-26',DATE '1998-03-26',DATE '1998-03-06',32.96,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10909,N'SANTG',1,DATE '1998-02-26',DATE '1998-03-26',DATE '1998-03-10',53.05,
	N'Sante Gourmet',N'Erling Skakkes gate 78',N'Stavern',
	NULL,N'4110',N'Norway');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10910,N'WILMK',1,DATE '1998-02-26',DATE '1998-03-26',DATE '1998-03-04',38.11,
	N'Wilman Kala',N'Keskuskatu 45',N'Helsinki',
	NULL,N'21240',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10911,N'GODOS',3,DATE '1998-02-26',DATE '1998-03-26',DATE '1998-03-05',38.19,
	N'Godos Cocina Tipica',N'C/ Romero, 33',N'Sevilla',
	NULL,N'41101',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10912,N'HUNGO',2,DATE '1998-02-26',DATE '1998-03-26',DATE '1998-03-18',580.91,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10913,N'QUEEN',4,DATE '1998-02-26',DATE '1998-03-26',DATE '1998-03-04',33.05,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10914,N'QUEEN',6,DATE '1998-02-27',DATE '1998-03-27',DATE '1998-03-02',21.19,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10915,N'TORTU',2,DATE '1998-02-27',DATE '1998-03-27',DATE '1998-03-02',3.51,
	N'Tortuga Restaurante',N'Avda. Azteca 123',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10916,N'RANCH',1,DATE '1998-02-27',DATE '1998-03-27',DATE '1998-03-09',63.77,
	N'Rancho grande',N'Av. del Libertador 900',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10917,N'ROMEY',4,DATE '1998-03-02',DATE '1998-03-30',DATE '1998-03-11',8.29,
	N'Romero y tomillo',N'Gran Via, 1',N'Madrid',
	NULL,N'28001',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10918,N'BOTTM',3,DATE '1998-03-02',DATE '1998-03-30',DATE '1998-03-11',48.83,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10919,N'LINOD',2,DATE '1998-03-02',DATE '1998-03-30',DATE '1998-03-04',19.80,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10920,N'AROUT',4,DATE '1998-03-03',DATE '1998-03-31',DATE '1998-03-09',29.61,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10921,N'VAFFE',1,DATE '1998-03-03',DATE '1998-04-14',DATE '1998-03-09',176.48,
	N'Vaffeljernet',N'Smagsloget 45',N'Arhus',
	NULL,N'8200',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10922,N'HANAR',5,DATE '1998-03-03',DATE '1998-03-31',DATE '1998-03-05',62.74,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10923,N'LAMAI',7,DATE '1998-03-03',DATE '1998-04-14',DATE '1998-03-13',68.26,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10924,N'BERGS',3,DATE '1998-03-04',DATE '1998-04-01',DATE '1998-04-08',151.52,
	N'Berglunds snabbkop',N'Berguvsvagen  8',N'Lulea',
	NULL,N'S-958 22',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10925,N'HANAR',3,DATE '1998-03-04',DATE '1998-04-01',DATE '1998-03-13',2.27,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10926,N'ANATR',4,DATE '1998-03-04',DATE '1998-04-01',DATE '1998-03-11',39.92,
	N'Ana Trujillo Emparedados y helados',N'Avda. de la Constitucion 2222',N'Mexico D.F.',
	NULL,N'05021',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10927,N'LACOR',4,DATE '1998-03-05',DATE '1998-04-02',DATE '1998-04-08',19.79,
	N'La corne d''abondance',N'67, avenue de l''Europe',N'Versailles',
	NULL,N'78000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10928,N'GALED',1,DATE '1998-03-05',DATE '1998-04-02',DATE '1998-03-18',1.36,
	N'Galeria del gastronomo',N'Rambla de Cataluna, 23',N'Barcelona',
	NULL,N'8022',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10929,N'FRANK',6,DATE '1998-03-05',DATE '1998-04-02',DATE '1998-03-12',33.93,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10930,N'SUPRD',4,DATE '1998-03-06',DATE '1998-04-17',DATE '1998-03-18',15.55,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10931,N'RICSU',4,DATE '1998-03-06',DATE '1998-03-20',DATE '1998-03-19',13.60,
	N'Richter Supermarkt',N'Starenweg 5',N'Geneve',
	NULL,N'1204',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10932,N'BONAP',8,DATE '1998-03-06',DATE '1998-04-03',DATE '1998-03-24',134.64,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10933,N'ISLAT',6,DATE '1998-03-06',DATE '1998-04-03',DATE '1998-03-16',54.15,
	N'Island Trading',N'Garden House Crowther Way',N'Cowes',
	N'Isle of Wight',N'PO31 7PJ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10934,N'LEHMS',3,DATE '1998-03-09',DATE '1998-04-06',DATE '1998-03-12',32.01,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10935,N'WELLI',4,DATE '1998-03-09',DATE '1998-04-06',DATE '1998-03-18',47.59,
	N'Wellington Importadora',N'Rua do Mercado, 12',N'Resende',
	N'SP',N'08737-363',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10936,N'GREAL',3,DATE '1998-03-09',DATE '1998-04-06',DATE '1998-03-18',33.68,
	N'Great Lakes Food Market',N'2732 Baker Blvd.',N'Eugene',
	N'OR',N'97403',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10937,N'CACTU',7,DATE '1998-03-10',DATE '1998-03-24',DATE '1998-03-13',31.51,
	N'Cactus Comidas para llevar',N'Cerrito 333',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10938,N'QUICK',3,DATE '1998-03-10',DATE '1998-04-07',DATE '1998-03-16',31.89,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10939,N'MAGAA',2,DATE '1998-03-10',DATE '1998-04-07',DATE '1998-03-13',76.33,
	N'Magazzini Alimentari Riuniti',N'Via Ludovico il Moro 22',N'Bergamo',
	NULL,N'24100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10940,N'BONAP',8,DATE '1998-03-11',DATE '1998-04-08',DATE '1998-03-23',19.77,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10941,N'SAVEA',7,DATE '1998-03-11',DATE '1998-04-08',DATE '1998-03-20',400.81,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10942,N'REGGC',9,DATE '1998-03-11',DATE '1998-04-08',DATE '1998-03-18',17.95,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10943,N'BSBEV',4,DATE '1998-03-11',DATE '1998-04-08',DATE '1998-03-19',2.17,
	N'B''s Beverages',N'Fauntleroy Circus',N'London',
	NULL,N'EC2 5NT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10944,N'BOTTM',6,DATE '1998-03-12',DATE '1998-03-26',DATE '1998-03-13',52.92,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10945,N'MORGK',4,DATE '1998-03-12',DATE '1998-04-09',DATE '1998-03-18',10.22,
	N'Morgenstern Gesundkost',N'Heerstr. 22',N'Leipzig',
	NULL,N'04179',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10946,N'VAFFE',1,DATE '1998-03-12',DATE '1998-04-09',DATE '1998-03-19',27.20,
	N'Vaffeljernet',N'Smagsloget 45',N'Arhus',
	NULL,N'8200',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10947,N'BSBEV',3,DATE '1998-03-13',DATE '1998-04-10',DATE '1998-03-16',3.26,
	N'B''s Beverages',N'Fauntleroy Circus',N'London',
	NULL,N'EC2 5NT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10948,N'GODOS',3,DATE '1998-03-13',DATE '1998-04-10',DATE '1998-03-19',23.39,
	N'Godos Cocina Tipica',N'C/ Romero, 33',N'Sevilla',
	NULL,N'41101',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10949,N'BOTTM',2,DATE '1998-03-13',DATE '1998-04-10',DATE '1998-03-17',74.44,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10950,N'MAGAA',1,DATE '1998-03-16',DATE '1998-04-13',DATE '1998-03-23',2.50,
	N'Magazzini Alimentari Riuniti',N'Via Ludovico il Moro 22',N'Bergamo',
	NULL,N'24100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10951,N'RICSU',9,DATE '1998-03-16',DATE '1998-04-27',DATE '1998-04-07',30.85,
	N'Richter Supermarkt',N'Starenweg 5',N'Geneve',
	NULL,N'1204',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10952,N'ALFKI',1,DATE '1998-03-16',DATE '1998-04-27',DATE '1998-03-24',40.42,
	N'Alfred''s Futterkiste',N'Obere Str. 57',N'Berlin',
	NULL,N'12209',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10953,N'AROUT',9,DATE '1998-03-16',DATE '1998-03-30',DATE '1998-03-25',23.72,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10954,N'LINOD',5,DATE '1998-03-17',DATE '1998-04-28',DATE '1998-03-20',27.91,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10955,N'FOLKO',8,DATE '1998-03-17',DATE '1998-04-14',DATE '1998-03-20',3.26,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10956,N'BLAUS',6,DATE '1998-03-17',DATE '1998-04-28',DATE '1998-03-20',44.65,
	N'Blauer See Delikatessen',N'Forsterstr. 57',N'Mannheim',
	NULL,N'68306',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10957,N'HILAA',8,DATE '1998-03-18',DATE '1998-04-15',DATE '1998-03-27',105.36,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10958,N'OCEAN',7,DATE '1998-03-18',DATE '1998-04-15',DATE '1998-03-27',49.56,
	N'Oceano Atlantico Ltda.',N'Ing. Gustavo Moncada 8585 Piso 20-A',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10959,N'GOURL',6,DATE '1998-03-18',DATE '1998-04-29',DATE '1998-03-23',4.98,
	N'Gourmet Lanchonetes',N'Av. Brasil, 442',N'Campinas',
	N'SP',N'04876-786',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10960,N'HILAA',3,DATE '1998-03-19',DATE '1998-04-02',DATE '1998-04-08',2.08,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10961,N'QUEEN',8,DATE '1998-03-19',DATE '1998-04-16',DATE '1998-03-30',104.47,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10962,N'QUICK',8,DATE '1998-03-19',DATE '1998-04-16',DATE '1998-03-23',275.79,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10963,N'FURIB',9,DATE '1998-03-19',DATE '1998-04-16',DATE '1998-03-26',2.70,
	N'Furia Bacalhau e Frutos do Mar',N'Jardim das rosas n. 32',N'Lisboa',
	NULL,N'1675',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10964,N'SPECD',3,DATE '1998-03-20',DATE '1998-04-17',DATE '1998-03-24',87.38,
	N'Specialites du monde',N'25, rue Lauriston',N'Paris',
	NULL,N'75016',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10965,N'OLDWO',6,DATE '1998-03-20',DATE '1998-04-17',DATE '1998-03-30',144.38,
	N'Old World Delicatessen',N'2743 Bering St.',N'Anchorage',
	N'AK',N'99508',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10966,N'CHOPS',4,DATE '1998-03-20',DATE '1998-04-17',DATE '1998-04-08',27.19,
	N'Chop-suey Chinese',N'Hauptstr. 31',N'Bern',
	NULL,N'3012',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10967,N'TOMSP',2,DATE '1998-03-23',DATE '1998-04-20',DATE '1998-04-02',62.22,
	N'Toms Spezialitaten',N'Luisenstr. 48',N'Munster',
	NULL,N'44087',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10968,N'ERNSH',1,DATE '1998-03-23',DATE '1998-04-20',DATE '1998-04-01',74.60,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10969,N'COMMI',1,DATE '1998-03-23',DATE '1998-04-20',DATE '1998-03-30',0.21,
	N'Comercio Mineiro',N'Av. dos Lusiadas, 23',N'Sao Paulo',
	N'SP',N'05432-043',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10970,N'BOLID',9,DATE '1998-03-24',DATE '1998-04-07',DATE '1998-04-24',16.16,
	N'Bolido Comidas preparadas',N'C/ Araquil, 67',N'Madrid',
	NULL,N'28023',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10971,N'FRANR',2,DATE '1998-03-24',DATE '1998-04-21',DATE '1998-04-02',121.82,
	N'France restauration',N'54, rue Royale',N'Nantes',
	NULL,N'44000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10972,N'LACOR',4,DATE '1998-03-24',DATE '1998-04-21',DATE '1998-03-26',0.02,
	N'La corne d''abondance',N'67, avenue de l''Europe',N'Versailles',
	NULL,N'78000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10973,N'LACOR',6,DATE '1998-03-24',DATE '1998-04-21',DATE '1998-03-27',15.17,
	N'La corne d''abondance',N'67, avenue de l''Europe',N'Versailles',
	NULL,N'78000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10974,N'SPLIR',3,DATE '1998-03-25',DATE '1998-04-08',DATE '1998-04-03',12.96,
	N'Split Rail Beer & Ale',N'P.O. Box 555',N'Lander',
	N'WY',N'82520',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10975,N'BOTTM',1,DATE '1998-03-25',DATE '1998-04-22',DATE '1998-03-27',32.27,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10976,N'HILAA',1,DATE '1998-03-25',DATE '1998-05-06',DATE '1998-04-03',37.97,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10977,N'FOLKO',8,DATE '1998-03-26',DATE '1998-04-23',DATE '1998-04-10',208.50,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10978,N'MAISD',9,DATE '1998-03-26',DATE '1998-04-23',DATE '1998-04-23',32.82,
	N'Maison Dewey',N'Rue Joseph-Bens 532',N'Bruxelles',
	NULL,N'B-1180',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10979,N'ERNSH',8,DATE '1998-03-26',DATE '1998-04-23',DATE '1998-03-31',353.07,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10980,N'FOLKO',4,DATE '1998-03-27',DATE '1998-05-08',DATE '1998-04-17',1.26,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10981,N'HANAR',1,DATE '1998-03-27',DATE '1998-04-24',DATE '1998-04-02',193.37,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10982,N'BOTTM',2,DATE '1998-03-27',DATE '1998-04-24',DATE '1998-04-08',14.01,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10983,N'SAVEA',2,DATE '1998-03-27',DATE '1998-04-24',DATE '1998-04-06',657.54,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10984,N'SAVEA',1,DATE '1998-03-30',DATE '1998-04-27',DATE '1998-04-03',211.22,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10985,N'HUNGO',2,DATE '1998-03-30',DATE '1998-04-27',DATE '1998-04-02',91.51,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10986,N'OCEAN',8,DATE '1998-03-30',DATE '1998-04-27',DATE '1998-04-21',217.86,
	N'Oceano Atlantico Ltda.',N'Ing. Gustavo Moncada 8585 Piso 20-A',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10987,N'EASTC',8,DATE '1998-03-31',DATE '1998-04-28',DATE '1998-04-06',185.48,
	N'Eastern Connection',N'35 King George',N'London',
	NULL,N'WX3 6FW',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10988,N'RATTC',3,DATE '1998-03-31',DATE '1998-04-28',DATE '1998-04-10',61.14,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10989,N'QUEDE',2,DATE '1998-03-31',DATE '1998-04-28',DATE '1998-04-02',34.76,
	N'Que Delicia',N'Rua da Panificadora, 12',N'Rio de Janeiro',
	N'RJ',N'02389-673',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10990,N'ERNSH',2,DATE '1998-04-01',DATE '1998-05-13',DATE '1998-04-07',117.61,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10991,N'QUICK',1,DATE '1998-04-01',DATE '1998-04-29',DATE '1998-04-07',38.51,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10992,N'THEBI',1,DATE '1998-04-01',DATE '1998-04-29',DATE '1998-04-03',4.27,
	N'The Big Cheese',N'89 Jefferson Way Suite 2',N'Portland',
	N'OR',N'97201',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10993,N'FOLKO',7,DATE '1998-04-01',DATE '1998-04-29',DATE '1998-04-10',8.81,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10994,N'VAFFE',2,DATE '1998-04-02',DATE '1998-04-16',DATE '1998-04-09',65.53,
	N'Vaffeljernet',N'Smagsloget 45',N'Arhus',
	NULL,N'8200',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10995,N'PERIC',1,DATE '1998-04-02',DATE '1998-04-30',DATE '1998-04-06',46.00,
	N'Pericles Comidas clasicas',N'Calle Dr. Jorge Cash 321',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10996,N'QUICK',4,DATE '1998-04-02',DATE '1998-04-30',DATE '1998-04-10',1.12,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10997,N'LILAS',8,DATE '1998-04-03',DATE '1998-05-15',DATE '1998-04-13',73.91,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10998,N'WOLZA',8,DATE '1998-04-03',DATE '1998-04-17',DATE '1998-04-17',20.31,
	N'Wolski Zajazd',N'ul. Filtrowa 68',N'Warszawa',
	NULL,N'01-012',N'Poland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (10999,N'OTTIK',6,DATE '1998-04-03',DATE '1998-05-01',DATE '1998-04-10',96.35,
	N'Ottilies Kaseladen',N'Mehrheimerstr. 369',N'Koln',
	NULL,N'50739',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11000,N'RATTC',2,DATE '1998-04-06',DATE '1998-05-04',DATE '1998-04-14',55.12,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11001,N'FOLKO',2,DATE '1998-04-06',DATE '1998-05-04',DATE '1998-04-14',197.30,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11002,N'SAVEA',4,DATE '1998-04-06',DATE '1998-05-04',DATE '1998-04-16',141.16,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11003,N'THECR',3,DATE '1998-04-06',DATE '1998-05-04',DATE '1998-04-08',14.91,
	N'The Cracker Box',N'55 Grizzly Peak Rd.',N'Butte',
	N'MT',N'59801',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11004,N'MAISD',3,DATE '1998-04-07',DATE '1998-05-05',DATE '1998-04-20',44.84,
	N'Maison Dewey',N'Rue Joseph-Bens 532',N'Bruxelles',
	NULL,N'B-1180',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11005,N'WILMK',2,DATE '1998-04-07',DATE '1998-05-05',DATE '1998-04-10',0.75,
	N'Wilman Kala',N'Keskuskatu 45',N'Helsinki',
	NULL,N'21240',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11006,N'GREAL',3,DATE '1998-04-07',DATE '1998-05-05',DATE '1998-04-15',25.19,
	N'Great Lakes Food Market',N'2732 Baker Blvd.',N'Eugene',
	N'OR',N'97403',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11007,N'PRINI',8,DATE '1998-04-08',DATE '1998-05-06',DATE '1998-04-13',202.24,
	N'Princesa Isabel Vinhos',N'Estrada da saude n. 58',N'Lisboa',
	NULL,N'1756',N'Portugal');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11008,N'ERNSH',7,DATE '1998-04-08',DATE '1998-05-06',NULL,79.46,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11009,N'GODOS',2,DATE '1998-04-08',DATE '1998-05-06',DATE '1998-04-10',59.11,
	N'Godos Cocina Tipica',N'C/ Romero, 33',N'Sevilla',
	NULL,N'41101',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11010,N'REGGC',2,DATE '1998-04-09',DATE '1998-05-07',DATE '1998-04-21',28.71,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11011,N'ALFKI',3,DATE '1998-04-09',DATE '1998-05-07',DATE '1998-04-13',1.21,
	N'Alfred''s Futterkiste',N'Obere Str. 57',N'Berlin',
	NULL,N'12209',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11012,N'FRANK',1,DATE '1998-04-09',DATE '1998-04-23',DATE '1998-04-17',242.95,
	N'Frankenversand',N'Berliner Platz 43',N'Munchen',
	NULL,N'80805',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11013,N'ROMEY',2,DATE '1998-04-09',DATE '1998-05-07',DATE '1998-04-10',32.99,
	N'Romero y tomillo',N'Gran Via, 1',N'Madrid',
	NULL,N'28001',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11014,N'LINOD',2,DATE '1998-04-10',DATE '1998-05-08',DATE '1998-04-15',23.60,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11015,N'SANTG',2,DATE '1998-04-10',DATE '1998-04-24',DATE '1998-04-20',4.62,
	N'Sante Gourmet',N'Erling Skakkes gate 78',N'Stavern',
	NULL,N'4110',N'Norway');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11016,N'AROUT',9,DATE '1998-04-10',DATE '1998-05-08',DATE '1998-04-13',33.80,
	N'Around the Horn',N'Brook Farm Stratford St. Mary',N'Colchester',
	N'Essex',N'CO7 6JX',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11017,N'ERNSH',9,DATE '1998-04-13',DATE '1998-05-11',DATE '1998-04-20',754.26,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11018,N'LONEP',4,DATE '1998-04-13',DATE '1998-05-11',DATE '1998-04-16',11.65,
	N'Lonesome Pine Restaurant',N'89 Chiaroscuro Rd.',N'Portland',
	N'OR',N'97219',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11019,N'RANCH',6,DATE '1998-04-13',DATE '1998-05-11',NULL,3.17,
	N'Rancho grande',N'Av. del Libertador 900',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11020,N'OTTIK',2,DATE '1998-04-14',DATE '1998-05-12',DATE '1998-04-16',43.30,
	N'Ottilies Kaseladen',N'Mehrheimerstr. 369',N'Koln',
	NULL,N'50739',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11021,N'QUICK',3,DATE '1998-04-14',DATE '1998-05-12',DATE '1998-04-21',297.18,
	N'QUICK-Stop',N'Taucherstra?e 10',N'Cunewalde',
	NULL,N'01307',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11022,N'HANAR',9,DATE '1998-04-14',DATE '1998-05-12',DATE '1998-05-04',6.27,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11023,N'BSBEV',1,DATE '1998-04-14',DATE '1998-04-28',DATE '1998-04-24',123.83,
	N'B''s Beverages',N'Fauntleroy Circus',N'London',
	NULL,N'EC2 5NT',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11024,N'EASTC',4,DATE '1998-04-15',DATE '1998-05-13',DATE '1998-04-20',74.36,
	N'Eastern Connection',N'35 King George',N'London',
	NULL,N'WX3 6FW',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11025,N'WARTH',6,DATE '1998-04-15',DATE '1998-05-13',DATE '1998-04-24',29.17,
	N'Wartian Herkku',N'Torikatu 38',N'Oulu',
	NULL,N'90110',N'Finland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11026,N'FRANS',4,DATE '1998-04-15',DATE '1998-05-13',DATE '1998-04-28',47.09,
	N'Franchi S.p.A.',N'Via Monte Bianco 34',N'Torino',
	NULL,N'10100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11027,N'BOTTM',1,DATE '1998-04-16',DATE '1998-05-14',DATE '1998-04-20',52.52,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11028,N'KOENE',2,DATE '1998-04-16',DATE '1998-05-14',DATE '1998-04-22',29.59,
	N'Koniglich Essen',N'Maubelstr. 90',N'Brandenburg',
	NULL,N'14776',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11029,N'CHOPS',4,DATE '1998-04-16',DATE '1998-05-14',DATE '1998-04-27',47.84,
	N'Chop-suey Chinese',N'Hauptstr. 31',N'Bern',
	NULL,N'3012',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11030,N'SAVEA',7,DATE '1998-04-17',DATE '1998-05-15',DATE '1998-04-27',830.75,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11031,N'SAVEA',6,DATE '1998-04-17',DATE '1998-05-15',DATE '1998-04-24',227.22,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11032,N'WHITC',2,DATE '1998-04-17',DATE '1998-05-15',DATE '1998-04-23',606.19,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11033,N'RICSU',7,DATE '1998-04-17',DATE '1998-05-15',DATE '1998-04-23',84.74,
	N'Richter Supermarkt',N'Starenweg 5',N'Geneve',
	NULL,N'1204',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11034,N'OLDWO',8,DATE '1998-04-20',DATE '1998-06-01',DATE '1998-04-27',40.32,
	N'Old World Delicatessen',N'2743 Bering St.',N'Anchorage',
	N'AK',N'99508',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11035,N'SUPRD',2,DATE '1998-04-20',DATE '1998-05-18',DATE '1998-04-24',0.17,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11036,N'DRACD',8,DATE '1998-04-20',DATE '1998-05-18',DATE '1998-04-22',149.47,
	N'Drachenblut Delikatessen',N'Walserweg 21',N'Aachen',
	NULL,N'52066',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11037,N'GODOS',7,DATE '1998-04-21',DATE '1998-05-19',DATE '1998-04-27',3.20,
	N'Godos Cocina Tipica',N'C/ Romero, 33',N'Sevilla',
	NULL,N'41101',N'Spain');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11038,N'SUPRD',1,DATE '1998-04-21',DATE '1998-05-19',DATE '1998-04-30',29.59,
	N'Supremes delices',N'Boulevard Tirou, 255',N'Charleroi',
	NULL,N'B-6000',N'Belgium');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11039,N'LINOD',1,DATE '1998-04-21',DATE '1998-05-19',NULL,65.00,
	N'LINO-Delicateses',N'Ave. 5 de Mayo Porlamar',N'I. de Margarita',
	N'Nueva Esparta',N'4980',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11040,N'GREAL',4,DATE '1998-04-22',DATE '1998-05-20',NULL,18.84,
	N'Great Lakes Food Market',N'2732 Baker Blvd.',N'Eugene',
	N'OR',N'97403',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11041,N'CHOPS',3,DATE '1998-04-22',DATE '1998-05-20',DATE '1998-04-28',48.22,
	N'Chop-suey Chinese',N'Hauptstr. 31',N'Bern',
	NULL,N'3012',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11042,N'COMMI',2,DATE '1998-04-22',DATE '1998-05-06',DATE '1998-05-01',29.99,
	N'Comercio Mineiro',N'Av. dos Lusiadas, 23',N'Sao Paulo',
	N'SP',N'05432-043',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11043,N'SPECD',5,DATE '1998-04-22',DATE '1998-05-20',DATE '1998-04-29',8.80,
	N'Specialites du monde',N'25, rue Lauriston',N'Paris',
	NULL,N'75016',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11044,N'WOLZA',4,DATE '1998-04-23',DATE '1998-05-21',DATE '1998-05-01',8.72,
	N'Wolski Zajazd',N'ul. Filtrowa 68',N'Warszawa',
	NULL,N'01-012',N'Poland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11045,N'BOTTM',6,DATE '1998-04-23',DATE '1998-05-21',NULL,70.58,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11046,N'WANDK',8,DATE '1998-04-23',DATE '1998-05-21',DATE '1998-04-24',71.64,
	N'Die Wandernde Kuh',N'Adenauerallee 900',N'Stuttgart',
	NULL,N'70563',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11047,N'EASTC',7,DATE '1998-04-24',DATE '1998-05-22',DATE '1998-05-01',46.62,
	N'Eastern Connection',N'35 King George',N'London',
	NULL,N'WX3 6FW',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11048,N'BOTTM',7,DATE '1998-04-24',DATE '1998-05-22',DATE '1998-04-30',24.12,
	N'Bottom-Dollar Markets',N'23 Tsawassen Blvd.',N'Tsawassen',
	N'BC',N'T2F 8M4',N'Canada');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11049,N'GOURL',3,DATE '1998-04-24',DATE '1998-05-22',DATE '1998-05-04',8.34,
	N'Gourmet Lanchonetes',N'Av. Brasil, 442',N'Campinas',
	N'SP',N'04876-786',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11050,N'FOLKO',8,DATE '1998-04-27',DATE '1998-05-25',DATE '1998-05-05',59.41,
	N'Folk och fa HB',N'Akergatan 24',N'Bracke',
	NULL,N'S-844 67',N'Sweden');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11051,N'LAMAI',7,DATE '1998-04-27',DATE '1998-05-25',NULL,2.79,
	N'La maison d''Asie',N'1 rue Alsace-Lorraine',N'Toulouse',
	NULL,N'31000',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11052,N'HANAR',3,DATE '1998-04-27',DATE '1998-05-25',DATE '1998-05-01',67.26,
	N'Hanari Carnes',N'Rua do Paco, 67',N'Rio de Janeiro',
	N'RJ',N'05454-876',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11053,N'PICCO',2,DATE '1998-04-27',DATE '1998-05-25',DATE '1998-04-29',53.05,
	N'Piccolo und mehr',N'Geislweg 14',N'Salzburg',
	NULL,N'5020',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11054,N'CACTU',8,DATE '1998-04-28',DATE '1998-05-26',NULL,0.33,
	N'Cactus Comidas para llevar',N'Cerrito 333',N'Buenos Aires',
	NULL,N'1010',N'Argentina');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11055,N'HILAA',7,DATE '1998-04-28',DATE '1998-05-26',DATE '1998-05-05',120.92,
	N'HILARION-Abastos',N'Carrera 22 con Ave. Carlos Soublette #8-35',N'San Cristobal',
	N'Tachira',N'5022',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11056,N'EASTC',8,DATE '1998-04-28',DATE '1998-05-12',DATE '1998-05-01',278.96,
	N'Eastern Connection',N'35 King George',N'London',
	NULL,N'WX3 6FW',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11057,N'NORTS',3,DATE '1998-04-29',DATE '1998-05-27',DATE '1998-05-01',4.13,
	N'North/South',N'South House 300 Queensbridge',N'London',
	NULL,N'SW7 1RZ',N'UK');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11058,N'BLAUS',9,DATE '1998-04-29',DATE '1998-05-27',NULL,31.14,
	N'Blauer See Delikatessen',N'Forsterstr. 57',N'Mannheim',
	NULL,N'68306',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11059,N'RICAR',2,DATE '1998-04-29',DATE '1998-06-10',NULL,85.80,
	N'Ricardo Adocicados',N'Av. Copacabana, 267',N'Rio de Janeiro',
	N'RJ',N'02389-890',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11060,N'FRANS',2,DATE '1998-04-30',DATE '1998-05-28',DATE '1998-05-04',10.98,
	N'Franchi S.p.A.',N'Via Monte Bianco 34',N'Torino',
	NULL,N'10100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11061,N'GREAL',4,DATE '1998-04-30',DATE '1998-06-11',NULL,14.01,
	N'Great Lakes Food Market',N'2732 Baker Blvd.',N'Eugene',
	N'OR',N'97403',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11062,N'REGGC',4,DATE '1998-04-30',DATE '1998-05-28',NULL,29.93,
	N'Reggiani Caseifici',N'Strada Provinciale 124',N'Reggio Emilia',
	NULL,N'42100',N'Italy');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11063,N'HUNGO',3,DATE '1998-04-30',DATE '1998-05-28',DATE '1998-05-06',81.73,
	N'Hungry Owl All-Night Grocers',N'8 Johnstown Road',N'Cork',
	N'Co. Cork',NULL,N'Ireland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11064,N'SAVEA',1,DATE '1998-05-01',DATE '1998-05-29',DATE '1998-05-04',30.09,
	N'Save-a-lot Markets',N'187 Suffolk Ln.',N'Boise',
	N'ID',N'83720',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11065,N'LILAS',8,DATE '1998-05-01',DATE '1998-05-29',NULL,12.91,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11066,N'WHITC',7,DATE '1998-05-01',DATE '1998-05-29',DATE '1998-05-04',44.72,
	N'White Clover Markets',N'1029 - 12th Ave. S.',N'Seattle',
	N'WA',N'98124',N'USA');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11067,N'DRACD',1,DATE '1998-05-04',DATE '1998-05-18',DATE '1998-05-06',7.98,
	N'Drachenblut Delikatessen',N'Walserweg 21',N'Aachen',
	NULL,N'52066',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11068,N'QUEEN',8,DATE '1998-05-04',DATE '1998-06-01',NULL,81.75,
	N'Queen Cozinha',N'Alameda dos Canarios, 891',N'Sao Paulo',
	N'SP',N'05487-020',N'Brazil');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11069,N'TORTU',1,DATE '1998-05-04',DATE '1998-06-01',DATE '1998-05-06',15.67,
	N'Tortuga Restaurante',N'Avda. Azteca 123',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11070,N'LEHMS',2,DATE '1998-05-05',DATE '1998-06-02',NULL,136.00,
	N'Lehmanns Marktstand',N'Magazinweg 7',N'Frankfurt a.M.',
	NULL,N'60528',N'Germany');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11071,N'LILAS',1,DATE '1998-05-05',DATE '1998-06-02',NULL,0.93,
	N'LILA-Supermercado',N'Carrera 52 con Ave. Bolivar #65-98 Llano Largo',N'Barquisimeto',
	N'Lara',N'3508',N'Venezuela');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11072,N'ERNSH',4,DATE '1998-05-05',DATE '1998-06-02',NULL,258.64,
	N'Ernst Handel',N'Kirchgasse 6',N'Graz',
	NULL,N'8010',N'Austria');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11073,N'PERIC',2,DATE '1998-05-05',DATE '1998-06-02',NULL,24.95,
	N'Pericles Comidas clasicas',N'Calle Dr. Jorge Cash 321',N'Mexico D.F.',
	NULL,N'05033',N'Mexico');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11074,N'SIMOB',7,DATE '1998-05-06',DATE '1998-06-03',NULL,18.44,
	N'Simons bistro',N'Vinb?ltet 34',N'Kobenhavn',
	NULL,N'1734',N'Denmark');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11075,N'RICSU',8,DATE '1998-05-06',DATE '1998-06-03',NULL,6.19,
	N'Richter Supermarkt',N'Starenweg 5',N'Geneve',
	NULL,N'1204',N'Switzerland');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11076,N'BONAP',4,DATE '1998-05-06',DATE '1998-06-03',NULL,38.28,
	N'Bon app''',N'12, rue des Bouchers',N'Marseille',
	NULL,N'13008',N'France');
INSERT INTO "Orders"
("OrderID","CustomerID","EmployeeID","OrderDate","RequiredDate",
	"ShippedDate","Freight","ShipName","ShipAddress",
	"ShipCity","ShipRegion","ShipPostalCode","ShipCountry")
VALUES (11077,N'RATTC',1,DATE '1998-05-06',DATE '1998-06-03',NULL,8.53,
	N'Rattlesnake Canyon Grocery',N'2817 Milton Dr.',N'Albuquerque',
	N'NM',N'87110',N'USA');  

INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(1,'Chai',1,1,'10 boxes x 20 bags',18,39,0,10,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(2,'Chang',1,1,'24 - 12 oz bottles',19,17,40,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(3,'Aniseed Syrup',1,2,'12 - 550 ml bottles',10,13,70,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(4,'Chef Anton''s Cajun Seasoning',2,2,'48 - 6 oz jars',22,53,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued", "DiscontinuedDate") VALUES(5,'Chef Anton''s Gumbo Mix',2,2,'36 boxes',21.35,0,0,0,1, DATE '1996-07-04');
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(6,'Grandma''s Boysenberry Spread',3,2,'12 - 8 oz jars',25,120,0,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(7,'Uncle Bob''s Organic Dried Pears',3,7,'12 - 1 lb pkgs.',30,15,0,10,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(8,'Northwoods Cranberry Sauce',3,2,'12 - 12 oz jars',40,6,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued", "DiscontinuedDate") VALUES(9,'Mishi Kobe Niku',4,6,'18 - 500 g pkgs.',97,29,0,0,1, DATE '1996-07-04');
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(10,'Ikura',4,8,'12 - 200 ml jars',31,31,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(11,'Queso Cabrales',5,4,'1 kg pkg.',21,22,30,30,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(12,'Queso Manchego La Pastora',5,4,'10 - 500 g pkgs.',38,86,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(13,'Konbu',6,8,'2 kg box',6,24,0,5,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(14,'Tofu',6,7,'40 - 100 g pkgs.',23.25,35,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(15,'Genen Shouyu',6,2,'24 - 250 ml bottles',15.5,39,0,5,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(16,'Pavlova',7,3,'32 - 500 g boxes',17.45,29,0,10,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued", "DiscontinuedDate") VALUES(17,'Alice Mutton',7,6,'20 - 1 kg tins',39,0,0,0,1, DATE '1996-07-04');
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(18,'Carnarvon Tigers',7,8,'16 kg pkg.',62.5,42,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(19,'Teatime Chocolate Biscuits',8,3,'10 boxes x 12 pieces',9.2,25,0,5,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(20,'Sir Rodney''s Marmalade',8,3,'30 gift boxes',81,40,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(21,'Sir Rodney''s Scones',8,3,'24 pkgs. x 4 pieces',10,3,40,5,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(22,'Gustaf''s Knackebrod',9,5,'24 - 500 g pkgs.',21,104,0,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(23,'Tunnbrod',9,5,'12 - 250 g pkgs.',9,61,0,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued", "DiscontinuedDate") VALUES(24,'Guarana Fantastica',10,1,'12 - 355 ml cans',4.5,20,0,0,1, DATE '1996-07-04');
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(25,'NuNuCa Nu?-Nougat-Creme',11,3,'20 - 450 g glasses',14,76,0,30,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(26,'Gumbar Gummibarchen',11,3,'100 - 250 g bags',31.23,15,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(27,'Schoggi Schokolade',11,3,'100 - 100 g pieces',43.9,49,0,30,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued", "DiscontinuedDate") VALUES(28,'Rossle Sauerkraut',12,7,'25 - 825 g cans',45.6,26,0,0,1, DATE '1996-07-04');
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued", "DiscontinuedDate") VALUES(29,'Thuringer Rostbratwurst',12,6,'50 bags x 30 sausgs.',123.79,0,0,0,1, DATE '1996-07-04');
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(30,'Nord-Ost Matjeshering',13,8,'10 - 200 g glasses',25.89,10,0,15,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(31,'Gorgonzola Telino',14,4,'12 - 100 g pkgs',12.5,0,70,20,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(32,'Mascarpone Fabioli',14,4,'24 - 200 g pkgs.',32,9,40,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(33,'Geitost',15,4,'500 g',2.5,112,0,20,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(34,'Sasquatch Ale',16,1,'24 - 12 oz bottles',14,111,0,15,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(35,'Steeleye Stout',16,1,'24 - 12 oz bottles',18,20,0,15,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(36,'Inlagd Sill',17,8,'24 - 250 g jars',19,112,0,20,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(37,'Gravad lax',17,8,'12 - 500 g pkgs.',26,11,50,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(38,'Cote de Blaye',18,1,'12 - 75 cl bottles',263.5,17,0,15,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(39,'Chartreuse verte',18,1,'750 cc per bottle',18,69,0,5,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(40,'Boston Crab Meat',19,8,'24 - 4 oz tins',18.4,123,0,30,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(41,'Jack''s New England Clam Chowder',19,8,'12 - 12 oz cans',9.65,85,0,10,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued", "DiscontinuedDate") VALUES(42,'Singaporean Hokkien Fried Mee',20,5,'32 - 1 kg pkgs.',14,26,0,0,1, DATE '1996-07-04');
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(43,'Ipoh Coffee',20,1,'16 - 500 g tins',46,17,10,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(44,'Gula Malacca',20,2,'20 - 2 kg bags',19.45,27,0,15,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(45,'Rogede sild',21,8,'1k pkg.',9.5,5,70,15,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(46,'Spegesild',21,8,'4 - 450 g glasses',12,95,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(47,'Zaanse koeken',22,3,'10 - 4 oz boxes',9.5,36,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(48,'Chocolade',22,3,'10 pkgs.',12.75,15,70,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(49,'Maxilaku',23,3,'24 - 50 g pkgs.',20,10,60,15,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(50,'Valkoinen suklaa',23,3,'12 - 100 g bars',16.25,65,0,30,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(51,'Manjimup Dried Apples',24,7,'50 - 300 g pkgs.',53,20,0,10,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(52,'Filo Mix',24,5,'16 - 2 kg boxes',7,38,0,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued", "DiscontinuedDate") VALUES(53,'Perth Pasties',24,6,'48 pieces',32.8,0,0,0,1, DATE '1996-07-04');
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(54,'Tourtiere',25,6,'16 pies',7.45,21,0,10,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(55,'Pate chinois',25,6,'24 boxes x 2 pies',24,115,0,20,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(56,'Gnocchi di nonna Alice',26,5,'24 - 250 g pkgs.',38,21,10,30,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(57,'Ravioli Angelo',26,5,'24 - 250 g pkgs.',19.5,36,0,20,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(58,'Escargots de Bourgogne',27,8,'24 pieces',13.25,62,0,20,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(59,'Raclette Courdavault',28,4,'5 kg pkg.',55,79,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(60,'Camembert Pierrot',28,4,'15 - 300 g rounds',34,19,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(61,'Sirop d''erable',29,2,'24 - 500 ml bottles',28.5,113,0,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(62,'Tarte au sucre',29,3,'48 pies',49.3,17,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(63,'Vegie-spread',7,2,'15 - 625 g jars',43.9,24,0,5,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(64,'Wimmers gute Semmelknodel',12,5,'20 bags x 4 pieces',33.25,22,80,30,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(65,'Louisiana Fiery Hot Pepper Sauce',2,2,'32 - 8 oz bottles',21.05,76,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(66,'Louisiana Hot Spiced Okra',2,2,'24 - 8 oz jars',17,4,100,20,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(67,'Laughing Lumberjack Lager',16,1,'24 - 12 oz bottles',14,52,0,10,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(68,'Scottish Longbreads',8,3,'10 boxes x 8 pieces',12.5,6,10,15,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(69,'Gudbrandsdalsost',15,4,'10 kg pkg.',36,26,0,15,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(70,'Outback Lager',7,1,'24 - 355 ml bottles',15,15,10,30,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(71,'Flotemysost',15,4,'10 - 500 g pkgs.',21.5,26,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(72,'Mozzarella di Giovanni',14,4,'24 - 200 g pkgs.',34.8,14,0,0,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(73,'Rod Kaviar',17,8,'24 - 150 g jars',15,101,0,5,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(74,'Longlife Tofu',4,7,'5 kg pkg.',10,4,20,5,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(75,'Rhonbrau Klosterbier',12,1,'24 - 0.5 l bottles',7.75,125,0,25,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(76,'Lakkalikoori',23,1,'500 ml',18,57,0,20,0);
INSERT INTO "Products"("ProductID","ProductName","SupplierID","CategoryID","QuantityPerUnit","UnitPrice","UnitsInStock","UnitsOnOrder","ReorderLevel","Discontinued") VALUES(77,'Original Frankfurter grune So?e',12,2,'12 boxes',13,32,0,15,0);
INSERT INTO "OrderDetails" VALUES(10248,11,14,12,0);
INSERT INTO "OrderDetails" VALUES(10248,42,9.8,10,0);
INSERT INTO "OrderDetails" VALUES(10248,72,34.8,5,0);
INSERT INTO "OrderDetails" VALUES(10249,14,18.6,9,0);
INSERT INTO "OrderDetails" VALUES(10249,51,42.4,40,0);
INSERT INTO "OrderDetails" VALUES(10250,41,7.7,10,0);
INSERT INTO "OrderDetails" VALUES(10250,51,42.4,35,0.15);
INSERT INTO "OrderDetails" VALUES(10250,65,16.8,15,0.15);
INSERT INTO "OrderDetails" VALUES(10251,22,16.8,6,0.05);
INSERT INTO "OrderDetails" VALUES(10251,57,15.6,15,0.05);
INSERT INTO "OrderDetails" VALUES(10251,65,16.8,20,0);
INSERT INTO "OrderDetails" VALUES(10252,20,64.8,40,0.05);
INSERT INTO "OrderDetails" VALUES(10252,33,2,25,0.05);
INSERT INTO "OrderDetails" VALUES(10252,60,27.2,40,0);
INSERT INTO "OrderDetails" VALUES(10253,31,10,20,0);
INSERT INTO "OrderDetails" VALUES(10253,39,14.4,42,0);
INSERT INTO "OrderDetails" VALUES(10253,49,16,40,0);
INSERT INTO "OrderDetails" VALUES(10254,24,3.6,15,0.15);
INSERT INTO "OrderDetails" VALUES(10254,55,19.2,21,0.15);
INSERT INTO "OrderDetails" VALUES(10254,74,8,21,0);
INSERT INTO "OrderDetails" VALUES(10255,2,15.2,20,0);
INSERT INTO "OrderDetails" VALUES(10255,16,13.9,35,0);
INSERT INTO "OrderDetails" VALUES(10255,36,15.2,25,0);
INSERT INTO "OrderDetails" VALUES(10255,59,44,30,0);
INSERT INTO "OrderDetails" VALUES(10256,53,26.2,15,0);
INSERT INTO "OrderDetails" VALUES(10256,77,10.4,12,0);
INSERT INTO "OrderDetails" VALUES(10257,27,35.1,25,0);
INSERT INTO "OrderDetails" VALUES(10257,39,14.4,6,0);
INSERT INTO "OrderDetails" VALUES(10257,77,10.4,15,0);
INSERT INTO "OrderDetails" VALUES(10258,2,15.2,50,0.2);
INSERT INTO "OrderDetails" VALUES(10258,5,17,65,0.2);
INSERT INTO "OrderDetails" VALUES(10258,32,25.6,6,0.2);
INSERT INTO "OrderDetails" VALUES(10259,21,8,10,0);
INSERT INTO "OrderDetails" VALUES(10259,37,20.8,1,0);
INSERT INTO "OrderDetails" VALUES(10260,41,7.7,16,0.25);
INSERT INTO "OrderDetails" VALUES(10260,57,15.6,50,0);
INSERT INTO "OrderDetails" VALUES(10260,62,39.4,15,0.25);
INSERT INTO "OrderDetails" VALUES(10260,70,12,21,0.25);
INSERT INTO "OrderDetails" VALUES(10261,21,8,20,0);
INSERT INTO "OrderDetails" VALUES(10261,35,14.4,20,0);
INSERT INTO "OrderDetails" VALUES(10262,5,17,12,0.2);
INSERT INTO "OrderDetails" VALUES(10262,7,24,15,0);
INSERT INTO "OrderDetails" VALUES(10262,56,30.4,2,0);
INSERT INTO "OrderDetails" VALUES(10263,16,13.9,60,0.25);
INSERT INTO "OrderDetails" VALUES(10263,24,3.6,28,0);
INSERT INTO "OrderDetails" VALUES(10263,30,20.7,60,0.25);
INSERT INTO "OrderDetails" VALUES(10263,74,8,36,0.25);
INSERT INTO "OrderDetails" VALUES(10264,2,15.2,35,0);
INSERT INTO "OrderDetails" VALUES(10264,41,7.7,25,0.15);
INSERT INTO "OrderDetails" VALUES(10265,17,31.2,30,0);
INSERT INTO "OrderDetails" VALUES(10265,70,12,20,0);
INSERT INTO "OrderDetails" VALUES(10266,12,30.4,12,0.05);
INSERT INTO "OrderDetails" VALUES(10267,40,14.7,50,0);
INSERT INTO "OrderDetails" VALUES(10267,59,44,70,0.15);
INSERT INTO "OrderDetails" VALUES(10267,76,14.4,15,0.15);
INSERT INTO "OrderDetails" VALUES(10268,29,99,10,0);
INSERT INTO "OrderDetails" VALUES(10268,72,27.8,4,0);
INSERT INTO "OrderDetails" VALUES(10269,33,2,60,0.05);
INSERT INTO "OrderDetails" VALUES(10269,72,27.8,20,0.05);
INSERT INTO "OrderDetails" VALUES(10270,36,15.2,30,0);
INSERT INTO "OrderDetails" VALUES(10270,43,36.8,25,0);
INSERT INTO "OrderDetails" VALUES(10271,33,2,24,0);
INSERT INTO "OrderDetails" VALUES(10272,20,64.8,6,0);
INSERT INTO "OrderDetails" VALUES(10272,31,10,40,0);
INSERT INTO "OrderDetails" VALUES(10272,72,27.8,24,0);
INSERT INTO "OrderDetails" VALUES(10273,10,24.8,24,0.05);
INSERT INTO "OrderDetails" VALUES(10273,31,10,15,0.05);
INSERT INTO "OrderDetails" VALUES(10273,33,2,20,0);
INSERT INTO "OrderDetails" VALUES(10273,40,14.7,60,0.05);
INSERT INTO "OrderDetails" VALUES(10273,76,14.4,33,0.05);
INSERT INTO "OrderDetails" VALUES(10274,71,17.2,20,0);
INSERT INTO "OrderDetails" VALUES(10274,72,27.8,7,0);
INSERT INTO "OrderDetails" VALUES(10275,24,3.6,12,0.05);
INSERT INTO "OrderDetails" VALUES(10275,59,44,6,0.05);
INSERT INTO "OrderDetails" VALUES(10276,10,24.8,15,0);
INSERT INTO "OrderDetails" VALUES(10276,13,4.8,10,0);
INSERT INTO "OrderDetails" VALUES(10277,28,36.4,20,0);
INSERT INTO "OrderDetails" VALUES(10277,62,39.4,12,0);
INSERT INTO "OrderDetails" VALUES(10278,44,15.5,16,0);
INSERT INTO "OrderDetails" VALUES(10278,59,44,15,0);
INSERT INTO "OrderDetails" VALUES(10278,63,35.1,8,0);
INSERT INTO "OrderDetails" VALUES(10278,73,12,25,0);
INSERT INTO "OrderDetails" VALUES(10279,17,31.2,15,0.25);
INSERT INTO "OrderDetails" VALUES(10280,24,3.6,12,0);
INSERT INTO "OrderDetails" VALUES(10280,55,19.2,20,0);
INSERT INTO "OrderDetails" VALUES(10280,75,6.2,30,0);
INSERT INTO "OrderDetails" VALUES(10281,19,7.3,1,0);
INSERT INTO "OrderDetails" VALUES(10281,24,3.6,6,0);
INSERT INTO "OrderDetails" VALUES(10281,35,14.4,4,0);
INSERT INTO "OrderDetails" VALUES(10282,30,20.7,6,0);
INSERT INTO "OrderDetails" VALUES(10282,57,15.6,2,0);
INSERT INTO "OrderDetails" VALUES(10283,15,12.4,20,0);
INSERT INTO "OrderDetails" VALUES(10283,19,7.3,18,0);
INSERT INTO "OrderDetails" VALUES(10283,60,27.2,35,0);
INSERT INTO "OrderDetails" VALUES(10283,72,27.8,3,0);
INSERT INTO "OrderDetails" VALUES(10284,27,35.1,15,0.25);
INSERT INTO "OrderDetails" VALUES(10284,44,15.5,21,0);
INSERT INTO "OrderDetails" VALUES(10284,60,27.2,20,0.25);
INSERT INTO "OrderDetails" VALUES(10284,67,11.2,5,0.25);
INSERT INTO "OrderDetails" VALUES(10285,1,14.4,45,0.2);
INSERT INTO "OrderDetails" VALUES(10285,40,14.7,40,0.2);
INSERT INTO "OrderDetails" VALUES(10285,53,26.2,36,0.2);
INSERT INTO "OrderDetails" VALUES(10286,35,14.4,100,0);
INSERT INTO "OrderDetails" VALUES(10286,62,39.4,40,0);
INSERT INTO "OrderDetails" VALUES(10287,16,13.9,40,0.15);
INSERT INTO "OrderDetails" VALUES(10287,34,11.2,20,0);
INSERT INTO "OrderDetails" VALUES(10287,46,9.6,15,0.15);
INSERT INTO "OrderDetails" VALUES(10288,54,5.9,10,0.1);
INSERT INTO "OrderDetails" VALUES(10288,68,10,3,0.1);
INSERT INTO "OrderDetails" VALUES(10289,3,8,30,0);
INSERT INTO "OrderDetails" VALUES(10289,64,26.6,9,0);
INSERT INTO "OrderDetails" VALUES(10290,5,17,20,0);
INSERT INTO "OrderDetails" VALUES(10290,29,99,15,0);
INSERT INTO "OrderDetails" VALUES(10290,49,16,15,0);
INSERT INTO "OrderDetails" VALUES(10290,77,10.4,10,0);
INSERT INTO "OrderDetails" VALUES(10291,13,4.8,20,0.1);
INSERT INTO "OrderDetails" VALUES(10291,44,15.5,24,0.1);
INSERT INTO "OrderDetails" VALUES(10291,51,42.4,2,0.1);
INSERT INTO "OrderDetails" VALUES(10292,20,64.8,20,0);
INSERT INTO "OrderDetails" VALUES(10293,18,50,12,0);
INSERT INTO "OrderDetails" VALUES(10293,24,3.6,10,0);
INSERT INTO "OrderDetails" VALUES(10293,63,35.1,5,0);
INSERT INTO "OrderDetails" VALUES(10293,75,6.2,6,0);
INSERT INTO "OrderDetails" VALUES(10294,1,14.4,18,0);
INSERT INTO "OrderDetails" VALUES(10294,17,31.2,15,0);
INSERT INTO "OrderDetails" VALUES(10294,43,36.8,15,0);
INSERT INTO "OrderDetails" VALUES(10294,60,27.2,21,0);
INSERT INTO "OrderDetails" VALUES(10294,75,6.2,6,0);
INSERT INTO "OrderDetails" VALUES(10295,56,30.4,4,0);
INSERT INTO "OrderDetails" VALUES(10296,11,16.8,12,0);
INSERT INTO "OrderDetails" VALUES(10296,16,13.9,30,0);
INSERT INTO "OrderDetails" VALUES(10296,69,28.8,15,0);
INSERT INTO "OrderDetails" VALUES(10297,39,14.4,60,0);
INSERT INTO "OrderDetails" VALUES(10297,72,27.8,20,0);
INSERT INTO "OrderDetails" VALUES(10298,2,15.2,40,0);
INSERT INTO "OrderDetails" VALUES(10298,36,15.2,40,0.25);
INSERT INTO "OrderDetails" VALUES(10298,59,44,30,0.25);
INSERT INTO "OrderDetails" VALUES(10298,62,39.4,15,0);
INSERT INTO "OrderDetails" VALUES(10299,19,7.3,15,0);
INSERT INTO "OrderDetails" VALUES(10299,70,12,20,0);
INSERT INTO "OrderDetails" VALUES(10300,66,13.6,30,0);
INSERT INTO "OrderDetails" VALUES(10300,68,10,20,0);
INSERT INTO "OrderDetails" VALUES(10301,40,14.7,10,0);
INSERT INTO "OrderDetails" VALUES(10301,56,30.4,20,0);
INSERT INTO "OrderDetails" VALUES(10302,17,31.2,40,0);
INSERT INTO "OrderDetails" VALUES(10302,28,36.4,28,0);
INSERT INTO "OrderDetails" VALUES(10302,43,36.8,12,0);
INSERT INTO "OrderDetails" VALUES(10303,40,14.7,40,0.1);
INSERT INTO "OrderDetails" VALUES(10303,65,16.8,30,0.1);
INSERT INTO "OrderDetails" VALUES(10303,68,10,15,0.1);
INSERT INTO "OrderDetails" VALUES(10304,49,16,30,0);
INSERT INTO "OrderDetails" VALUES(10304,59,44,10,0);
INSERT INTO "OrderDetails" VALUES(10304,71,17.2,2,0);
INSERT INTO "OrderDetails" VALUES(10305,18,50,25,0.1);
INSERT INTO "OrderDetails" VALUES(10305,29,99,25,0.1);
INSERT INTO "OrderDetails" VALUES(10305,39,14.4,30,0.1);
INSERT INTO "OrderDetails" VALUES(10306,30,20.7,10,0);
INSERT INTO "OrderDetails" VALUES(10306,53,26.2,10,0);
INSERT INTO "OrderDetails" VALUES(10306,54,5.9,5,0);
INSERT INTO "OrderDetails" VALUES(10307,62,39.4,10,0);
INSERT INTO "OrderDetails" VALUES(10307,68,10,3,0);
INSERT INTO "OrderDetails" VALUES(10308,69,28.8,1,0);
INSERT INTO "OrderDetails" VALUES(10308,70,12,5,0);
INSERT INTO "OrderDetails" VALUES(10309,4,17.6,20,0);
INSERT INTO "OrderDetails" VALUES(10309,6,20,30,0);
INSERT INTO "OrderDetails" VALUES(10309,42,11.2,2,0);
INSERT INTO "OrderDetails" VALUES(10309,43,36.8,20,0);
INSERT INTO "OrderDetails" VALUES(10309,71,17.2,3,0);
INSERT INTO "OrderDetails" VALUES(10310,16,13.9,10,0);
INSERT INTO "OrderDetails" VALUES(10310,62,39.4,5,0);
INSERT INTO "OrderDetails" VALUES(10311,42,11.2,6,0);
INSERT INTO "OrderDetails" VALUES(10311,69,28.8,7,0);
INSERT INTO "OrderDetails" VALUES(10312,28,36.4,4,0);
INSERT INTO "OrderDetails" VALUES(10312,43,36.8,24,0);
INSERT INTO "OrderDetails" VALUES(10312,53,26.2,20,0);
INSERT INTO "OrderDetails" VALUES(10312,75,6.2,10,0);
INSERT INTO "OrderDetails" VALUES(10313,36,15.2,12,0);
INSERT INTO "OrderDetails" VALUES(10314,32,25.6,40,0.1);
INSERT INTO "OrderDetails" VALUES(10314,58,10.6,30,0.1);
INSERT INTO "OrderDetails" VALUES(10314,62,39.4,25,0.1);
INSERT INTO "OrderDetails" VALUES(10315,34,11.2,14,0);
INSERT INTO "OrderDetails" VALUES(10315,70,12,30,0);
INSERT INTO "OrderDetails" VALUES(10316,41,7.7,10,0);
INSERT INTO "OrderDetails" VALUES(10316,62,39.4,70,0);
INSERT INTO "OrderDetails" VALUES(10317,1,14.4,20,0);
INSERT INTO "OrderDetails" VALUES(10318,41,7.7,20,0);
INSERT INTO "OrderDetails" VALUES(10318,76,14.4,6,0);
INSERT INTO "OrderDetails" VALUES(10319,17,31.2,8,0);
INSERT INTO "OrderDetails" VALUES(10319,28,36.4,14,0);
INSERT INTO "OrderDetails" VALUES(10319,76,14.4,30,0);
INSERT INTO "OrderDetails" VALUES(10320,71,17.2,30,0);
INSERT INTO "OrderDetails" VALUES(10321,35,14.4,10,0);
INSERT INTO "OrderDetails" VALUES(10322,52,5.6,20,0);
INSERT INTO "OrderDetails" VALUES(10323,15,12.4,5,0);
INSERT INTO "OrderDetails" VALUES(10323,25,11.2,4,0);
INSERT INTO "OrderDetails" VALUES(10323,39,14.4,4,0);
INSERT INTO "OrderDetails" VALUES(10324,16,13.9,21,0.15);
INSERT INTO "OrderDetails" VALUES(10324,35,14.4,70,0.15);
INSERT INTO "OrderDetails" VALUES(10324,46,9.6,30,0);
INSERT INTO "OrderDetails" VALUES(10324,59,44,40,0.15);
INSERT INTO "OrderDetails" VALUES(10324,63,35.1,80,0.15);
INSERT INTO "OrderDetails" VALUES(10325,6,20,6,0);
INSERT INTO "OrderDetails" VALUES(10325,13,4.8,12,0);
INSERT INTO "OrderDetails" VALUES(10325,14,18.6,9,0);
INSERT INTO "OrderDetails" VALUES(10325,31,10,4,0);
INSERT INTO "OrderDetails" VALUES(10325,72,27.8,40,0);
INSERT INTO "OrderDetails" VALUES(10326,4,17.6,24,0);
INSERT INTO "OrderDetails" VALUES(10326,57,15.6,16,0);
INSERT INTO "OrderDetails" VALUES(10326,75,6.2,50,0);
INSERT INTO "OrderDetails" VALUES(10327,2,15.2,25,0.2);
INSERT INTO "OrderDetails" VALUES(10327,11,16.8,50,0.2);
INSERT INTO "OrderDetails" VALUES(10327,30,20.7,35,0.2);
INSERT INTO "OrderDetails" VALUES(10327,58,10.6,30,0.2);
INSERT INTO "OrderDetails" VALUES(10328,59,44,9,0);
INSERT INTO "OrderDetails" VALUES(10328,65,16.8,40,0);
INSERT INTO "OrderDetails" VALUES(10328,68,10,10,0);
INSERT INTO "OrderDetails" VALUES(10329,19,7.3,10,0.05);
INSERT INTO "OrderDetails" VALUES(10329,30,20.7,8,0.05);
INSERT INTO "OrderDetails" VALUES(10329,38,210.8,20,0.05);
INSERT INTO "OrderDetails" VALUES(10329,56,30.4,12,0.05);
INSERT INTO "OrderDetails" VALUES(10330,26,24.9,50,0.15);
INSERT INTO "OrderDetails" VALUES(10330,72,27.8,25,0.15);
INSERT INTO "OrderDetails" VALUES(10331,54,5.9,15,0);
INSERT INTO "OrderDetails" VALUES(10332,18,50,40,0.2);
INSERT INTO "OrderDetails" VALUES(10332,42,11.2,10,0.2);
INSERT INTO "OrderDetails" VALUES(10332,47,7.6,16,0.2);
INSERT INTO "OrderDetails" VALUES(10333,14,18.6,10,0);
INSERT INTO "OrderDetails" VALUES(10333,21,8,10,0.1);
INSERT INTO "OrderDetails" VALUES(10333,71,17.2,40,0.1);
INSERT INTO "OrderDetails" VALUES(10334,52,5.6,8,0);
INSERT INTO "OrderDetails" VALUES(10334,68,10,10,0);
INSERT INTO "OrderDetails" VALUES(10335,2,15.2,7,0.2);
INSERT INTO "OrderDetails" VALUES(10335,31,10,25,0.2);
INSERT INTO "OrderDetails" VALUES(10335,32,25.6,6,0.2);
INSERT INTO "OrderDetails" VALUES(10335,51,42.4,48,0.2);
INSERT INTO "OrderDetails" VALUES(10336,4,17.6,18,0.1);
INSERT INTO "OrderDetails" VALUES(10337,23,7.2,40,0);
INSERT INTO "OrderDetails" VALUES(10337,26,24.9,24,0);
INSERT INTO "OrderDetails" VALUES(10337,36,15.2,20,0);
INSERT INTO "OrderDetails" VALUES(10337,37,20.8,28,0);
INSERT INTO "OrderDetails" VALUES(10337,72,27.8,25,0);
INSERT INTO "OrderDetails" VALUES(10338,17,31.2,20,0);
INSERT INTO "OrderDetails" VALUES(10338,30,20.7,15,0);
INSERT INTO "OrderDetails" VALUES(10339,4,17.6,10,0);
INSERT INTO "OrderDetails" VALUES(10339,17,31.2,70,0.05);
INSERT INTO "OrderDetails" VALUES(10339,62,39.4,28,0);
INSERT INTO "OrderDetails" VALUES(10340,18,50,20,0.05);
INSERT INTO "OrderDetails" VALUES(10340,41,7.7,12,0.05);
INSERT INTO "OrderDetails" VALUES(10340,43,36.8,40,0.05);
INSERT INTO "OrderDetails" VALUES(10341,33,2,8,0);
INSERT INTO "OrderDetails" VALUES(10341,59,44,9,0.15);
INSERT INTO "OrderDetails" VALUES(10342,2,15.2,24,0.2);
INSERT INTO "OrderDetails" VALUES(10342,31,10,56,0.2);
INSERT INTO "OrderDetails" VALUES(10342,36,15.2,40,0.2);
INSERT INTO "OrderDetails" VALUES(10342,55,19.2,40,0.2);
INSERT INTO "OrderDetails" VALUES(10343,64,26.6,50,0);
INSERT INTO "OrderDetails" VALUES(10343,68,10,4,0.05);
INSERT INTO "OrderDetails" VALUES(10343,76,14.4,15,0);
INSERT INTO "OrderDetails" VALUES(10344,4,17.6,35,0);
INSERT INTO "OrderDetails" VALUES(10344,8,32,70,0.25);
INSERT INTO "OrderDetails" VALUES(10345,8,32,70,0);
INSERT INTO "OrderDetails" VALUES(10345,19,7.3,80,0);
INSERT INTO "OrderDetails" VALUES(10345,42,11.2,9,0);
INSERT INTO "OrderDetails" VALUES(10346,17,31.2,36,0.1);
INSERT INTO "OrderDetails" VALUES(10346,56,30.4,20,0);
INSERT INTO "OrderDetails" VALUES(10347,25,11.2,10,0);
INSERT INTO "OrderDetails" VALUES(10347,39,14.4,50,0.15);
INSERT INTO "OrderDetails" VALUES(10347,40,14.7,4,0);
INSERT INTO "OrderDetails" VALUES(10347,75,6.2,6,0.15);
INSERT INTO "OrderDetails" VALUES(10348,1,14.4,15,0.15);
INSERT INTO "OrderDetails" VALUES(10348,23,7.2,25,0);
INSERT INTO "OrderDetails" VALUES(10349,54,5.9,24,0);
INSERT INTO "OrderDetails" VALUES(10350,50,13,15,0.1);
INSERT INTO "OrderDetails" VALUES(10350,69,28.8,18,0.1);
INSERT INTO "OrderDetails" VALUES(10351,38,210.8,20,0.05);
INSERT INTO "OrderDetails" VALUES(10351,41,7.7,13,0);
INSERT INTO "OrderDetails" VALUES(10351,44,15.5,77,0.05);
INSERT INTO "OrderDetails" VALUES(10351,65,16.8,10,0.05);
INSERT INTO "OrderDetails" VALUES(10352,24,3.6,10,0);
INSERT INTO "OrderDetails" VALUES(10352,54,5.9,20,0.15);
INSERT INTO "OrderDetails" VALUES(10353,11,16.8,12,0.2);
INSERT INTO "OrderDetails" VALUES(10353,38,210.8,50,0.2);
INSERT INTO "OrderDetails" VALUES(10354,1,14.4,12,0);
INSERT INTO "OrderDetails" VALUES(10354,29,99,4,0);
INSERT INTO "OrderDetails" VALUES(10355,24,3.6,25,0);
INSERT INTO "OrderDetails" VALUES(10355,57,15.6,25,0);
INSERT INTO "OrderDetails" VALUES(10356,31,10,30,0);
INSERT INTO "OrderDetails" VALUES(10356,55,19.2,12,0);
INSERT INTO "OrderDetails" VALUES(10356,69,28.8,20,0);
INSERT INTO "OrderDetails" VALUES(10357,10,24.8,30,0.2);
INSERT INTO "OrderDetails" VALUES(10357,26,24.9,16,0);
INSERT INTO "OrderDetails" VALUES(10357,60,27.2,8,0.2);
INSERT INTO "OrderDetails" VALUES(10358,24,3.6,10,0.05);
INSERT INTO "OrderDetails" VALUES(10358,34,11.2,10,0.05);
INSERT INTO "OrderDetails" VALUES(10358,36,15.2,20,0.05);
INSERT INTO "OrderDetails" VALUES(10359,16,13.9,56,0.05);
INSERT INTO "OrderDetails" VALUES(10359,31,10,70,0.05);
INSERT INTO "OrderDetails" VALUES(10359,60,27.2,80,0.05);
INSERT INTO "OrderDetails" VALUES(10360,28,36.4,30,0);
INSERT INTO "OrderDetails" VALUES(10360,29,99,35,0);
INSERT INTO "OrderDetails" VALUES(10360,38,210.8,10,0);
INSERT INTO "OrderDetails" VALUES(10360,49,16,35,0);
INSERT INTO "OrderDetails" VALUES(10360,54,5.9,28,0);
INSERT INTO "OrderDetails" VALUES(10361,39,14.4,54,0.1);
INSERT INTO "OrderDetails" VALUES(10361,60,27.2,55,0.1);
INSERT INTO "OrderDetails" VALUES(10362,25,11.2,50,0);
INSERT INTO "OrderDetails" VALUES(10362,51,42.4,20,0);
INSERT INTO "OrderDetails" VALUES(10362,54,5.9,24,0);
INSERT INTO "OrderDetails" VALUES(10363,31,10,20,0);
INSERT INTO "OrderDetails" VALUES(10363,75,6.2,12,0);
INSERT INTO "OrderDetails" VALUES(10363,76,14.4,12,0);
INSERT INTO "OrderDetails" VALUES(10364,69,28.8,30,0);
INSERT INTO "OrderDetails" VALUES(10364,71,17.2,5,0);
INSERT INTO "OrderDetails" VALUES(10365,11,16.8,24,0);
INSERT INTO "OrderDetails" VALUES(10366,65,16.8,5,0);
INSERT INTO "OrderDetails" VALUES(10366,77,10.4,5,0);
INSERT INTO "OrderDetails" VALUES(10367,34,11.2,36,0);
INSERT INTO "OrderDetails" VALUES(10367,54,5.9,18,0);
INSERT INTO "OrderDetails" VALUES(10367,65,16.8,15,0);
INSERT INTO "OrderDetails" VALUES(10367,77,10.4,7,0);
INSERT INTO "OrderDetails" VALUES(10368,21,8,5,0.1);
INSERT INTO "OrderDetails" VALUES(10368,28,36.4,13,0.1);
INSERT INTO "OrderDetails" VALUES(10368,57,15.6,25,0);
INSERT INTO "OrderDetails" VALUES(10368,64,26.6,35,0.1);
INSERT INTO "OrderDetails" VALUES(10369,29,99,20,0);
INSERT INTO "OrderDetails" VALUES(10369,56,30.4,18,0.25);
INSERT INTO "OrderDetails" VALUES(10370,1,14.4,15,0.15);
INSERT INTO "OrderDetails" VALUES(10370,64,26.6,30,0);
INSERT INTO "OrderDetails" VALUES(10370,74,8,20,0.15);
INSERT INTO "OrderDetails" VALUES(10371,36,15.2,6,0.2);
INSERT INTO "OrderDetails" VALUES(10372,20,64.8,12,0.25);
INSERT INTO "OrderDetails" VALUES(10372,38,210.8,40,0.25);
INSERT INTO "OrderDetails" VALUES(10372,60,27.2,70,0.25);
INSERT INTO "OrderDetails" VALUES(10372,72,27.8,42,0.25);
INSERT INTO "OrderDetails" VALUES(10373,58,10.6,80,0.2);
INSERT INTO "OrderDetails" VALUES(10373,71,17.2,50,0.2);
INSERT INTO "OrderDetails" VALUES(10374,31,10,30,0);
INSERT INTO "OrderDetails" VALUES(10374,58,10.6,15,0);
INSERT INTO "OrderDetails" VALUES(10375,14,18.6,15,0);
INSERT INTO "OrderDetails" VALUES(10375,54,5.9,10,0);
INSERT INTO "OrderDetails" VALUES(10376,31,10,42,0.05);
INSERT INTO "OrderDetails" VALUES(10377,28,36.4,20,0.15);
INSERT INTO "OrderDetails" VALUES(10377,39,14.4,20,0.15);
INSERT INTO "OrderDetails" VALUES(10378,71,17.2,6,0);
INSERT INTO "OrderDetails" VALUES(10379,41,7.7,8,0.1);
INSERT INTO "OrderDetails" VALUES(10379,63,35.1,16,0.1);
INSERT INTO "OrderDetails" VALUES(10379,65,16.8,20,0.1);
INSERT INTO "OrderDetails" VALUES(10380,30,20.7,18,0.1);
INSERT INTO "OrderDetails" VALUES(10380,53,26.2,20,0.1);
INSERT INTO "OrderDetails" VALUES(10380,60,27.2,6,0.1);
INSERT INTO "OrderDetails" VALUES(10380,70,12,30,0);
INSERT INTO "OrderDetails" VALUES(10381,74,8,14,0);
INSERT INTO "OrderDetails" VALUES(10382,5,17,32,0);
INSERT INTO "OrderDetails" VALUES(10382,18,50,9,0);
INSERT INTO "OrderDetails" VALUES(10382,29,99,14,0);
INSERT INTO "OrderDetails" VALUES(10382,33,2,60,0);
INSERT INTO "OrderDetails" VALUES(10382,74,8,50,0);
INSERT INTO "OrderDetails" VALUES(10383,13,4.8,20,0);
INSERT INTO "OrderDetails" VALUES(10383,50,13,15,0);
INSERT INTO "OrderDetails" VALUES(10383,56,30.4,20,0);
INSERT INTO "OrderDetails" VALUES(10384,20,64.8,28,0);
INSERT INTO "OrderDetails" VALUES(10384,60,27.2,15,0);
INSERT INTO "OrderDetails" VALUES(10385,7,24,10,0.2);
INSERT INTO "OrderDetails" VALUES(10385,60,27.2,20,0.2);
INSERT INTO "OrderDetails" VALUES(10385,68,10,8,0.2);
INSERT INTO "OrderDetails" VALUES(10386,24,3.6,15,0);
INSERT INTO "OrderDetails" VALUES(10386,34,11.2,10,0);
INSERT INTO "OrderDetails" VALUES(10387,24,3.6,15,0);
INSERT INTO "OrderDetails" VALUES(10387,28,36.4,6,0);
INSERT INTO "OrderDetails" VALUES(10387,59,44,12,0);
INSERT INTO "OrderDetails" VALUES(10387,71,17.2,15,0);
INSERT INTO "OrderDetails" VALUES(10388,45,7.6,15,0.2);
INSERT INTO "OrderDetails" VALUES(10388,52,5.6,20,0.2);
INSERT INTO "OrderDetails" VALUES(10388,53,26.2,40,0);
INSERT INTO "OrderDetails" VALUES(10389,10,24.8,16,0);
INSERT INTO "OrderDetails" VALUES(10389,55,19.2,15,0);
INSERT INTO "OrderDetails" VALUES(10389,62,39.4,20,0);
INSERT INTO "OrderDetails" VALUES(10389,70,12,30,0);
INSERT INTO "OrderDetails" VALUES(10390,31,10,60,0.1);
INSERT INTO "OrderDetails" VALUES(10390,35,14.4,40,0.1);
INSERT INTO "OrderDetails" VALUES(10390,46,9.6,45,0);
INSERT INTO "OrderDetails" VALUES(10390,72,27.8,24,0.1);
INSERT INTO "OrderDetails" VALUES(10391,13,4.8,18,0);
INSERT INTO "OrderDetails" VALUES(10392,69,28.8,50,0);
INSERT INTO "OrderDetails" VALUES(10393,2,15.2,25,0.25);
INSERT INTO "OrderDetails" VALUES(10393,14,18.6,42,0.25);
INSERT INTO "OrderDetails" VALUES(10393,25,11.2,7,0.25);
INSERT INTO "OrderDetails" VALUES(10393,26,24.9,70,0.25);
INSERT INTO "OrderDetails" VALUES(10393,31,10,32,0);
INSERT INTO "OrderDetails" VALUES(10394,13,4.8,10,0);
INSERT INTO "OrderDetails" VALUES(10394,62,39.4,10,0);
INSERT INTO "OrderDetails" VALUES(10395,46,9.6,28,0.1);
INSERT INTO "OrderDetails" VALUES(10395,53,26.2,70,0.1);
INSERT INTO "OrderDetails" VALUES(10395,69,28.8,8,0);
INSERT INTO "OrderDetails" VALUES(10396,23,7.2,40,0);
INSERT INTO "OrderDetails" VALUES(10396,71,17.2,60,0);
INSERT INTO "OrderDetails" VALUES(10396,72,27.8,21,0);
INSERT INTO "OrderDetails" VALUES(10397,21,8,10,0.15);
INSERT INTO "OrderDetails" VALUES(10397,51,42.4,18,0.15);
INSERT INTO "OrderDetails" VALUES(10398,35,14.4,30,0);
INSERT INTO "OrderDetails" VALUES(10398,55,19.2,120,0.1);
INSERT INTO "OrderDetails" VALUES(10399,68,10,60,0);
INSERT INTO "OrderDetails" VALUES(10399,71,17.2,30,0);
INSERT INTO "OrderDetails" VALUES(10399,76,14.4,35,0);
INSERT INTO "OrderDetails" VALUES(10399,77,10.4,14,0);
INSERT INTO "OrderDetails" VALUES(10400,29,99,21,0);
INSERT INTO "OrderDetails" VALUES(10400,35,14.4,35,0);
INSERT INTO "OrderDetails" VALUES(10400,49,16,30,0);
INSERT INTO "OrderDetails" VALUES(10401,30,20.7,18,0);
INSERT INTO "OrderDetails" VALUES(10401,56,30.4,70,0);
INSERT INTO "OrderDetails" VALUES(10401,65,16.8,20,0);
INSERT INTO "OrderDetails" VALUES(10401,71,17.2,60,0);
INSERT INTO "OrderDetails" VALUES(10402,23,7.2,60,0);
INSERT INTO "OrderDetails" VALUES(10402,63,35.1,65,0);
INSERT INTO "OrderDetails" VALUES(10403,16,13.9,21,0.15);
INSERT INTO "OrderDetails" VALUES(10403,48,10.2,70,0.15);
INSERT INTO "OrderDetails" VALUES(10404,26,24.9,30,0.05);
INSERT INTO "OrderDetails" VALUES(10404,42,11.2,40,0.05);
INSERT INTO "OrderDetails" VALUES(10404,49,16,30,0.05);
INSERT INTO "OrderDetails" VALUES(10405,3,8,50,0);
INSERT INTO "OrderDetails" VALUES(10406,1,14.4,10,0);
INSERT INTO "OrderDetails" VALUES(10406,21,8,30,0.1);
INSERT INTO "OrderDetails" VALUES(10406,28,36.4,42,0.1);
INSERT INTO "OrderDetails" VALUES(10406,36,15.2,5,0.1);
INSERT INTO "OrderDetails" VALUES(10406,40,14.7,2,0.1);
INSERT INTO "OrderDetails" VALUES(10407,11,16.8,30,0);
INSERT INTO "OrderDetails" VALUES(10407,69,28.8,15,0);
INSERT INTO "OrderDetails" VALUES(10407,71,17.2,15,0);
INSERT INTO "OrderDetails" VALUES(10408,37,20.8,10,0);
INSERT INTO "OrderDetails" VALUES(10408,54,5.9,6,0);
INSERT INTO "OrderDetails" VALUES(10408,62,39.4,35,0);
INSERT INTO "OrderDetails" VALUES(10409,14,18.6,12,0);
INSERT INTO "OrderDetails" VALUES(10409,21,8,12,0);
INSERT INTO "OrderDetails" VALUES(10410,33,2,49,0);
INSERT INTO "OrderDetails" VALUES(10410,59,44,16,0);
INSERT INTO "OrderDetails" VALUES(10411,41,7.7,25,0.2);
INSERT INTO "OrderDetails" VALUES(10411,44,15.5,40,0.2);
INSERT INTO "OrderDetails" VALUES(10411,59,44,9,0.2);
INSERT INTO "OrderDetails" VALUES(10412,14,18.6,20,0.1);
INSERT INTO "OrderDetails" VALUES(10413,1,14.4,24,0);
INSERT INTO "OrderDetails" VALUES(10413,62,39.4,40,0);
INSERT INTO "OrderDetails" VALUES(10413,76,14.4,14,0);
INSERT INTO "OrderDetails" VALUES(10414,19,7.3,18,0.05);
INSERT INTO "OrderDetails" VALUES(10414,33,2,50,0);
INSERT INTO "OrderDetails" VALUES(10415,17,31.2,2,0);
INSERT INTO "OrderDetails" VALUES(10415,33,2,20,0);
INSERT INTO "OrderDetails" VALUES(10416,19,7.3,20,0);
INSERT INTO "OrderDetails" VALUES(10416,53,26.2,10,0);
INSERT INTO "OrderDetails" VALUES(10416,57,15.6,20,0);
INSERT INTO "OrderDetails" VALUES(10417,38,210.8,50,0);
INSERT INTO "OrderDetails" VALUES(10417,46,9.6,2,0.25);
INSERT INTO "OrderDetails" VALUES(10417,68,10,36,0.25);
INSERT INTO "OrderDetails" VALUES(10417,77,10.4,35,0);
INSERT INTO "OrderDetails" VALUES(10418,2,15.2,60,0);
INSERT INTO "OrderDetails" VALUES(10418,47,7.6,55,0);
INSERT INTO "OrderDetails" VALUES(10418,61,22.8,16,0);
INSERT INTO "OrderDetails" VALUES(10418,74,8,15,0);
INSERT INTO "OrderDetails" VALUES(10419,60,27.2,60,0.05);
INSERT INTO "OrderDetails" VALUES(10419,69,28.8,20,0.05);
INSERT INTO "OrderDetails" VALUES(10420,9,77.6,20,0.1);
INSERT INTO "OrderDetails" VALUES(10420,13,4.8,2,0.1);
INSERT INTO "OrderDetails" VALUES(10420,70,12,8,0.1);
INSERT INTO "OrderDetails" VALUES(10420,73,12,20,0.1);
INSERT INTO "OrderDetails" VALUES(10421,19,7.3,4,0.15);
INSERT INTO "OrderDetails" VALUES(10421,26,24.9,30,0);
INSERT INTO "OrderDetails" VALUES(10421,53,26.2,15,0.15);
INSERT INTO "OrderDetails" VALUES(10421,77,10.4,10,0.15);
INSERT INTO "OrderDetails" VALUES(10422,26,24.9,2,0);
INSERT INTO "OrderDetails" VALUES(10423,31,10,14,0);
INSERT INTO "OrderDetails" VALUES(10423,59,44,20,0);
INSERT INTO "OrderDetails" VALUES(10424,35,14.4,60,0.2);
INSERT INTO "OrderDetails" VALUES(10424,38,210.8,49,0.2);
INSERT INTO "OrderDetails" VALUES(10424,68,10,30,0.2);
INSERT INTO "OrderDetails" VALUES(10425,55,19.2,10,0.25);
INSERT INTO "OrderDetails" VALUES(10425,76,14.4,20,0.25);
INSERT INTO "OrderDetails" VALUES(10426,56,30.4,5,0);
INSERT INTO "OrderDetails" VALUES(10426,64,26.6,7,0);
INSERT INTO "OrderDetails" VALUES(10427,14,18.6,35,0);
INSERT INTO "OrderDetails" VALUES(10428,46,9.6,20,0);
INSERT INTO "OrderDetails" VALUES(10429,50,13,40,0);
INSERT INTO "OrderDetails" VALUES(10429,63,35.1,35,0.25);
INSERT INTO "OrderDetails" VALUES(10430,17,31.2,45,0.2);
INSERT INTO "OrderDetails" VALUES(10430,21,8,50,0);
INSERT INTO "OrderDetails" VALUES(10430,56,30.4,30,0);
INSERT INTO "OrderDetails" VALUES(10430,59,44,70,0.2);
INSERT INTO "OrderDetails" VALUES(10431,17,31.2,50,0.25);
INSERT INTO "OrderDetails" VALUES(10431,40,14.7,50,0.25);
INSERT INTO "OrderDetails" VALUES(10431,47,7.6,30,0.25);
INSERT INTO "OrderDetails" VALUES(10432,26,24.9,10,0);
INSERT INTO "OrderDetails" VALUES(10432,54,5.9,40,0);
INSERT INTO "OrderDetails" VALUES(10433,56,30.4,28,0);
INSERT INTO "OrderDetails" VALUES(10434,11,16.8,6,0);
INSERT INTO "OrderDetails" VALUES(10434,76,14.4,18,0.15);
INSERT INTO "OrderDetails" VALUES(10435,2,15.2,10,0);
INSERT INTO "OrderDetails" VALUES(10435,22,16.8,12,0);
INSERT INTO "OrderDetails" VALUES(10435,72,27.8,10,0);
INSERT INTO "OrderDetails" VALUES(10436,46,9.6,5,0);
INSERT INTO "OrderDetails" VALUES(10436,56,30.4,40,0.1);
INSERT INTO "OrderDetails" VALUES(10436,64,26.6,30,0.1);
INSERT INTO "OrderDetails" VALUES(10436,75,6.2,24,0.1);
INSERT INTO "OrderDetails" VALUES(10437,53,26.2,15,0);
INSERT INTO "OrderDetails" VALUES(10438,19,7.3,15,0.2);
INSERT INTO "OrderDetails" VALUES(10438,34,11.2,20,0.2);
INSERT INTO "OrderDetails" VALUES(10438,57,15.6,15,0.2);
INSERT INTO "OrderDetails" VALUES(10439,12,30.4,15,0);
INSERT INTO "OrderDetails" VALUES(10439,16,13.9,16,0);
INSERT INTO "OrderDetails" VALUES(10439,64,26.6,6,0);
INSERT INTO "OrderDetails" VALUES(10439,74,8,30,0);
INSERT INTO "OrderDetails" VALUES(10440,2,15.2,45,0.15);
INSERT INTO "OrderDetails" VALUES(10440,16,13.9,49,0.15);
INSERT INTO "OrderDetails" VALUES(10440,29,99,24,0.15);
INSERT INTO "OrderDetails" VALUES(10440,61,22.8,90,0.15);
INSERT INTO "OrderDetails" VALUES(10441,27,35.1,50,0);
INSERT INTO "OrderDetails" VALUES(10442,11,16.8,30,0);
INSERT INTO "OrderDetails" VALUES(10442,54,5.9,80,0);
INSERT INTO "OrderDetails" VALUES(10442,66,13.6,60,0);
INSERT INTO "OrderDetails" VALUES(10443,11,16.8,6,0.2);
INSERT INTO "OrderDetails" VALUES(10443,28,36.4,12,0);
INSERT INTO "OrderDetails" VALUES(10444,17,31.2,10,0);
INSERT INTO "OrderDetails" VALUES(10444,26,24.9,15,0);
INSERT INTO "OrderDetails" VALUES(10444,35,14.4,8,0);
INSERT INTO "OrderDetails" VALUES(10444,41,7.7,30,0);
INSERT INTO "OrderDetails" VALUES(10445,39,14.4,6,0);
INSERT INTO "OrderDetails" VALUES(10445,54,5.9,15,0);
INSERT INTO "OrderDetails" VALUES(10446,19,7.3,12,0.1);
INSERT INTO "OrderDetails" VALUES(10446,24,3.6,20,0.1);
INSERT INTO "OrderDetails" VALUES(10446,31,10,3,0.1);
INSERT INTO "OrderDetails" VALUES(10446,52,5.6,15,0.1);
INSERT INTO "OrderDetails" VALUES(10447,19,7.3,40,0);
INSERT INTO "OrderDetails" VALUES(10447,65,16.8,35,0);
INSERT INTO "OrderDetails" VALUES(10447,71,17.2,2,0);
INSERT INTO "OrderDetails" VALUES(10448,26,24.9,6,0);
INSERT INTO "OrderDetails" VALUES(10448,40,14.7,20,0);
INSERT INTO "OrderDetails" VALUES(10449,10,24.8,14,0);
INSERT INTO "OrderDetails" VALUES(10449,52,5.6,20,0);
INSERT INTO "OrderDetails" VALUES(10449,62,39.4,35,0);
INSERT INTO "OrderDetails" VALUES(10450,10,24.8,20,0.2);
INSERT INTO "OrderDetails" VALUES(10450,54,5.9,6,0.2);
INSERT INTO "OrderDetails" VALUES(10451,55,19.2,120,0.1);
INSERT INTO "OrderDetails" VALUES(10451,64,26.6,35,0.1);
INSERT INTO "OrderDetails" VALUES(10451,65,16.8,28,0.1);
INSERT INTO "OrderDetails" VALUES(10451,77,10.4,55,0.1);
INSERT INTO "OrderDetails" VALUES(10452,28,36.4,15,0);
INSERT INTO "OrderDetails" VALUES(10452,44,15.5,100,0.05);
INSERT INTO "OrderDetails" VALUES(10453,48,10.2,15,0.1);
INSERT INTO "OrderDetails" VALUES(10453,70,12,25,0.1);
INSERT INTO "OrderDetails" VALUES(10454,16,13.9,20,0.2);
INSERT INTO "OrderDetails" VALUES(10454,33,2,20,0.2);
INSERT INTO "OrderDetails" VALUES(10454,46,9.6,10,0.2);
INSERT INTO "OrderDetails" VALUES(10455,39,14.4,20,0);
INSERT INTO "OrderDetails" VALUES(10455,53,26.2,50,0);
INSERT INTO "OrderDetails" VALUES(10455,61,22.8,25,0);
INSERT INTO "OrderDetails" VALUES(10455,71,17.2,30,0);
INSERT INTO "OrderDetails" VALUES(10456,21,8,40,0.15);
INSERT INTO "OrderDetails" VALUES(10456,49,16,21,0.15);
INSERT INTO "OrderDetails" VALUES(10457,59,44,36,0);
INSERT INTO "OrderDetails" VALUES(10458,26,24.9,30,0);
INSERT INTO "OrderDetails" VALUES(10458,28,36.4,30,0);
INSERT INTO "OrderDetails" VALUES(10458,43,36.8,20,0);
INSERT INTO "OrderDetails" VALUES(10458,56,30.4,15,0);
INSERT INTO "OrderDetails" VALUES(10458,71,17.2,50,0);
INSERT INTO "OrderDetails" VALUES(10459,7,24,16,0.05);
INSERT INTO "OrderDetails" VALUES(10459,46,9.6,20,0.05);
INSERT INTO "OrderDetails" VALUES(10459,72,27.8,40,0);
INSERT INTO "OrderDetails" VALUES(10460,68,10,21,0.25);
INSERT INTO "OrderDetails" VALUES(10460,75,6.2,4,0.25);
INSERT INTO "OrderDetails" VALUES(10461,21,8,40,0.25);
INSERT INTO "OrderDetails" VALUES(10461,30,20.7,28,0.25);
INSERT INTO "OrderDetails" VALUES(10461,55,19.2,60,0.25);
INSERT INTO "OrderDetails" VALUES(10462,13,4.8,1,0);
INSERT INTO "OrderDetails" VALUES(10462,23,7.2,21,0);
INSERT INTO "OrderDetails" VALUES(10463,19,7.3,21,0);
INSERT INTO "OrderDetails" VALUES(10463,42,11.2,50,0);
INSERT INTO "OrderDetails" VALUES(10464,4,17.6,16,0.2);
INSERT INTO "OrderDetails" VALUES(10464,43,36.8,3,0);
INSERT INTO "OrderDetails" VALUES(10464,56,30.4,30,0.2);
INSERT INTO "OrderDetails" VALUES(10464,60,27.2,20,0);
INSERT INTO "OrderDetails" VALUES(10465,24,3.6,25,0);
INSERT INTO "OrderDetails" VALUES(10465,29,99,18,0.1);
INSERT INTO "OrderDetails" VALUES(10465,40,14.7,20,0);
INSERT INTO "OrderDetails" VALUES(10465,45,7.6,30,0.1);
INSERT INTO "OrderDetails" VALUES(10465,50,13,25,0);
INSERT INTO "OrderDetails" VALUES(10466,11,16.8,10,0);
INSERT INTO "OrderDetails" VALUES(10466,46,9.6,5,0);
INSERT INTO "OrderDetails" VALUES(10467,24,3.6,28,0);
INSERT INTO "OrderDetails" VALUES(10467,25,11.2,12,0);
INSERT INTO "OrderDetails" VALUES(10468,30,20.7,8,0);
INSERT INTO "OrderDetails" VALUES(10468,43,36.8,15,0);
INSERT INTO "OrderDetails" VALUES(10469,2,15.2,40,0.15);
INSERT INTO "OrderDetails" VALUES(10469,16,13.9,35,0.15);
INSERT INTO "OrderDetails" VALUES(10469,44,15.5,2,0.15);
INSERT INTO "OrderDetails" VALUES(10470,18,50,30,0);
INSERT INTO "OrderDetails" VALUES(10470,23,7.2,15,0);
INSERT INTO "OrderDetails" VALUES(10470,64,26.6,8,0);
INSERT INTO "OrderDetails" VALUES(10471,7,24,30,0);
INSERT INTO "OrderDetails" VALUES(10471,56,30.4,20,0);
INSERT INTO "OrderDetails" VALUES(10472,24,3.6,80,0.05);
INSERT INTO "OrderDetails" VALUES(10472,51,42.4,18,0);
INSERT INTO "OrderDetails" VALUES(10473,33,2,12,0);
INSERT INTO "OrderDetails" VALUES(10473,71,17.2,12,0);
INSERT INTO "OrderDetails" VALUES(10474,14,18.6,12,0);
INSERT INTO "OrderDetails" VALUES(10474,28,36.4,18,0);
INSERT INTO "OrderDetails" VALUES(10474,40,14.7,21,0);
INSERT INTO "OrderDetails" VALUES(10474,75,6.2,10,0);
INSERT INTO "OrderDetails" VALUES(10475,31,10,35,0.15);
INSERT INTO "OrderDetails" VALUES(10475,66,13.6,60,0.15);
INSERT INTO "OrderDetails" VALUES(10475,76,14.4,42,0.15);
INSERT INTO "OrderDetails" VALUES(10476,55,19.2,2,0.05);
INSERT INTO "OrderDetails" VALUES(10476,70,12,12,0);
INSERT INTO "OrderDetails" VALUES(10477,1,14.4,15,0);
INSERT INTO "OrderDetails" VALUES(10477,21,8,21,0.25);
INSERT INTO "OrderDetails" VALUES(10477,39,14.4,20,0.25);
INSERT INTO "OrderDetails" VALUES(10478,10,24.8,20,0.05);
INSERT INTO "OrderDetails" VALUES(10479,38,210.8,30,0);
INSERT INTO "OrderDetails" VALUES(10479,53,26.2,28,0);
INSERT INTO "OrderDetails" VALUES(10479,59,44,60,0);
INSERT INTO "OrderDetails" VALUES(10479,64,26.6,30,0);
INSERT INTO "OrderDetails" VALUES(10480,47,7.6,30,0);
INSERT INTO "OrderDetails" VALUES(10480,59,44,12,0);
INSERT INTO "OrderDetails" VALUES(10481,49,16,24,0);
INSERT INTO "OrderDetails" VALUES(10481,60,27.2,40,0);
INSERT INTO "OrderDetails" VALUES(10482,40,14.7,10,0);
INSERT INTO "OrderDetails" VALUES(10483,34,11.2,35,0.05);
INSERT INTO "OrderDetails" VALUES(10483,77,10.4,30,0.05);
INSERT INTO "OrderDetails" VALUES(10484,21,8,14,0);
INSERT INTO "OrderDetails" VALUES(10484,40,14.7,10,0);
INSERT INTO "OrderDetails" VALUES(10484,51,42.4,3,0);
INSERT INTO "OrderDetails" VALUES(10485,2,15.2,20,0.1);
INSERT INTO "OrderDetails" VALUES(10485,3,8,20,0.1);
INSERT INTO "OrderDetails" VALUES(10485,55,19.2,30,0.1);
INSERT INTO "OrderDetails" VALUES(10485,70,12,60,0.1);
INSERT INTO "OrderDetails" VALUES(10486,11,16.8,5,0);
INSERT INTO "OrderDetails" VALUES(10486,51,42.4,25,0);
INSERT INTO "OrderDetails" VALUES(10486,74,8,16,0);
INSERT INTO "OrderDetails" VALUES(10487,19,7.3,5,0);
INSERT INTO "OrderDetails" VALUES(10487,26,24.9,30,0);
INSERT INTO "OrderDetails" VALUES(10487,54,5.9,24,0.25);
INSERT INTO "OrderDetails" VALUES(10488,59,44,30,0);
INSERT INTO "OrderDetails" VALUES(10488,73,12,20,0.2);
INSERT INTO "OrderDetails" VALUES(10489,11,16.8,15,0.25);
INSERT INTO "OrderDetails" VALUES(10489,16,13.9,18,0);
INSERT INTO "OrderDetails" VALUES(10490,59,44,60,0);
INSERT INTO "OrderDetails" VALUES(10490,68,10,30,0);
INSERT INTO "OrderDetails" VALUES(10490,75,6.2,36,0);
INSERT INTO "OrderDetails" VALUES(10491,44,15.5,15,0.15);
INSERT INTO "OrderDetails" VALUES(10491,77,10.4,7,0.15);
INSERT INTO "OrderDetails" VALUES(10492,25,11.2,60,0.05);
INSERT INTO "OrderDetails" VALUES(10492,42,11.2,20,0.05);
INSERT INTO "OrderDetails" VALUES(10493,65,16.8,15,0.1);
INSERT INTO "OrderDetails" VALUES(10493,66,13.6,10,0.1);
INSERT INTO "OrderDetails" VALUES(10493,69,28.8,10,0.1);
INSERT INTO "OrderDetails" VALUES(10494,56,30.4,30,0);
INSERT INTO "OrderDetails" VALUES(10495,23,7.2,10,0);
INSERT INTO "OrderDetails" VALUES(10495,41,7.7,20,0);
INSERT INTO "OrderDetails" VALUES(10495,77,10.4,5,0);
INSERT INTO "OrderDetails" VALUES(10496,31,10,20,0.05);
INSERT INTO "OrderDetails" VALUES(10497,56,30.4,14,0);
INSERT INTO "OrderDetails" VALUES(10497,72,27.8,25,0);
INSERT INTO "OrderDetails" VALUES(10497,77,10.4,25,0);
INSERT INTO "OrderDetails" VALUES(10498,24,4.5,14,0);
INSERT INTO "OrderDetails" VALUES(10498,40,18.4,5,0);
INSERT INTO "OrderDetails" VALUES(10498,42,14,30,0);
INSERT INTO "OrderDetails" VALUES(10499,28,45.6,20,0);
INSERT INTO "OrderDetails" VALUES(10499,49,20,25,0);
INSERT INTO "OrderDetails" VALUES(10500,15,15.5,12,0.05);
INSERT INTO "OrderDetails" VALUES(10500,28,45.6,8,0.05);
INSERT INTO "OrderDetails" VALUES(10501,54,7.45,20,0);
INSERT INTO "OrderDetails" VALUES(10502,45,9.5,21,0);
INSERT INTO "OrderDetails" VALUES(10502,53,32.8,6,0);
INSERT INTO "OrderDetails" VALUES(10502,67,14,30,0);
INSERT INTO "OrderDetails" VALUES(10503,14,23.25,70,0);
INSERT INTO "OrderDetails" VALUES(10503,65,21.05,20,0);
INSERT INTO "OrderDetails" VALUES(10504,2,19,12,0);
INSERT INTO "OrderDetails" VALUES(10504,21,10,12,0);
INSERT INTO "OrderDetails" VALUES(10504,53,32.8,10,0);
INSERT INTO "OrderDetails" VALUES(10504,61,28.5,25,0);
INSERT INTO "OrderDetails" VALUES(10505,62,49.3,3,0);
INSERT INTO "OrderDetails" VALUES(10506,25,14,18,0.1);
INSERT INTO "OrderDetails" VALUES(10506,70,15,14,0.1);
INSERT INTO "OrderDetails" VALUES(10507,43,46,15,0.15);
INSERT INTO "OrderDetails" VALUES(10507,48,12.75,15,0.15);
INSERT INTO "OrderDetails" VALUES(10508,13,6,10,0);
INSERT INTO "OrderDetails" VALUES(10508,39,18,10,0);
INSERT INTO "OrderDetails" VALUES(10509,28,45.6,3,0);
INSERT INTO "OrderDetails" VALUES(10510,29,123.79,36,0);
INSERT INTO "OrderDetails" VALUES(10510,75,7.75,36,0.1);
INSERT INTO "OrderDetails" VALUES(10511,4,22,50,0.15);
INSERT INTO "OrderDetails" VALUES(10511,7,30,50,0.15);
INSERT INTO "OrderDetails" VALUES(10511,8,40,10,0.15);
INSERT INTO "OrderDetails" VALUES(10512,24,4.5,10,0.15);
INSERT INTO "OrderDetails" VALUES(10512,46,12,9,0.15);
INSERT INTO "OrderDetails" VALUES(10512,47,9.5,6,0.15);
INSERT INTO "OrderDetails" VALUES(10512,60,34,12,0.15);
INSERT INTO "OrderDetails" VALUES(10513,21,10,40,0.2);
INSERT INTO "OrderDetails" VALUES(10513,32,32,50,0.2);
INSERT INTO "OrderDetails" VALUES(10513,61,28.5,15,0.2);
INSERT INTO "OrderDetails" VALUES(10514,20,81,39,0);
INSERT INTO "OrderDetails" VALUES(10514,28,45.6,35,0);
INSERT INTO "OrderDetails" VALUES(10514,56,38,70,0);
INSERT INTO "OrderDetails" VALUES(10514,65,21.05,39,0);
INSERT INTO "OrderDetails" VALUES(10514,75,7.75,50,0);
INSERT INTO "OrderDetails" VALUES(10515,9,97,16,0.15);
INSERT INTO "OrderDetails" VALUES(10515,16,17.45,50,0);
INSERT INTO "OrderDetails" VALUES(10515,27,43.9,120,0);
INSERT INTO "OrderDetails" VALUES(10515,33,2.5,16,0.15);
INSERT INTO "OrderDetails" VALUES(10515,60,34,84,0.15);
INSERT INTO "OrderDetails" VALUES(10516,18,62.5,25,0.1);
INSERT INTO "OrderDetails" VALUES(10516,41,9.65,80,0.1);
INSERT INTO "OrderDetails" VALUES(10516,42,14,20,0);
INSERT INTO "OrderDetails" VALUES(10517,52,7,6,0);
INSERT INTO "OrderDetails" VALUES(10517,59,55,4,0);
INSERT INTO "OrderDetails" VALUES(10517,70,15,6,0);
INSERT INTO "OrderDetails" VALUES(10518,24,4.5,5,0);
INSERT INTO "OrderDetails" VALUES(10518,38,263.5,15,0);
INSERT INTO "OrderDetails" VALUES(10518,44,19.45,9,0);
INSERT INTO "OrderDetails" VALUES(10519,10,31,16,0.05);
INSERT INTO "OrderDetails" VALUES(10519,56,38,40,0);
INSERT INTO "OrderDetails" VALUES(10519,60,34,10,0.05);
INSERT INTO "OrderDetails" VALUES(10520,24,4.5,8,0);
INSERT INTO "OrderDetails" VALUES(10520,53,32.8,5,0);
INSERT INTO "OrderDetails" VALUES(10521,35,18,3,0);
INSERT INTO "OrderDetails" VALUES(10521,41,9.65,10,0);
INSERT INTO "OrderDetails" VALUES(10521,68,12.5,6,0);
INSERT INTO "OrderDetails" VALUES(10522,1,18,40,0.2);
INSERT INTO "OrderDetails" VALUES(10522,8,40,24,0);
INSERT INTO "OrderDetails" VALUES(10522,30,25.89,20,0.2);
INSERT INTO "OrderDetails" VALUES(10522,40,18.4,25,0.2);
INSERT INTO "OrderDetails" VALUES(10523,17,39,25,0.1);
INSERT INTO "OrderDetails" VALUES(10523,20,81,15,0.1);
INSERT INTO "OrderDetails" VALUES(10523,37,26,18,0.1);
INSERT INTO "OrderDetails" VALUES(10523,41,9.65,6,0.1);
INSERT INTO "OrderDetails" VALUES(10524,10,31,2,0);
INSERT INTO "OrderDetails" VALUES(10524,30,25.89,10,0);
INSERT INTO "OrderDetails" VALUES(10524,43,46,60,0);
INSERT INTO "OrderDetails" VALUES(10524,54,7.45,15,0);
INSERT INTO "OrderDetails" VALUES(10525,36,19,30,0);
INSERT INTO "OrderDetails" VALUES(10525,40,18.4,15,0.1);
INSERT INTO "OrderDetails" VALUES(10526,1,18,8,0.15);
INSERT INTO "OrderDetails" VALUES(10526,13,6,10,0);
INSERT INTO "OrderDetails" VALUES(10526,56,38,30,0.15);
INSERT INTO "OrderDetails" VALUES(10527,4,22,50,0.1);
INSERT INTO "OrderDetails" VALUES(10527,36,19,30,0.1);
INSERT INTO "OrderDetails" VALUES(10528,11,21,3,0);
INSERT INTO "OrderDetails" VALUES(10528,33,2.5,8,0.2);
INSERT INTO "OrderDetails" VALUES(10528,72,34.8,9,0);
INSERT INTO "OrderDetails" VALUES(10529,55,24,14,0);
INSERT INTO "OrderDetails" VALUES(10529,68,12.5,20,0);
INSERT INTO "OrderDetails" VALUES(10529,69,36,10,0);
INSERT INTO "OrderDetails" VALUES(10530,17,39,40,0);
INSERT INTO "OrderDetails" VALUES(10530,43,46,25,0);
INSERT INTO "OrderDetails" VALUES(10530,61,28.5,20,0);
INSERT INTO "OrderDetails" VALUES(10530,76,18,50,0);
INSERT INTO "OrderDetails" VALUES(10531,59,55,2,0);
INSERT INTO "OrderDetails" VALUES(10532,30,25.89,15,0);
INSERT INTO "OrderDetails" VALUES(10532,66,17,24,0);
INSERT INTO "OrderDetails" VALUES(10533,4,22,50,0.05);
INSERT INTO "OrderDetails" VALUES(10533,72,34.8,24,0);
INSERT INTO "OrderDetails" VALUES(10533,73,15,24,0.05);
INSERT INTO "OrderDetails" VALUES(10534,30,25.89,10,0);
INSERT INTO "OrderDetails" VALUES(10534,40,18.4,10,0.2);
INSERT INTO "OrderDetails" VALUES(10534,54,7.45,10,0.2);
INSERT INTO "OrderDetails" VALUES(10535,11,21,50,0.1);
INSERT INTO "OrderDetails" VALUES(10535,40,18.4,10,0.1);
INSERT INTO "OrderDetails" VALUES(10535,57,19.5,5,0.1);
INSERT INTO "OrderDetails" VALUES(10535,59,55,15,0.1);
INSERT INTO "OrderDetails" VALUES(10536,12,38,15,0.25);
INSERT INTO "OrderDetails" VALUES(10536,31,12.5,20,0);
INSERT INTO "OrderDetails" VALUES(10536,33,2.5,30,0);
INSERT INTO "OrderDetails" VALUES(10536,60,34,35,0.25);
INSERT INTO "OrderDetails" VALUES(10537,31,12.5,30,0);
INSERT INTO "OrderDetails" VALUES(10537,51,53,6,0);
INSERT INTO "OrderDetails" VALUES(10537,58,13.25,20,0);
INSERT INTO "OrderDetails" VALUES(10537,72,34.8,21,0);
INSERT INTO "OrderDetails" VALUES(10537,73,15,9,0);
INSERT INTO "OrderDetails" VALUES(10538,70,15,7,0);
INSERT INTO "OrderDetails" VALUES(10538,72,34.8,1,0);
INSERT INTO "OrderDetails" VALUES(10539,13,6,8,0);
INSERT INTO "OrderDetails" VALUES(10539,21,10,15,0);
INSERT INTO "OrderDetails" VALUES(10539,33,2.5,15,0);
INSERT INTO "OrderDetails" VALUES(10539,49,20,6,0);
INSERT INTO "OrderDetails" VALUES(10540,3,10,60,0);
INSERT INTO "OrderDetails" VALUES(10540,26,31.23,40,0);
INSERT INTO "OrderDetails" VALUES(10540,38,263.5,30,0);
INSERT INTO "OrderDetails" VALUES(10540,68,12.5,35,0);
INSERT INTO "OrderDetails" VALUES(10541,24,4.5,35,0.1);
INSERT INTO "OrderDetails" VALUES(10541,38,263.5,4,0.1);
INSERT INTO "OrderDetails" VALUES(10541,65,21.05,36,0.1);
INSERT INTO "OrderDetails" VALUES(10541,71,21.5,9,0.1);
INSERT INTO "OrderDetails" VALUES(10542,11,21,15,0.05);
INSERT INTO "OrderDetails" VALUES(10542,54,7.45,24,0.05);
INSERT INTO "OrderDetails" VALUES(10543,12,38,30,0.15);
INSERT INTO "OrderDetails" VALUES(10543,23,9,70,0.15);
INSERT INTO "OrderDetails" VALUES(10544,28,45.6,7,0);
INSERT INTO "OrderDetails" VALUES(10544,67,14,7,0);
INSERT INTO "OrderDetails" VALUES(10545,11,21,10,0);
INSERT INTO "OrderDetails" VALUES(10546,7,30,10,0);
INSERT INTO "OrderDetails" VALUES(10546,35,18,30,0);
INSERT INTO "OrderDetails" VALUES(10546,62,49.3,40,0);
INSERT INTO "OrderDetails" VALUES(10547,32,32,24,0.15);
INSERT INTO "OrderDetails" VALUES(10547,36,19,60,0);
INSERT INTO "OrderDetails" VALUES(10548,34,14,10,0.25);
INSERT INTO "OrderDetails" VALUES(10548,41,9.65,14,0);
INSERT INTO "OrderDetails" VALUES(10549,31,12.5,55,0.15);
INSERT INTO "OrderDetails" VALUES(10549,45,9.5,100,0.15);
INSERT INTO "OrderDetails" VALUES(10549,51,53,48,0.15);
INSERT INTO "OrderDetails" VALUES(10550,17,39,8,0.1);
INSERT INTO "OrderDetails" VALUES(10550,19,9.2,10,0);
INSERT INTO "OrderDetails" VALUES(10550,21,10,6,0.1);
INSERT INTO "OrderDetails" VALUES(10550,61,28.5,10,0.1);
INSERT INTO "OrderDetails" VALUES(10551,16,17.45,40,0.15);
INSERT INTO "OrderDetails" VALUES(10551,35,18,20,0.15);
INSERT INTO "OrderDetails" VALUES(10551,44,19.45,40,0);
INSERT INTO "OrderDetails" VALUES(10552,69,36,18,0);
INSERT INTO "OrderDetails" VALUES(10552,75,7.75,30,0);
INSERT INTO "OrderDetails" VALUES(10553,11,21,15,0);
INSERT INTO "OrderDetails" VALUES(10553,16,17.45,14,0);
INSERT INTO "OrderDetails" VALUES(10553,22,21,24,0);
INSERT INTO "OrderDetails" VALUES(10553,31,12.5,30,0);
INSERT INTO "OrderDetails" VALUES(10553,35,18,6,0);
INSERT INTO "OrderDetails" VALUES(10554,16,17.45,30,0.05);
INSERT INTO "OrderDetails" VALUES(10554,23,9,20,0.05);
INSERT INTO "OrderDetails" VALUES(10554,62,49.3,20,0.05);
INSERT INTO "OrderDetails" VALUES(10554,77,13,10,0.05);
INSERT INTO "OrderDetails" VALUES(10555,14,23.25,30,0.2);
INSERT INTO "OrderDetails" VALUES(10555,19,9.2,35,0.2);
INSERT INTO "OrderDetails" VALUES(10555,24,4.5,18,0.2);
INSERT INTO "OrderDetails" VALUES(10555,51,53,20,0.2);
INSERT INTO "OrderDetails" VALUES(10555,56,38,40,0.2);
INSERT INTO "OrderDetails" VALUES(10556,72,34.8,24,0);
INSERT INTO "OrderDetails" VALUES(10557,64,33.25,30,0);
INSERT INTO "OrderDetails" VALUES(10557,75,7.75,20,0);
INSERT INTO "OrderDetails" VALUES(10558,47,9.5,25,0);
INSERT INTO "OrderDetails" VALUES(10558,51,53,20,0);
INSERT INTO "OrderDetails" VALUES(10558,52,7,30,0);
INSERT INTO "OrderDetails" VALUES(10558,53,32.8,18,0);
INSERT INTO "OrderDetails" VALUES(10558,73,15,3,0);
INSERT INTO "OrderDetails" VALUES(10559,41,9.65,12,0.05);
INSERT INTO "OrderDetails" VALUES(10559,55,24,18,0.05);
INSERT INTO "OrderDetails" VALUES(10560,30,25.89,20,0);
INSERT INTO "OrderDetails" VALUES(10560,62,49.3,15,0.25);
INSERT INTO "OrderDetails" VALUES(10561,44,19.45,10,0);
INSERT INTO "OrderDetails" VALUES(10561,51,53,50,0);
INSERT INTO "OrderDetails" VALUES(10562,33,2.5,20,0.1);
INSERT INTO "OrderDetails" VALUES(10562,62,49.3,10,0.1);
INSERT INTO "OrderDetails" VALUES(10563,36,19,25,0);
INSERT INTO "OrderDetails" VALUES(10563,52,7,70,0);
INSERT INTO "OrderDetails" VALUES(10564,17,39,16,0.05);
INSERT INTO "OrderDetails" VALUES(10564,31,12.5,6,0.05);
INSERT INTO "OrderDetails" VALUES(10564,55,24,25,0.05);
INSERT INTO "OrderDetails" VALUES(10565,24,4.5,25,0.1);
INSERT INTO "OrderDetails" VALUES(10565,64,33.25,18,0.1);
INSERT INTO "OrderDetails" VALUES(10566,11,21,35,0.15);
INSERT INTO "OrderDetails" VALUES(10566,18,62.5,18,0.15);
INSERT INTO "OrderDetails" VALUES(10566,76,18,10,0);
INSERT INTO "OrderDetails" VALUES(10567,31,12.5,60,0.2);
INSERT INTO "OrderDetails" VALUES(10567,51,53,3,0);
INSERT INTO "OrderDetails" VALUES(10567,59,55,40,0.2);
INSERT INTO "OrderDetails" VALUES(10568,10,31,5,0);
INSERT INTO "OrderDetails" VALUES(10569,31,12.5,35,0.2);
INSERT INTO "OrderDetails" VALUES(10569,76,18,30,0);
INSERT INTO "OrderDetails" VALUES(10570,11,21,15,0.05);
INSERT INTO "OrderDetails" VALUES(10570,56,38,60,0.05);
INSERT INTO "OrderDetails" VALUES(10571,14,23.25,11,0.15);
INSERT INTO "OrderDetails" VALUES(10571,42,14,28,0.15);
INSERT INTO "OrderDetails" VALUES(10572,16,17.45,12,0.1);
INSERT INTO "OrderDetails" VALUES(10572,32,32,10,0.1);
INSERT INTO "OrderDetails" VALUES(10572,40,18.4,50,0);
INSERT INTO "OrderDetails" VALUES(10572,75,7.75,15,0.1);
INSERT INTO "OrderDetails" VALUES(10573,17,39,18,0);
INSERT INTO "OrderDetails" VALUES(10573,34,14,40,0);
INSERT INTO "OrderDetails" VALUES(10573,53,32.8,25,0);
INSERT INTO "OrderDetails" VALUES(10574,33,2.5,14,0);
INSERT INTO "OrderDetails" VALUES(10574,40,18.4,2,0);
INSERT INTO "OrderDetails" VALUES(10574,62,49.3,10,0);
INSERT INTO "OrderDetails" VALUES(10574,64,33.25,6,0);
INSERT INTO "OrderDetails" VALUES(10575,59,55,12,0);
INSERT INTO "OrderDetails" VALUES(10575,63,43.9,6,0);
INSERT INTO "OrderDetails" VALUES(10575,72,34.8,30,0);
INSERT INTO "OrderDetails" VALUES(10575,76,18,10,0);
INSERT INTO "OrderDetails" VALUES(10576,1,18,10,0);
INSERT INTO "OrderDetails" VALUES(10576,31,12.5,20,0);
INSERT INTO "OrderDetails" VALUES(10576,44,19.45,21,0);
INSERT INTO "OrderDetails" VALUES(10577,39,18,10,0);
INSERT INTO "OrderDetails" VALUES(10577,75,7.75,20,0);
INSERT INTO "OrderDetails" VALUES(10577,77,13,18,0);
INSERT INTO "OrderDetails" VALUES(10578,35,18,20,0);
INSERT INTO "OrderDetails" VALUES(10578,57,19.5,6,0);
INSERT INTO "OrderDetails" VALUES(10579,15,15.5,10,0);
INSERT INTO "OrderDetails" VALUES(10579,75,7.75,21,0);
INSERT INTO "OrderDetails" VALUES(10580,14,23.25,15,0.05);
INSERT INTO "OrderDetails" VALUES(10580,41,9.65,9,0.05);
INSERT INTO "OrderDetails" VALUES(10580,65,21.05,30,0.05);
INSERT INTO "OrderDetails" VALUES(10581,75,7.75,50,0.2);
INSERT INTO "OrderDetails" VALUES(10582,57,19.5,4,0);
INSERT INTO "OrderDetails" VALUES(10582,76,18,14,0);
INSERT INTO "OrderDetails" VALUES(10583,29,123.79,10,0);
INSERT INTO "OrderDetails" VALUES(10583,60,34,24,0.15);
INSERT INTO "OrderDetails" VALUES(10583,69,36,10,0.15);
INSERT INTO "OrderDetails" VALUES(10584,31,12.5,50,0.05);
INSERT INTO "OrderDetails" VALUES(10585,47,9.5,15,0);
INSERT INTO "OrderDetails" VALUES(10586,52,7,4,0.15);
INSERT INTO "OrderDetails" VALUES(10587,26,31.23,6,0);
INSERT INTO "OrderDetails" VALUES(10587,35,18,20,0);
INSERT INTO "OrderDetails" VALUES(10587,77,13,20,0);
INSERT INTO "OrderDetails" VALUES(10588,18,62.5,40,0.2);
INSERT INTO "OrderDetails" VALUES(10588,42,14,100,0.2);
INSERT INTO "OrderDetails" VALUES(10589,35,18,4,0);
INSERT INTO "OrderDetails" VALUES(10590,1,18,20,0);
INSERT INTO "OrderDetails" VALUES(10590,77,13,60,0.05);
INSERT INTO "OrderDetails" VALUES(10591,3,10,14,0);
INSERT INTO "OrderDetails" VALUES(10591,7,30,10,0);
INSERT INTO "OrderDetails" VALUES(10591,54,7.45,50,0);
INSERT INTO "OrderDetails" VALUES(10592,15,15.5,25,0.05);
INSERT INTO "OrderDetails" VALUES(10592,26,31.23,5,0.05);
INSERT INTO "OrderDetails" VALUES(10593,20,81,21,0.2);
INSERT INTO "OrderDetails" VALUES(10593,69,36,20,0.2);
INSERT INTO "OrderDetails" VALUES(10593,76,18,4,0.2);
INSERT INTO "OrderDetails" VALUES(10594,52,7,24,0);
INSERT INTO "OrderDetails" VALUES(10594,58,13.25,30,0);
INSERT INTO "OrderDetails" VALUES(10595,35,18,30,0.25);
INSERT INTO "OrderDetails" VALUES(10595,61,28.5,120,0.25);
INSERT INTO "OrderDetails" VALUES(10595,69,36,65,0.25);
INSERT INTO "OrderDetails" VALUES(10596,56,38,5,0.2);
INSERT INTO "OrderDetails" VALUES(10596,63,43.9,24,0.2);
INSERT INTO "OrderDetails" VALUES(10596,75,7.75,30,0.2);
INSERT INTO "OrderDetails" VALUES(10597,24,4.5,35,0.2);
INSERT INTO "OrderDetails" VALUES(10597,57,19.5,20,0);
INSERT INTO "OrderDetails" VALUES(10597,65,21.05,12,0.2);
INSERT INTO "OrderDetails" VALUES(10598,27,43.9,50,0);
INSERT INTO "OrderDetails" VALUES(10598,71,21.5,9,0);
INSERT INTO "OrderDetails" VALUES(10599,62,49.3,10,0);
INSERT INTO "OrderDetails" VALUES(10600,54,7.45,4,0);
INSERT INTO "OrderDetails" VALUES(10600,73,15,30,0);
INSERT INTO "OrderDetails" VALUES(10601,13,6,60,0);
INSERT INTO "OrderDetails" VALUES(10601,59,55,35,0);
INSERT INTO "OrderDetails" VALUES(10602,77,13,5,0.25);
INSERT INTO "OrderDetails" VALUES(10603,22,21,48,0);
INSERT INTO "OrderDetails" VALUES(10603,49,20,25,0.05);
INSERT INTO "OrderDetails" VALUES(10604,48,12.75,6,0.1);
INSERT INTO "OrderDetails" VALUES(10604,76,18,10,0.1);
INSERT INTO "OrderDetails" VALUES(10605,16,17.45,30,0.05);
INSERT INTO "OrderDetails" VALUES(10605,59,55,20,0.05);
INSERT INTO "OrderDetails" VALUES(10605,60,34,70,0.05);
INSERT INTO "OrderDetails" VALUES(10605,71,21.5,15,0.05);
INSERT INTO "OrderDetails" VALUES(10606,4,22,20,0.2);
INSERT INTO "OrderDetails" VALUES(10606,55,24,20,0.2);
INSERT INTO "OrderDetails" VALUES(10606,62,49.3,10,0.2);
INSERT INTO "OrderDetails" VALUES(10607,7,30,45,0);
INSERT INTO "OrderDetails" VALUES(10607,17,39,100,0);
INSERT INTO "OrderDetails" VALUES(10607,33,2.5,14,0);
INSERT INTO "OrderDetails" VALUES(10607,40,18.4,42,0);
INSERT INTO "OrderDetails" VALUES(10607,72,34.8,12,0);
INSERT INTO "OrderDetails" VALUES(10608,56,38,28,0);
INSERT INTO "OrderDetails" VALUES(10609,1,18,3,0);
INSERT INTO "OrderDetails" VALUES(10609,10,31,10,0);
INSERT INTO "OrderDetails" VALUES(10609,21,10,6,0);
INSERT INTO "OrderDetails" VALUES(10610,36,19,21,0.25);
INSERT INTO "OrderDetails" VALUES(10611,1,18,6,0);
INSERT INTO "OrderDetails" VALUES(10611,2,19,10,0);
INSERT INTO "OrderDetails" VALUES(10611,60,34,15,0);
INSERT INTO "OrderDetails" VALUES(10612,10,31,70,0);
INSERT INTO "OrderDetails" VALUES(10612,36,19,55,0);
INSERT INTO "OrderDetails" VALUES(10612,49,20,18,0);
INSERT INTO "OrderDetails" VALUES(10612,60,34,40,0);
INSERT INTO "OrderDetails" VALUES(10612,76,18,80,0);
INSERT INTO "OrderDetails" VALUES(10613,13,6,8,0.1);
INSERT INTO "OrderDetails" VALUES(10613,75,7.75,40,0);
INSERT INTO "OrderDetails" VALUES(10614,11,21,14,0);
INSERT INTO "OrderDetails" VALUES(10614,21,10,8,0);
INSERT INTO "OrderDetails" VALUES(10614,39,18,5,0);
INSERT INTO "OrderDetails" VALUES(10615,55,24,5,0);
INSERT INTO "OrderDetails" VALUES(10616,38,263.5,15,0.05);
INSERT INTO "OrderDetails" VALUES(10616,56,38,14,0);
INSERT INTO "OrderDetails" VALUES(10616,70,15,15,0.05);
INSERT INTO "OrderDetails" VALUES(10616,71,21.5,15,0.05);
INSERT INTO "OrderDetails" VALUES(10617,59,55,30,0.15);
INSERT INTO "OrderDetails" VALUES(10618,6,25,70,0);
INSERT INTO "OrderDetails" VALUES(10618,56,38,20,0);
INSERT INTO "OrderDetails" VALUES(10618,68,12.5,15,0);
INSERT INTO "OrderDetails" VALUES(10619,21,10,42,0);
INSERT INTO "OrderDetails" VALUES(10619,22,21,40,0);
INSERT INTO "OrderDetails" VALUES(10620,24,4.5,5,0);
INSERT INTO "OrderDetails" VALUES(10620,52,7,5,0);
INSERT INTO "OrderDetails" VALUES(10621,19,9.2,5,0);
INSERT INTO "OrderDetails" VALUES(10621,23,9,10,0);
INSERT INTO "OrderDetails" VALUES(10621,70,15,20,0);
INSERT INTO "OrderDetails" VALUES(10621,71,21.5,15,0);
INSERT INTO "OrderDetails" VALUES(10622,2,19,20,0);
INSERT INTO "OrderDetails" VALUES(10622,68,12.5,18,0.2);
INSERT INTO "OrderDetails" VALUES(10623,14,23.25,21,0);
INSERT INTO "OrderDetails" VALUES(10623,19,9.2,15,0.1);
INSERT INTO "OrderDetails" VALUES(10623,21,10,25,0.1);
INSERT INTO "OrderDetails" VALUES(10623,24,4.5,3,0);
INSERT INTO "OrderDetails" VALUES(10623,35,18,30,0.1);
INSERT INTO "OrderDetails" VALUES(10624,28,45.6,10,0);
INSERT INTO "OrderDetails" VALUES(10624,29,123.79,6,0);
INSERT INTO "OrderDetails" VALUES(10624,44,19.45,10,0);
INSERT INTO "OrderDetails" VALUES(10625,14,23.25,3,0);
INSERT INTO "OrderDetails" VALUES(10625,42,14,5,0);
INSERT INTO "OrderDetails" VALUES(10625,60,34,10,0);
INSERT INTO "OrderDetails" VALUES(10626,53,32.8,12,0);
INSERT INTO "OrderDetails" VALUES(10626,60,34,20,0);
INSERT INTO "OrderDetails" VALUES(10626,71,21.5,20,0);
INSERT INTO "OrderDetails" VALUES(10627,62,49.3,15,0);
INSERT INTO "OrderDetails" VALUES(10627,73,15,35,0.15);
INSERT INTO "OrderDetails" VALUES(10628,1,18,25,0);
INSERT INTO "OrderDetails" VALUES(10629,29,123.79,20,0);
INSERT INTO "OrderDetails" VALUES(10629,64,33.25,9,0);
INSERT INTO "OrderDetails" VALUES(10630,55,24,12,0.05);
INSERT INTO "OrderDetails" VALUES(10630,76,18,35,0);
INSERT INTO "OrderDetails" VALUES(10631,75,7.75,8,0.1);
INSERT INTO "OrderDetails" VALUES(10632,2,19,30,0.05);
INSERT INTO "OrderDetails" VALUES(10632,33,2.5,20,0.05);
INSERT INTO "OrderDetails" VALUES(10633,12,38,36,0.15);
INSERT INTO "OrderDetails" VALUES(10633,13,6,13,0.15);
INSERT INTO "OrderDetails" VALUES(10633,26,31.23,35,0.15);
INSERT INTO "OrderDetails" VALUES(10633,62,49.3,80,0.15);
INSERT INTO "OrderDetails" VALUES(10634,7,30,35,0);
INSERT INTO "OrderDetails" VALUES(10634,18,62.5,50,0);
INSERT INTO "OrderDetails" VALUES(10634,51,53,15,0);
INSERT INTO "OrderDetails" VALUES(10634,75,7.75,2,0);
INSERT INTO "OrderDetails" VALUES(10635,4,22,10,0.1);
INSERT INTO "OrderDetails" VALUES(10635,5,21.35,15,0.1);
INSERT INTO "OrderDetails" VALUES(10635,22,21,40,0);
INSERT INTO "OrderDetails" VALUES(10636,4,22,25,0);
INSERT INTO "OrderDetails" VALUES(10636,58,13.25,6,0);
INSERT INTO "OrderDetails" VALUES(10637,11,21,10,0);
INSERT INTO "OrderDetails" VALUES(10637,50,16.25,25,0.05);
INSERT INTO "OrderDetails" VALUES(10637,56,38,60,0.05);
INSERT INTO "OrderDetails" VALUES(10638,45,9.5,20,0);
INSERT INTO "OrderDetails" VALUES(10638,65,21.05,21,0);
INSERT INTO "OrderDetails" VALUES(10638,72,34.8,60,0);
INSERT INTO "OrderDetails" VALUES(10639,18,62.5,8,0);
INSERT INTO "OrderDetails" VALUES(10640,69,36,20,0.25);
INSERT INTO "OrderDetails" VALUES(10640,70,15,15,0.25);
INSERT INTO "OrderDetails" VALUES(10641,2,19,50,0);
INSERT INTO "OrderDetails" VALUES(10641,40,18.4,60,0);
INSERT INTO "OrderDetails" VALUES(10642,21,10,30,0.2);
INSERT INTO "OrderDetails" VALUES(10642,61,28.5,20,0.2);
INSERT INTO "OrderDetails" VALUES(10643,28,45.6,15,0.25);
INSERT INTO "OrderDetails" VALUES(10643,39,18,21,0.25);
INSERT INTO "OrderDetails" VALUES(10643,46,12,2,0.25);
INSERT INTO "OrderDetails" VALUES(10644,18,62.5,4,0.1);
INSERT INTO "OrderDetails" VALUES(10644,43,46,20,0);
INSERT INTO "OrderDetails" VALUES(10644,46,12,21,0.1);
INSERT INTO "OrderDetails" VALUES(10645,18,62.5,20,0);
INSERT INTO "OrderDetails" VALUES(10645,36,19,15,0);
INSERT INTO "OrderDetails" VALUES(10646,1,18,15,0.25);
INSERT INTO "OrderDetails" VALUES(10646,10,31,18,0.25);
INSERT INTO "OrderDetails" VALUES(10646,71,21.5,30,0.25);
INSERT INTO "OrderDetails" VALUES(10646,77,13,35,0.25);
INSERT INTO "OrderDetails" VALUES(10647,19,9.2,30,0);
INSERT INTO "OrderDetails" VALUES(10647,39,18,20,0);
INSERT INTO "OrderDetails" VALUES(10648,22,21,15,0);
INSERT INTO "OrderDetails" VALUES(10648,24,4.5,15,0.15);
INSERT INTO "OrderDetails" VALUES(10649,28,45.6,20,0);
INSERT INTO "OrderDetails" VALUES(10649,72,34.8,15,0);
INSERT INTO "OrderDetails" VALUES(10650,30,25.89,30,0);
INSERT INTO "OrderDetails" VALUES(10650,53,32.8,25,0.05);
INSERT INTO "OrderDetails" VALUES(10650,54,7.45,30,0);
INSERT INTO "OrderDetails" VALUES(10651,19,9.2,12,0.25);
INSERT INTO "OrderDetails" VALUES(10651,22,21,20,0.25);
INSERT INTO "OrderDetails" VALUES(10652,30,25.89,2,0.25);
INSERT INTO "OrderDetails" VALUES(10652,42,14,20,0);
INSERT INTO "OrderDetails" VALUES(10653,16,17.45,30,0.1);
INSERT INTO "OrderDetails" VALUES(10653,60,34,20,0.1);
INSERT INTO "OrderDetails" VALUES(10654,4,22,12,0.1);
INSERT INTO "OrderDetails" VALUES(10654,39,18,20,0.1);
INSERT INTO "OrderDetails" VALUES(10654,54,7.45,6,0.1);
INSERT INTO "OrderDetails" VALUES(10655,41,9.65,20,0.2);
INSERT INTO "OrderDetails" VALUES(10656,14,23.25,3,0.1);
INSERT INTO "OrderDetails" VALUES(10656,44,19.45,28,0.1);
INSERT INTO "OrderDetails" VALUES(10656,47,9.5,6,0.1);
INSERT INTO "OrderDetails" VALUES(10657,15,15.5,50,0);
INSERT INTO "OrderDetails" VALUES(10657,41,9.65,24,0);
INSERT INTO "OrderDetails" VALUES(10657,46,12,45,0);
INSERT INTO "OrderDetails" VALUES(10657,47,9.5,10,0);
INSERT INTO "OrderDetails" VALUES(10657,56,38,45,0);
INSERT INTO "OrderDetails" VALUES(10657,60,34,30,0);
INSERT INTO "OrderDetails" VALUES(10658,21,10,60,0);
INSERT INTO "OrderDetails" VALUES(10658,40,18.4,70,0.05);
INSERT INTO "OrderDetails" VALUES(10658,60,34,55,0.05);
INSERT INTO "OrderDetails" VALUES(10658,77,13,70,0.05);
INSERT INTO "OrderDetails" VALUES(10659,31,12.5,20,0.05);
INSERT INTO "OrderDetails" VALUES(10659,40,18.4,24,0.05);
INSERT INTO "OrderDetails" VALUES(10659,70,15,40,0.05);
INSERT INTO "OrderDetails" VALUES(10660,20,81,21,0);
INSERT INTO "OrderDetails" VALUES(10661,39,18,3,0.2);
INSERT INTO "OrderDetails" VALUES(10661,58,13.25,49,0.2);
INSERT INTO "OrderDetails" VALUES(10662,68,12.5,10,0);
INSERT INTO "OrderDetails" VALUES(10663,40,18.4,30,0.05);
INSERT INTO "OrderDetails" VALUES(10663,42,14,30,0.05);
INSERT INTO "OrderDetails" VALUES(10663,51,53,20,0.05);
INSERT INTO "OrderDetails" VALUES(10664,10,31,24,0.15);
INSERT INTO "OrderDetails" VALUES(10664,56,38,12,0.15);
INSERT INTO "OrderDetails" VALUES(10664,65,21.05,15,0.15);
INSERT INTO "OrderDetails" VALUES(10665,51,53,20,0);
INSERT INTO "OrderDetails" VALUES(10665,59,55,1,0);
INSERT INTO "OrderDetails" VALUES(10665,76,18,10,0);
INSERT INTO "OrderDetails" VALUES(10666,29,123.79,36,0);
INSERT INTO "OrderDetails" VALUES(10666,65,21.05,10,0);
INSERT INTO "OrderDetails" VALUES(10667,69,36,45,0.2);
INSERT INTO "OrderDetails" VALUES(10667,71,21.5,14,0.2);
INSERT INTO "OrderDetails" VALUES(10668,31,12.5,8,0.1);
INSERT INTO "OrderDetails" VALUES(10668,55,24,4,0.1);
INSERT INTO "OrderDetails" VALUES(10668,64,33.25,15,0.1);
INSERT INTO "OrderDetails" VALUES(10669,36,19,30,0);
INSERT INTO "OrderDetails" VALUES(10670,23,9,32,0);
INSERT INTO "OrderDetails" VALUES(10670,46,12,60,0);
INSERT INTO "OrderDetails" VALUES(10670,67,14,25,0);
INSERT INTO "OrderDetails" VALUES(10670,73,15,50,0);
INSERT INTO "OrderDetails" VALUES(10670,75,7.75,25,0);
INSERT INTO "OrderDetails" VALUES(10671,16,17.45,10,0);
INSERT INTO "OrderDetails" VALUES(10671,62,49.3,10,0);
INSERT INTO "OrderDetails" VALUES(10671,65,21.05,12,0);
INSERT INTO "OrderDetails" VALUES(10672,38,263.5,15,0.1);
INSERT INTO "OrderDetails" VALUES(10672,71,21.5,12,0);
INSERT INTO "OrderDetails" VALUES(10673,16,17.45,3,0);
INSERT INTO "OrderDetails" VALUES(10673,42,14,6,0);
INSERT INTO "OrderDetails" VALUES(10673,43,46,6,0);
INSERT INTO "OrderDetails" VALUES(10674,23,9,5,0);
INSERT INTO "OrderDetails" VALUES(10675,14,23.25,30,0);
INSERT INTO "OrderDetails" VALUES(10675,53,32.8,10,0);
INSERT INTO "OrderDetails" VALUES(10675,58,13.25,30,0);
INSERT INTO "OrderDetails" VALUES(10676,10,31,2,0);
INSERT INTO "OrderDetails" VALUES(10676,19,9.2,7,0);
INSERT INTO "OrderDetails" VALUES(10676,44,19.45,21,0);
INSERT INTO "OrderDetails" VALUES(10677,26,31.23,30,0.15);
INSERT INTO "OrderDetails" VALUES(10677,33,2.5,8,0.15);
INSERT INTO "OrderDetails" VALUES(10678,12,38,100,0);
INSERT INTO "OrderDetails" VALUES(10678,33,2.5,30,0);
INSERT INTO "OrderDetails" VALUES(10678,41,9.65,120,0);
INSERT INTO "OrderDetails" VALUES(10678,54,7.45,30,0);
INSERT INTO "OrderDetails" VALUES(10679,59,55,12,0);
INSERT INTO "OrderDetails" VALUES(10680,16,17.45,50,0.25);
INSERT INTO "OrderDetails" VALUES(10680,31,12.5,20,0.25);
INSERT INTO "OrderDetails" VALUES(10680,42,14,40,0.25);
INSERT INTO "OrderDetails" VALUES(10681,19,9.2,30,0.1);
INSERT INTO "OrderDetails" VALUES(10681,21,10,12,0.1);
INSERT INTO "OrderDetails" VALUES(10681,64,33.25,28,0);
INSERT INTO "OrderDetails" VALUES(10682,33,2.5,30,0);
INSERT INTO "OrderDetails" VALUES(10682,66,17,4,0);
INSERT INTO "OrderDetails" VALUES(10682,75,7.75,30,0);
INSERT INTO "OrderDetails" VALUES(10683,52,7,9,0);
INSERT INTO "OrderDetails" VALUES(10684,40,18.4,20,0);
INSERT INTO "OrderDetails" VALUES(10684,47,9.5,40,0);
INSERT INTO "OrderDetails" VALUES(10684,60,34,30,0);
INSERT INTO "OrderDetails" VALUES(10685,10,31,20,0);
INSERT INTO "OrderDetails" VALUES(10685,41,9.65,4,0);
INSERT INTO "OrderDetails" VALUES(10685,47,9.5,15,0);
INSERT INTO "OrderDetails" VALUES(10686,17,39,30,0.2);
INSERT INTO "OrderDetails" VALUES(10686,26,31.23,15,0);
INSERT INTO "OrderDetails" VALUES(10687,9,97,50,0.25);
INSERT INTO "OrderDetails" VALUES(10687,29,123.79,10,0);
INSERT INTO "OrderDetails" VALUES(10687,36,19,6,0.25);
INSERT INTO "OrderDetails" VALUES(10688,10,31,18,0.1);
INSERT INTO "OrderDetails" VALUES(10688,28,45.6,60,0.1);
INSERT INTO "OrderDetails" VALUES(10688,34,14,14,0);
INSERT INTO "OrderDetails" VALUES(10689,1,18,35,0.25);
INSERT INTO "OrderDetails" VALUES(10690,56,38,20,0.25);
INSERT INTO "OrderDetails" VALUES(10690,77,13,30,0.25);
INSERT INTO "OrderDetails" VALUES(10691,1,18,30,0);
INSERT INTO "OrderDetails" VALUES(10691,29,123.79,40,0);
INSERT INTO "OrderDetails" VALUES(10691,43,46,40,0);
INSERT INTO "OrderDetails" VALUES(10691,44,19.45,24,0);
INSERT INTO "OrderDetails" VALUES(10691,62,49.3,48,0);
INSERT INTO "OrderDetails" VALUES(10692,63,43.9,20,0);
INSERT INTO "OrderDetails" VALUES(10693,9,97,6,0);
INSERT INTO "OrderDetails" VALUES(10693,54,7.45,60,0.15);
INSERT INTO "OrderDetails" VALUES(10693,69,36,30,0.15);
INSERT INTO "OrderDetails" VALUES(10693,73,15,15,0.15);
INSERT INTO "OrderDetails" VALUES(10694,7,30,90,0);
INSERT INTO "OrderDetails" VALUES(10694,59,55,25,0);
INSERT INTO "OrderDetails" VALUES(10694,70,15,50,0);
INSERT INTO "OrderDetails" VALUES(10695,8,40,10,0);
INSERT INTO "OrderDetails" VALUES(10695,12,38,4,0);
INSERT INTO "OrderDetails" VALUES(10695,24,4.5,20,0);
INSERT INTO "OrderDetails" VALUES(10696,17,39,20,0);
INSERT INTO "OrderDetails" VALUES(10696,46,12,18,0);
INSERT INTO "OrderDetails" VALUES(10697,19,9.2,7,0.25);
INSERT INTO "OrderDetails" VALUES(10697,35,18,9,0.25);
INSERT INTO "OrderDetails" VALUES(10697,58,13.25,30,0.25);
INSERT INTO "OrderDetails" VALUES(10697,70,15,30,0.25);
INSERT INTO "OrderDetails" VALUES(10698,11,21,15,0);
INSERT INTO "OrderDetails" VALUES(10698,17,39,8,0.05);
INSERT INTO "OrderDetails" VALUES(10698,29,123.79,12,0.05);
INSERT INTO "OrderDetails" VALUES(10698,65,21.05,65,0.05);
INSERT INTO "OrderDetails" VALUES(10698,70,15,8,0.05);
INSERT INTO "OrderDetails" VALUES(10699,47,9.5,12,0);
INSERT INTO "OrderDetails" VALUES(10700,1,18,5,0.2);
INSERT INTO "OrderDetails" VALUES(10700,34,14,12,0.2);
INSERT INTO "OrderDetails" VALUES(10700,68,12.5,40,0.2);
INSERT INTO "OrderDetails" VALUES(10700,71,21.5,60,0.2);
INSERT INTO "OrderDetails" VALUES(10701,59,55,42,0.15);
INSERT INTO "OrderDetails" VALUES(10701,71,21.5,20,0.15);
INSERT INTO "OrderDetails" VALUES(10701,76,18,35,0.15);
INSERT INTO "OrderDetails" VALUES(10702,3,10,6,0);
INSERT INTO "OrderDetails" VALUES(10702,76,18,15,0);
INSERT INTO "OrderDetails" VALUES(10703,2,19,5,0);
INSERT INTO "OrderDetails" VALUES(10703,59,55,35,0);
INSERT INTO "OrderDetails" VALUES(10703,73,15,35,0);
INSERT INTO "OrderDetails" VALUES(10704,4,22,6,0);
INSERT INTO "OrderDetails" VALUES(10704,24,4.5,35,0);
INSERT INTO "OrderDetails" VALUES(10704,48,12.75,24,0);
INSERT INTO "OrderDetails" VALUES(10705,31,12.5,20,0);
INSERT INTO "OrderDetails" VALUES(10705,32,32,4,0);
INSERT INTO "OrderDetails" VALUES(10706,16,17.45,20,0);
INSERT INTO "OrderDetails" VALUES(10706,43,46,24,0);
INSERT INTO "OrderDetails" VALUES(10706,59,55,8,0);
INSERT INTO "OrderDetails" VALUES(10707,55,24,21,0);
INSERT INTO "OrderDetails" VALUES(10707,57,19.5,40,0);
INSERT INTO "OrderDetails" VALUES(10707,70,15,28,0.15);
INSERT INTO "OrderDetails" VALUES(10708,5,21.35,4,0);
INSERT INTO "OrderDetails" VALUES(10708,36,19,5,0);
INSERT INTO "OrderDetails" VALUES(10709,8,40,40,0);
INSERT INTO "OrderDetails" VALUES(10709,51,53,28,0);
INSERT INTO "OrderDetails" VALUES(10709,60,34,10,0);
INSERT INTO "OrderDetails" VALUES(10710,19,9.2,5,0);
INSERT INTO "OrderDetails" VALUES(10710,47,9.5,5,0);
INSERT INTO "OrderDetails" VALUES(10711,19,9.2,12,0);
INSERT INTO "OrderDetails" VALUES(10711,41,9.65,42,0);
INSERT INTO "OrderDetails" VALUES(10711,53,32.8,120,0);
INSERT INTO "OrderDetails" VALUES(10712,53,32.8,3,0.05);
INSERT INTO "OrderDetails" VALUES(10712,56,38,30,0);
INSERT INTO "OrderDetails" VALUES(10713,10,31,18,0);
INSERT INTO "OrderDetails" VALUES(10713,26,31.23,30,0);
INSERT INTO "OrderDetails" VALUES(10713,45,9.5,110,0);
INSERT INTO "OrderDetails" VALUES(10713,46,12,24,0);
INSERT INTO "OrderDetails" VALUES(10714,2,19,30,0.25);
INSERT INTO "OrderDetails" VALUES(10714,17,39,27,0.25);
INSERT INTO "OrderDetails" VALUES(10714,47,9.5,50,0.25);
INSERT INTO "OrderDetails" VALUES(10714,56,38,18,0.25);
INSERT INTO "OrderDetails" VALUES(10714,58,13.25,12,0.25);
INSERT INTO "OrderDetails" VALUES(10715,10,31,21,0);
INSERT INTO "OrderDetails" VALUES(10715,71,21.5,30,0);
INSERT INTO "OrderDetails" VALUES(10716,21,10,5,0);
INSERT INTO "OrderDetails" VALUES(10716,51,53,7,0);
INSERT INTO "OrderDetails" VALUES(10716,61,28.5,10,0);
INSERT INTO "OrderDetails" VALUES(10717,21,10,32,0.05);
INSERT INTO "OrderDetails" VALUES(10717,54,7.45,15,0);
INSERT INTO "OrderDetails" VALUES(10717,69,36,25,0.05);
INSERT INTO "OrderDetails" VALUES(10718,12,38,36,0);
INSERT INTO "OrderDetails" VALUES(10718,16,17.45,20,0);
INSERT INTO "OrderDetails" VALUES(10718,36,19,40,0);
INSERT INTO "OrderDetails" VALUES(10718,62,49.3,20,0);
INSERT INTO "OrderDetails" VALUES(10719,18,62.5,12,0.25);
INSERT INTO "OrderDetails" VALUES(10719,30,25.89,3,0.25);
INSERT INTO "OrderDetails" VALUES(10719,54,7.45,40,0.25);
INSERT INTO "OrderDetails" VALUES(10720,35,18,21,0);
INSERT INTO "OrderDetails" VALUES(10720,71,21.5,8,0);
INSERT INTO "OrderDetails" VALUES(10721,44,19.45,50,0.05);
INSERT INTO "OrderDetails" VALUES(10722,2,19,3,0);
INSERT INTO "OrderDetails" VALUES(10722,31,12.5,50,0);
INSERT INTO "OrderDetails" VALUES(10722,68,12.5,45,0);
INSERT INTO "OrderDetails" VALUES(10722,75,7.75,42,0);
INSERT INTO "OrderDetails" VALUES(10723,26,31.23,15,0);
INSERT INTO "OrderDetails" VALUES(10724,10,31,16,0);
INSERT INTO "OrderDetails" VALUES(10724,61,28.5,5,0);
INSERT INTO "OrderDetails" VALUES(10725,41,9.65,12,0);
INSERT INTO "OrderDetails" VALUES(10725,52,7,4,0);
INSERT INTO "OrderDetails" VALUES(10725,55,24,6,0);
INSERT INTO "OrderDetails" VALUES(10726,4,22,25,0);
INSERT INTO "OrderDetails" VALUES(10726,11,21,5,0);
INSERT INTO "OrderDetails" VALUES(10727,17,39,20,0.05);
INSERT INTO "OrderDetails" VALUES(10727,56,38,10,0.05);
INSERT INTO "OrderDetails" VALUES(10727,59,55,10,0.05);
INSERT INTO "OrderDetails" VALUES(10728,30,25.89,15,0);
INSERT INTO "OrderDetails" VALUES(10728,40,18.4,6,0);
INSERT INTO "OrderDetails" VALUES(10728,55,24,12,0);
INSERT INTO "OrderDetails" VALUES(10728,60,34,15,0);
INSERT INTO "OrderDetails" VALUES(10729,1,18,50,0);
INSERT INTO "OrderDetails" VALUES(10729,21,10,30,0);
INSERT INTO "OrderDetails" VALUES(10729,50,16.25,40,0);
INSERT INTO "OrderDetails" VALUES(10730,16,17.45,15,0.05);
INSERT INTO "OrderDetails" VALUES(10730,31,12.5,3,0.05);
INSERT INTO "OrderDetails" VALUES(10730,65,21.05,10,0.05);
INSERT INTO "OrderDetails" VALUES(10731,21,10,40,0.05);
INSERT INTO "OrderDetails" VALUES(10731,51,53,30,0.05);
INSERT INTO "OrderDetails" VALUES(10732,76,18,20,0);
INSERT INTO "OrderDetails" VALUES(10733,14,23.25,16,0);
INSERT INTO "OrderDetails" VALUES(10733,28,45.6,20,0);
INSERT INTO "OrderDetails" VALUES(10733,52,7,25,0);
INSERT INTO "OrderDetails" VALUES(10734,6,25,30,0);
INSERT INTO "OrderDetails" VALUES(10734,30,25.89,15,0);
INSERT INTO "OrderDetails" VALUES(10734,76,18,20,0);
INSERT INTO "OrderDetails" VALUES(10735,61,28.5,20,0.1);
INSERT INTO "OrderDetails" VALUES(10735,77,13,2,0.1);
INSERT INTO "OrderDetails" VALUES(10736,65,21.05,40,0);
INSERT INTO "OrderDetails" VALUES(10736,75,7.75,20,0);
INSERT INTO "OrderDetails" VALUES(10737,13,6,4,0);
INSERT INTO "OrderDetails" VALUES(10737,41,9.65,12,0);
INSERT INTO "OrderDetails" VALUES(10738,16,17.45,3,0);
INSERT INTO "OrderDetails" VALUES(10739,36,19,6,0);
INSERT INTO "OrderDetails" VALUES(10739,52,7,18,0);
INSERT INTO "OrderDetails" VALUES(10740,28,45.6,5,0.2);
INSERT INTO "OrderDetails" VALUES(10740,35,18,35,0.2);
INSERT INTO "OrderDetails" VALUES(10740,45,9.5,40,0.2);
INSERT INTO "OrderDetails" VALUES(10740,56,38,14,0.2);
INSERT INTO "OrderDetails" VALUES(10741,2,19,15,0.2);
INSERT INTO "OrderDetails" VALUES(10742,3,10,20,0);
INSERT INTO "OrderDetails" VALUES(10742,60,34,50,0);
INSERT INTO "OrderDetails" VALUES(10742,72,34.8,35,0);
INSERT INTO "OrderDetails" VALUES(10743,46,12,28,0.05);
INSERT INTO "OrderDetails" VALUES(10744,40,18.4,50,0.2);
INSERT INTO "OrderDetails" VALUES(10745,18,62.5,24,0);
INSERT INTO "OrderDetails" VALUES(10745,44,19.45,16,0);
INSERT INTO "OrderDetails" VALUES(10745,59,55,45,0);
INSERT INTO "OrderDetails" VALUES(10745,72,34.8,7,0);
INSERT INTO "OrderDetails" VALUES(10746,13,6,6,0);
INSERT INTO "OrderDetails" VALUES(10746,42,14,28,0);
INSERT INTO "OrderDetails" VALUES(10746,62,49.3,9,0);
INSERT INTO "OrderDetails" VALUES(10746,69,36,40,0);
INSERT INTO "OrderDetails" VALUES(10747,31,12.5,8,0);
INSERT INTO "OrderDetails" VALUES(10747,41,9.65,35,0);
INSERT INTO "OrderDetails" VALUES(10747,63,43.9,9,0);
INSERT INTO "OrderDetails" VALUES(10747,69,36,30,0);
INSERT INTO "OrderDetails" VALUES(10748,23,9,44,0);
INSERT INTO "OrderDetails" VALUES(10748,40,18.4,40,0);
INSERT INTO "OrderDetails" VALUES(10748,56,38,28,0);
INSERT INTO "OrderDetails" VALUES(10749,56,38,15,0);
INSERT INTO "OrderDetails" VALUES(10749,59,55,6,0);
INSERT INTO "OrderDetails" VALUES(10749,76,18,10,0);
INSERT INTO "OrderDetails" VALUES(10750,14,23.25,5,0.15);
INSERT INTO "OrderDetails" VALUES(10750,45,9.5,40,0.15);
INSERT INTO "OrderDetails" VALUES(10750,59,55,25,0.15);
INSERT INTO "OrderDetails" VALUES(10751,26,31.23,12,0.1);
INSERT INTO "OrderDetails" VALUES(10751,30,25.89,30,0);
INSERT INTO "OrderDetails" VALUES(10751,50,16.25,20,0.1);
INSERT INTO "OrderDetails" VALUES(10751,73,15,15,0);
INSERT INTO "OrderDetails" VALUES(10752,1,18,8,0);
INSERT INTO "OrderDetails" VALUES(10752,69,36,3,0);
INSERT INTO "OrderDetails" VALUES(10753,45,9.5,4,0);
INSERT INTO "OrderDetails" VALUES(10753,74,10,5,0);
INSERT INTO "OrderDetails" VALUES(10754,40,18.4,3,0);
INSERT INTO "OrderDetails" VALUES(10755,47,9.5,30,0.25);
INSERT INTO "OrderDetails" VALUES(10755,56,38,30,0.25);
INSERT INTO "OrderDetails" VALUES(10755,57,19.5,14,0.25);
INSERT INTO "OrderDetails" VALUES(10755,69,36,25,0.25);
INSERT INTO "OrderDetails" VALUES(10756,18,62.5,21,0.2);
INSERT INTO "OrderDetails" VALUES(10756,36,19,20,0.2);
INSERT INTO "OrderDetails" VALUES(10756,68,12.5,6,0.2);
INSERT INTO "OrderDetails" VALUES(10756,69,36,20,0.2);
INSERT INTO "OrderDetails" VALUES(10757,34,14,30,0);
INSERT INTO "OrderDetails" VALUES(10757,59,55,7,0);
INSERT INTO "OrderDetails" VALUES(10757,62,49.3,30,0);
INSERT INTO "OrderDetails" VALUES(10757,64,33.25,24,0);
INSERT INTO "OrderDetails" VALUES(10758,26,31.23,20,0);
INSERT INTO "OrderDetails" VALUES(10758,52,7,60,0);
INSERT INTO "OrderDetails" VALUES(10758,70,15,40,0);
INSERT INTO "OrderDetails" VALUES(10759,32,32,10,0);
INSERT INTO "OrderDetails" VALUES(10760,25,14,12,0.25);
INSERT INTO "OrderDetails" VALUES(10760,27,43.9,40,0);
INSERT INTO "OrderDetails" VALUES(10760,43,46,30,0.25);
INSERT INTO "OrderDetails" VALUES(10761,25,14,35,0.25);
INSERT INTO "OrderDetails" VALUES(10761,75,7.75,18,0);
INSERT INTO "OrderDetails" VALUES(10762,39,18,16,0);
INSERT INTO "OrderDetails" VALUES(10762,47,9.5,30,0);
INSERT INTO "OrderDetails" VALUES(10762,51,53,28,0);
INSERT INTO "OrderDetails" VALUES(10762,56,38,60,0);
INSERT INTO "OrderDetails" VALUES(10763,21,10,40,0);
INSERT INTO "OrderDetails" VALUES(10763,22,21,6,0);
INSERT INTO "OrderDetails" VALUES(10763,24,4.5,20,0);
INSERT INTO "OrderDetails" VALUES(10764,3,10,20,0.1);
INSERT INTO "OrderDetails" VALUES(10764,39,18,130,0.1);
INSERT INTO "OrderDetails" VALUES(10765,65,21.05,80,0.1);
INSERT INTO "OrderDetails" VALUES(10766,2,19,40,0);
INSERT INTO "OrderDetails" VALUES(10766,7,30,35,0);
INSERT INTO "OrderDetails" VALUES(10766,68,12.5,40,0);
INSERT INTO "OrderDetails" VALUES(10767,42,14,2,0);
INSERT INTO "OrderDetails" VALUES(10768,22,21,4,0);
INSERT INTO "OrderDetails" VALUES(10768,31,12.5,50,0);
INSERT INTO "OrderDetails" VALUES(10768,60,34,15,0);
INSERT INTO "OrderDetails" VALUES(10768,71,21.5,12,0);
INSERT INTO "OrderDetails" VALUES(10769,41,9.65,30,0.05);
INSERT INTO "OrderDetails" VALUES(10769,52,7,15,0.05);
INSERT INTO "OrderDetails" VALUES(10769,61,28.5,20,0);
INSERT INTO "OrderDetails" VALUES(10769,62,49.3,15,0);
INSERT INTO "OrderDetails" VALUES(10770,11,21,15,0.25);
INSERT INTO "OrderDetails" VALUES(10771,71,21.5,16,0);
INSERT INTO "OrderDetails" VALUES(10772,29,123.79,18,0);
INSERT INTO "OrderDetails" VALUES(10772,59,55,25,0);
INSERT INTO "OrderDetails" VALUES(10773,17,39,33,0);
INSERT INTO "OrderDetails" VALUES(10773,31,12.5,70,0.2);
INSERT INTO "OrderDetails" VALUES(10773,75,7.75,7,0.2);
INSERT INTO "OrderDetails" VALUES(10774,31,12.5,2,0.25);
INSERT INTO "OrderDetails" VALUES(10774,66,17,50,0);
INSERT INTO "OrderDetails" VALUES(10775,10,31,6,0);
INSERT INTO "OrderDetails" VALUES(10775,67,14,3,0);
INSERT INTO "OrderDetails" VALUES(10776,31,12.5,16,0.05);
INSERT INTO "OrderDetails" VALUES(10776,42,14,12,0.05);
INSERT INTO "OrderDetails" VALUES(10776,45,9.5,27,0.05);
INSERT INTO "OrderDetails" VALUES(10776,51,53,120,0.05);
INSERT INTO "OrderDetails" VALUES(10777,42,14,20,0.2);
INSERT INTO "OrderDetails" VALUES(10778,41,9.65,10,0);
INSERT INTO "OrderDetails" VALUES(10779,16,17.45,20,0);
INSERT INTO "OrderDetails" VALUES(10779,62,49.3,20,0);
INSERT INTO "OrderDetails" VALUES(10780,70,15,35,0);
INSERT INTO "OrderDetails" VALUES(10780,77,13,15,0);
INSERT INTO "OrderDetails" VALUES(10781,54,7.45,3,0.2);
INSERT INTO "OrderDetails" VALUES(10781,56,38,20,0.2);
INSERT INTO "OrderDetails" VALUES(10781,74,10,35,0);
INSERT INTO "OrderDetails" VALUES(10782,31,12.5,1,0);
INSERT INTO "OrderDetails" VALUES(10783,31,12.5,10,0);
INSERT INTO "OrderDetails" VALUES(10783,38,263.5,5,0);
INSERT INTO "OrderDetails" VALUES(10784,36,19,30,0);
INSERT INTO "OrderDetails" VALUES(10784,39,18,2,0.15);
INSERT INTO "OrderDetails" VALUES(10784,72,34.8,30,0.15);
INSERT INTO "OrderDetails" VALUES(10785,10,31,10,0);
INSERT INTO "OrderDetails" VALUES(10785,75,7.75,10,0);
INSERT INTO "OrderDetails" VALUES(10786,8,40,30,0.2);
INSERT INTO "OrderDetails" VALUES(10786,30,25.89,15,0.2);
INSERT INTO "OrderDetails" VALUES(10786,75,7.75,42,0.2);
INSERT INTO "OrderDetails" VALUES(10787,2,19,15,0.05);
INSERT INTO "OrderDetails" VALUES(10787,29,123.79,20,0.05);
INSERT INTO "OrderDetails" VALUES(10788,19,9.2,50,0.05);
INSERT INTO "OrderDetails" VALUES(10788,75,7.75,40,0.05);
INSERT INTO "OrderDetails" VALUES(10789,18,62.5,30,0);
INSERT INTO "OrderDetails" VALUES(10789,35,18,15,0);
INSERT INTO "OrderDetails" VALUES(10789,63,43.9,30,0);
INSERT INTO "OrderDetails" VALUES(10789,68,12.5,18,0);
INSERT INTO "OrderDetails" VALUES(10790,7,30,3,0.15);
INSERT INTO "OrderDetails" VALUES(10790,56,38,20,0.15);
INSERT INTO "OrderDetails" VALUES(10791,29,123.79,14,0.05);
INSERT INTO "OrderDetails" VALUES(10791,41,9.65,20,0.05);
INSERT INTO "OrderDetails" VALUES(10792,2,19,10,0);
INSERT INTO "OrderDetails" VALUES(10792,54,7.45,3,0);
INSERT INTO "OrderDetails" VALUES(10792,68,12.5,15,0);
INSERT INTO "OrderDetails" VALUES(10793,41,9.65,14,0);
INSERT INTO "OrderDetails" VALUES(10793,52,7,8,0);
INSERT INTO "OrderDetails" VALUES(10794,14,23.25,15,0.2);
INSERT INTO "OrderDetails" VALUES(10794,54,7.45,6,0.2);
INSERT INTO "OrderDetails" VALUES(10795,16,17.45,65,0);
INSERT INTO "OrderDetails" VALUES(10795,17,39,35,0.25);
INSERT INTO "OrderDetails" VALUES(10796,26,31.23,21,0.2);
INSERT INTO "OrderDetails" VALUES(10796,44,19.45,10,0);
INSERT INTO "OrderDetails" VALUES(10796,64,33.25,35,0.2);
INSERT INTO "OrderDetails" VALUES(10796,69,36,24,0.2);
INSERT INTO "OrderDetails" VALUES(10797,11,21,20,0);
INSERT INTO "OrderDetails" VALUES(10798,62,49.3,2,0);
INSERT INTO "OrderDetails" VALUES(10798,72,34.8,10,0);
INSERT INTO "OrderDetails" VALUES(10799,13,6,20,0.15);
INSERT INTO "OrderDetails" VALUES(10799,24,4.5,20,0.15);
INSERT INTO "OrderDetails" VALUES(10799,59,55,25,0);
INSERT INTO "OrderDetails" VALUES(10800,11,21,50,0.1);
INSERT INTO "OrderDetails" VALUES(10800,51,53,10,0.1);
INSERT INTO "OrderDetails" VALUES(10800,54,7.45,7,0.1);
INSERT INTO "OrderDetails" VALUES(10801,17,39,40,0.25);
INSERT INTO "OrderDetails" VALUES(10801,29,123.79,20,0.25);
INSERT INTO "OrderDetails" VALUES(10802,30,25.89,25,0.25);
INSERT INTO "OrderDetails" VALUES(10802,51,53,30,0.25);
INSERT INTO "OrderDetails" VALUES(10802,55,24,60,0.25);
INSERT INTO "OrderDetails" VALUES(10802,62,49.3,5,0.25);
INSERT INTO "OrderDetails" VALUES(10803,19,9.2,24,0.05);
INSERT INTO "OrderDetails" VALUES(10803,25,14,15,0.05);
INSERT INTO "OrderDetails" VALUES(10803,59,55,15,0.05);
INSERT INTO "OrderDetails" VALUES(10804,10,31,36,0);
INSERT INTO "OrderDetails" VALUES(10804,28,45.6,24,0);
INSERT INTO "OrderDetails" VALUES(10804,49,20,4,0.15);
INSERT INTO "OrderDetails" VALUES(10805,34,14,10,0);
INSERT INTO "OrderDetails" VALUES(10805,38,263.5,10,0);
INSERT INTO "OrderDetails" VALUES(10806,2,19,20,0.25);
INSERT INTO "OrderDetails" VALUES(10806,65,21.05,2,0);
INSERT INTO "OrderDetails" VALUES(10806,74,10,15,0.25);
INSERT INTO "OrderDetails" VALUES(10807,40,18.4,1,0);
INSERT INTO "OrderDetails" VALUES(10808,56,38,20,0.15);
INSERT INTO "OrderDetails" VALUES(10808,76,18,50,0.15);
INSERT INTO "OrderDetails" VALUES(10809,52,7,20,0);
INSERT INTO "OrderDetails" VALUES(10810,13,6,7,0);
INSERT INTO "OrderDetails" VALUES(10810,25,14,5,0);
INSERT INTO "OrderDetails" VALUES(10810,70,15,5,0);
INSERT INTO "OrderDetails" VALUES(10811,19,9.2,15,0);
INSERT INTO "OrderDetails" VALUES(10811,23,9,18,0);
INSERT INTO "OrderDetails" VALUES(10811,40,18.4,30,0);
INSERT INTO "OrderDetails" VALUES(10812,31,12.5,16,0.1);
INSERT INTO "OrderDetails" VALUES(10812,72,34.8,40,0.1);
INSERT INTO "OrderDetails" VALUES(10812,77,13,20,0);
INSERT INTO "OrderDetails" VALUES(10813,2,19,12,0.2);
INSERT INTO "OrderDetails" VALUES(10813,46,12,35,0);
INSERT INTO "OrderDetails" VALUES(10814,41,9.65,20,0);
INSERT INTO "OrderDetails" VALUES(10814,43,46,20,0.15);
INSERT INTO "OrderDetails" VALUES(10814,48,12.75,8,0.15);
INSERT INTO "OrderDetails" VALUES(10814,61,28.5,30,0.15);
INSERT INTO "OrderDetails" VALUES(10815,33,2.5,16,0);
INSERT INTO "OrderDetails" VALUES(10816,38,263.5,30,0.05);
INSERT INTO "OrderDetails" VALUES(10816,62,49.3,20,0.05);
INSERT INTO "OrderDetails" VALUES(10817,26,31.23,40,0.15);
INSERT INTO "OrderDetails" VALUES(10817,38,263.5,30,0);
INSERT INTO "OrderDetails" VALUES(10817,40,18.4,60,0.15);
INSERT INTO "OrderDetails" VALUES(10817,62,49.3,25,0.15);
INSERT INTO "OrderDetails" VALUES(10818,32,32,20,0);
INSERT INTO "OrderDetails" VALUES(10818,41,9.65,20,0);
INSERT INTO "OrderDetails" VALUES(10819,43,46,7,0);
INSERT INTO "OrderDetails" VALUES(10819,75,7.75,20,0);
INSERT INTO "OrderDetails" VALUES(10820,56,38,30,0);
INSERT INTO "OrderDetails" VALUES(10821,35,18,20,0);
INSERT INTO "OrderDetails" VALUES(10821,51,53,6,0);
INSERT INTO "OrderDetails" VALUES(10822,62,49.3,3,0);
INSERT INTO "OrderDetails" VALUES(10822,70,15,6,0);
INSERT INTO "OrderDetails" VALUES(10823,11,21,20,0.1);
INSERT INTO "OrderDetails" VALUES(10823,57,19.5,15,0);
INSERT INTO "OrderDetails" VALUES(10823,59,55,40,0.1);
INSERT INTO "OrderDetails" VALUES(10823,77,13,15,0.1);
INSERT INTO "OrderDetails" VALUES(10824,41,9.65,12,0);
INSERT INTO "OrderDetails" VALUES(10824,70,15,9,0);
INSERT INTO "OrderDetails" VALUES(10825,26,31.23,12,0);
INSERT INTO "OrderDetails" VALUES(10825,53,32.8,20,0);
INSERT INTO "OrderDetails" VALUES(10826,31,12.5,35,0);
INSERT INTO "OrderDetails" VALUES(10826,57,19.5,15,0);
INSERT INTO "OrderDetails" VALUES(10827,10,31,15,0);
INSERT INTO "OrderDetails" VALUES(10827,39,18,21,0);
INSERT INTO "OrderDetails" VALUES(10828,20,81,5,0);
INSERT INTO "OrderDetails" VALUES(10828,38,263.5,2,0);
INSERT INTO "OrderDetails" VALUES(10829,2,19,10,0);
INSERT INTO "OrderDetails" VALUES(10829,8,40,20,0);
INSERT INTO "OrderDetails" VALUES(10829,13,6,10,0);
INSERT INTO "OrderDetails" VALUES(10829,60,34,21,0);
INSERT INTO "OrderDetails" VALUES(10830,6,25,6,0);
INSERT INTO "OrderDetails" VALUES(10830,39,18,28,0);
INSERT INTO "OrderDetails" VALUES(10830,60,34,30,0);
INSERT INTO "OrderDetails" VALUES(10830,68,12.5,24,0);
INSERT INTO "OrderDetails" VALUES(10831,19,9.2,2,0);
INSERT INTO "OrderDetails" VALUES(10831,35,18,8,0);
INSERT INTO "OrderDetails" VALUES(10831,38,263.5,8,0);
INSERT INTO "OrderDetails" VALUES(10831,43,46,9,0);
INSERT INTO "OrderDetails" VALUES(10832,13,6,3,0.2);
INSERT INTO "OrderDetails" VALUES(10832,25,14,10,0.2);
INSERT INTO "OrderDetails" VALUES(10832,44,19.45,16,0.2);
INSERT INTO "OrderDetails" VALUES(10832,64,33.25,3,0);
INSERT INTO "OrderDetails" VALUES(10833,7,30,20,0.1);
INSERT INTO "OrderDetails" VALUES(10833,31,12.5,9,0.1);
INSERT INTO "OrderDetails" VALUES(10833,53,32.8,9,0.1);
INSERT INTO "OrderDetails" VALUES(10834,29,123.79,8,0.05);
INSERT INTO "OrderDetails" VALUES(10834,30,25.89,20,0.05);
INSERT INTO "OrderDetails" VALUES(10835,59,55,15,0);
INSERT INTO "OrderDetails" VALUES(10835,77,13,2,0.2);
INSERT INTO "OrderDetails" VALUES(10836,22,21,52,0);
INSERT INTO "OrderDetails" VALUES(10836,35,18,6,0);
INSERT INTO "OrderDetails" VALUES(10836,57,19.5,24,0);
INSERT INTO "OrderDetails" VALUES(10836,60,34,60,0);
INSERT INTO "OrderDetails" VALUES(10836,64,33.25,30,0);
INSERT INTO "OrderDetails" VALUES(10837,13,6,6,0);
INSERT INTO "OrderDetails" VALUES(10837,40,18.4,25,0);
INSERT INTO "OrderDetails" VALUES(10837,47,9.5,40,0.25);
INSERT INTO "OrderDetails" VALUES(10837,76,18,21,0.25);
INSERT INTO "OrderDetails" VALUES(10838,1,18,4,0.25);
INSERT INTO "OrderDetails" VALUES(10838,18,62.5,25,0.25);
INSERT INTO "OrderDetails" VALUES(10838,36,19,50,0.25);
INSERT INTO "OrderDetails" VALUES(10839,58,13.25,30,0.1);
INSERT INTO "OrderDetails" VALUES(10839,72,34.8,15,0.1);
INSERT INTO "OrderDetails" VALUES(10840,25,14,6,0.2);
INSERT INTO "OrderDetails" VALUES(10840,39,18,10,0.2);
INSERT INTO "OrderDetails" VALUES(10841,10,31,16,0);
INSERT INTO "OrderDetails" VALUES(10841,56,38,30,0);
INSERT INTO "OrderDetails" VALUES(10841,59,55,50,0);
INSERT INTO "OrderDetails" VALUES(10841,77,13,15,0);
INSERT INTO "OrderDetails" VALUES(10842,11,21,15,0);
INSERT INTO "OrderDetails" VALUES(10842,43,46,5,0);
INSERT INTO "OrderDetails" VALUES(10842,68,12.5,20,0);
INSERT INTO "OrderDetails" VALUES(10842,70,15,12,0);
INSERT INTO "OrderDetails" VALUES(10843,51,53,4,0.25);
INSERT INTO "OrderDetails" VALUES(10844,22,21,35,0);
INSERT INTO "OrderDetails" VALUES(10845,23,9,70,0.1);
INSERT INTO "OrderDetails" VALUES(10845,35,18,25,0.1);
INSERT INTO "OrderDetails" VALUES(10845,42,14,42,0.1);
INSERT INTO "OrderDetails" VALUES(10845,58,13.25,60,0.1);
INSERT INTO "OrderDetails" VALUES(10845,64,33.25,48,0);
INSERT INTO "OrderDetails" VALUES(10846,4,22,21,0);
INSERT INTO "OrderDetails" VALUES(10846,70,15,30,0);
INSERT INTO "OrderDetails" VALUES(10846,74,10,20,0);
INSERT INTO "OrderDetails" VALUES(10847,1,18,80,0.2);
INSERT INTO "OrderDetails" VALUES(10847,19,9.2,12,0.2);
INSERT INTO "OrderDetails" VALUES(10847,37,26,60,0.2);
INSERT INTO "OrderDetails" VALUES(10847,45,9.5,36,0.2);
INSERT INTO "OrderDetails" VALUES(10847,60,34,45,0.2);
INSERT INTO "OrderDetails" VALUES(10847,71,21.5,55,0.2);
INSERT INTO "OrderDetails" VALUES(10848,5,21.35,30,0);
INSERT INTO "OrderDetails" VALUES(10848,9,97,3,0);
INSERT INTO "OrderDetails" VALUES(10849,3,10,49,0);
INSERT INTO "OrderDetails" VALUES(10849,26,31.23,18,0.15);
INSERT INTO "OrderDetails" VALUES(10850,25,14,20,0.15);
INSERT INTO "OrderDetails" VALUES(10850,33,2.5,4,0.15);
INSERT INTO "OrderDetails" VALUES(10850,70,15,30,0.15);
INSERT INTO "OrderDetails" VALUES(10851,2,19,5,0.05);
INSERT INTO "OrderDetails" VALUES(10851,25,14,10,0.05);
INSERT INTO "OrderDetails" VALUES(10851,57,19.5,10,0.05);
INSERT INTO "OrderDetails" VALUES(10851,59,55,42,0.05);
INSERT INTO "OrderDetails" VALUES(10852,2,19,15,0);
INSERT INTO "OrderDetails" VALUES(10852,17,39,6,0);
INSERT INTO "OrderDetails" VALUES(10852,62,49.3,50,0);
INSERT INTO "OrderDetails" VALUES(10853,18,62.5,10,0);
INSERT INTO "OrderDetails" VALUES(10854,10,31,100,0.15);
INSERT INTO "OrderDetails" VALUES(10854,13,6,65,0.15);
INSERT INTO "OrderDetails" VALUES(10855,16,17.45,50,0);
INSERT INTO "OrderDetails" VALUES(10855,31,12.5,14,0);
INSERT INTO "OrderDetails" VALUES(10855,56,38,24,0);
INSERT INTO "OrderDetails" VALUES(10855,65,21.05,15,0.15);
INSERT INTO "OrderDetails" VALUES(10856,2,19,20,0);
INSERT INTO "OrderDetails" VALUES(10856,42,14,20,0);
INSERT INTO "OrderDetails" VALUES(10857,3,10,30,0);
INSERT INTO "OrderDetails" VALUES(10857,26,31.23,35,0.25);
INSERT INTO "OrderDetails" VALUES(10857,29,123.79,10,0.25);
INSERT INTO "OrderDetails" VALUES(10858,7,30,5,0);
INSERT INTO "OrderDetails" VALUES(10858,27,43.9,10,0);
INSERT INTO "OrderDetails" VALUES(10858,70,15,4,0);
INSERT INTO "OrderDetails" VALUES(10859,24,4.5,40,0.25);
INSERT INTO "OrderDetails" VALUES(10859,54,7.45,35,0.25);
INSERT INTO "OrderDetails" VALUES(10859,64,33.25,30,0.25);
INSERT INTO "OrderDetails" VALUES(10860,51,53,3,0);
INSERT INTO "OrderDetails" VALUES(10860,76,18,20,0);
INSERT INTO "OrderDetails" VALUES(10861,17,39,42,0);
INSERT INTO "OrderDetails" VALUES(10861,18,62.5,20,0);
INSERT INTO "OrderDetails" VALUES(10861,21,10,40,0);
INSERT INTO "OrderDetails" VALUES(10861,33,2.5,35,0);
INSERT INTO "OrderDetails" VALUES(10861,62,49.3,3,0);
INSERT INTO "OrderDetails" VALUES(10862,11,21,25,0);
INSERT INTO "OrderDetails" VALUES(10862,52,7,8,0);
INSERT INTO "OrderDetails" VALUES(10863,1,18,20,0.15);
INSERT INTO "OrderDetails" VALUES(10863,58,13.25,12,0.15);
INSERT INTO "OrderDetails" VALUES(10864,35,18,4,0);
INSERT INTO "OrderDetails" VALUES(10864,67,14,15,0);
INSERT INTO "OrderDetails" VALUES(10865,38,263.5,60,0.05);
INSERT INTO "OrderDetails" VALUES(10865,39,18,80,0.05);
INSERT INTO "OrderDetails" VALUES(10866,2,19,21,0.25);
INSERT INTO "OrderDetails" VALUES(10866,24,4.5,6,0.25);
INSERT INTO "OrderDetails" VALUES(10866,30,25.89,40,0.25);
INSERT INTO "OrderDetails" VALUES(10867,53,32.8,3,0);
INSERT INTO "OrderDetails" VALUES(10868,26,31.23,20,0);
INSERT INTO "OrderDetails" VALUES(10868,35,18,30,0);
INSERT INTO "OrderDetails" VALUES(10868,49,20,42,0.1);
INSERT INTO "OrderDetails" VALUES(10869,1,18,40,0);
INSERT INTO "OrderDetails" VALUES(10869,11,21,10,0);
INSERT INTO "OrderDetails" VALUES(10869,23,9,50,0);
INSERT INTO "OrderDetails" VALUES(10869,68,12.5,20,0);
INSERT INTO "OrderDetails" VALUES(10870,35,18,3,0);
INSERT INTO "OrderDetails" VALUES(10870,51,53,2,0);
INSERT INTO "OrderDetails" VALUES(10871,6,25,50,0.05);
INSERT INTO "OrderDetails" VALUES(10871,16,17.45,12,0.05);
INSERT INTO "OrderDetails" VALUES(10871,17,39,16,0.05);
INSERT INTO "OrderDetails" VALUES(10872,55,24,10,0.05);
INSERT INTO "OrderDetails" VALUES(10872,62,49.3,20,0.05);
INSERT INTO "OrderDetails" VALUES(10872,64,33.25,15,0.05);
INSERT INTO "OrderDetails" VALUES(10872,65,21.05,21,0.05);
INSERT INTO "OrderDetails" VALUES(10873,21,10,20,0);
INSERT INTO "OrderDetails" VALUES(10873,28,45.6,3,0);
INSERT INTO "OrderDetails" VALUES(10874,10,31,10,0);
INSERT INTO "OrderDetails" VALUES(10875,19,9.2,25,0);
INSERT INTO "OrderDetails" VALUES(10875,47,9.5,21,0.1);
INSERT INTO "OrderDetails" VALUES(10875,49,20,15,0);
INSERT INTO "OrderDetails" VALUES(10876,46,12,21,0);
INSERT INTO "OrderDetails" VALUES(10876,64,33.25,20,0);
INSERT INTO "OrderDetails" VALUES(10877,16,17.45,30,0.25);
INSERT INTO "OrderDetails" VALUES(10877,18,62.5,25,0);
INSERT INTO "OrderDetails" VALUES(10878,20,81,20,0.05);
INSERT INTO "OrderDetails" VALUES(10879,40,18.4,12,0);
INSERT INTO "OrderDetails" VALUES(10879,65,21.05,10,0);
INSERT INTO "OrderDetails" VALUES(10879,76,18,10,0);
INSERT INTO "OrderDetails" VALUES(10880,23,9,30,0.2);
INSERT INTO "OrderDetails" VALUES(10880,61,28.5,30,0.2);
INSERT INTO "OrderDetails" VALUES(10880,70,15,50,0.2);
INSERT INTO "OrderDetails" VALUES(10881,73,15,10,0);
INSERT INTO "OrderDetails" VALUES(10882,42,14,25,0);
INSERT INTO "OrderDetails" VALUES(10882,49,20,20,0.15);
INSERT INTO "OrderDetails" VALUES(10882,54,7.45,32,0.15);
INSERT INTO "OrderDetails" VALUES(10883,24,4.5,8,0);
INSERT INTO "OrderDetails" VALUES(10884,21,10,40,0.05);
INSERT INTO "OrderDetails" VALUES(10884,56,38,21,0.05);
INSERT INTO "OrderDetails" VALUES(10884,65,21.05,12,0.05);
INSERT INTO "OrderDetails" VALUES(10885,2,19,20,0);
INSERT INTO "OrderDetails" VALUES(10885,24,4.5,12,0);
INSERT INTO "OrderDetails" VALUES(10885,70,15,30,0);
INSERT INTO "OrderDetails" VALUES(10885,77,13,25,0);
INSERT INTO "OrderDetails" VALUES(10886,10,31,70,0);
INSERT INTO "OrderDetails" VALUES(10886,31,12.5,35,0);
INSERT INTO "OrderDetails" VALUES(10886,77,13,40,0);
INSERT INTO "OrderDetails" VALUES(10887,25,14,5,0);
INSERT INTO "OrderDetails" VALUES(10888,2,19,20,0);
INSERT INTO "OrderDetails" VALUES(10888,68,12.5,18,0);
INSERT INTO "OrderDetails" VALUES(10889,11,21,40,0);
INSERT INTO "OrderDetails" VALUES(10889,38,263.5,40,0);
INSERT INTO "OrderDetails" VALUES(10890,17,39,15,0);
INSERT INTO "OrderDetails" VALUES(10890,34,14,10,0);
INSERT INTO "OrderDetails" VALUES(10890,41,9.65,14,0);
INSERT INTO "OrderDetails" VALUES(10891,30,25.89,15,0.05);
INSERT INTO "OrderDetails" VALUES(10892,59,55,40,0.05);
INSERT INTO "OrderDetails" VALUES(10893,8,40,30,0);
INSERT INTO "OrderDetails" VALUES(10893,24,4.5,10,0);
INSERT INTO "OrderDetails" VALUES(10893,29,123.79,24,0);
INSERT INTO "OrderDetails" VALUES(10893,30,25.89,35,0);
INSERT INTO "OrderDetails" VALUES(10893,36,19,20,0);
INSERT INTO "OrderDetails" VALUES(10894,13,6,28,0.05);
INSERT INTO "OrderDetails" VALUES(10894,69,36,50,0.05);
INSERT INTO "OrderDetails" VALUES(10894,75,7.75,120,0.05);
INSERT INTO "OrderDetails" VALUES(10895,24,4.5,110,0);
INSERT INTO "OrderDetails" VALUES(10895,39,18,45,0);
INSERT INTO "OrderDetails" VALUES(10895,40,18.4,91,0);
INSERT INTO "OrderDetails" VALUES(10895,60,34,100,0);
INSERT INTO "OrderDetails" VALUES(10896,45,9.5,15,0);
INSERT INTO "OrderDetails" VALUES(10896,56,38,16,0);
INSERT INTO "OrderDetails" VALUES(10897,29,123.79,80,0);
INSERT INTO "OrderDetails" VALUES(10897,30,25.89,36,0);
INSERT INTO "OrderDetails" VALUES(10898,13,6,5,0);
INSERT INTO "OrderDetails" VALUES(10899,39,18,8,0.15);
INSERT INTO "OrderDetails" VALUES(10900,70,15,3,0.25);
INSERT INTO "OrderDetails" VALUES(10901,41,9.65,30,0);
INSERT INTO "OrderDetails" VALUES(10901,71,21.5,30,0);
INSERT INTO "OrderDetails" VALUES(10902,55,24,30,0.15);
INSERT INTO "OrderDetails" VALUES(10902,62,49.3,6,0.15);
INSERT INTO "OrderDetails" VALUES(10903,13,6,40,0);
INSERT INTO "OrderDetails" VALUES(10903,65,21.05,21,0);
INSERT INTO "OrderDetails" VALUES(10903,68,12.5,20,0);
INSERT INTO "OrderDetails" VALUES(10904,58,13.25,15,0);
INSERT INTO "OrderDetails" VALUES(10904,62,49.3,35,0);
INSERT INTO "OrderDetails" VALUES(10905,1,18,20,0.05);
INSERT INTO "OrderDetails" VALUES(10906,61,28.5,15,0);
INSERT INTO "OrderDetails" VALUES(10907,75,7.75,14,0);
INSERT INTO "OrderDetails" VALUES(10908,7,30,20,0.05);
INSERT INTO "OrderDetails" VALUES(10908,52,7,14,0.05);
INSERT INTO "OrderDetails" VALUES(10909,7,30,12,0);
INSERT INTO "OrderDetails" VALUES(10909,16,17.45,15,0);
INSERT INTO "OrderDetails" VALUES(10909,41,9.65,5,0);
INSERT INTO "OrderDetails" VALUES(10910,19,9.2,12,0);
INSERT INTO "OrderDetails" VALUES(10910,49,20,10,0);
INSERT INTO "OrderDetails" VALUES(10910,61,28.5,5,0);
INSERT INTO "OrderDetails" VALUES(10911,1,18,10,0);
INSERT INTO "OrderDetails" VALUES(10911,17,39,12,0);
INSERT INTO "OrderDetails" VALUES(10911,67,14,15,0);
INSERT INTO "OrderDetails" VALUES(10912,11,21,40,0.25);
INSERT INTO "OrderDetails" VALUES(10912,29,123.79,60,0.25);
INSERT INTO "OrderDetails" VALUES(10913,4,22,30,0.25);
INSERT INTO "OrderDetails" VALUES(10913,33,2.5,40,0.25);
INSERT INTO "OrderDetails" VALUES(10913,58,13.25,15,0);
INSERT INTO "OrderDetails" VALUES(10914,71,21.5,25,0);
INSERT INTO "OrderDetails" VALUES(10915,17,39,10,0);
INSERT INTO "OrderDetails" VALUES(10915,33,2.5,30,0);
INSERT INTO "OrderDetails" VALUES(10915,54,7.45,10,0);
INSERT INTO "OrderDetails" VALUES(10916,16,17.45,6,0);
INSERT INTO "OrderDetails" VALUES(10916,32,32,6,0);
INSERT INTO "OrderDetails" VALUES(10916,57,19.5,20,0);
INSERT INTO "OrderDetails" VALUES(10917,30,25.89,1,0);
INSERT INTO "OrderDetails" VALUES(10917,60,34,10,0);
INSERT INTO "OrderDetails" VALUES(10918,1,18,60,0.25);
INSERT INTO "OrderDetails" VALUES(10918,60,34,25,0.25);
INSERT INTO "OrderDetails" VALUES(10919,16,17.45,24,0);
INSERT INTO "OrderDetails" VALUES(10919,25,14,24,0);
INSERT INTO "OrderDetails" VALUES(10919,40,18.4,20,0);
INSERT INTO "OrderDetails" VALUES(10920,50,16.25,24,0);
INSERT INTO "OrderDetails" VALUES(10921,35,18,10,0);
INSERT INTO "OrderDetails" VALUES(10921,63,43.9,40,0);
INSERT INTO "OrderDetails" VALUES(10922,17,39,15,0);
INSERT INTO "OrderDetails" VALUES(10922,24,4.5,35,0);
INSERT INTO "OrderDetails" VALUES(10923,42,14,10,0.2);
INSERT INTO "OrderDetails" VALUES(10923,43,46,10,0.2);
INSERT INTO "OrderDetails" VALUES(10923,67,14,24,0.2);
INSERT INTO "OrderDetails" VALUES(10924,10,31,20,0.1);
INSERT INTO "OrderDetails" VALUES(10924,28,45.6,30,0.1);
INSERT INTO "OrderDetails" VALUES(10924,75,7.75,6,0);
INSERT INTO "OrderDetails" VALUES(10925,36,19,25,0.15);
INSERT INTO "OrderDetails" VALUES(10925,52,7,12,0.15);
INSERT INTO "OrderDetails" VALUES(10926,11,21,2,0);
INSERT INTO "OrderDetails" VALUES(10926,13,6,10,0);
INSERT INTO "OrderDetails" VALUES(10926,19,9.2,7,0);
INSERT INTO "OrderDetails" VALUES(10926,72,34.8,10,0);
INSERT INTO "OrderDetails" VALUES(10927,20,81,5,0);
INSERT INTO "OrderDetails" VALUES(10927,52,7,5,0);
INSERT INTO "OrderDetails" VALUES(10927,76,18,20,0);
INSERT INTO "OrderDetails" VALUES(10928,47,9.5,5,0);
INSERT INTO "OrderDetails" VALUES(10928,76,18,5,0);
INSERT INTO "OrderDetails" VALUES(10929,21,10,60,0);
INSERT INTO "OrderDetails" VALUES(10929,75,7.75,49,0);
INSERT INTO "OrderDetails" VALUES(10929,77,13,15,0);
INSERT INTO "OrderDetails" VALUES(10930,21,10,36,0);
INSERT INTO "OrderDetails" VALUES(10930,27,43.9,25,0);
INSERT INTO "OrderDetails" VALUES(10930,55,24,25,0.2);
INSERT INTO "OrderDetails" VALUES(10930,58,13.25,30,0.2);
INSERT INTO "OrderDetails" VALUES(10931,13,6,42,0.15);
INSERT INTO "OrderDetails" VALUES(10931,57,19.5,30,0);
INSERT INTO "OrderDetails" VALUES(10932,16,17.45,30,0.1);
INSERT INTO "OrderDetails" VALUES(10932,62,49.3,14,0.1);
INSERT INTO "OrderDetails" VALUES(10932,72,34.8,16,0);
INSERT INTO "OrderDetails" VALUES(10932,75,7.75,20,0.1);
INSERT INTO "OrderDetails" VALUES(10933,53,32.8,2,0);
INSERT INTO "OrderDetails" VALUES(10933,61,28.5,30,0);
INSERT INTO "OrderDetails" VALUES(10934,6,25,20,0);
INSERT INTO "OrderDetails" VALUES(10935,1,18,21,0);
INSERT INTO "OrderDetails" VALUES(10935,18,62.5,4,0.25);
INSERT INTO "OrderDetails" VALUES(10935,23,9,8,0.25);
INSERT INTO "OrderDetails" VALUES(10936,36,19,30,0.2);
INSERT INTO "OrderDetails" VALUES(10937,28,45.6,8,0);
INSERT INTO "OrderDetails" VALUES(10937,34,14,20,0);
INSERT INTO "OrderDetails" VALUES(10938,13,6,20,0.25);
INSERT INTO "OrderDetails" VALUES(10938,43,46,24,0.25);
INSERT INTO "OrderDetails" VALUES(10938,60,34,49,0.25);
INSERT INTO "OrderDetails" VALUES(10938,71,21.5,35,0.25);
INSERT INTO "OrderDetails" VALUES(10939,2,19,10,0.15);
INSERT INTO "OrderDetails" VALUES(10939,67,14,40,0.15);
INSERT INTO "OrderDetails" VALUES(10940,7,30,8,0);
INSERT INTO "OrderDetails" VALUES(10940,13,6,20,0);
INSERT INTO "OrderDetails" VALUES(10941,31,12.5,44,0.25);
INSERT INTO "OrderDetails" VALUES(10941,62,49.3,30,0.25);
INSERT INTO "OrderDetails" VALUES(10941,68,12.5,80,0.25);
INSERT INTO "OrderDetails" VALUES(10941,72,34.8,50,0);
INSERT INTO "OrderDetails" VALUES(10942,49,20,28,0);
INSERT INTO "OrderDetails" VALUES(10943,13,6,15,0);
INSERT INTO "OrderDetails" VALUES(10943,22,21,21,0);
INSERT INTO "OrderDetails" VALUES(10943,46,12,15,0);
INSERT INTO "OrderDetails" VALUES(10944,11,21,5,0.25);
INSERT INTO "OrderDetails" VALUES(10944,44,19.45,18,0.25);
INSERT INTO "OrderDetails" VALUES(10944,56,38,18,0);
INSERT INTO "OrderDetails" VALUES(10945,13,6,20,0);
INSERT INTO "OrderDetails" VALUES(10945,31,12.5,10,0);
INSERT INTO "OrderDetails" VALUES(10946,10,31,25,0);
INSERT INTO "OrderDetails" VALUES(10946,24,4.5,25,0);
INSERT INTO "OrderDetails" VALUES(10946,77,13,40,0);
INSERT INTO "OrderDetails" VALUES(10947,59,55,4,0);
INSERT INTO "OrderDetails" VALUES(10948,50,16.25,9,0);
INSERT INTO "OrderDetails" VALUES(10948,51,53,40,0);
INSERT INTO "OrderDetails" VALUES(10948,55,24,4,0);
INSERT INTO "OrderDetails" VALUES(10949,6,25,12,0);
INSERT INTO "OrderDetails" VALUES(10949,10,31,30,0);
INSERT INTO "OrderDetails" VALUES(10949,17,39,6,0);
INSERT INTO "OrderDetails" VALUES(10949,62,49.3,60,0);
INSERT INTO "OrderDetails" VALUES(10950,4,22,5,0);
INSERT INTO "OrderDetails" VALUES(10951,33,2.5,15,0.05);
INSERT INTO "OrderDetails" VALUES(10951,41,9.65,6,0.05);
INSERT INTO "OrderDetails" VALUES(10951,75,7.75,50,0.05);
INSERT INTO "OrderDetails" VALUES(10952,6,25,16,0.05);
INSERT INTO "OrderDetails" VALUES(10952,28,45.6,2,0);
INSERT INTO "OrderDetails" VALUES(10953,20,81,50,0.05);
INSERT INTO "OrderDetails" VALUES(10953,31,12.5,50,0.05);
INSERT INTO "OrderDetails" VALUES(10954,16,17.45,28,0.15);
INSERT INTO "OrderDetails" VALUES(10954,31,12.5,25,0.15);
INSERT INTO "OrderDetails" VALUES(10954,45,9.5,30,0);
INSERT INTO "OrderDetails" VALUES(10954,60,34,24,0.15);
INSERT INTO "OrderDetails" VALUES(10955,75,7.75,12,0.2);
INSERT INTO "OrderDetails" VALUES(10956,21,10,12,0);
INSERT INTO "OrderDetails" VALUES(10956,47,9.5,14,0);
INSERT INTO "OrderDetails" VALUES(10956,51,53,8,0);
INSERT INTO "OrderDetails" VALUES(10957,30,25.89,30,0);
INSERT INTO "OrderDetails" VALUES(10957,35,18,40,0);
INSERT INTO "OrderDetails" VALUES(10957,64,33.25,8,0);
INSERT INTO "OrderDetails" VALUES(10958,5,21.35,20,0);
INSERT INTO "OrderDetails" VALUES(10958,7,30,6,0);
INSERT INTO "OrderDetails" VALUES(10958,72,34.8,5,0);
INSERT INTO "OrderDetails" VALUES(10959,75,7.75,20,0.15);
INSERT INTO "OrderDetails" VALUES(10960,24,4.5,10,0.25);
INSERT INTO "OrderDetails" VALUES(10960,41,9.65,24,0);
INSERT INTO "OrderDetails" VALUES(10961,52,7,6,0.05);
INSERT INTO "OrderDetails" VALUES(10961,76,18,60,0);
INSERT INTO "OrderDetails" VALUES(10962,7,30,45,0);
INSERT INTO "OrderDetails" VALUES(10962,13,6,77,0);
INSERT INTO "OrderDetails" VALUES(10962,53,32.8,20,0);
INSERT INTO "OrderDetails" VALUES(10962,69,36,9,0);
INSERT INTO "OrderDetails" VALUES(10962,76,18,44,0);
INSERT INTO "OrderDetails" VALUES(10963,60,34,2,0.15);
INSERT INTO "OrderDetails" VALUES(10964,18,62.5,6,0);
INSERT INTO "OrderDetails" VALUES(10964,38,263.5,5,0);
INSERT INTO "OrderDetails" VALUES(10964,69,36,10,0);
INSERT INTO "OrderDetails" VALUES(10965,51,53,16,0);
INSERT INTO "OrderDetails" VALUES(10966,37,26,8,0);
INSERT INTO "OrderDetails" VALUES(10966,56,38,12,0.15);
INSERT INTO "OrderDetails" VALUES(10966,62,49.3,12,0.15);
INSERT INTO "OrderDetails" VALUES(10967,19,9.2,12,0);
INSERT INTO "OrderDetails" VALUES(10967,49,20,40,0);
INSERT INTO "OrderDetails" VALUES(10968,12,38,30,0);
INSERT INTO "OrderDetails" VALUES(10968,24,4.5,30,0);
INSERT INTO "OrderDetails" VALUES(10968,64,33.25,4,0);
INSERT INTO "OrderDetails" VALUES(10969,46,12,9,0);
INSERT INTO "OrderDetails" VALUES(10970,52,7,40,0.2);
INSERT INTO "OrderDetails" VALUES(10971,29,123.79,14,0);
INSERT INTO "OrderDetails" VALUES(10972,17,39,6,0);
INSERT INTO "OrderDetails" VALUES(10972,33,2.5,7,0);
INSERT INTO "OrderDetails" VALUES(10973,26,31.23,5,0);
INSERT INTO "OrderDetails" VALUES(10973,41,9.65,6,0);
INSERT INTO "OrderDetails" VALUES(10973,75,7.75,10,0);
INSERT INTO "OrderDetails" VALUES(10974,63,43.9,10,0);
INSERT INTO "OrderDetails" VALUES(10975,8,40,16,0);
INSERT INTO "OrderDetails" VALUES(10975,75,7.75,10,0);
INSERT INTO "OrderDetails" VALUES(10976,28,45.6,20,0);
INSERT INTO "OrderDetails" VALUES(10977,39,18,30,0);
INSERT INTO "OrderDetails" VALUES(10977,47,9.5,30,0);
INSERT INTO "OrderDetails" VALUES(10977,51,53,10,0);
INSERT INTO "OrderDetails" VALUES(10977,63,43.9,20,0);
INSERT INTO "OrderDetails" VALUES(10978,8,40,20,0.15);
INSERT INTO "OrderDetails" VALUES(10978,21,10,40,0.15);
INSERT INTO "OrderDetails" VALUES(10978,40,18.4,10,0);
INSERT INTO "OrderDetails" VALUES(10978,44,19.45,6,0.15);
INSERT INTO "OrderDetails" VALUES(10979,7,30,18,0);
INSERT INTO "OrderDetails" VALUES(10979,12,38,20,0);
INSERT INTO "OrderDetails" VALUES(10979,24,4.5,80,0);
INSERT INTO "OrderDetails" VALUES(10979,27,43.9,30,0);
INSERT INTO "OrderDetails" VALUES(10979,31,12.5,24,0);
INSERT INTO "OrderDetails" VALUES(10979,63,43.9,35,0);
INSERT INTO "OrderDetails" VALUES(10980,75,7.75,40,0.2);
INSERT INTO "OrderDetails" VALUES(10981,38,263.5,60,0);
INSERT INTO "OrderDetails" VALUES(10982,7,30,20,0);
INSERT INTO "OrderDetails" VALUES(10982,43,46,9,0);
INSERT INTO "OrderDetails" VALUES(10983,13,6,84,0.15);
INSERT INTO "OrderDetails" VALUES(10983,57,19.5,15,0);
INSERT INTO "OrderDetails" VALUES(10984,16,17.45,55,0);
INSERT INTO "OrderDetails" VALUES(10984,24,4.5,20,0);
INSERT INTO "OrderDetails" VALUES(10984,36,19,40,0);
INSERT INTO "OrderDetails" VALUES(10985,16,17.45,36,0.1);
INSERT INTO "OrderDetails" VALUES(10985,18,62.5,8,0.1);
INSERT INTO "OrderDetails" VALUES(10985,32,32,35,0.1);
INSERT INTO "OrderDetails" VALUES(10986,11,21,30,0);
INSERT INTO "OrderDetails" VALUES(10986,20,81,15,0);
INSERT INTO "OrderDetails" VALUES(10986,76,18,10,0);
INSERT INTO "OrderDetails" VALUES(10986,77,13,15,0);
INSERT INTO "OrderDetails" VALUES(10987,7,30,60,0);
INSERT INTO "OrderDetails" VALUES(10987,43,46,6,0);
INSERT INTO "OrderDetails" VALUES(10987,72,34.8,20,0);
INSERT INTO "OrderDetails" VALUES(10988,7,30,60,0);
INSERT INTO "OrderDetails" VALUES(10988,62,49.3,40,0.1);
INSERT INTO "OrderDetails" VALUES(10989,6,25,40,0);
INSERT INTO "OrderDetails" VALUES(10989,11,21,15,0);
INSERT INTO "OrderDetails" VALUES(10989,41,9.65,4,0);
INSERT INTO "OrderDetails" VALUES(10990,21,10,65,0);
INSERT INTO "OrderDetails" VALUES(10990,34,14,60,0.15);
INSERT INTO "OrderDetails" VALUES(10990,55,24,65,0.15);
INSERT INTO "OrderDetails" VALUES(10990,61,28.5,66,0.15);
INSERT INTO "OrderDetails" VALUES(10991,2,19,50,0.2);
INSERT INTO "OrderDetails" VALUES(10991,70,15,20,0.2);
INSERT INTO "OrderDetails" VALUES(10991,76,18,90,0.2);
INSERT INTO "OrderDetails" VALUES(10992,72,34.8,2,0);
INSERT INTO "OrderDetails" VALUES(10993,29,123.79,50,0.25);
INSERT INTO "OrderDetails" VALUES(10993,41,9.65,35,0.25);
INSERT INTO "OrderDetails" VALUES(10994,59,55,18,0.05);
INSERT INTO "OrderDetails" VALUES(10995,51,53,20,0);
INSERT INTO "OrderDetails" VALUES(10995,60,34,4,0);
INSERT INTO "OrderDetails" VALUES(10996,42,14,40,0);
INSERT INTO "OrderDetails" VALUES(10997,32,32,50,0);
INSERT INTO "OrderDetails" VALUES(10997,46,12,20,0.25);
INSERT INTO "OrderDetails" VALUES(10997,52,7,20,0.25);
INSERT INTO "OrderDetails" VALUES(10998,24,4.5,12,0);
INSERT INTO "OrderDetails" VALUES(10998,61,28.5,7,0);
INSERT INTO "OrderDetails" VALUES(10998,74,10,20,0);
INSERT INTO "OrderDetails" VALUES(10998,75,7.75,30,0);
INSERT INTO "OrderDetails" VALUES(10999,41,9.65,20,0.05);
INSERT INTO "OrderDetails" VALUES(10999,51,53,15,0.05);
INSERT INTO "OrderDetails" VALUES(10999,77,13,21,0.05);
INSERT INTO "OrderDetails" VALUES(11000,4,22,25,0.25);
INSERT INTO "OrderDetails" VALUES(11000,24,4.5,30,0.25);
INSERT INTO "OrderDetails" VALUES(11000,77,13,30,0);
INSERT INTO "OrderDetails" VALUES(11001,7,30,60,0);
INSERT INTO "OrderDetails" VALUES(11001,22,21,25,0);
INSERT INTO "OrderDetails" VALUES(11001,46,12,25,0);
INSERT INTO "OrderDetails" VALUES(11001,55,24,6,0);
INSERT INTO "OrderDetails" VALUES(11002,13,6,56,0);
INSERT INTO "OrderDetails" VALUES(11002,35,18,15,0.15);
INSERT INTO "OrderDetails" VALUES(11002,42,14,24,0.15);
INSERT INTO "OrderDetails" VALUES(11002,55,24,40,0);
INSERT INTO "OrderDetails" VALUES(11003,1,18,4,0);
INSERT INTO "OrderDetails" VALUES(11003,40,18.4,10,0);
INSERT INTO "OrderDetails" VALUES(11003,52,7,10,0);
INSERT INTO "OrderDetails" VALUES(11004,26,31.23,6,0);
INSERT INTO "OrderDetails" VALUES(11004,76,18,6,0);
INSERT INTO "OrderDetails" VALUES(11005,1,18,2,0);
INSERT INTO "OrderDetails" VALUES(11005,59,55,10,0);
INSERT INTO "OrderDetails" VALUES(11006,1,18,8,0);
INSERT INTO "OrderDetails" VALUES(11006,29,123.79,2,0.25);
INSERT INTO "OrderDetails" VALUES(11007,8,40,30,0);
INSERT INTO "OrderDetails" VALUES(11007,29,123.79,10,0);
INSERT INTO "OrderDetails" VALUES(11007,42,14,14,0);
INSERT INTO "OrderDetails" VALUES(11008,28,45.6,70,0.05);
INSERT INTO "OrderDetails" VALUES(11008,34,14,90,0.05);
INSERT INTO "OrderDetails" VALUES(11008,71,21.5,21,0);
INSERT INTO "OrderDetails" VALUES(11009,24,4.5,12,0);
INSERT INTO "OrderDetails" VALUES(11009,36,19,18,0.25);
INSERT INTO "OrderDetails" VALUES(11009,60,34,9,0);
INSERT INTO "OrderDetails" VALUES(11010,7,30,20,0);
INSERT INTO "OrderDetails" VALUES(11010,24,4.5,10,0);
INSERT INTO "OrderDetails" VALUES(11011,58,13.25,40,0.05);
INSERT INTO "OrderDetails" VALUES(11011,71,21.5,20,0);
INSERT INTO "OrderDetails" VALUES(11012,19,9.2,50,0.05);
INSERT INTO "OrderDetails" VALUES(11012,60,34,36,0.05);
INSERT INTO "OrderDetails" VALUES(11012,71,21.5,60,0.05);
INSERT INTO "OrderDetails" VALUES(11013,23,9,10,0);
INSERT INTO "OrderDetails" VALUES(11013,42,14,4,0);
INSERT INTO "OrderDetails" VALUES(11013,45,9.5,20,0);
INSERT INTO "OrderDetails" VALUES(11013,68,12.5,2,0);
INSERT INTO "OrderDetails" VALUES(11014,41,9.65,28,0.1);
INSERT INTO "OrderDetails" VALUES(11015,30,25.89,15,0);
INSERT INTO "OrderDetails" VALUES(11015,77,13,18,0);
INSERT INTO "OrderDetails" VALUES(11016,31,12.5,15,0);
INSERT INTO "OrderDetails" VALUES(11016,36,19,16,0);
INSERT INTO "OrderDetails" VALUES(11017,3,10,25,0);
INSERT INTO "OrderDetails" VALUES(11017,59,55,110,0);
INSERT INTO "OrderDetails" VALUES(11017,70,15,30,0);
INSERT INTO "OrderDetails" VALUES(11018,12,38,20,0);
INSERT INTO "OrderDetails" VALUES(11018,18,62.5,10,0);
INSERT INTO "OrderDetails" VALUES(11018,56,38,5,0);
INSERT INTO "OrderDetails" VALUES(11019,46,12,3,0);
INSERT INTO "OrderDetails" VALUES(11019,49,20,2,0);
INSERT INTO "OrderDetails" VALUES(11020,10,31,24,0.15);
INSERT INTO "OrderDetails" VALUES(11021,2,19,11,0.25);
INSERT INTO "OrderDetails" VALUES(11021,20,81,15,0);
INSERT INTO "OrderDetails" VALUES(11021,26,31.23,63,0);
INSERT INTO "OrderDetails" VALUES(11021,51,53,44,0.25);
INSERT INTO "OrderDetails" VALUES(11021,72,34.8,35,0);
INSERT INTO "OrderDetails" VALUES(11022,19,9.2,35,0);
INSERT INTO "OrderDetails" VALUES(11022,69,36,30,0);
INSERT INTO "OrderDetails" VALUES(11023,7,30,4,0);
INSERT INTO "OrderDetails" VALUES(11023,43,46,30,0);
INSERT INTO "OrderDetails" VALUES(11024,26,31.23,12,0);
INSERT INTO "OrderDetails" VALUES(11024,33,2.5,30,0);
INSERT INTO "OrderDetails" VALUES(11024,65,21.05,21,0);
INSERT INTO "OrderDetails" VALUES(11024,71,21.5,50,0);
INSERT INTO "OrderDetails" VALUES(11025,1,18,10,0.1);
INSERT INTO "OrderDetails" VALUES(11025,13,6,20,0.1);
INSERT INTO "OrderDetails" VALUES(11026,18,62.5,8,0);
INSERT INTO "OrderDetails" VALUES(11026,51,53,10,0);
INSERT INTO "OrderDetails" VALUES(11027,24,4.5,30,0.25);
INSERT INTO "OrderDetails" VALUES(11027,62,49.3,21,0.25);
INSERT INTO "OrderDetails" VALUES(11028,55,24,35,0);
INSERT INTO "OrderDetails" VALUES(11028,59,55,24,0);
INSERT INTO "OrderDetails" VALUES(11029,56,38,20,0);
INSERT INTO "OrderDetails" VALUES(11029,63,43.9,12,0);
INSERT INTO "OrderDetails" VALUES(11030,2,19,100,0.25);
INSERT INTO "OrderDetails" VALUES(11030,5,21.35,70,0);
INSERT INTO "OrderDetails" VALUES(11030,29,123.79,60,0.25);
INSERT INTO "OrderDetails" VALUES(11030,59,55,100,0.25);
INSERT INTO "OrderDetails" VALUES(11031,1,18,45,0);
INSERT INTO "OrderDetails" VALUES(11031,13,6,80,0);
INSERT INTO "OrderDetails" VALUES(11031,24,4.5,21,0);
INSERT INTO "OrderDetails" VALUES(11031,64,33.25,20,0);
INSERT INTO "OrderDetails" VALUES(11031,71,21.5,16,0);
INSERT INTO "OrderDetails" VALUES(11032,36,19,35,0);
INSERT INTO "OrderDetails" VALUES(11032,38,263.5,25,0);
INSERT INTO "OrderDetails" VALUES(11032,59,55,30,0);
INSERT INTO "OrderDetails" VALUES(11033,53,32.8,70,0.1);
INSERT INTO "OrderDetails" VALUES(11033,69,36,36,0.1);
INSERT INTO "OrderDetails" VALUES(11034,21,10,15,0.1);
INSERT INTO "OrderDetails" VALUES(11034,44,19.45,12,0);
INSERT INTO "OrderDetails" VALUES(11034,61,28.5,6,0);
INSERT INTO "OrderDetails" VALUES(11035,1,18,10,0);
INSERT INTO "OrderDetails" VALUES(11035,35,18,60,0);
INSERT INTO "OrderDetails" VALUES(11035,42,14,30,0);
INSERT INTO "OrderDetails" VALUES(11035,54,7.45,10,0);
INSERT INTO "OrderDetails" VALUES(11036,13,6,7,0);
INSERT INTO "OrderDetails" VALUES(11036,59,55,30,0);
INSERT INTO "OrderDetails" VALUES(11037,70,15,4,0);
INSERT INTO "OrderDetails" VALUES(11038,40,18.4,5,0.2);
INSERT INTO "OrderDetails" VALUES(11038,52,7,2,0);
INSERT INTO "OrderDetails" VALUES(11038,71,21.5,30,0);
INSERT INTO "OrderDetails" VALUES(11039,28,45.6,20,0);
INSERT INTO "OrderDetails" VALUES(11039,35,18,24,0);
INSERT INTO "OrderDetails" VALUES(11039,49,20,60,0);
INSERT INTO "OrderDetails" VALUES(11039,57,19.5,28,0);
INSERT INTO "OrderDetails" VALUES(11040,21,10,20,0);
INSERT INTO "OrderDetails" VALUES(11041,2,19,30,0.2);
INSERT INTO "OrderDetails" VALUES(11041,63,43.9,30,0);
INSERT INTO "OrderDetails" VALUES(11042,44,19.45,15,0);
INSERT INTO "OrderDetails" VALUES(11042,61,28.5,4,0);
INSERT INTO "OrderDetails" VALUES(11043,11,21,10,0);
INSERT INTO "OrderDetails" VALUES(11044,62,49.3,12,0);
INSERT INTO "OrderDetails" VALUES(11045,33,2.5,15,0);
INSERT INTO "OrderDetails" VALUES(11045,51,53,24,0);
INSERT INTO "OrderDetails" VALUES(11046,12,38,20,0.05);
INSERT INTO "OrderDetails" VALUES(11046,32,32,15,0.05);
INSERT INTO "OrderDetails" VALUES(11046,35,18,18,0.05);
INSERT INTO "OrderDetails" VALUES(11047,1,18,25,0.25);
INSERT INTO "OrderDetails" VALUES(11047,5,21.35,30,0.25);
INSERT INTO "OrderDetails" VALUES(11048,68,12.5,42,0);
INSERT INTO "OrderDetails" VALUES(11049,2,19,10,0.2);
INSERT INTO "OrderDetails" VALUES(11049,12,38,4,0.2);
INSERT INTO "OrderDetails" VALUES(11050,76,18,50,0.1);
INSERT INTO "OrderDetails" VALUES(11051,24,4.5,10,0.2);
INSERT INTO "OrderDetails" VALUES(11052,43,46,30,0.2);
INSERT INTO "OrderDetails" VALUES(11052,61,28.5,10,0.2);
INSERT INTO "OrderDetails" VALUES(11053,18,62.5,35,0.2);
INSERT INTO "OrderDetails" VALUES(11053,32,32,20,0);
INSERT INTO "OrderDetails" VALUES(11053,64,33.25,25,0.2);
INSERT INTO "OrderDetails" VALUES(11054,33,2.5,10,0);
INSERT INTO "OrderDetails" VALUES(11054,67,14,20,0);
INSERT INTO "OrderDetails" VALUES(11055,24,4.5,15,0);
INSERT INTO "OrderDetails" VALUES(11055,25,14,15,0);
INSERT INTO "OrderDetails" VALUES(11055,51,53,20,0);
INSERT INTO "OrderDetails" VALUES(11055,57,19.5,20,0);
INSERT INTO "OrderDetails" VALUES(11056,7,30,40,0);
INSERT INTO "OrderDetails" VALUES(11056,55,24,35,0);
INSERT INTO "OrderDetails" VALUES(11056,60,34,50,0);
INSERT INTO "OrderDetails" VALUES(11057,70,15,3,0);
INSERT INTO "OrderDetails" VALUES(11058,21,10,3,0);
INSERT INTO "OrderDetails" VALUES(11058,60,34,21,0);
INSERT INTO "OrderDetails" VALUES(11058,61,28.5,4,0);
INSERT INTO "OrderDetails" VALUES(11059,13,6,30,0);
INSERT INTO "OrderDetails" VALUES(11059,17,39,12,0);
INSERT INTO "OrderDetails" VALUES(11059,60,34,35,0);
INSERT INTO "OrderDetails" VALUES(11060,60,34,4,0);
INSERT INTO "OrderDetails" VALUES(11060,77,13,10,0);
INSERT INTO "OrderDetails" VALUES(11061,60,34,15,0);
INSERT INTO "OrderDetails" VALUES(11062,53,32.8,10,0.2);
INSERT INTO "OrderDetails" VALUES(11062,70,15,12,0.2);
INSERT INTO "OrderDetails" VALUES(11063,34,14,30,0);
INSERT INTO "OrderDetails" VALUES(11063,40,18.4,40,0.1);
INSERT INTO "OrderDetails" VALUES(11063,41,9.65,30,0.1);
INSERT INTO "OrderDetails" VALUES(11064,17,39,77,0.1);
INSERT INTO "OrderDetails" VALUES(11064,41,9.65,12,0);
INSERT INTO "OrderDetails" VALUES(11064,53,32.8,25,0.1);
INSERT INTO "OrderDetails" VALUES(11064,55,24,4,0.1);
INSERT INTO "OrderDetails" VALUES(11064,68,12.5,55,0);
INSERT INTO "OrderDetails" VALUES(11065,30,25.89,4,0.25);
INSERT INTO "OrderDetails" VALUES(11065,54,7.45,20,0.25);
INSERT INTO "OrderDetails" VALUES(11066,16,17.45,3,0);
INSERT INTO "OrderDetails" VALUES(11066,19,9.2,42,0);
INSERT INTO "OrderDetails" VALUES(11066,34,14,35,0);
INSERT INTO "OrderDetails" VALUES(11067,41,9.65,9,0);
INSERT INTO "OrderDetails" VALUES(11068,28,45.6,8,0.15);
INSERT INTO "OrderDetails" VALUES(11068,43,46,36,0.15);
INSERT INTO "OrderDetails" VALUES(11068,77,13,28,0.15);
INSERT INTO "OrderDetails" VALUES(11069,39,18,20,0);
INSERT INTO "OrderDetails" VALUES(11070,1,18,40,0.15);
INSERT INTO "OrderDetails" VALUES(11070,2,19,20,0.15);
INSERT INTO "OrderDetails" VALUES(11070,16,17.45,30,0.15);
INSERT INTO "OrderDetails" VALUES(11070,31,12.5,20,0);
INSERT INTO "OrderDetails" VALUES(11071,7,30,15,0.05);
INSERT INTO "OrderDetails" VALUES(11071,13,6,10,0.05);
INSERT INTO "OrderDetails" VALUES(11072,2,19,8,0);
INSERT INTO "OrderDetails" VALUES(11072,41,9.65,40,0);
INSERT INTO "OrderDetails" VALUES(11072,50,16.25,22,0);
INSERT INTO "OrderDetails" VALUES(11072,64,33.25,130,0);
INSERT INTO "OrderDetails" VALUES(11073,11,21,10,0);
INSERT INTO "OrderDetails" VALUES(11073,24,4.5,20,0);
INSERT INTO "OrderDetails" VALUES(11074,16,17.45,14,0.05);
INSERT INTO "OrderDetails" VALUES(11075,2,19,10,0.15);
INSERT INTO "OrderDetails" VALUES(11075,46,12,30,0.15);
INSERT INTO "OrderDetails" VALUES(11075,76,18,2,0.15);
INSERT INTO "OrderDetails" VALUES(11076,6,25,20,0.25);
INSERT INTO "OrderDetails" VALUES(11076,14,23.25,20,0.25);
INSERT INTO "OrderDetails" VALUES(11076,19,9.2,10,0.25);
INSERT INTO "OrderDetails" VALUES(11077,2,19,24,0.2);
INSERT INTO "OrderDetails" VALUES(11077,3,10,4,0);
INSERT INTO "OrderDetails" VALUES(11077,4,22,1,0);
INSERT INTO "OrderDetails" VALUES(11077,6,25,1,0.02);
INSERT INTO "OrderDetails" VALUES(11077,7,30,1,0.05);
INSERT INTO "OrderDetails" VALUES(11077,8,40,2,0.1);
INSERT INTO "OrderDetails" VALUES(11077,10,31,1,0);
INSERT INTO "OrderDetails" VALUES(11077,12,38,2,0.05);
INSERT INTO "OrderDetails" VALUES(11077,13,6,4,0);
INSERT INTO "OrderDetails" VALUES(11077,14,23.25,1,0.03);
INSERT INTO "OrderDetails" VALUES(11077,16,17.45,2,0.03);
INSERT INTO "OrderDetails" VALUES(11077,20,81,1,0.04);
INSERT INTO "OrderDetails" VALUES(11077,23,9,2,0);
INSERT INTO "OrderDetails" VALUES(11077,32,32,1,0);
INSERT INTO "OrderDetails" VALUES(11077,39,18,2,0.05);
INSERT INTO "OrderDetails" VALUES(11077,41,9.65,3,0);
INSERT INTO "OrderDetails" VALUES(11077,46,12,3,0.02);
INSERT INTO "OrderDetails" VALUES(11077,52,7,2,0);
INSERT INTO "OrderDetails" VALUES(11077,55,24,2,0);
INSERT INTO "OrderDetails" VALUES(11077,60,34,2,0.06);
INSERT INTO "OrderDetails" VALUES(11077,64,33.25,2,0.03);
INSERT INTO "OrderDetails" VALUES(11077,66,17,1,0);
INSERT INTO "OrderDetails" VALUES(11077,73,15,2,0.01);
INSERT INTO "OrderDetails" VALUES(11077,75,7.75,4,0);
INSERT INTO "OrderDetails" VALUES(11077,77,13,2,0);
INSERT INTO "EmployeesTerritories" Values (1,'06897');
INSERT INTO "EmployeesTerritories" Values (1,'19713');
INSERT INTO "EmployeesTerritories" Values (2,'01581');
INSERT INTO "EmployeesTerritories" Values (2,'01730');
INSERT INTO "EmployeesTerritories" Values (2,'01833');
INSERT INTO "EmployeesTerritories" Values (2,'02116');
INSERT INTO "EmployeesTerritories" Values (2,'02139');
INSERT INTO "EmployeesTerritories" Values (2,'02184');
INSERT INTO "EmployeesTerritories" Values (2,'40222');
INSERT INTO "EmployeesTerritories" Values (3,'30346');
INSERT INTO "EmployeesTerritories" Values (3,'31406');
INSERT INTO "EmployeesTerritories" Values (3,'32859');
INSERT INTO "EmployeesTerritories" Values (3,'33607');
INSERT INTO "EmployeesTerritories" Values (4,'20852');
INSERT INTO "EmployeesTerritories" Values (4,'27403');
INSERT INTO "EmployeesTerritories" Values (4,'27511');
INSERT INTO "EmployeesTerritories" Values (5,'02903');
INSERT INTO "EmployeesTerritories" Values (5,'07960');
INSERT INTO "EmployeesTerritories" Values (5,'08837');
INSERT INTO "EmployeesTerritories" Values (5,'10019');
INSERT INTO "EmployeesTerritories" Values (5,'10038');
INSERT INTO "EmployeesTerritories" Values (5,'11747');
INSERT INTO "EmployeesTerritories" Values (5,'14450');
INSERT INTO "EmployeesTerritories" Values (6,'85014');
INSERT INTO "EmployeesTerritories" Values (6,'85251');
INSERT INTO "EmployeesTerritories" Values (6,'98004');
INSERT INTO "EmployeesTerritories" Values (6,'98052');
INSERT INTO "EmployeesTerritories" Values (6,'98104');
INSERT INTO "EmployeesTerritories" Values (8,'19428');
INSERT INTO "EmployeesTerritories" Values (8,'44122');
INSERT INTO "EmployeesTerritories" Values (8,'45839');
INSERT INTO "EmployeesTerritories" Values (8,'53404');
INSERT INTO "EmployeesTerritories" Values (9,'03049');
INSERT INTO "EmployeesTerritories" Values (9,'03801');
INSERT INTO "EmployeesTerritories" Values (9,'48075');
INSERT INTO "EmployeesTerritories" Values (9,'48084');
INSERT INTO "EmployeesTerritories" Values (9,'48304');
INSERT INTO "EmployeesTerritories" Values (9,'55113');
INSERT INTO "EmployeesTerritories" Values (9,'55439');


	CREATE TABLE "InternationalOrders"(
	"OrderID" numeric(10) NOT NULL,
	"CustomsDescription" VARCHAR2(100),
	"ExciseTax" numeric NOT NULL,
	CONSTRAINT "PK_InternationalOrders" PRIMARY KEY 
	(
		"OrderID"
	),
	CONSTRAINT "FK_InternationalOrders_Orders" FOREIGN KEY 
	(
		"OrderID"
	) REFERENCES "Orders" (
		"OrderID"
	)
 );
 
 INSERT INTO "InternationalOrders" VALUES(10248, 'Singaporean Hokkien Fried Mee and other products for personal use', 201.0209);
INSERT INTO "InternationalOrders" VALUES(10249, 'Tofu and other products for personal use', 851.3233);
INSERT INTO "InternationalOrders" VALUES(10250, 'Manjimup Dried Apples and other products for personal use', 828.2973);
INSERT INTO "InternationalOrders" VALUES(10251, 'Ravioli Angelo and other products for personal use', 306.4654);
INSERT INTO "InternationalOrders" VALUES(10252, 'Sir Rodney''s Marmalade and other products for personal use', 1704.1086);
INSERT INTO "InternationalOrders" VALUES(10253, 'Maxilaku and other products for personal use', 660.0794);
INSERT INTO "InternationalOrders" VALUES(10254, 'Pate chinois and other products for personal use', 285.6324);
INSERT INTO "InternationalOrders" VALUES(10255, 'Raclette Courdavault and other products for personal use', 1137.8237);
INSERT INTO "InternationalOrders" VALUES(10256, 'Perth Pasties and other products for personal use', 236.5650);
INSERT INTO "InternationalOrders" VALUES(10257, 'Schoggi Schokolade and other products for personal use', 511.6438);
INSERT INTO "InternationalOrders" VALUES(10258, 'Mascarpone Fabioli and other products for personal use', 922.2289);
INSERT INTO "InternationalOrders" VALUES(10259, 'Sir Rodney''s Scones and other products for personal use', 46.0521);
INSERT INTO "InternationalOrders" VALUES(10260, 'Tarte au sucre and other products for personal use', 797.7787);
INSERT INTO "InternationalOrders" VALUES(10261, 'Steeleye Stout and other products for personal use', 204.6758);
INSERT INTO "InternationalOrders" VALUES(10263, 'Pavlova and other products for personal use', 1126.0823);
INSERT INTO "InternationalOrders" VALUES(10264, 'Jack''s New England Clam Chowder and other products for personal use', 330.9991);
INSERT INTO "InternationalOrders" VALUES(10265, 'Outback Lager and other products for personal use', 537.2739);
INSERT INTO "InternationalOrders" VALUES(10266, 'Queso Manchego La Pastora and other products for personal use', 166.6646);
INSERT INTO "InternationalOrders" VALUES(10267, 'Raclette Courdavault and other products for personal use', 1841.6252);
INSERT INTO "InternationalOrders" VALUES(10268, 'Thuringer Rostbratwurst and other products for personal use', 503.1004);
INSERT INTO "InternationalOrders" VALUES(10270, 'Ipoh Coffee and other products for personal use', 628.6470);
INSERT INTO "InternationalOrders" VALUES(10273, 'Lakkalikoori and other products for personal use', 978.7888);
INSERT INTO "InternationalOrders" VALUES(10274, 'Mozzarella di Giovanni and other products for personal use', 246.0678);
INSERT INTO "InternationalOrders" VALUES(10275, 'Raclette Courdavault and other products for personal use', 140.3491);
INSERT INTO "InternationalOrders" VALUES(10276, 'Konbu and other products for personal use', 191.8835);
INSERT INTO "InternationalOrders" VALUES(10277, 'Tarte au sucre and other products for personal use', 548.6042);
INSERT INTO "InternationalOrders" VALUES(10278, 'Vegie-spread and other products for personal use', 680.1815);
INSERT INTO "InternationalOrders" VALUES(10279, 'Alice Mutton and other products for personal use', 213.8131);
INSERT INTO "InternationalOrders" VALUES(10280, 'Rhonbrau Klosterbier and other products for personal use', 280.1500);
INSERT INTO "InternationalOrders" VALUES(10281, 'Teatime Chocolate Biscuits and other products for personal use', 39.5189);
INSERT INTO "InternationalOrders" VALUES(10282, 'Ravioli Angelo and other products for personal use', 70.9969);
INSERT INTO "InternationalOrders" VALUES(10283, 'Teatime Chocolate Biscuits and other products for personal use', 646.3734);
INSERT INTO "InternationalOrders" VALUES(10284, 'Schoggi Schokolade and other products for personal use', 663.3688);
INSERT INTO "InternationalOrders" VALUES(10285, 'Perth Pasties and other products for personal use', 995.6015);
INSERT INTO "InternationalOrders" VALUES(10286, 'Tarte au sucre and other products for personal use', 1377.9066);
INSERT INTO "InternationalOrders" VALUES(10287, 'Spegesild and other products for personal use', 422.1438);
INSERT INTO "InternationalOrders" VALUES(10288, 'Tourtiere and other products for personal use', 40.6610);
INSERT INTO "InternationalOrders" VALUES(10289, 'Wimmers gute Semmelknodel and other products for personal use', 219.0214);
INSERT INTO "InternationalOrders" VALUES(10290, 'Thuringer Rostbratwurst and other products for personal use', 990.9414);
INSERT INTO "InternationalOrders" VALUES(10291, 'Manjimup Dried Apples and other products for personal use', 252.5553);
INSERT INTO "InternationalOrders" VALUES(10292, 'Sir Rodney''s Marmalade and other products for personal use', 592.0978);
INSERT INTO "InternationalOrders" VALUES(10293, 'Vegie-spread and other products for personal use', 387.7418);
INSERT INTO "InternationalOrders" VALUES(10295, 'Gnocchi di nonna Alice and other products for personal use', 55.5549);
INSERT INTO "InternationalOrders" VALUES(10296, 'Queso Cabrales and other products for personal use', 479.9830);
INSERT INTO "InternationalOrders" VALUES(10297, 'Mozzarella di Giovanni and other products for personal use', 648.7491);
INSERT INTO "InternationalOrders" VALUES(10298, 'Tarte au sucre and other products for personal use', 1428.6187);
INSERT INTO "InternationalOrders" VALUES(10299, 'Teatime Chocolate Biscuits and other products for personal use', 159.6745);
INSERT INTO "InternationalOrders" VALUES(10300, 'Scottish Longbreads and other products for personal use', 277.7743);
INSERT INTO "InternationalOrders" VALUES(10301, 'Gnocchi di nonna Alice and other products for personal use', 344.9335);
INSERT INTO "InternationalOrders" VALUES(10302, 'Rossle Sauerkraut and other products for personal use', 1237.5575);
INSERT INTO "InternationalOrders" VALUES(10303, 'Scottish Longbreads and other products for personal use', 567.4270);
INSERT INTO "InternationalOrders" VALUES(10304, 'Raclette Courdavault and other products for personal use', 436.0325);
INSERT INTO "InternationalOrders" VALUES(10306, 'Tourtiere and other products for personal use', 227.7475);
INSERT INTO "InternationalOrders" VALUES(10308, 'Outback Lager and other products for personal use', 40.5697);
INSERT INTO "InternationalOrders" VALUES(10309, 'Singaporean Hokkien Fried Mee and other products for personal use', 804.9972);
INSERT INTO "InternationalOrders" VALUES(10311, 'Singaporean Hokkien Fried Mee and other products for personal use', 122.8055);
INSERT INTO "InternationalOrders" VALUES(10312, 'Rossle Sauerkraut and other products for personal use', 737.7465);
INSERT INTO "InternationalOrders" VALUES(10313, 'Inlagd Sill and other products for personal use', 83.3323);
INSERT INTO "InternationalOrders" VALUES(10315, 'Sasquatch Ale and other products for personal use', 236.1081);
INSERT INTO "InternationalOrders" VALUES(10318, 'Lakkalikoori and other products for personal use', 109.8305);
INSERT INTO "InternationalOrders" VALUES(10319, 'Rossle Sauerkraut and other products for personal use', 544.2183);
INSERT INTO "InternationalOrders" VALUES(10320, 'Flotemysost and other products for personal use', 235.7426);
INSERT INTO "InternationalOrders" VALUES(10321, 'Steeleye Stout and other products for personal use', 65.7886);
INSERT INTO "InternationalOrders" VALUES(10322, 'Filo Mix and other products for personal use', 51.1689);
INSERT INTO "InternationalOrders" VALUES(10323, 'NuNuCa Nu?-Nougat-Creme and other products for personal use', 75.1087);
INSERT INTO "InternationalOrders" VALUES(10325, 'Tofu and other products for personal use', 683.9278);
INSERT INTO "InternationalOrders" VALUES(10326, 'Rhonbrau Klosterbier and other products for personal use', 448.6420);
INSERT INTO "InternationalOrders" VALUES(10327, 'Queso Cabrales and other products for personal use', 1033.6584);
INSERT INTO "InternationalOrders" VALUES(10328, 'Scottish Longbreads and other products for personal use', 533.6190);
INSERT INTO "InternationalOrders" VALUES(10330, 'Mozzarella di Giovanni and other products for personal use', 886.3192);
INSERT INTO "InternationalOrders" VALUES(10331, 'Tourtiere and other products for personal use', 40.4326);
INSERT INTO "InternationalOrders" VALUES(10332, 'Zaanse koeken and other products for personal use', 1020.4550);
INSERT INTO "InternationalOrders" VALUES(10333, 'Tofu and other products for personal use', 435.8498);
INSERT INTO "InternationalOrders" VALUES(10334, 'Scottish Longbreads and other products for personal use', 66.1541);
INSERT INTO "InternationalOrders" VALUES(10335, 'Mascarpone Fabioli and other products for personal use', 1162.8143);
INSERT INTO "InternationalOrders" VALUES(10336, 'Chef Anton''s Cajun Seasoning and other products for personal use', 144.7350);
INSERT INTO "InternationalOrders" VALUES(10337, 'Tunnbrod and other products for personal use', 1127.0874);
INSERT INTO "InternationalOrders" VALUES(10339, 'Tarte au sucre and other products for personal use', 1582.2169);
INSERT INTO "InternationalOrders" VALUES(10340, 'Jack''s New England Clam Chowder and other products for personal use', 1171.5861);
INSERT INTO "InternationalOrders" VALUES(10341, 'Raclette Courdavault and other products for personal use', 188.2286);
INSERT INTO "InternationalOrders" VALUES(10342, 'Pate chinois and other products for personal use', 1051.1563);
INSERT INTO "InternationalOrders" VALUES(10343, 'Wimmers gute Semmelknodel and other products for personal use', 724.5888);
INSERT INTO "InternationalOrders" VALUES(10345, 'Teatime Chocolate Biscuits and other products for personal use', 1336.2404);
INSERT INTO "InternationalOrders" VALUES(10347, 'Rhonbrau Klosterbier and other products for personal use', 423.9713);
INSERT INTO "InternationalOrders" VALUES(10348, 'Tunnbrod and other products for personal use', 180.9188);
INSERT INTO "InternationalOrders" VALUES(10350, 'Valkoinen suklaa and other products for personal use', 325.9279);
INSERT INTO "InternationalOrders" VALUES(10351, 'Louisiana Fiery Hot Pepper Sauce and other products for personal use', 2593.9000);
INSERT INTO "InternationalOrders" VALUES(10352, 'Tourtiere and other products for personal use', 70.3573);
INSERT INTO "InternationalOrders" VALUES(10353, 'Queso Cabrales and other products for personal use', 4907.4673);
INSERT INTO "InternationalOrders" VALUES(10354, 'Thuringer Rostbratwurst and other products for personal use', 259.8651);
INSERT INTO "InternationalOrders" VALUES(10355, 'Ravioli Angelo and other products for personal use', 219.2955);
INSERT INTO "InternationalOrders" VALUES(10356, 'Pate chinois and other products for personal use', 505.4761);
INSERT INTO "InternationalOrders" VALUES(10357, 'Ikura and other products for personal use', 621.3372);
INSERT INTO "InternationalOrders" VALUES(10358, 'Sasquatch Ale and other products for personal use', 206.5032);
INSERT INTO "InternationalOrders" VALUES(10359, 'Pavlova and other products for personal use', 1669.5696);
INSERT INTO "InternationalOrders" VALUES(10360, 'Tourtiere and other products for personal use', 3376.3280);
INSERT INTO "InternationalOrders" VALUES(10361, 'Chartreuse verte and other products for personal use', 1038.7296);
INSERT INTO "InternationalOrders" VALUES(10362, 'Tourtiere and other products for personal use', 707.9589);
INSERT INTO "InternationalOrders" VALUES(10363, 'Rhonbrau Klosterbier and other products for personal use', 204.3103);
INSERT INTO "InternationalOrders" VALUES(10364, 'Gudbrandsdalsost and other products for personal use', 434.0223);
INSERT INTO "InternationalOrders" VALUES(10365, 'Queso Cabrales and other products for personal use', 184.2082);
INSERT INTO "InternationalOrders" VALUES(10366, 'Original Frankfurter grune So?e and other products for personal use', 62.1337);
INSERT INTO "InternationalOrders" VALUES(10367, 'Tourtiere and other products for personal use', 381.1173);
INSERT INTO "InternationalOrders" VALUES(10368, 'Wimmers gute Semmelknodel and other products for personal use', 837.9828);
INSERT INTO "InternationalOrders" VALUES(10370, 'Wimmers gute Semmelknodel and other products for personal use', 536.3602);
INSERT INTO "InternationalOrders" VALUES(10371, 'Inlagd Sill and other products for personal use', 41.6661);
INSERT INTO "InternationalOrders" VALUES(10372, 'Sir Rodney''s Marmalade and other products for personal use', 5610.8576);
INSERT INTO "InternationalOrders" VALUES(10373, 'Flotemysost and other products for personal use', 780.3264);
INSERT INTO "InternationalOrders" VALUES(10374, 'Gorgonzola Telino and other products for personal use', 209.7013);
INSERT INTO "InternationalOrders" VALUES(10376, 'Gorgonzola Telino and other products for personal use', 191.8835);
INSERT INTO "InternationalOrders" VALUES(10377, 'Rossle Sauerkraut and other products for personal use', 464.1754);
INSERT INTO "InternationalOrders" VALUES(10378, 'Flotemysost and other products for personal use', 47.1485);
INSERT INTO "InternationalOrders" VALUES(10379, 'Vegie-spread and other products for personal use', 438.2255);
INSERT INTO "InternationalOrders" VALUES(10380, 'Perth Pasties and other products for personal use', 648.6578);
INSERT INTO "InternationalOrders" VALUES(10381, 'Longlife Tofu and other products for personal use', 51.1689);
INSERT INTO "InternationalOrders" VALUES(10382, 'Thuringer Rostbratwurst and other products for personal use', 1324.9102);
INSERT INTO "InternationalOrders" VALUES(10383, 'Valkoinen suklaa and other products for personal use', 410.7222);
INSERT INTO "InternationalOrders" VALUES(10384, 'Sir Rodney''s Marmalade and other products for personal use', 1015.3381);
INSERT INTO "InternationalOrders" VALUES(10386, 'Sasquatch Ale and other products for personal use', 75.8397);
INSERT INTO "InternationalOrders" VALUES(10387, 'Rossle Sauerkraut and other products for personal use', 483.5465);
INSERT INTO "InternationalOrders" VALUES(10388, 'Rogede sild and other products for personal use', 582.0467);
INSERT INTO "InternationalOrders" VALUES(10389, 'Tarte au sucre and other products for personal use', 837.3432);
INSERT INTO "InternationalOrders" VALUES(10390, 'Steeleye Stout and other products for personal use', 1039.4606);
INSERT INTO "InternationalOrders" VALUES(10391, 'Konbu and other products for personal use', 39.4732);
INSERT INTO "InternationalOrders" VALUES(10392, 'Gudbrandsdalsost and other products for personal use', 657.8864);
INSERT INTO "InternationalOrders" VALUES(10395, 'Spegesild and other products for personal use', 1065.9588);
INSERT INTO "InternationalOrders" VALUES(10396, 'Tunnbrod and other products for personal use', 869.7807);
INSERT INTO "InternationalOrders" VALUES(10397, 'Sir Rodney''s Scones and other products for personal use', 385.2291);
INSERT INTO "InternationalOrders" VALUES(10399, 'Scottish Longbreads and other products for personal use', 806.6419);
INSERT INTO "InternationalOrders" VALUES(10400, 'Thuringer Rostbratwurst and other products for personal use', 1399.3793);
INSERT INTO "InternationalOrders" VALUES(10402, 'Vegie-spread and other products for personal use', 1239.7048);
INSERT INTO "InternationalOrders" VALUES(10403, 'Pavlova and other products for personal use', 459.5611);
INSERT INTO "InternationalOrders" VALUES(10404, 'Singaporean Hokkien Fried Mee and other products for personal use', 765.2498);
INSERT INTO "InternationalOrders" VALUES(10405, 'Aniseed Syrup and other products for personal use', 182.7462);
INSERT INTO "InternationalOrders" VALUES(10406, 'Sir Rodney''s Scones and other products for personal use', 922.0461);
INSERT INTO "InternationalOrders" VALUES(10407, 'Queso Cabrales and other products for personal use', 545.4975);
INSERT INTO "InternationalOrders" VALUES(10408, 'Tourtiere and other products for personal use', 741.2187);
INSERT INTO "InternationalOrders" VALUES(10409, 'Tofu and other products for personal use', 145.8315);
INSERT INTO "InternationalOrders" VALUES(10410, 'Raclette Courdavault and other products for personal use', 366.4062);
INSERT INTO "InternationalOrders" VALUES(10411, 'Raclette Courdavault and other products for personal use', 552.1221);
INSERT INTO "InternationalOrders" VALUES(10412, 'Tofu and other products for personal use', 169.9540);
INSERT INTO "InternationalOrders" VALUES(10413, 'Tarte au sucre and other products for personal use', 970.0170);
INSERT INTO "InternationalOrders" VALUES(10414, 'Teatime Chocolate Biscuits and other products for personal use', 105.7187);
INSERT INTO "InternationalOrders" VALUES(10416, 'Teatime Chocolate Biscuits and other products for personal use', 328.9432);
INSERT INTO "InternationalOrders" VALUES(10417, 'Spegesild and other products for personal use', 5154.9057);
INSERT INTO "InternationalOrders" VALUES(10418, 'Zaanse koeken and other products for personal use', 829.1197);
INSERT INTO "InternationalOrders" VALUES(10419, 'Gudbrandsdalsost and other products for personal use', 1008.7592);
INSERT INTO "InternationalOrders" VALUES(10420, 'Rod Kaviar and other products for personal use', 866.9481);
INSERT INTO "InternationalOrders" VALUES(10421, 'Teatime Chocolate Biscuits and other products for personal use', 581.6813);
INSERT INTO "InternationalOrders" VALUES(10422, 'Gumbar Gummibarchen and other products for personal use', 22.7519);
INSERT INTO "InternationalOrders" VALUES(10423, 'Raclette Courdavault and other products for personal use', 466.0029);
INSERT INTO "InternationalOrders" VALUES(10424, 'Steeleye Stout and other products for personal use', 5250.8475);
INSERT INTO "InternationalOrders" VALUES(10425, 'Pate chinois and other products for personal use', 219.2955);
INSERT INTO "InternationalOrders" VALUES(10426, 'Wimmers gute Semmelknodel and other products for personal use', 154.5119);
INSERT INTO "InternationalOrders" VALUES(10427, 'Tofu and other products for personal use', 297.4195);
INSERT INTO "InternationalOrders" VALUES(10428, 'Spegesild and other products for personal use', 87.7182);
INSERT INTO "InternationalOrders" VALUES(10429, 'Vegie-spread and other products for personal use', 798.8295);
INSERT INTO "InternationalOrders" VALUES(10430, 'Sir Rodney''s Scones and other products for personal use', 2647.9929);
INSERT INTO "InternationalOrders" VALUES(10431, 'Zaanse koeken and other products for personal use', 1152.6719);
INSERT INTO "InternationalOrders" VALUES(10433, 'Gnocchi di nonna Alice and other products for personal use', 388.8840);
INSERT INTO "InternationalOrders" VALUES(10434, 'Queso Cabrales and other products for personal use', 164.4716);
INSERT INTO "InternationalOrders" VALUES(10435, 'Mozzarella di Giovanni and other products for personal use', 288.5563);
INSERT INTO "InternationalOrders" VALUES(10436, 'Wimmers gute Semmelknodel and other products for personal use', 1010.0384);
INSERT INTO "InternationalOrders" VALUES(10437, 'Perth Pasties and other products for personal use', 179.5482);
INSERT INTO "InternationalOrders" VALUES(10438, 'Teatime Chocolate Biscuits and other products for personal use', 259.2712);
INSERT INTO "InternationalOrders" VALUES(10439, 'Wimmers gute Semmelknodel and other products for personal use', 492.5011);
INSERT INTO "InternationalOrders" VALUES(10442, 'Tourtiere and other products for personal use', 818.7031);
INSERT INTO "InternationalOrders" VALUES(10443, 'Rossle Sauerkraut and other products for personal use', 245.6109);
INSERT INTO "InternationalOrders" VALUES(10444, 'Steeleye Stout and other products for personal use', 471.3482);
INSERT INTO "InternationalOrders" VALUES(10445, 'Tourtiere and other products for personal use', 79.9058);
INSERT INTO "InternationalOrders" VALUES(10446, 'Teatime Chocolate Biscuits and other products for personal use', 124.9984);
INSERT INTO "InternationalOrders" VALUES(10447, 'Teatime Chocolate Biscuits and other products for personal use', 417.7579);
INSERT INTO "InternationalOrders" VALUES(10448, 'Gumbar Gummibarchen and other products for personal use', 202.5742);
INSERT INTO "InternationalOrders" VALUES(10449, 'Tarte au sucre and other products for personal use', 839.8103);
INSERT INTO "InternationalOrders" VALUES(10450, 'Tourtiere and other products for personal use', 242.7784);
INSERT INTO "InternationalOrders" VALUES(10451, 'Wimmers gute Semmelknodel and other products for personal use', 1954.1968);
INSERT INTO "InternationalOrders" VALUES(10453, 'Outback Lager and other products for personal use', 206.9601);
INSERT INTO "InternationalOrders" VALUES(10454, 'Spegesild and other products for personal use', 189.1423);
INSERT INTO "InternationalOrders" VALUES(10455, 'Sirop d''erable and other products for personal use', 1226.2272);
INSERT INTO "InternationalOrders" VALUES(10456, 'Sir Rodney''s Scones and other products for personal use', 299.7038);
INSERT INTO "InternationalOrders" VALUES(10457, 'Raclette Courdavault and other products for personal use', 723.6751);
INSERT INTO "InternationalOrders" VALUES(10458, 'Rossle Sauerkraut and other products for personal use', 1777.6640);
INSERT INTO "InternationalOrders" VALUES(10459, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 771.1891);
INSERT INTO "InternationalOrders" VALUES(10460, 'Scottish Longbreads and other products for personal use', 107.2720);
INSERT INTO "InternationalOrders" VALUES(10461, 'Sir Rodney''s Scones and other products for personal use', 937.3054);
INSERT INTO "InternationalOrders" VALUES(10462, 'Tunnbrod and other products for personal use', 71.2710);
INSERT INTO "InternationalOrders" VALUES(10463, 'Teatime Chocolate Biscuits and other products for personal use', 325.8822);
INSERT INTO "InternationalOrders" VALUES(10464, 'Ipoh Coffee and other products for personal use', 844.2876);
INSERT INTO "InternationalOrders" VALUES(10465, 'Valkoinen suklaa and other products for personal use', 1242.2175);
INSERT INTO "InternationalOrders" VALUES(10466, 'Spegesild and other products for personal use', 98.6830);
INSERT INTO "InternationalOrders" VALUES(10467, 'NuNuCa Nu?-Nougat-Creme and other products for personal use', 107.4548);
INSERT INTO "InternationalOrders" VALUES(10468, 'Nord-Ost Matjeshering and other products for personal use', 327.8467);
INSERT INTO "InternationalOrders" VALUES(10470, 'Wimmers gute Semmelknodel and other products for personal use', 831.8608);
INSERT INTO "InternationalOrders" VALUES(10471, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 606.7175);
INSERT INTO "InternationalOrders" VALUES(10472, 'Manjimup Dried Apples and other products for personal use', 480.2571);
INSERT INTO "InternationalOrders" VALUES(10473, 'Geitost and other products for personal use', 105.2618);
INSERT INTO "InternationalOrders" VALUES(10474, 'Tofu and other products for personal use', 570.6708);
INSERT INTO "InternationalOrders" VALUES(10475, 'Louisiana Hot Spiced Okra and other products for personal use', 809.0176);
INSERT INTO "InternationalOrders" VALUES(10476, 'Pate chinois and other products for personal use', 83.3323);
INSERT INTO "InternationalOrders" VALUES(10477, 'Sir Rodney''s Scones and other products for personal use', 307.0137);
INSERT INTO "InternationalOrders" VALUES(10478, 'Ikura and other products for personal use', 226.6053);
INSERT INTO "InternationalOrders" VALUES(10480, 'Zaanse koeken and other products for personal use', 345.3904);
INSERT INTO "InternationalOrders" VALUES(10481, 'Maxilaku and other products for personal use', 672.5061);
INSERT INTO "InternationalOrders" VALUES(10484, 'Sir Rodney''s Scones and other products for personal use', 176.4415);
INSERT INTO "InternationalOrders" VALUES(10485, 'Pate chinois and other products for personal use', 804.0834);
INSERT INTO "InternationalOrders" VALUES(10486, 'Queso Cabrales and other products for personal use', 581.1330);
INSERT INTO "InternationalOrders" VALUES(10487, 'Tourtiere and other products for personal use', 422.6463);
INSERT INTO "InternationalOrders" VALUES(10488, 'Rod Kaviar and other products for personal use', 712.7103);
INSERT INTO "InternationalOrders" VALUES(10489, 'Queso Cabrales and other products for personal use', 229.4379);
INSERT INTO "InternationalOrders" VALUES(10490, 'Scottish Longbreads and other products for personal use', 1445.1572);
INSERT INTO "InternationalOrders" VALUES(10491, 'Original Frankfurter grune So?e and other products for personal use', 139.4811);
INSERT INTO "InternationalOrders" VALUES(10492, 'Singaporean Hokkien Fried Mee and other products for personal use', 409.3516);
INSERT INTO "InternationalOrders" VALUES(10493, 'Louisiana Hot Spiced Okra and other products for personal use', 308.8411);
INSERT INTO "InternationalOrders" VALUES(10494, 'Gnocchi di nonna Alice and other products for personal use', 416.6614);
INSERT INTO "InternationalOrders" VALUES(10495, 'Tunnbrod and other products for personal use', 127.0086);
INSERT INTO "InternationalOrders" VALUES(10496, 'Gorgonzola Telino and other products for personal use', 91.3731);
INSERT INTO "InternationalOrders" VALUES(10497, 'Original Frankfurter grune So?e and other products for personal use', 630.7486);
INSERT INTO "InternationalOrders" VALUES(10498, 'Singaporean Hokkien Fried Mee and other products for personal use', 262.6977);
INSERT INTO "InternationalOrders" VALUES(10499, 'Rossle Sauerkraut and other products for personal use', 645.0942);
INSERT INTO "InternationalOrders" VALUES(10500, 'Rossle Sauerkraut and other products for personal use', 251.6416);
INSERT INTO "InternationalOrders" VALUES(10501, 'Tourtiere and other products for personal use', 68.0730);
INSERT INTO "InternationalOrders" VALUES(10502, 'Rogede sild and other products for personal use', 372.9394);
INSERT INTO "InternationalOrders" VALUES(10503, 'Tofu and other products for personal use', 935.8891);
INSERT INTO "InternationalOrders" VALUES(10505, 'Tarte au sucre and other products for personal use', 67.5704);
INSERT INTO "InternationalOrders" VALUES(10506, 'Outback Lager and other products for personal use', 211.0719);
INSERT INTO "InternationalOrders" VALUES(10507, 'Ipoh Coffee and other products for personal use', 402.6128);
INSERT INTO "InternationalOrders" VALUES(10508, 'Konbu and other products for personal use', 109.6477);
INSERT INTO "InternationalOrders" VALUES(10509, 'Rossle Sauerkraut and other products for personal use', 62.4992);
INSERT INTO "InternationalOrders" VALUES(10511, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 1370.5967);
INSERT INTO "InternationalOrders" VALUES(10512, 'Zaanse koeken and other products for personal use', 282.3429);
INSERT INTO "InternationalOrders" VALUES(10513, 'Sirop d''erable and other products for personal use', 1109.0412);
INSERT INTO "InternationalOrders" VALUES(10514, 'Sir Rodney''s Marmalade and other products for personal use', 3939.7575);
INSERT INTO "InternationalOrders" VALUES(10515, 'Schoggi Schokolade and other products for personal use', 4837.5212);
INSERT INTO "InternationalOrders" VALUES(10516, 'Singaporean Hokkien Fried Mee and other products for personal use', 1194.4751);
INSERT INTO "InternationalOrders" VALUES(10517, 'Raclette Courdavault and other products for personal use', 160.8167);
INSERT INTO "InternationalOrders" VALUES(10518, 'Gula Malacca and other products for personal use', 1896.0150);
INSERT INTO "InternationalOrders" VALUES(10519, 'Ikura and other products for personal use', 1076.3753);
INSERT INTO "InternationalOrders" VALUES(10520, 'Perth Pasties and other products for personal use', 91.3731);
INSERT INTO "InternationalOrders" VALUES(10521, 'Steeleye Stout and other products for personal use', 103.0232);
INSERT INTO "InternationalOrders" VALUES(10522, 'Northwoods Cranberry Sauce and other products for personal use', 1214.2573);
INSERT INTO "InternationalOrders" VALUES(10523, 'Sir Rodney''s Marmalade and other products for personal use', 1240.8012);
INSERT INTO "InternationalOrders" VALUES(10524, 'Tourtiere and other products for personal use', 1458.6119);
INSERT INTO "InternationalOrders" VALUES(10525, 'Inlagd Sill and other products for personal use', 386.5083);
INSERT INTO "InternationalOrders" VALUES(10526, 'Konbu and other products for personal use', 614.0273);
INSERT INTO "InternationalOrders" VALUES(10527, 'Inlagd Sill and other products for personal use', 762.9655);
INSERT INTO "InternationalOrders" VALUES(10529, 'Scottish Longbreads and other products for personal use', 432.1948);
INSERT INTO "InternationalOrders" VALUES(10530, 'Sirop d''erable and other products for personal use', 1909.6981);
INSERT INTO "InternationalOrders" VALUES(10531, 'Raclette Courdavault and other products for personal use', 50.2552);
INSERT INTO "InternationalOrders" VALUES(10532, 'Nord-Ost Matjeshering and other products for personal use', 363.8249);
INSERT INTO "InternationalOrders" VALUES(10533, 'Rod Kaviar and other products for personal use', 1048.5979);
INSERT INTO "InternationalOrders" VALUES(10534, 'Tourtiere and other products for personal use', 236.3823);
INSERT INTO "InternationalOrders" VALUES(10535, 'Ravioli Angelo and other products for personal use', 985.2306);
INSERT INTO "InternationalOrders" VALUES(10536, 'Queso Manchego La Pastora and other products for personal use', 952.5647);
INSERT INTO "InternationalOrders" VALUES(10537, 'Rod Kaviar and other products for personal use', 833.2314);
INSERT INTO "InternationalOrders" VALUES(10538, 'Outback Lager and other products for personal use', 63.8698);
INSERT INTO "InternationalOrders" VALUES(10539, 'Sir Rodney''s Scones and other products for personal use', 162.4157);
INSERT INTO "InternationalOrders" VALUES(10540, 'Scottish Longbreads and other products for personal use', 4656.2369);
INSERT INTO "InternationalOrders" VALUES(10541, 'Louisiana Fiery Hot Pepper Sauce and other products for personal use', 988.1089);
INSERT INTO "InternationalOrders" VALUES(10542, 'Tourtiere and other products for personal use', 225.6002);
INSERT INTO "InternationalOrders" VALUES(10543, 'Tunnbrod and other products for personal use', 808.6521);
INSERT INTO "InternationalOrders" VALUES(10546, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 1284.7060);
INSERT INTO "InternationalOrders" VALUES(10547, 'Mascarpone Fabioli and other products for personal use', 871.6995);
INSERT INTO "InternationalOrders" VALUES(10548, 'Sasquatch Ale and other products for personal use', 125.6837);
INSERT INTO "InternationalOrders" VALUES(10549, 'Rogede sild and other products for personal use', 1910.3834);
INSERT INTO "InternationalOrders" VALUES(10550, 'Teatime Chocolate Biscuits and other products for personal use', 342.1923);
INSERT INTO "InternationalOrders" VALUES(10551, 'Steeleye Stout and other products for personal use', 838.8052);
INSERT INTO "InternationalOrders" VALUES(10552, 'Rhonbrau Klosterbier and other products for personal use', 402.2701);
INSERT INTO "InternationalOrders" VALUES(10553, 'Steeleye Stout and other products for personal use', 706.4512);
INSERT INTO "InternationalOrders" VALUES(10554, 'Tunnbrod and other products for personal use', 831.2669);
INSERT INTO "InternationalOrders" VALUES(10556, 'Mozzarella di Giovanni and other products for personal use', 381.5741);
INSERT INTO "InternationalOrders" VALUES(10557, 'Wimmers gute Semmelknodel and other products for personal use', 526.5376);
INSERT INTO "InternationalOrders" VALUES(10558, 'Zaanse koeken and other products for personal use', 979.0173);
INSERT INTO "InternationalOrders" VALUES(10559, 'Pate chinois and other products for personal use', 250.2710);
INSERT INTO "InternationalOrders" VALUES(10560, 'Tarte au sucre and other products for personal use', 574.4171);
INSERT INTO "InternationalOrders" VALUES(10561, 'Manjimup Dried Apples and other products for personal use', 1299.5541);
INSERT INTO "InternationalOrders" VALUES(10562, 'Tarte au sucre and other products for personal use', 248.0780);
INSERT INTO "InternationalOrders" VALUES(10563, 'Inlagd Sill and other products for personal use', 440.8753);
INSERT INTO "InternationalOrders" VALUES(10565, 'Wimmers gute Semmelknodel and other products for personal use', 324.8314);
INSERT INTO "InternationalOrders" VALUES(10566, 'Queso Cabrales and other products for personal use', 932.0058);
INSERT INTO "InternationalOrders" VALUES(10567, 'Raclette Courdavault and other products for personal use', 1420.3951);
INSERT INTO "InternationalOrders" VALUES(10568, 'Ikura and other products for personal use', 70.8142);
INSERT INTO "InternationalOrders" VALUES(10570, 'Queso Cabrales and other products for personal use', 1185.5662);
INSERT INTO "InternationalOrders" VALUES(10571, 'Tofu and other products for personal use', 295.9347);
INSERT INTO "InternationalOrders" VALUES(10572, 'Rhonbrau Klosterbier and other products for personal use', 715.2916);
INSERT INTO "InternationalOrders" VALUES(10573, 'Sasquatch Ale and other products for personal use', 951.1941);
INSERT INTO "InternationalOrders" VALUES(10575, 'Vegie-spread and other products for personal use', 981.0731);
INSERT INTO "InternationalOrders" VALUES(10576, 'Gula Malacca and other products for personal use', 383.0589);
INSERT INTO "InternationalOrders" VALUES(10578, 'Steeleye Stout and other products for personal use', 217.9249);
INSERT INTO "InternationalOrders" VALUES(10580, 'Tofu and other products for personal use', 487.5213);
INSERT INTO "InternationalOrders" VALUES(10581, 'Rhonbrau Klosterbier and other products for personal use', 177.0354);
INSERT INTO "InternationalOrders" VALUES(10582, 'Ravioli Angelo and other products for personal use', 150.7656);
INSERT INTO "InternationalOrders" VALUES(10583, 'Thuringer Rostbratwurst and other products for personal use', 1102.8278);
INSERT INTO "InternationalOrders" VALUES(10584, 'Gorgonzola Telino and other products for personal use', 285.5410);
INSERT INTO "InternationalOrders" VALUES(10585, 'Zaanse koeken and other products for personal use', 65.1033);
INSERT INTO "InternationalOrders" VALUES(10586, 'Filo Mix and other products for personal use', 12.7922);
INSERT INTO "InternationalOrders" VALUES(10587, 'Steeleye Stout and other products for personal use', 368.8641);
INSERT INTO "InternationalOrders" VALUES(10588, 'Singaporean Hokkien Fried Mee and other products for personal use', 1781.7758);
INSERT INTO "InternationalOrders" VALUES(10590, 'Original Frankfurter grune So?e and other products for personal use', 520.8268);
INSERT INTO "InternationalOrders" VALUES(10591, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 371.2033);
INSERT INTO "InternationalOrders" VALUES(10592, 'Gumbar Gummibarchen and other products for personal use', 248.3750);
INSERT INTO "InternationalOrders" VALUES(10593, 'Sir Rodney''s Marmalade and other products for personal use', 1138.9659);
INSERT INTO "InternationalOrders" VALUES(10595, 'Steeleye Stout and other products for personal use', 2878.2532);
INSERT INTO "InternationalOrders" VALUES(10597, 'Ravioli Angelo and other products for personal use', 365.5382);
INSERT INTO "InternationalOrders" VALUES(10599, 'Tarte au sucre and other products for personal use', 225.2347);
INSERT INTO "InternationalOrders" VALUES(10601, 'Raclette Courdavault and other products for personal use', 1043.9378);
INSERT INTO "InternationalOrders" VALUES(10602, 'Original Frankfurter grune So?e and other products for personal use', 29.6963);
INSERT INTO "InternationalOrders" VALUES(10604, 'Lakkalikoori and other products for personal use', 117.1860);
INSERT INTO "InternationalOrders" VALUES(10605, 'Raclette Courdavault and other products for personal use', 1976.4005);
INSERT INTO "InternationalOrders" VALUES(10606, 'Tarte au sucre and other products for personal use', 645.5511);
INSERT INTO "InternationalOrders" VALUES(10608, 'Gnocchi di nonna Alice and other products for personal use', 486.1050);
INSERT INTO "InternationalOrders" VALUES(10609, 'Sir Rodney''s Scones and other products for personal use', 193.7110);
INSERT INTO "InternationalOrders" VALUES(10610, 'Inlagd Sill and other products for personal use', 182.2894);
INSERT INTO "InternationalOrders" VALUES(10611, 'Chang and other products for personal use', 369.1474);
INSERT INTO "InternationalOrders" VALUES(10613, 'Rhonbrau Klosterbier and other products for personal use', 163.5579);
INSERT INTO "InternationalOrders" VALUES(10614, 'Sir Rodney''s Scones and other products for personal use', 211.9856);
INSERT INTO "InternationalOrders" VALUES(10615, 'Pate chinois and other products for personal use', 54.8239);
INSERT INTO "InternationalOrders" VALUES(10618, 'Scottish Longbreads and other products for personal use', 1232.3949);
INSERT INTO "InternationalOrders" VALUES(10619, 'Sir Rodney''s Scones and other products for personal use', 575.6506);
INSERT INTO "InternationalOrders" VALUES(10620, 'Guarana Fantastica and other products for personal use', 26.2698);
INSERT INTO "InternationalOrders" VALUES(10621, 'Tunnbrod and other products for personal use', 346.5325);
INSERT INTO "InternationalOrders" VALUES(10622, 'Scottish Longbreads and other products for personal use', 276.4037);
INSERT INTO "InternationalOrders" VALUES(10623, 'Tofu and other products for personal use', 653.2036);
INSERT INTO "InternationalOrders" VALUES(10625, 'Tofu and other products for personal use', 219.1813);
INSERT INTO "InternationalOrders" VALUES(10626, 'Perth Pasties and other products for personal use', 686.9431);
INSERT INTO "InternationalOrders" VALUES(10628, 'Chai and other products for personal use', 205.5895);
INSERT INTO "InternationalOrders" VALUES(10629, 'Wimmers gute Semmelknodel and other products for personal use', 1267.8248);
INSERT INTO "InternationalOrders" VALUES(10630, 'Pate chinois and other products for personal use', 419.4026);
INSERT INTO "InternationalOrders" VALUES(10631, 'Rhonbrau Klosterbier and other products for personal use', 28.3257);
INSERT INTO "InternationalOrders" VALUES(10632, 'Geitost and other products for personal use', 283.2567);
INSERT INTO "InternationalOrders" VALUES(10633, 'Tarte au sucre and other products for personal use', 2961.8824);
INSERT INTO "InternationalOrders" VALUES(10634, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 2277.7033);
INSERT INTO "InternationalOrders" VALUES(10635, 'Gustaf''s Knackebrod and other products for personal use', 630.5887);
INSERT INTO "InternationalOrders" VALUES(10636, 'Escargots de Bourgogne and other products for personal use', 287.5969);
INSERT INTO "InternationalOrders" VALUES(10637, 'Valkoinen suklaa and other products for personal use', 1323.1969);
INSERT INTO "InternationalOrders" VALUES(10638, 'Rogede sild and other products for personal use', 1242.6972);
INSERT INTO "InternationalOrders" VALUES(10639, 'Carnarvon Tigers and other products for personal use', 228.4328);
INSERT INTO "InternationalOrders" VALUES(10640, 'Outback Lager and other products for personal use', 431.7380);
INSERT INTO "InternationalOrders" VALUES(10641, 'Chang and other products for personal use', 938.4019);
INSERT INTO "InternationalOrders" VALUES(10642, 'Sirop d''rable and other products for personal use', 397.4731);
INSERT INTO "InternationalOrders" VALUES(10643, 'Spegesild and other products for personal use', 496.1560);
INSERT INTO "InternationalOrders" VALUES(10644, 'Spegesild and other products for personal use', 649.6629);
INSERT INTO "InternationalOrders" VALUES(10645, 'Inlagd Sill and other products for personal use', 701.2887);
INSERT INTO "InternationalOrders" VALUES(10646, 'Original Frankfurter grune So?e and other products for personal use', 880.8368);
INSERT INTO "InternationalOrders" VALUES(10647, 'Teatime Chocolate Biscuits and other products for personal use', 290.5665);
INSERT INTO "InternationalOrders" VALUES(10648, 'Gustaf''s Knackebrod and other products for personal use', 174.7511);
INSERT INTO "InternationalOrders" VALUES(10649, 'Rossle Sauerkraut and other products for personal use', 655.1452);
INSERT INTO "InternationalOrders" VALUES(10650, 'Tourtiere and other products for personal use', 831.5867);
INSERT INTO "InternationalOrders" VALUES(10651, 'Teatime Chocolate Biscuits and other products for personal use', 242.3215);
INSERT INTO "InternationalOrders" VALUES(10652, 'Singaporean Hokkien Fried Mee and other products for personal use', 151.5789);
INSERT INTO "InternationalOrders" VALUES(10653, 'Pavlova and other products for personal use', 549.8377);
INSERT INTO "InternationalOrders" VALUES(10654, 'Tourtiere and other products for personal use', 305.5060);
INSERT INTO "InternationalOrders" VALUES(10655, 'Jack''s New England Clam Chowder and other products for personal use', 88.1751);
INSERT INTO "InternationalOrders" VALUES(10658, 'Sir Rodney''s Scones and other products for personal use', 2132.6485);
INSERT INTO "InternationalOrders" VALUES(10659, 'Outback Lager and other products for personal use', 590.0876);
INSERT INTO "InternationalOrders" VALUES(10661, 'Escargots de Bourgogne and other products for personal use', 321.2907);
INSERT INTO "InternationalOrders" VALUES(10663, 'Singaporean Hokkien Fried Mee and other products for personal use', 928.3509);
INSERT INTO "InternationalOrders" VALUES(10664, 'Louisiana Fiery Hot Pepper Sauce and other products for personal use', 692.4940);
INSERT INTO "InternationalOrders" VALUES(10666, 'Thuringer Rostbratwurst and other products for personal use', 2132.1642);
INSERT INTO "InternationalOrders" VALUES(10667, 'Gudbrandsdalsost and other products for personal use', 877.6388);
INSERT INTO "InternationalOrders" VALUES(10668, 'Wimmers gute Semmelknodel and other products for personal use', 317.4074);
INSERT INTO "InternationalOrders" VALUES(10669, 'Inlagd Sill and other products for personal use', 260.4134);
INSERT INTO "InternationalOrders" VALUES(10670, 'Tunnbrod and other products for personal use', 1051.5903);
INSERT INTO "InternationalOrders" VALUES(10671, 'Tarte au sucre and other products for personal use', 420.3620);
INSERT INTO "InternationalOrders" VALUES(10672, 'Flotemysost and other products for personal use', 1923.6325);
INSERT INTO "InternationalOrders" VALUES(10673, 'Singaporean Hokkien Fried Mee and other products for personal use', 188.3885);
INSERT INTO "InternationalOrders" VALUES(10674, 'Tunnbrod and other products for personal use', 20.5590);
INSERT INTO "InternationalOrders" VALUES(10675, 'Tofu and other products for personal use', 650.1197);
INSERT INTO "InternationalOrders" VALUES(10676, 'Teatime Chocolate Biscuits and other products for personal use', 244.3546);
INSERT INTO "InternationalOrders" VALUES(10677, 'Gumbar Gummibarchen and other products for personal use', 437.1747);
INSERT INTO "InternationalOrders" VALUES(10679, 'Raclette Courdavault and other products for personal use', 301.5313);
INSERT INTO "InternationalOrders" VALUES(10682, 'Rhonbrau Klosterbier and other products for personal use', 171.5530);
INSERT INTO "InternationalOrders" VALUES(10683, 'Filo Mix and other products for personal use', 28.7825);
INSERT INTO "InternationalOrders" VALUES(10684, 'Zaanse koeken and other products for personal use', 807.7383);
INSERT INTO "InternationalOrders" VALUES(10685, 'Zaanse koeken and other products for personal use', 365.9950);
INSERT INTO "InternationalOrders" VALUES(10686, 'Gumbar Gummibarchen and other products for personal use', 748.5514);
INSERT INTO "InternationalOrders" VALUES(10687, 'Thuringer Rostbratwurst and other products for personal use', 2833.4346);
INSERT INTO "InternationalOrders" VALUES(10688, 'Sasquatch Ale and other products for personal use', 1594.4609);
INSERT INTO "InternationalOrders" VALUES(10689, 'Chai and other products for personal use', 287.8253);
INSERT INTO "InternationalOrders" VALUES(10690, 'Original Frankfurter grune So?e and other products for personal use', 525.3954);
INSERT INTO "InternationalOrders" VALUES(10691, 'Thuringer Rostbratwurst and other products for personal use', 4643.9472);
INSERT INTO "InternationalOrders" VALUES(10692, 'Vegie-spread and other products for personal use', 401.1280);
INSERT INTO "InternationalOrders" VALUES(10694, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 2204.3764);
INSERT INTO "InternationalOrders" VALUES(10695, 'Queso Manchego La Pastora and other products for personal use', 293.3077);
INSERT INTO "InternationalOrders" VALUES(10697, 'Teatime Chocolate Biscuits and other products for personal use', 490.6279);
INSERT INTO "InternationalOrders" VALUES(10698, 'Thuringer Rostbratwurst and other products for personal use', 1645.0496);
INSERT INTO "InternationalOrders" VALUES(10699, 'Zaanse koeken and other products for personal use', 52.0827);
INSERT INTO "InternationalOrders" VALUES(10701, 'Raclette Courdavault and other products for personal use', 1539.6370);
INSERT INTO "InternationalOrders" VALUES(10702, 'Lakkalikoori and other products for personal use', 150.7656);
INSERT INTO "InternationalOrders" VALUES(10703, 'Rod Kaviar and other products for personal use', 1162.7229);
INSERT INTO "InternationalOrders" VALUES(10704, 'Guarana Fantastica and other products for personal use', 272.0635);
INSERT INTO "InternationalOrders" VALUES(10705, 'Mascarpone Fabioli and other products for personal use', 172.6952);
INSERT INTO "InternationalOrders" VALUES(10707, 'Ravioli Angelo and other products for personal use', 778.4989);
INSERT INTO "InternationalOrders" VALUES(10709, 'Northwoods Cranberry Sauce and other products for personal use', 1564.3077);
INSERT INTO "InternationalOrders" VALUES(10710, 'Zaanse koeken and other products for personal use', 42.7169);
INSERT INTO "InternationalOrders" VALUES(10712, 'Perth Pasties and other products for personal use', 565.7823);
INSERT INTO "InternationalOrders" VALUES(10715, 'Ikura and other products for personal use', 592.0978);
INSERT INTO "InternationalOrders" VALUES(10716, 'Sirop d''erable and other products for personal use', 322.5471);
INSERT INTO "InternationalOrders" VALUES(10717, 'Tourtiere and other products for personal use', 608.4307);
INSERT INTO "InternationalOrders" VALUES(10718, 'Tarte au sucre and other products for personal use', 1582.1255);
INSERT INTO "InternationalOrders" VALUES(10720, 'Steeleye Stout and other products for personal use', 251.2761);
INSERT INTO "InternationalOrders" VALUES(10721, 'Gula Malacca and other products for personal use', 444.3018);
INSERT INTO "InternationalOrders" VALUES(10724, 'Sirop d''erable and other products for personal use', 291.7087);
INSERT INTO "InternationalOrders" VALUES(10725, 'Pate chinois and other products for personal use', 131.4859);
INSERT INTO "InternationalOrders" VALUES(10726, 'Queso Cabrales and other products for personal use', 299.2470);
INSERT INTO "InternationalOrders" VALUES(10727, 'Raclette Courdavault and other products for personal use', 781.2401);
INSERT INTO "InternationalOrders" VALUES(10728, 'Pate chinois and other products for personal use', 592.4404);
INSERT INTO "InternationalOrders" VALUES(10729, 'Valkoinen suklaa and other products for personal use', 845.2013);
INSERT INTO "InternationalOrders" VALUES(10730, 'Pavlova and other products for personal use', 232.8872);
INSERT INTO "InternationalOrders" VALUES(10731, 'Sir Rodney''s Scones and other products for personal use', 909.1625);
INSERT INTO "InternationalOrders" VALUES(10732, 'Lakkalikoori and other products for personal use', 164.4716);
INSERT INTO "InternationalOrders" VALUES(10733, 'Tofu and other products for personal use', 666.5669);
INSERT INTO "InternationalOrders" VALUES(10734, 'Nord-Ost Matjeshering and other products for personal use', 684.5445);
INSERT INTO "InternationalOrders" VALUES(10736, 'Rhonbrau Klosterbier and other products for personal use', 455.4950);
INSERT INTO "InternationalOrders" VALUES(10737, 'Konbu and other products for personal use', 63.8698);
INSERT INTO "InternationalOrders" VALUES(10738, 'Pavlova and other products for personal use', 23.9169);
INSERT INTO "InternationalOrders" VALUES(10739, 'Inlagd Sill and other products for personal use', 109.6477);
INSERT INTO "InternationalOrders" VALUES(10741, 'Chang and other products for personal use', 130.2067);
INSERT INTO "InternationalOrders" VALUES(10742, 'Mozzarella di Giovanni and other products for personal use', 1424.5069);
INSERT INTO "InternationalOrders" VALUES(10743, 'Spegesild and other products for personal use', 153.5068);
INSERT INTO "InternationalOrders" VALUES(10744, 'Boston Crab Meat and other products for personal use', 420.3163);
INSERT INTO "InternationalOrders" VALUES(10745, 'Raclette Courdavault and other products for personal use', 2069.5097);
INSERT INTO "InternationalOrders" VALUES(10746, 'Tarte au sucre and other products for personal use', 1056.1362);
INSERT INTO "InternationalOrders" VALUES(10747, 'Vegie-spread and other products for personal use', 873.9153);
INSERT INTO "InternationalOrders" VALUES(10749, 'Raclette Courdavault and other products for personal use', 493.4148);
INSERT INTO "InternationalOrders" VALUES(10750, 'Tofu and other products for personal use', 854.9097);
INSERT INTO "InternationalOrders" VALUES(10751, 'Valkoinen suklaa and other products for personal use', 777.3385);
INSERT INTO "InternationalOrders" VALUES(10752, 'Gudbrandsdalsost and other products for personal use', 115.1301);
INSERT INTO "InternationalOrders" VALUES(10753, 'Rogede sild and other products for personal use', 40.2042);
INSERT INTO "InternationalOrders" VALUES(10754, 'Boston Crab Meat and other products for personal use', 25.2190);
INSERT INTO "InternationalOrders" VALUES(10755, 'Zaanse koeken and other products for personal use', 1186.9368);
INSERT INTO "InternationalOrders" VALUES(10758, 'Outback Lager and other products for personal use', 751.3611);
INSERT INTO "InternationalOrders" VALUES(10759, 'Mascarpone Fabioli and other products for personal use', 146.1970);
INSERT INTO "InternationalOrders" VALUES(10760, 'Schoggi Schokolade and other products for personal use', 1509.4839);
INSERT INTO "InternationalOrders" VALUES(10762, 'Zaanse koeken and other products for personal use', 1981.4260);
INSERT INTO "InternationalOrders" VALUES(10763, 'Sir Rodney''s Scones and other products for personal use', 281.4292);
INSERT INTO "InternationalOrders" VALUES(10764, 'Chartreuse verte and other products for personal use', 1160.4386);
INSERT INTO "InternationalOrders" VALUES(10765, 'Louisiana Fiery Hot Pepper Sauce and other products for personal use', 769.3616);
INSERT INTO "InternationalOrders" VALUES(10766, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 1055.3595);
INSERT INTO "InternationalOrders" VALUES(10767, 'Singaporean Hokkien Fried Mee and other products for personal use', 12.7922);
INSERT INTO "InternationalOrders" VALUES(10768, 'Gustaf''s Knackebrod and other products for personal use', 674.7905);
INSERT INTO "InternationalOrders" VALUES(10769, 'Tarte au sucre and other products for personal use', 778.4989);
INSERT INTO "InternationalOrders" VALUES(10770, 'Queso Cabrales and other products for personal use', 143.9127);
INSERT INTO "InternationalOrders" VALUES(10771, 'Flotemysost and other products for personal use', 157.1618);
INSERT INTO "InternationalOrders" VALUES(10772, 'Thuringer Rostbratwurst and other products for personal use', 1646.1872);
INSERT INTO "InternationalOrders" VALUES(10773, 'Rhonbrau Klosterbier and other products for personal use', 1012.5283);
INSERT INTO "InternationalOrders" VALUES(10774, 'Louisiana Hot Spiced Okra and other products for personal use', 399.7574);
INSERT INTO "InternationalOrders" VALUES(10776, 'Singaporean Hokkien Fried Mee and other products for personal use', 3190.9776);
INSERT INTO "InternationalOrders" VALUES(10777, 'Singaporean Hokkien Fried Mee and other products for personal use', 127.9224);
INSERT INTO "InternationalOrders" VALUES(10778, 'Jack''s New England Clam Chowder and other products for personal use', 44.0875);
INSERT INTO "InternationalOrders" VALUES(10779, 'Tarte au sucre and other products for personal use', 609.9155);
INSERT INTO "InternationalOrders" VALUES(10780, 'Outback Lager and other products for personal use', 328.9432);
INSERT INTO "InternationalOrders" VALUES(10781, 'Tourtiere and other products for personal use', 517.3317);
INSERT INTO "InternationalOrders" VALUES(10782, 'Gorgonzola Telino and other products for personal use', 5.7108);
INSERT INTO "InternationalOrders" VALUES(10783, 'Gorgonzola Telino and other products for personal use', 659.0286);
INSERT INTO "InternationalOrders" VALUES(10784, 'Mozzarella di Giovanni and other products for personal use', 753.8282);
INSERT INTO "InternationalOrders" VALUES(10785, 'Rhonbrau Klosterbier and other products for personal use', 177.0354);
INSERT INTO "InternationalOrders" VALUES(10786, 'Rhonbrau Klosterbier and other products for personal use', 874.3722);
INSERT INTO "InternationalOrders" VALUES(10787, 'Thuringer Rostbratwurst and other products for personal use', 1261.3145);
INSERT INTO "InternationalOrders" VALUES(10788, 'Teatime Chocolate Biscuits and other products for personal use', 351.7865);
INSERT INTO "InternationalOrders" VALUES(10789, 'Vegie-spread and other products for personal use', 1684.4634);
INSERT INTO "InternationalOrders" VALUES(10790, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 388.3357);
INSERT INTO "InternationalOrders" VALUES(10791, 'Thuringer Rostbratwurst and other products for personal use', 879.9505);
INSERT INTO "InternationalOrders" VALUES(10792, 'Tourtiere and other products for personal use', 182.6777);
INSERT INTO "InternationalOrders" VALUES(10793, 'Jack''s New England Clam Chowder and other products for personal use', 87.3070);
INSERT INTO "InternationalOrders" VALUES(10794, 'Tourtiere and other products for personal use', 179.7538);
INSERT INTO "InternationalOrders" VALUES(10795, 'Pavlova and other products for personal use', 1141.8213);
INSERT INTO "InternationalOrders" VALUES(10796, 'Wimmers gute Semmelknodel and other products for personal use', 1314.8957);
INSERT INTO "InternationalOrders" VALUES(10797, 'Queso Cabrales and other products for personal use', 191.8835);
INSERT INTO "InternationalOrders" VALUES(10798, 'Tarte au sucre and other products for personal use', 204.0362);
INSERT INTO "InternationalOrders" VALUES(10799, 'Raclette Courdavault and other products for personal use', 724.1319);
INSERT INTO "InternationalOrders" VALUES(10800, 'Tourtiere and other products for personal use', 745.6732);
INSERT INTO "InternationalOrders" VALUES(10801, 'Thuringer Rostbratwurst and other products for personal use', 1843.8181);
INSERT INTO "InternationalOrders" VALUES(10802, 'Tarte au sucre and other products for personal use', 1792.6263);
INSERT INTO "InternationalOrders" VALUES(10803, 'Teatime Chocolate Biscuits and other products for personal use', 573.7318);
INSERT INTO "InternationalOrders" VALUES(10804, 'Rossle Sauerkraut and other products for personal use', 1046.4049);
INSERT INTO "InternationalOrders" VALUES(10806, 'Louisiana Fiery Hot Pepper Sauce and other products for personal use', 261.3728);
INSERT INTO "InternationalOrders" VALUES(10807, 'Boston Crab Meat and other products for personal use', 8.4063);
INSERT INTO "InternationalOrders" VALUES(10809, 'Filo Mix and other products for personal use', 63.9612);
INSERT INTO "InternationalOrders" VALUES(10810, 'Outback Lager and other products for personal use', 85.4339);
INSERT INTO "InternationalOrders" VALUES(10811, 'Tunnbrod and other products for personal use', 389.2495);
INSERT INTO "InternationalOrders" VALUES(10812, 'Original Frankfurter grune So?e and other products for personal use', 846.1151);
INSERT INTO "InternationalOrders" VALUES(10813, 'Spegesild and other products for personal use', 296.0489);
INSERT INTO "InternationalOrders" VALUES(10814, 'Sirop d''erable and other products for personal use', 945.7117);
INSERT INTO "InternationalOrders" VALUES(10817, 'Tarte au sucre and other products for personal use', 5249.7053);
INSERT INTO "InternationalOrders" VALUES(10818, 'Mascarpone Fabioli and other products for personal use', 380.5690);
INSERT INTO "InternationalOrders" VALUES(10819, 'Rhonbrau Klosterbier and other products for personal use', 217.9249);
INSERT INTO "InternationalOrders" VALUES(10823, 'Ravioli Angelo and other products for personal use', 1419.7098);
INSERT INTO "InternationalOrders" VALUES(10824, 'Outback Lager and other products for personal use', 114.5819);
INSERT INTO "InternationalOrders" VALUES(10825, 'Perth Pasties and other products for personal use', 470.9188);
INSERT INTO "InternationalOrders" VALUES(10826, 'Ravioli Angelo and other products for personal use', 333.5119);
INSERT INTO "InternationalOrders" VALUES(10827, 'Ikura and other products for personal use', 385.1377);
INSERT INTO "InternationalOrders" VALUES(10828, 'Sir Rodney''s Marmalade and other products for personal use', 425.7987);
INSERT INTO "InternationalOrders" VALUES(10829, 'Northwoods Cranberry Sauce and other products for personal use', 805.9109);
INSERT INTO "InternationalOrders" VALUES(10830, 'Scottish Longbreads and other products for personal use', 901.8527);
INSERT INTO "InternationalOrders" VALUES(10831, 'Teatime Chocolate Biscuits and other products for personal use', 1226.4100);
INSERT INTO "InternationalOrders" VALUES(10832, 'Wimmers gute Semmelknodel and other products for personal use', 259.9337);
INSERT INTO "InternationalOrders" VALUES(10833, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 460.3834);
INSERT INTO "InternationalOrders" VALUES(10834, 'Thuringer Rostbratwurst and other products for personal use', 689.0081);
INSERT INTO "InternationalOrders" VALUES(10835, 'Raclette Courdavault and other products for personal use', 388.7926);
INSERT INTO "InternationalOrders" VALUES(10836, 'Wimmers gute Semmelknodel and other products for personal use', 2149.7810);
INSERT INTO "InternationalOrders" VALUES(10837, 'Zaanse koeken and other products for personal use', 572.9094);
INSERT INTO "InternationalOrders" VALUES(10838, 'Inlagd Sill and other products for personal use', 1180.7691);
INSERT INTO "InternationalOrders" VALUES(10839, 'Mozzarella di Giovanni and other products for personal use', 420.0879);
INSERT INTO "InternationalOrders" VALUES(10840, 'NuNuCa Nu?-Nougat-Creme and other products for personal use', 120.6125);
INSERT INTO "InternationalOrders" VALUES(10841, 'Raclette Courdavault and other products for personal use', 2092.9012);
INSERT INTO "InternationalOrders" VALUES(10842, 'Scottish Longbreads and other products for personal use', 445.4439);
INSERT INTO "InternationalOrders" VALUES(10843, 'Manjimup Dried Apples and other products for personal use', 96.8555);
INSERT INTO "InternationalOrders" VALUES(10844, 'Gustaf''s Knackebrod and other products for personal use', 335.7962);
INSERT INTO "InternationalOrders" VALUES(10845, 'Wimmers gute Semmelknodel and other products for personal use', 1854.4174);
INSERT INTO "InternationalOrders" VALUES(10846, 'Outback Lager and other products for personal use', 508.0345);
INSERT INTO "InternationalOrders" VALUES(10848, 'Mishi Kobe Niku and other products for personal use', 425.5703);
INSERT INTO "InternationalOrders" VALUES(10849, 'Gumbar Gummibarchen and other products for personal use', 480.6866);
INSERT INTO "InternationalOrders" VALUES(10850, 'Outback Lager and other products for personal use', 338.0805);
INSERT INTO "InternationalOrders" VALUES(10851, 'Ravioli Angelo and other products for personal use', 1251.8117);
INSERT INTO "InternationalOrders" VALUES(10853, 'Carnarvon Tigers and other products for personal use', 285.5410);
INSERT INTO "InternationalOrders" VALUES(10854, 'Konbu and other products for personal use', 1594.4609);
INSERT INTO "InternationalOrders" VALUES(10856, 'Singaporean Hokkien Fried Mee and other products for personal use', 301.5313);
INSERT INTO "InternationalOrders" VALUES(10857, 'Thuringer Rostbratwurst and other products for personal use', 1201.9905);
INSERT INTO "InternationalOrders" VALUES(10858, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 296.5058);
INSERT INTO "InternationalOrders" VALUES(10859, 'Wimmers gute Semmelknodel and other products for personal use', 657.0869);
INSERT INTO "InternationalOrders" VALUES(10860, 'Manjimup Dried Apples and other products for personal use', 237.1132);
INSERT INTO "InternationalOrders" VALUES(10862, 'Queso Cabrales and other products for personal use', 265.4389);
INSERT INTO "InternationalOrders" VALUES(10863, 'Escargots de Bourgogne and other products for personal use', 237.1132);
INSERT INTO "InternationalOrders" VALUES(10864, 'Steeleye Stout and other products for personal use', 128.8361);
INSERT INTO "InternationalOrders" VALUES(10865, 'Cote de Blaye and other products for personal use', 7880.9312);
INSERT INTO "InternationalOrders" VALUES(10866, 'Nord-Ost Matjeshering and other products for personal use', 667.7547);
INSERT INTO "InternationalOrders" VALUES(10868, 'Steeleye Stout and other products for personal use', 915.8327);
INSERT INTO "InternationalOrders" VALUES(10869, 'Tunnbrod and other products for personal use', 744.6909);
INSERT INTO "InternationalOrders" VALUES(10870, 'Steeleye Stout and other products for personal use', 73.0985);
INSERT INTO "InternationalOrders" VALUES(10871, 'Pavlova and other products for personal use', 951.8337);
INSERT INTO "InternationalOrders" VALUES(10872, 'Wimmers gute Semmelknodel and other products for personal use', 989.9363);
INSERT INTO "InternationalOrders" VALUES(10873, 'Sir Rodney''s Scones and other products for personal use', 153.8723);
INSERT INTO "InternationalOrders" VALUES(10874, 'Ikura and other products for personal use', 141.6283);
INSERT INTO "InternationalOrders" VALUES(10875, 'Zaanse koeken and other products for personal use', 333.2834);
INSERT INTO "InternationalOrders" VALUES(10876, 'Wimmers gute Semmelknodel and other products for personal use', 418.9457);
INSERT INTO "InternationalOrders" VALUES(10877, 'Pavlova and other products for personal use', 953.0216);
INSERT INTO "InternationalOrders" VALUES(10878, 'Sir Rodney''s Marmalade and other products for personal use', 740.1222);
INSERT INTO "InternationalOrders" VALUES(10879, 'Louisiana Fiery Hot Pepper Sauce and other products for personal use', 279.2819);
INSERT INTO "InternationalOrders" VALUES(10880, 'Tunnbrod and other products for personal use', 856.6230);
INSERT INTO "InternationalOrders" VALUES(10881, 'Rod Kaviar and other products for personal use', 68.5298);
INSERT INTO "InternationalOrders" VALUES(10885, 'Outback Lager and other products for personal use', 552.3505);
INSERT INTO "InternationalOrders" VALUES(10886, 'Original Frankfurter grune So?e and other products for personal use', 1428.8471);
INSERT INTO "InternationalOrders" VALUES(10887, 'NuNuCa Nu?-Nougat-Creme and other products for personal use', 31.9806);
INSERT INTO "InternationalOrders" VALUES(10888, 'Scottish Longbreads and other products for personal use', 276.4037);
INSERT INTO "InternationalOrders" VALUES(10890, 'Sasquatch Ale and other products for personal use', 392.9501);
INSERT INTO "InternationalOrders" VALUES(10891, 'Nord-Ost Matjeshering and other products for personal use', 177.4237);
INSERT INTO "InternationalOrders" VALUES(10892, 'Raclette Courdavault and other products for personal use', 1005.1043);
INSERT INTO "InternationalOrders" VALUES(10893, 'Thuringer Rostbratwurst and other products for personal use', 2513.7247);
INSERT INTO "InternationalOrders" VALUES(10895, 'Guarana Fantastica and other products for personal use', 2914.5283);
INSERT INTO "InternationalOrders" VALUES(10896, 'Rogede sild and other products for personal use', 342.8776);
INSERT INTO "InternationalOrders" VALUES(10897, 'Thuringer Rostbratwurst and other products for personal use', 4950.2482);
INSERT INTO "InternationalOrders" VALUES(10898, 'Konbu and other products for personal use', 13.7060);
INSERT INTO "InternationalOrders" VALUES(10899, 'Chartreuse verte and other products for personal use', 65.7886);
INSERT INTO "InternationalOrders" VALUES(10900, 'Outback Lager and other products for personal use', 20.5590);
INSERT INTO "InternationalOrders" VALUES(10901, 'Jack''s New England Clam Chowder and other products for personal use', 426.9409);
INSERT INTO "InternationalOrders" VALUES(10902, 'Tarte au sucre and other products for personal use', 464.0841);
INSERT INTO "InternationalOrders" VALUES(10903, 'Scottish Longbreads and other products for personal use', 425.8216);
INSERT INTO "InternationalOrders" VALUES(10905, 'Chai and other products for personal use', 164.4716);
INSERT INTO "InternationalOrders" VALUES(10906, 'Sirop d''erable and other products for personal use', 195.3100);
INSERT INTO "InternationalOrders" VALUES(10907, 'Rhonbrau Klosterbier and other products for personal use', 49.5699);
INSERT INTO "InternationalOrders" VALUES(10908, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 318.8922);
INSERT INTO "InternationalOrders" VALUES(10909, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 306.0999);
INSERT INTO "InternationalOrders" VALUES(10910, 'Teatime Chocolate Biscuits and other products for personal use', 206.9144);
INSERT INTO "InternationalOrders" VALUES(10911, 'Laughing Lumberjack Lager and other products for personal use', 391.9907);
INSERT INTO "InternationalOrders" VALUES(10912, 'Thuringer Rostbratwurst and other products for personal use', 3777.0905);
INSERT INTO "InternationalOrders" VALUES(10913, 'Geitost and other products for personal use', 438.0199);
INSERT INTO "InternationalOrders" VALUES(10914, 'Flotemysost and other products for personal use', 245.5652);
INSERT INTO "InternationalOrders" VALUES(10915, 'Tourtiere and other products for personal use', 246.4790);
INSERT INTO "InternationalOrders" VALUES(10916, 'Ravioli Angelo and other products for personal use', 313.7296);
INSERT INTO "InternationalOrders" VALUES(10917, 'Nord-Ost Matjeshering and other products for personal use', 167.1625);
INSERT INTO "InternationalOrders" VALUES(10918, 'Chai and other products for personal use', 881.7506);
INSERT INTO "InternationalOrders" VALUES(10919, 'Pavlova and other products for personal use', 512.9687);
INSERT INTO "InternationalOrders" VALUES(10920, 'Valkoinen suklaa and other products for personal use', 178.1776);
INSERT INTO "InternationalOrders" VALUES(10921, 'Vegie-spread and other products for personal use', 884.4918);
INSERT INTO "InternationalOrders" VALUES(10922, 'Guarana Fantastica and other products for personal use', 339.2227);
INSERT INTO "InternationalOrders" VALUES(10923, 'Singaporean Hokkien Fried Mee and other products for personal use', 427.6262);
INSERT INTO "InternationalOrders" VALUES(10924, 'Rossle Sauerkraut and other products for personal use', 929.4930);
INSERT INTO "InternationalOrders" VALUES(10925, 'Inlagd Sill and other products for personal use', 255.3879);
INSERT INTO "InternationalOrders" VALUES(10926, 'Teatime Chocolate Biscuits and other products for personal use', 235.0117);
INSERT INTO "InternationalOrders" VALUES(10927, 'Sir Rodney''s Marmalade and other products for personal use', 365.4925);
INSERT INTO "InternationalOrders" VALUES(10928, 'Zaanse koeken and other products for personal use', 62.8190);
INSERT INTO "InternationalOrders" VALUES(10929, 'Sir Rodney''s Scones and other products for personal use', 536.7028);
INSERT INTO "InternationalOrders" VALUES(10930, 'Sir Rodney''s Scones and other products for personal use', 1121.6050);
INSERT INTO "InternationalOrders" VALUES(10931, 'Ravioli Angelo and other products for personal use', 382.3965);
INSERT INTO "InternationalOrders" VALUES(10932, 'Tarte au sucre and other products for personal use', 879.6947);
INSERT INTO "InternationalOrders" VALUES(10933, 'Sirop d''erable and other products for personal use', 420.5905);
INSERT INTO "InternationalOrders" VALUES(10934, 'Grandma''s Boysenberry Spread and other products for personal use', 228.4328);
INSERT INTO "InternationalOrders" VALUES(10935, 'Tunnbrod and other products for personal use', 319.8059);
INSERT INTO "InternationalOrders" VALUES(10937, 'Sasquatch Ale and other products for personal use', 294.5869);
INSERT INTO "InternationalOrders" VALUES(10938, 'Konbu and other products for personal use', 1664.1329);
INSERT INTO "InternationalOrders" VALUES(10939, 'Laughing Lumberjack Lager and other products for personal use', 342.6492);
INSERT INTO "InternationalOrders" VALUES(10940, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 164.4716);
INSERT INTO "InternationalOrders" VALUES(10942, 'Maxilaku and other products for personal use', 255.8447);
INSERT INTO "InternationalOrders" VALUES(10943, 'Spegesild and other products for personal use', 324.8314);
INSERT INTO "InternationalOrders" VALUES(10944, 'Queso Cabrales and other products for personal use', 520.4156);
INSERT INTO "InternationalOrders" VALUES(10945, 'Konbu and other products for personal use', 111.9321);
INSERT INTO "InternationalOrders" VALUES(10946, 'Original Frankfurter grune So?e and other products for personal use', 643.0383);
INSERT INTO "InternationalOrders" VALUES(10947, 'Raclette Courdavault and other products for personal use', 100.5104);
INSERT INTO "InternationalOrders" VALUES(10948, 'Valkoinen suklaa and other products for personal use', 1079.2307);
INSERT INTO "InternationalOrders" VALUES(10949, 'Tarte au sucre and other products for personal use', 2020.2596);
INSERT INTO "InternationalOrders" VALUES(10950, 'Chef Anton''s Cajun Seasoning and other products for personal use', 50.2552);
INSERT INTO "InternationalOrders" VALUES(10951, 'Rhonbrau Klosterbier and other products for personal use', 220.6204);
INSERT INTO "InternationalOrders" VALUES(10952, 'Rossle Sauerkraut and other products for personal use', 224.4124);
INSERT INTO "InternationalOrders" VALUES(10953, 'Sir Rodney''s Marmalade and other products for personal use', 2135.8466);
INSERT INTO "InternationalOrders" VALUES(10954, 'Rogede sild and other products for personal use', 869.0040);
INSERT INTO "InternationalOrders" VALUES(10955, 'Rhonbrau Klosterbier and other products for personal use', 42.4885);
INSERT INTO "InternationalOrders" VALUES(10956, 'Zaanse koeken and other products for personal use', 309.2980);
INSERT INTO "InternationalOrders" VALUES(10957, 'Wimmers gute Semmelknodel and other products for personal use', 805.3170);
INSERT INTO "InternationalOrders" VALUES(10958, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 356.8120);
INSERT INTO "InternationalOrders" VALUES(10959, 'Rhonbrau Klosterbier and other products for personal use', 70.8142);
INSERT INTO "InternationalOrders" VALUES(10960, 'Jack''s New England Clam Chowder and other products for personal use', 126.3690);
INSERT INTO "InternationalOrders" VALUES(10961, 'Lakkalikoori and other products for personal use', 512.6032);
INSERT INTO "InternationalOrders" VALUES(10962, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 1637.4062);
INSERT INTO "InternationalOrders" VALUES(10963, 'Camembert Pierrot and other products for personal use', 31.0669);
INSERT INTO "InternationalOrders" VALUES(10964, 'Gudbrandsdalsost and other products for personal use', 937.7166);
INSERT INTO "InternationalOrders" VALUES(10966, 'Tarte au sucre and other products for personal use', 573.6404);
INSERT INTO "InternationalOrders" VALUES(10967, 'Teatime Chocolate Biscuits and other products for personal use', 415.9304);
INSERT INTO "InternationalOrders" VALUES(10968, 'Wimmers gute Semmelknodel and other products for personal use', 643.2667);
INSERT INTO "InternationalOrders" VALUES(10969, 'Spegesild and other products for personal use', 49.3415);
INSERT INTO "InternationalOrders" VALUES(10970, 'Filo Mix and other products for personal use', 127.9224);
INSERT INTO "InternationalOrders" VALUES(10971, 'Thuringer Rostbratwurst and other products for personal use', 791.7755);
INSERT INTO "InternationalOrders" VALUES(10972, 'Geitost and other products for personal use', 114.9017);
INSERT INTO "InternationalOrders" VALUES(10973, 'Rhonbrau Klosterbier and other products for personal use', 133.1992);
INSERT INTO "InternationalOrders" VALUES(10975, 'Rhonbrau Klosterbier and other products for personal use', 327.8011);
INSERT INTO "InternationalOrders" VALUES(10976, 'Rossle Sauerkraut and other products for personal use', 416.6614);
INSERT INTO "InternationalOrders" VALUES(10977, 'Zaanse koeken and other products for personal use', 1020.1808);
INSERT INTO "InternationalOrders" VALUES(10978, 'Sir Rodney''s Scones and other products for personal use', 685.6182);
INSERT INTO "InternationalOrders" VALUES(10979, 'Vegie-spread and other products for personal use', 2199.1225);
INSERT INTO "InternationalOrders" VALUES(10980, 'Rhonbrau Klosterbier and other products for personal use', 141.6283);
INSERT INTO "InternationalOrders" VALUES(10981, 'Cote de Blaye and other products for personal use', 7223.0448);
INSERT INTO "InternationalOrders" VALUES(10982, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 463.2617);
INSERT INTO "InternationalOrders" VALUES(10985, 'Pavlova and other products for personal use', 1027.1252);
INSERT INTO "InternationalOrders" VALUES(10986, 'Sir Rodney''s Marmalade and other products for personal use', 1014.2416);
INSERT INTO "InternationalOrders" VALUES(10987, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 1266.4314);
INSERT INTO "InternationalOrders" VALUES(10989, 'Queso Cabrales and other products for personal use', 618.4132);
INSERT INTO "InternationalOrders" VALUES(10990, 'Sirop d''erable and other products for personal use', 2252.8042);
INSERT INTO "InternationalOrders" VALUES(10991, 'Outback Lager and other products for personal use', 1311.2042);
INSERT INTO "InternationalOrders" VALUES(10993, 'Thuringer Rostbratwurst and other products for personal use', 2982.0759);
INSERT INTO "InternationalOrders" VALUES(10994, 'Raclette Courdavault and other products for personal use', 452.2969);
INSERT INTO "InternationalOrders" VALUES(10995, 'Manjimup Dried Apples and other products for personal use', 546.4112);
INSERT INTO "InternationalOrders" VALUES(10996, 'Singaporean Hokkien Fried Mee and other products for personal use', 255.8447);
INSERT INTO "InternationalOrders" VALUES(10997, 'Spegesild and other products for personal use', 904.5938);
INSERT INTO "InternationalOrders" VALUES(10998, 'Sirop d''erable and other products for personal use', 313.4098);
INSERT INTO "InternationalOrders" VALUES(10999, 'Original Frankfurter grune So?e and other products for personal use', 576.1075);
INSERT INTO "InternationalOrders" VALUES(11001, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 1265.0608);
INSERT INTO "InternationalOrders" VALUES(11004, 'Lakkalikoori and other products for personal use', 134.9490);
INSERT INTO "InternationalOrders" VALUES(11005, 'Raclette Courdavault and other products for personal use', 267.7232);
INSERT INTO "InternationalOrders" VALUES(11007, 'Thuringer Rostbratwurst and other products for personal use', 1203.3383);
INSERT INTO "InternationalOrders" VALUES(11008, 'Sasquatch Ale and other products for personal use', 2240.2404);
INSERT INTO "InternationalOrders" VALUES(11009, 'Inlagd Sill and other products for personal use', 320.7196);
INSERT INTO "InternationalOrders" VALUES(11010, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 294.6783);
INSERT INTO "InternationalOrders" VALUES(11011, 'Flotemysost and other products for personal use', 438.5910);
INSERT INTO "InternationalOrders" VALUES(11012, 'Teatime Chocolate Biscuits and other products for personal use', 1358.7182);
INSERT INTO "InternationalOrders" VALUES(11013, 'Tunnbrod and other products for personal use', 164.9285);
INSERT INTO "InternationalOrders" VALUES(11014, 'Jack''s New England Clam Chowder and other products for personal use', 123.4451);
INSERT INTO "InternationalOrders" VALUES(11015, 'Original Frankfurter grune So?e and other products for personal use', 284.3303);
INSERT INTO "InternationalOrders" VALUES(11016, 'Inlagd Sill and other products for personal use', 224.5494);
INSERT INTO "InternationalOrders" VALUES(11017, 'Raclette Courdavault and other products for personal use', 3083.8427);
INSERT INTO "InternationalOrders" VALUES(11019, 'Spegesild and other products for personal use', 34.7218);
INSERT INTO "InternationalOrders" VALUES(11020, 'Ikura and other products for personal use', 339.9080);
INSERT INTO "InternationalOrders" VALUES(11021, 'Sir Rodney''s Marmalade and other products for personal use', 3171.3279);
INSERT INTO "InternationalOrders" VALUES(11022, 'Teatime Chocolate Biscuits and other products for personal use', 640.5255);
INSERT INTO "InternationalOrders" VALUES(11023, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 685.2984);
INSERT INTO "InternationalOrders" VALUES(11024, 'Louisiana Fiery Hot Pepper Sauce and other products for personal use', 898.5678);
INSERT INTO "InternationalOrders" VALUES(11025, 'Konbu and other products for personal use', 137.0597);
INSERT INTO "InternationalOrders" VALUES(11026, 'Manjimup Dried Apples and other products for personal use', 470.5715);
INSERT INTO "InternationalOrders" VALUES(11027, 'Tarte au sucre and other products for personal use', 534.6698);
INSERT INTO "InternationalOrders" VALUES(11028, 'Raclette Courdavault and other products for personal use', 986.8297);
INSERT INTO "InternationalOrders" VALUES(11029, 'Vegie-spread and other products for personal use', 587.8946);
INSERT INTO "InternationalOrders" VALUES(11033, 'Perth Pasties and other products for personal use', 1641.0612);
INSERT INTO "InternationalOrders" VALUES(11035, 'Tourtiere and other products for personal use', 801.5707);
INSERT INTO "InternationalOrders" VALUES(11036, 'Raclette Courdavault and other products for personal use', 773.0166);
INSERT INTO "InternationalOrders" VALUES(11037, 'Outback Lager and other products for personal use', 27.4119);
INSERT INTO "InternationalOrders" VALUES(11038, 'Flotemysost and other products for personal use', 343.1061);
INSERT INTO "InternationalOrders" VALUES(11039, 'Steeleye Stout and other products for personal use', 1411.7146);
INSERT INTO "InternationalOrders" VALUES(11041, 'Vegie-spread and other products for personal use', 862.1053);
INSERT INTO "InternationalOrders" VALUES(11042, 'Sirop d''erable and other products for personal use', 185.3732);
INSERT INTO "InternationalOrders" VALUES(11043, 'Queso Cabrales and other products for personal use', 95.9418);
INSERT INTO "InternationalOrders" VALUES(11044, 'Tarte au sucre and other products for personal use', 270.2817);
INSERT INTO "InternationalOrders" VALUES(11045, 'Manjimup Dried Apples and other products for personal use', 598.2655);
INSERT INTO "InternationalOrders" VALUES(11046, 'Steeleye Stout and other products for personal use', 714.5378);
INSERT INTO "InternationalOrders" VALUES(11047, 'Chef Anton''s Gumbo Mix and other products for personal use', 498.2119);
INSERT INTO "InternationalOrders" VALUES(11048, 'Scottish Longbreads and other products for personal use', 239.8544);
INSERT INTO "InternationalOrders" VALUES(11049, 'Queso Manchego La Pastora and other products for personal use', 156.2480);
INSERT INTO "InternationalOrders" VALUES(11050, 'Lakkalikoori and other products for personal use', 411.1790);
INSERT INTO "InternationalOrders" VALUES(11051, 'Guarana Fantastica and other products for personal use', 20.5590);
INSERT INTO "InternationalOrders" VALUES(11052, 'Sirop d''erable and other products for personal use', 760.6812);
INSERT INTO "InternationalOrders" VALUES(11053, 'Wimmers gute Semmelknodel and other products for personal use', 1671.5569);
INSERT INTO "InternationalOrders" VALUES(11054, 'Laughing Lumberjack Lager and other products for personal use', 139.3440);
INSERT INTO "InternationalOrders" VALUES(11055, 'Ravioli Angelo and other products for personal use', 789.2353);
INSERT INTO "InternationalOrders" VALUES(11056, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 1708.6773);
INSERT INTO "InternationalOrders" VALUES(11057, 'Outback Lager and other products for personal use', 20.5590);
INSERT INTO "InternationalOrders" VALUES(11058, 'Sirop d''erable and other products for personal use', 391.9907);
INSERT INTO "InternationalOrders" VALUES(11059, 'Konbu and other products for personal use', 839.7189);
INSERT INTO "InternationalOrders" VALUES(11060, 'Original Frankfurter grune So?e and other products for personal use', 121.5262);
INSERT INTO "InternationalOrders" VALUES(11062, 'Perth Pasties and other products for personal use', 232.0877);
INSERT INTO "InternationalOrders" VALUES(11063, 'Sasquatch Ale and other products for personal use', 660.3992);
INSERT INTO "InternationalOrders" VALUES(11065, 'Tourtiere and other products for personal use', 115.3860);
INSERT INTO "InternationalOrders" VALUES(11067, 'Jack''s New England Clam Chowder and other products for personal use', 39.6788);
INSERT INTO "InternationalOrders" VALUES(11068, 'Rossle Sauerkraut and other products for personal use', 1089.5330);
INSERT INTO "InternationalOrders" VALUES(11069, 'Chartreuse verte and other products for personal use', 164.4716);
INSERT INTO "InternationalOrders" VALUES(11070, 'Pavlova and other products for personal use', 855.9377);
INSERT INTO "InternationalOrders" VALUES(11071, 'Uncle Bob''s Organic Dried Pears and other products for personal use', 233.0014);
INSERT INTO "InternationalOrders" VALUES(11072, 'Wimmers gute Semmelknodel and other products for personal use', 2383.9246);
INSERT INTO "InternationalOrders" VALUES(11073, 'Queso Cabrales and other products for personal use', 137.0597);
INSERT INTO "InternationalOrders" VALUES(11074, 'Pavlova and other products for personal use', 111.6123);
INSERT INTO "InternationalOrders" VALUES(11075, 'Spegesild and other products for personal use', 267.7232);
INSERT INTO "InternationalOrders" VALUES(11076, 'Tofu and other products for personal use', 482.9069);

CREATE TABLE "PreviousEmployees" (
  "EmployeeID" NUMBER(10) NOT NULL,
  "LastName" VARCHAR2(20) NOT NULL,
  "FirstName" VARCHAR2(10) NOT NULL,
  "Title" VARCHAR2(30),
  "TitleOfCourtesy" VARCHAR2(25),
  "BirthDate" TIMESTAMP(6),
  "HireDate" TIMESTAMP(6),
  "Address" VARCHAR2(60),
  "City" VARCHAR2(15),
  "Region" VARCHAR2(15),
  "PostalCode" VARCHAR2(10),
  "Country" VARCHAR2(15),
  "HomePhone" VARCHAR2(24),
  "Extension" VARCHAR2(4),
  "Photo" BLOB,
  "Notes" VARCHAR2(4000),
  "PhotoPath" VARCHAR2(255),
  CONSTRAINT "CK_PreviousEmployeesBirthdate" CHECK (  "BirthDate"< ''),
  CONSTRAINT "PK_PreviousEmployees" PRIMARY KEY ("EmployeeID"))
TABLESPACE USERS
STORAGE (
  INITIAL 64K
  MAXEXTENTS UNLIMITED
)
LOGGING;


INSERT INTO "PreviousEmployees" VALUES
	(7, 'King', 'Robert', 'Sales Representative', 'Mr.', DATE '1960-05-29', DATE '1994-01-02', 'Edgeham Hollow Winchester Way', 'London', NULL, 'RG1 9SP', 'UK', '(71) 555-5598', 465, null, 'Robert King served in the Peace Corps and traveled extensively before completing his degree in English at the University of Michigan in 1992, the year he joined the company.  After completing a course entitled "Selling in Europe," he was transferred to the London office in March 1993.', 'http://accweb/emmployees/davolio.bmp');
	
	CREATE OR REPLACE PROCEDURE "EmployeesDelete"("E_ID_" NUMBER)
IS
BEGIN
	INSERT INTO "PreviousEmployees"("EmployeeID","LastName","FirstName","Title","TitleOfCourtesy","BirthDate","HireDate","Address","City","Region","PostalCode","Country","HomePhone","Extension","Photo","Notes","PhotoPath")
	SELECT "EmployeeID","LastName","FirstName","Title","TitleOfCourtesy","BirthDate","HireDate","Address","City","Region","PostalCode","Country","HomePhone","Extension","Photo","Notes","PhotoPath"
	FROM "Employees" WHERE "EmployeeID" = "E_ID_";

	DELETE FROM "EmployeesTerritories" WHERE "EmployeeID" = "E_ID_";

	DELETE FROM "Employees"	WHERE "EmployeeID" = "E_ID_";
 END;
/
CREATE OR REPLACE PROCEDURE "EmployeesInsert"(
           LastName NUMBER,
           FirstName varchar2,
           Title varchar2,
           TitleOfCourtesy varchar2,
           BirthDate timestamp,
           HireDate timestamp,
           Address varchar2,
           City varchar2,
           Region varchar2,
           PostalCode varchar2,
           Country varchar2,
           HomePhone varchar2,
           Extension varchar2,
           Photo blob,
           Notes varchar2,
           PhotoPath varchar2
)
IS
BEGIN
	INSERT INTO "Employees"
           ("LastName"
           ,"FirstName"
           ,"Title"
           ,"TitleOfCourtesy"
           ,"BirthDate"
           ,"HireDate"
           ,"Address"
           ,"City"
           ,"Region"
           ,"PostalCode"
           ,"Country"
           ,"HomePhone"
           ,"Extension"
           ,"Photo"
           ,"Notes"
           ,"PhotoPath")
     VALUES
           (LastName,
           FirstName,
           Title,
           TitleOfCourtesy,
           BirthDate,
           HireDate,
           Address,
           City,
           Region,
           PostalCode,
           Country,
           HomePhone,
           Extension,
           Photo,
           Notes,
           PhotoPath);
END;
/
CREATE OR REPLACE PROCEDURE "EmployeesUpdate"(
           EmployeeID NUMBER,
           LastName NUMBER,
           FirstName varchar2,
           Title varchar2,
           TitleOfCourtesy varchar2,
           BirthDate timestamp,
           HireDate timestamp,
           Address varchar2,
           City varchar2,
           Region varchar2,
           PostalCode varchar2,
           Country varchar2,
           HomePhone varchar2,
           Extension varchar2,
           Photo blob,
           Notes varchar2,
           PhotoPath varchar2
)
IS
BEGIN
	UPDATE "Employees" SET
           "LastName" = LastName,
           "FirstName" = FirstName,
           "Title" = Title,
           "TitleOfCourtesy" = TitleOfCourtesy,
           "BirthDate" = BirthDate,
           "HireDate" = HireDate,
           "Address" = Address,
           "City" = City,
           "Region" = Region,
           "PostalCode" = PostalCode,
           "Country" = Country,
           "HomePhone" = HomePhone,
           "Extension" = Extension,
           "Photo" = Photo,
           "Notes" = Notes,
           "PhotoPath" = PhotoPath
	WHERE "EmployeeID" = EmployeeID;
END;