unit DemoBase;

{$I ..\EntityDAC.Demo.inc}

interface

uses
{$IFDEF MSWINDOWS}
  Windows, ShellAPI,
{$ENDIF}
  Classes, SysUtils, DemoFrame, CategoryFrame,
  ComCtrls, StdCtrls, Controls, Graphics, Forms, HTMLConsts;

type
  TDemoType = (dtDemo, dtCategory);

  TDemo = class
  private
    function GetFullDFMName: string;
    function GetFullUnitName: string;
  protected
    FName: string;
    FFileName: string;
    FHint: string;
    FDescription: string;
    FFolder: string;
    FDemoType: TDemoType;
    FFrameClass: TDemoFrameClass;
    FFrame: TDemoFrame;
    FSelector: integer;
    FNode: TTreeNode;

    procedure DoLoadDemoPiece(Pattern: string; Strings: TStrings);
  public
    constructor Create(Name, Hint, Description: string; DemoType: TDemoType; FrameClass: TDemoFrameClass; Selector: integer; FileName: string = ''; Folder: string = '');
    destructor Destroy; override;

    procedure LoadDemoCode(Strings: TStrings);
    procedure LoadFormCode(Strings: TStrings);
    procedure LoadDemoPiece(Button: TButton; Strings: TStrings); overload;
    procedure LoadDemoPiece(ProcedureName: string; Strings: TStrings); overload;
    procedure OpenDemoFolder;
    procedure FreeFrame;

    property Name: string read FName;
    property Hint: string read FHint;
    property Description: string read FDescription;
    property DemoType: TDemoType read FDemoType;
    property FrameClass: TDemoFrameClass read FFrameClass;
    property Frame: TDemoFrame read FFrame;
    property Selector: integer read FSelector;
    property FullUnitName: string read GetFullUnitName;
    property FullDFMName: string read GetFullDFMName;

    property Node: TTreeNode read FNode;
  end;

  TDemos = class
  protected
    FDemoTreeNodes: TTreeNodes;
    FSelectedDemo: TDemo;
    FSupplementaryDemosDescription: TStrings;
    function GetSelectedDemo: TDemo;
    function GetItems(Index: integer): TDemo;

    function WrapDescription(Demo: TDemo): TStrings;
    function FindCategoryNode(const CategoryName: string): TTreeNode;
    procedure RegisterCategory(const ParentNode: TTreeNode; CategoryName, Hint: string; ImgIndex: integer = -1; SupplementaryDemo: boolean = False); overload;
  public
    constructor Create(TreeNodes: TTreeNodes); virtual;
    destructor Destroy; override;

    procedure RegisterCategory(CategoryName, Hint: string; ImgIndex: integer = -1; SupplementaryDemo: boolean = False); overload;
    procedure RegisterCategory(CategoryName, Hint, ParentCategory: string; ImgIndex: integer = -1; SupplementaryDemo: boolean = False); overload;
    procedure RegisterDemo(DemoName, DemoHint, DemoDescription, DemoCategory: string; FrameClass: TDemoFrameClass; Selector, ImgIndex: integer; FileName: string = ''; Folder: string = ''; SupplementaryDemo: boolean = False);
    procedure Clear;
    //Navigation
    function SelectDemo(DemoIndex: integer): TDemo;    //Create demo frame by DemoIndex

    function GetDemoIndex(AbsoluteIndex: integer): integer;

    property Items[Index: integer]: TDemo read GetItems; default;
    property SelectedDemo: TDemo read GetSelectedDemo;
  end;

implementation

const
  MainDemosHeader = 'Main Demo Projects';
  SupplementaryDemosHeader = 'Supplementary Demo Projects';
  SupplementaryDemosNote = '       Note, the demo projects listed below are separate projects. ' {$IFNDEF WEB} + #13#10 {$ENDIF} + '       This project contains only their descriptions and links to their folders.';

{$IFNDEF WEB}
procedure WriteTable(CategoryName, CategoryDescription: string; List: TStrings);
var
  strs: TStringList;
begin
  List.Add(' ');
  List.Add('    ' + CategoryName);
  strs := TStringList.Create;
  strs.Text := CategoryDescription;
  List.AddStrings(strs);
  strs.Free;
end;
{$ELSE}
procedure WriteTable(CategoryName, CategoryDescription: string; List: TStrings);
begin
  if CategoryName = '' then begin  // used to close virtual category
    List.Add('</table>');
    Exit;
  end;

  List.Add('<p><b><div style="padding-top: 10px;">' + CategoryName + '</div></b></p>');
  List.Add('<table class="xmldoctable" cellspacing="0">');
  List.Add('<tr>');
  List.Add('<th width="150">Demo</th>');
  List.Add('<th>Description</th>');
  List.Add('</tr>');
  if CategoryDescription <> '' then begin
    List.Add(CategoryDescription);
    List.Add('</table>');
  end;
end;
{$ENDIF}

constructor TDemos.Create(TreeNodes: TTreeNodes);
begin
  inherited Create;

  if not Assigned(TreeNodes) then
    raise Exception.Create('TreeNodes should be set');
  FDemoTreeNodes := TreeNodes;
  FSupplementaryDemosDescription := TStringList.Create;
{$IFNDEF WEB}
  FSupplementaryDemosDescription.Text :=  #13#10'     ' + SupplementaryDemosHeader;
  FSupplementaryDemosDescription.Add(SupplementaryDemosNote);
{$ENDIF}
end;

destructor TDemos.Destroy;
begin
  Clear;
  FSupplementaryDemosDescription.Free;

  inherited;
end;

function TDemos.FindCategoryNode(const CategoryName: string): TTreeNode;

  function DoFindCategoryNode(Node: TTreeNode): TTreeNode;
  begin
    Result := nil;
    if Node = nil then
      Exit;
    if Node.Text = CategoryName then begin
      Result := Node;
      Exit;
    end;
    Result := DoFindCategoryNode(Node.getNextSibling);
    if Result <> nil then
      Exit;
    Result := DoFindCategoryNode(Node.getFirstChild);
  end;

var
  RootNode: TTreeNode;
begin
  RootNode := FDemoTreeNodes.GetFirstNode;
  if RootNode <> nil then
    Result := DoFindCategoryNode(RootNode.getFirstChild)
  else
    Result := nil;
end;

procedure TDemos.RegisterCategory(const ParentNode: TTreeNode; CategoryName, Hint: string; ImgIndex: integer; SupplementaryDemo: boolean);
var
  Node: TTreeNode;
  Index: integer;
  Category: TDemo;
begin
  if SupplementaryDemo then
    WriteTable(CategoryName, Hint, FSupplementaryDemosDescription)
  else begin
    Category := TDemo.Create(CategoryName, Hint, '', dtCategory, TCategoryFrame, 0);
    Node := FDemoTreeNodes.AddChildObject(ParentNode, CategoryName, Category);
    Category.FNode := Node;
    if ImgIndex < 0 then
      Index := 0
    else
      Index := ImgIndex;
    Node.ImageIndex := Index;
    Node.SelectedIndex := Index;
    Node.StateIndex := Index;
  end;
end;

procedure TDemos.RegisterCategory(CategoryName, Hint: string; ImgIndex: integer = -1; SupplementaryDemo: boolean = False);
begin
  RegisterCategory(FDemoTreeNodes.GetFirstNode, CategoryName, Hint, ImgIndex, SupplementaryDemo);
end;

procedure TDemos.RegisterCategory(CategoryName, Hint, ParentCategory: string; ImgIndex: integer; SupplementaryDemo: boolean);
begin
  RegisterCategory(FindCategoryNode(ParentCategory), CategoryName, Hint, ImgIndex, SupplementaryDemo);
end;

procedure TDemos.RegisterDemo(DemoName, DemoHint, DemoDescription, DemoCategory: string; FrameClass: TDemoFrameClass; Selector, ImgIndex: integer; FileName: string = ''; Folder: string = ''; SupplementaryDemo: boolean = False);

{$IFNDEF WEB}
  function CompleteWithSpaces(s: string; ResultLength: word): string;
  var
    n, i: integer;
  begin
    result := s;
    n := ResultLength - Length(s);
    if n > 0 then
      for i := 1 to n do
        result := result + ' ';
  end;
{$ENDIF}

  function ToTableLine(Name, Description, DemoLink: string): string;
  begin
  {$IFDEF WEB}
    Result := Format('<tr height="23"> <td>' +
      '<b><a href="x:\%s">%s</a></b></td><td class="xmldoctable">%s</td></tr>'#13#10,
      [DemoLink, Name, Description]);
  {$ELSE}
    Result := CompleteWithSpaces(Name, 15) + '- ' + Description + {$IFDEF LINUX}#13{$ELSE}#13#10{$ENDIF};
  {$ENDIF}
  end;

var
  CategoryNode, DemoNode: TTreeNode;
  Index: integer;
  Category, Demo: TDemo;
begin
  if SupplementaryDemo then
    FSupplementaryDemosDescription.Text := FSupplementaryDemosDescription.Text + ToTableLine(DemoName, DemoDescription, '..\' + DemoCategory + '\' + DemoName)
  else begin
    CategoryNode := FindCategoryNode(DemoCategory);
    if not Assigned(CategoryNode) then
      raise Exception.Create('DemoCategory is wrong');

    Category := TDemo(CategoryNode.Data);
    Demo := TDemo.Create(DemoName, DemoHint, DemoDescription, dtDemo, FrameClass, Selector, FileName, Folder);
    DemoNode := FDemoTreeNodes.AddChildObject(CategoryNode, Demo.Name, Demo);
    Demo.FNode := DemoNode;
    if ImgIndex < 0 then
      Index := 1
    else
      Index := ImgIndex;
    DemoNode.ImageIndex := Index;
    DemoNode.SelectedIndex := Index;
    DemoNode.StateIndex := Index;

    Category.FDescription := Category.FDescription + ToTableLine(DemoName, DemoDescription, IntToStr(integer(DemoNode.Data)));
  end;
end;

procedure TDemos.Clear;
var
  i: integer;
begin
  for i := 0 to FDemoTreeNodes.Count - 1 do
    if FDemoTreeNodes[i].Data <> nil then
      TDemo(FDemoTreeNodes[i].Data).Free;
end;

function TDemos.GetSelectedDemo: TDemo;
begin
  if FSelectedDemo <> nil then
    Result := FSelectedDemo
  else
    raise Exception.Create('No selected demo');
end;

function TDemos.GetDemoIndex(AbsoluteIndex: integer): integer;
var
  i: integer;
begin
  Result := -1;
  for i := 0 to FDemoTreeNodes.Count - 1 do
    if integer(FDemoTreeNodes[i].Data) = AbsoluteIndex then begin
      if FDemoTreeNodes[i].Data <> nil then
        Result := i;
      Break;
    end;
end;

function TDemos.GetItems(Index: integer): TDemo;
var
  i: integer;
begin
  i := GetDemoIndex(Index);
  if i >= 0 then
    Result := TDemo(FDemoTreeNodes[i].Data)
  else
    Result := nil;
//    raise Exception.Create('Wrong demo index');
end;

function TDemos.WrapDescription(Demo: TDemo): TStrings;
var
  CatNode, RootNode: TTreeNode;

  procedure DoWrapDescription(Node: TTreeNode);
  begin
    if Node <> nil then begin
      CatNode := Node.getFirstChild;
      if CatNode <> nil then begin
        if TDemo(CatNode.Data).FDemoType = dtDemo then
          WriteTable(TDemo(Node.Data).Name, TDemo(Node.Data).Description, Result)
        else
          DoWrapDescription(CatNode);
      end;
      CatNode := Node.getNextSibling;
      if CatNode <> nil then
        DoWrapDescription(CatNode);
    end;
  end;

begin
  Result := TStringList.Create;
{$IFDEF WEB}
  Result.Add(HTMLHeader);
  Result.Add('<h3 class="dxH3" style = "text-align:center">' + MainDemosHeader + '</h3>');
{$ELSE}
  Result.Add('');
  Result.Add('     ' + MainDemosHeader);
  Result.Add('');
{$ENDIF}
  if Demo <> GetItems(integer(FDemoTreeNodes.GetFirstNode.Data)) then         //RootCategory
    WriteTable(Demo.Name, Demo.Description, Result)
  else begin
    RootNode := FDemoTreeNodes.GetFirstNode;
    DoWrapDescription(RootNode.getFirstChild);
{$IFDEF WEB}
    Result.Add('<h3 class="dxH3" style = "text-align:center"><br>' + SupplementaryDemosHeader + '</h3>');
    Result.Add('<p Align="center"><b>' + SupplementaryDemosNote + '</p></b>');
{$ENDIF}

    Result.AddStrings(FSupplementaryDemosDescription);
  end;
{$IFDEF WEB}
  Result.Add(HTMLFooter);
{$ENDIF}
end;

function TDemos.SelectDemo(DemoIndex: integer): TDemo;  //Init and show demo by DemoIndex
var
  Descriptions: TStrings;
begin
  Result := GetItems(DemoIndex);
  if (FSelectedDemo <> nil) and (Result <> FSelectedDemo) then
    if FSelectedDemo.DemoType <> dtCategory then
      FSelectedDemo.FreeFrame //In case of demo selection change we should free demo frame except category description
    else
      FSelectedDemo.Frame.Hide;
  FSelectedDemo := Result;
  with FSelectedDemo do
    if FFrame = nil then begin
      FFrame := FFrameClass.Create(nil);
      FFrame.Selector := Result.Selector;
      if DemoType = dtCategory then begin
        Descriptions := WrapDescription(FSelectedDemo);
        try
          TCategoryFrame(FFrame).SetDemoDescriptions(Descriptions);
        finally
          Descriptions.Free;
        end;
      end;
    end
    else
      FFrame.Show;
end;

{TDemo}

constructor TDemo.Create(Name, Hint, Description: string; DemoType: TDemoType; FrameClass: TDemoFrameClass; Selector: integer; FileName: string = ''; Folder: string = '');
begin
  inherited Create;

  FName := Name;
  if FileName = '' then
    FFileName := Name
  else
    FFileName := FileName;
  FHint := Hint;
  FDescription := Description;
  FFrameClass := FrameClass;
  FDemoType := DemoType;
  FSelector := Selector;
  if Folder = '' then
    FFolder := Name
  else
    FFolder := Folder;
end;

destructor TDemo.Destroy;
begin
  FreeFrame;

  inherited;
end;

procedure TDemo.DoLoadDemoPiece(Pattern: string; Strings: TStrings);
var
  lL: TStrings;
  i: Integer;
begin
  Strings.BeginUpdate;
  try
    lL := TStringList.Create;
    try
      Strings.Text := '';
      LoadDemoCode(lL);
      if lL.Count = 0 then
        Exit;

      i := 0;
      while (lL.Count > 0) and (i < lL.Count) and (pos(Pattern, UpperCase(lL[i])) <= 0) do
        Inc(i);
      if i < lL.Count then begin
        repeat
          Strings.Add(lL[i]);
          Inc(i);
        until (i = (lL.Count - 1)) or (UpperCase(Trim(lL[i])) = 'END.') or (pos(Copy(Pattern, 1, pos('.', Pattern)), UpperCase(lL[i])) > 0);
        i := Strings.Count - 1;
        while (i > 0) and (Trim(Strings[i]) = '') do begin
          Strings.Delete(i);
          Dec(i);
        end;
        while i > 0 do begin
          if (pos('INHERITED',     UpperCase(Strings[i])) > 0) or
             (pos('SHOWDEMOPIECE', UpperCase(Strings[i])) > 0)
          then begin
            Strings.Delete(i);
            Dec(i);
            while (i > 0) and (Trim(Strings[i]) = '') do begin
              Strings.Delete(i);
              Dec(i);
            end;
          end
          else
            Dec(i);
        end;
      end;
    finally
      lL.Free;
    end;
  finally
    if Strings.Count > 0 then
      Strings.Delete(0);
    Strings.EndUpdate;
  end;
end;

function TDemo.GetFullDFMName: string;
begin
{$IFDEF CLR}
  Result := Format('%s\%s\%s.nfm', [ExtractFilePath(Application.ExeName), FFrame.UnitFolder, FFrame.UnitName]);
{$ELSE}
{$IFDEF FPC}
  Result := Format('%s\%s\%s.lfm', [ExtractFilePath(Application.ExeName), FFrame.UnitFolder, FFrame.UnitName]);
{$ELSE}
  Result := Format('%s\%s\%s.dfm', [ExtractFilePath(Application.ExeName), FFrame.UnitFolder, FFrame.UnitName]);
{$ENDIF}
{$ENDIF}
end;

function TDemo.GetFullUnitName: string;
begin
{$IFDEF MSWINDOWS}
  Result := Format('%s\%s\%s.pas', [ExtractFilePath(Application.ExeName), FFrame.UnitFolder, FFrame.UnitName]);
{$ELSE}
  Result := Format('%s/%s/%s.pas', [ExtractFilePath(Application.ExeName), FFrame.UnitFolder, FFrame.UnitName]);
{$ENDIF}
end;

procedure TDemo.LoadDemoCode(Strings: TStrings);
var
  FileName: string;
begin
  if DemoType = dtCategory then
    Strings.Clear
  else begin
    FileName := GetFullUnitName;

    if FileExists(FileName) then
      Strings.LoadFromFile(FileName)
    else
      Strings.Clear;
  end;
end;

procedure TDemo.LoadFormCode(Strings: TStrings);
var
  FileName: string;
begin
  if DemoType = dtCategory then
    Strings.Clear
  else begin
    FileName := GetFullDFMName;

    if FileExists(FileName) then
      Strings.LoadFromFile(FileName)
    else
      Strings.Clear;
  end;
end;

procedure TDemo.LoadDemoPiece(Button: TButton; Strings: TStrings);
begin
  DoLoadDemoPiece('PROCEDURE ' + UpperCase(FFrame.ClassName) + '.' + UpperCase(Button.Name) + 'CLICK(', Strings);
end;

procedure TDemo.LoadDemoPiece(ProcedureName: string; Strings: TStrings);
begin
  DoLoadDemoPiece(' ' + UpperCase(FFrame.ClassName) + '.' + UpperCase(ProcedureName), Strings);
end;

procedure TDemo.OpenDemoFolder;
var
  FolderName: string;
begin
  if DemoType = dtDemo then begin
    FolderName := ExtractFilePath(Application.ExeName) + FFolder;
    ShellExecute(0, 'open', {$IFNDEF CLR}PChar{$ENDIF}(FolderName), '', '.', SW_SHOW);
  end;
end;

procedure TDemo.FreeFrame;
begin
  if FFrame <> nil then
    FFrame.Finalize;

  FFrame.Free;
  FFrame := nil;
end;


end.
