unit DataContextFrame;

{$I EntityDAC.Demo.inc}

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Grids, Forms,
  Dialogs, StdCtrls, ExtCtrls,
  EntityDAC.Context,
  EntityDAC.Dictionary,
  EntityDAC.Entity,
  EntityDAC.EntityAttributes,
  EntityDAC.EntityContext,
  EntityDAC.Enumerable,
  EntityDAC.Linq,
  EntityDAC.MetaData,
  DemoFrame;


type
  TDataContextFrame = class(TDemoFrame)
    pnTop: TPanel;
    mmSource: TMemo;
    pnButtons: TPanel;
  protected
    function GetUnitFolder: string; override;

    procedure EntityToMemo(Entity: TEntity; Memo: TMemo); overload;
    procedure EntitiesToMemo(Entities: IEntityEnumerable; Memo: TMemo); overload;
    procedure EntitiesToGrid(Entities: IEntityEnumerable; Grid: TStringGrid); overload;
  {$IFDEF USE_GENERICS}
    procedure EntityToMemo(Entity: TObject; Memo: TMemo); overload;
    procedure EntitiesToMemo(Entities: IObjectEnumerable; Memo: TMemo); overload;
    procedure EntitiesToMemo<T: class>(Entities: IObjectEnumerable<T>; Memo: TMemo); overload;
    procedure EntitiesToGrid(Entities: IObjectEnumerable; Grid: TStringGrid); overload;
    procedure EntitiesToGrid<T: class>(Entities: IObjectEnumerable<T>; Grid: TStringGrid); overload;
    procedure EntitiesToGrid<T: class>(Entities: IEntityEnumerable<T>; Grid: TStringGrid); overload;
  {$ENDIF}

    procedure ShowDemoPiece(Button: TButton); overload;
    procedure ShowDemoPiece(Button: TObject); overload;
  public
    procedure Initialize; override;
    procedure Finalize; override;
  end;

implementation

{$R *.dfm}

uses
  DB,
{$IFDEF USE_GENERICS}
  Rtti,
  EntityDAC.MetaEntity,
  EntityDAC.RttiUtils,
{$ENDIF}  
  EntityDAC.Values,
  EntityDAC.Utils,
  DemoForm;

type
  TInternalEDValue = class(TEDValue);

{ TDataContextFrame }

function TDataContextFrame.GetUnitFolder: string;
begin
  Result := 'DataContext';
end;

procedure TDataContextFrame.Initialize;
begin
  inherited;
end;

procedure TDataContextFrame.EntityToMemo(Entity: TEntity; Memo: TMemo);
var
  i: integer;
begin
  Memo.Lines.Add('---');
  if Entity <> nil then begin
    Memo.Lines.Add('MetaType: ' + Entity.MetaType.Name);
    for i := 0 to Entity.MetaType.MetaAttributes.Count - 1 do
      Memo.Lines.Add(Entity.MetaType.MetaAttributes[i].Name + ' = ' + Entity.Attributes[i].ToString);
  end
  else
    Memo.Lines.Add('NOT FOUND');
  Memo.Lines.Add('---');
end;

procedure TDataContextFrame.EntitiesToMemo(Entities: IEntityEnumerable; Memo: TMemo);
var
  i: integer;
  Entity: TEntity;
  str: string;
begin
  Memo.Lines.Add('---');
  if Entities.MetaType is TUnmappedMetaType then
    Memo.Lines.Add('MetaType: LINQ custom')
  else
    Memo.Lines.Add('MetaType: ' + Entities.MetaType.Name);
  Memo.Lines.Add('---');
  for Entity in Entities do begin
    str := '';
    for i := 0 to Entity.MetaType.MetaAttributes.Count - 1 do begin
      if str <> '' then
        str := str + ' |-| ';
      str := str + Entity.MetaType.MetaAttributes[i].Name + ' = ' + Entity.Attributes[i].ToString;
    end;
    Memo.Lines.Add(str);
  end;
  Memo.Lines.Add('---');
end;

procedure TDataContextFrame.EntitiesToGrid(Entities: IEntityEnumerable; Grid: TStringGrid);
var
  i, j: integer;
  MetaType: TMetaType;
begin
  MetaType := Entities.MetaType;
  Grid.ColCount := MetaType.MetaAttributes.Count;
  Grid.RowCount := Entities.Count + 1;

  for i := 0 to MetaType.MetaAttributes.Count - 1 do
    Grid.Cells[i, 0] := MetaType.MetaAttributes[i].Name;

  for i := 0 to Entities.Count - 1 do
    for j  := 0 to MetaType.MetaAttributes.Count - 1 do
      if Entities[i].Attributes[j].HasValue then
        Grid.Cells[j, i + 1] := Entities[i].Attributes[j].AsString
      else  
        Grid.Cells[j, i + 1] := '';
end;

{$IFDEF USE_GENERICS}
procedure TDataContextFrame.EntityToMemo(Entity: TObject; Memo: TMemo);
var
  i: integer;
  MetaEntity: TMetaEntity;
  Column: TColumnProperties;
  Value: TValue;
  EDValue: TEDValue;
begin
  Memo.Lines.Add('---');
  MetaEntity := ObjectMetaEntityManager.GetMetaEntity(Entity.ClassType);
  if MetaEntity <> nil then begin
    Memo.Lines.Add('MetaType: ' + MetaEntity.MetaType.Name);
    for i := 0 to MetaEntity.MetaType.MetaAttributes.Count - 1 do begin
      Column := TColumnProperties(MetaEntity.ColumnsByMetaAttribute[MetaEntity.MetaType.MetaAttributes[i]]);
      EDValue := TEDValue.CreateValue(Column.MetaAttribute.DataType, Column.IsNullable);
      try
        Value := TRttiUtils.GetInstanceMemberValue(Entity, Column.RttiMember, Column.IsNullable);
        TInternalEDValue(EDValue).ReadFromProperty(Value);
        Memo.Lines.Add(MetaEntity.MetaType.MetaAttributes[i].Name + ' = ' + VarToStr(EDValue.AsVariant));
      finally
        EDValue.Free;
      end;
    end;
  end
  else
    Memo.Lines.Add('NOT FOUND');
  Memo.Lines.Add('---');
end;

procedure TDataContextFrame.EntitiesToMemo(Entities: IObjectEnumerable; Memo: TMemo);
var
  i: integer;
  Obj: TObject;
  MetaType: TMetaType;
  Entity: TEntity;
  str: string;
  MetaEntity: TMetaEntity;
  Column: TColumnProperties;
  Value: TValue;
  EDValue: TEDValue;
begin
  MetaType := Entities.MetaType;

  if MetaType is TUnmappedMetaType then
    MetaEntity := nil
  else
    MetaEntity := TMappedMetaType(MetaType).MetaEntity as TMetaEntity;

  Memo.Lines.Add('---');
  for Obj in Entities do begin
    if Obj is TEntity then begin
      Entity := TEntity(Obj);
      str := '';
      for i := 0 to Entity.MetaType.MetaAttributes.Count - 1 do begin
        if str <> '' then
          str := str + ' |-| ';
        str := str + Entity.MetaType.MetaAttributes[i].Name + ' = ' + Entity.Attributes[i].ToString;
      end;
      Memo.Lines.Add(str);
    end
    else begin
      str := '';
      if MetaEntity = nil then
        Memo.Lines.Add('The entity = nil')
      else
        for i := 0 to MetaType.MetaAttributes.Count - 1 do begin
          if str <> '' then
            str := str + ' |-| ';
          Column := TColumnProperties(MetaEntity.ColumnsByMetaAttribute[MetaType.MetaAttributes[i]]);
          Value := TRttiUtils.GetInstanceMemberValue(Obj, Column.RttiMember, Column.IsNullable);
          EDValue := TEDValue.CreateValue(Column.MetaAttribute.DataType, Column.IsNullable);
          try
            TInternalEDValue(EDValue).ReadFromProperty(Value);
            str := str + MetaType.MetaAttributes[i].Name + ' = ' + VarToStr(EDValue.AsVariant)
          finally
            EDValue.Free;
          end;
        end;
      Memo.Lines.Add(str);
    end;
  end;
  Memo.Lines.Add('---');
end;

procedure TDataContextFrame.EntitiesToMemo<T>(Entities: IObjectEnumerable<T>; Memo: TMemo);
begin
  EntitiesToMemo(Entities as IObjectEnumerable, Memo);
end;

procedure TDataContextFrame.EntitiesToGrid(Entities: IObjectEnumerable; Grid: TStringGrid);
var
  i, j: integer;
  Obj: TObject;
  Entity: TEntity;
  MetaType: TMetaType;
  MetaEntity: TMetaEntity;
  Column: TColumnProperties;
  Value: TValue;
  EDValue: TEDValue;
begin
  MetaType := Entities.MetaType;

  if MetaType is TUnmappedMetaType then
    MetaEntity := nil
  else
    MetaEntity := TMappedMetaType(MetaType).MetaEntity as TMetaEntity;

  Grid.ColCount := MetaType.MetaAttributes.Count;
  Grid.RowCount := Entities.Count + 1;

  for i := 0 to MetaType.MetaAttributes.Count - 1 do
    Grid.Cells[i, 0] := MetaType.MetaAttributes[i].Name;

  for i := 0 to Entities.Count - 1 do begin
    Obj := Entities[i];
    if Obj is TEntity then begin
      Entity := TEntity(Obj);
      for j  := 0 to MetaType.MetaAttributes.Count - 1 do
        Grid.Cells[j, i + 1] := Entity.Attributes[j].AsString;
    end
    else begin
      for j := 0 to MetaType.MetaAttributes.Count - 1 do begin
        Column := TColumnProperties(MetaEntity.ColumnsByMetaAttribute[MetaType.MetaAttributes[j]]);
        Value := TRttiUtils.GetInstanceMemberValue(Obj, Column.RttiMember, Column.IsNullable);
        EDValue := TEDValue.CreateValue(Column.MetaAttribute.DataType, Column.IsNullable);
        try
          TInternalEDValue(EDValue).ReadFromProperty(Value);
          Grid.Cells[j, i + 1] := VarToStr(EDValue.AsVariant);
        finally
          EDValue.Free;
        end;
      end;
    end;
  end;
end;

procedure TDataContextFrame.EntitiesToGrid<T>(Entities: IObjectEnumerable<T>; Grid: TStringGrid);
begin
  EntitiesToGrid(Entities as IObjectEnumerable, Grid);
end;

procedure TDataContextFrame.EntitiesToGrid<T>(Entities: IEntityEnumerable<T>; Grid: TStringGrid);
begin
  EntitiesToGrid(Entities as IEntityEnumerable, Grid);
end;
{$ENDIF}

procedure TDataContextFrame.ShowDemoPiece(Button: TButton);
begin
  inherited ShowDemoPiece(Button, mmSource);
end;

procedure TDataContextFrame.ShowDemoPiece(Button: TObject);
begin
  ShowDemoPiece(Button as TButton);
end;

procedure TDataContextFrame.Finalize;
begin
  inherited;
end;

end.
