unit uEntityDemoForm;

{$I EntityDAC.Demo.inc}

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ImgList, Menus, StdCtrls, Buttons, ExtCtrls, ComCtrls, ToolWin,
  EntityDAC.MetaData,
{$IFDEF VER14P}
  EntityDAC.MetaEntity,
{$ENDIF}  
  EntityDAC.Context,
  EntityDAC.DataProvider,
  EntityDAC.EntityConnection,
  EntityDAC.SQLDialect,
  DemoForm;

type
  TEntityDemoForm = class(TDemoForm)
    procedure cbProviderChange(Sender: TObject);
    procedure cbDialectChange(Sender: TObject);
    procedure lbAboutClick(Sender: TObject); override;
  protected
    FIniFileName: string;

    function ApplicationTitle: string; override;
    function ProductName: string; override;
    procedure RegisterDemos; override;

    procedure PopulateProvidersList; override;
    function LoadConnectionSettings: boolean; override;
    procedure SaveConnectionSettings; override;
    procedure SetDefaultConnectionSettings; override;
    function GetProvider: TDataProvider; override;

    procedure BeforeConnect(Sender: TObject);
    procedure DoExecScript(const FileName: string);
  public
    function ProductColor: TColor; override;
    procedure ExecCreateScript; override;
    procedure ExecDropScript; override;
    function GetConnection: TEntityConnection; override;
    procedure Connect(Connection: TEntityConnection); override;
    procedure Disconnect; override;
  end;

var
  EntityDemoForm: TEntityDemoForm;

implementation

{$R *.dfm}

uses
  IniFiles,
  LINQFrame, LINQFrameSelect, LINQFrameWhere, LINQFrameOrder, LINQFrameGroup,
  LINQFrameJoin, LINQFrameSet, LINQFramePart, LINQFrameAgg, LINQFrameQuant,
  GetEntityFrameM, {$IFDEF USE_GENERICS}GetEntityFrameE, GetEntityFrameG, GetEntityFrameO,{$ENDIF}
  GetEntitiesFrameM, {$IFDEF USE_GENERICS}GetEntitiesFrameE, GetEntitiesFrameG, GetEntitiesFrameO,{$ENDIF}
  GetEntityReferenceFrameM, {$IFDEF USE_GENERICS}GetEntityReferenceFrameE, GetEntityReferenceFrameG, GetEntityReferenceFrameO,{$ENDIF}
  GetEntityCollectionFrameM, {$IFDEF USE_GENERICS}GetEntityCollectionFrameE, GetEntityCollectionFrameG, GetEntityCollectionFrameO,{$ENDIF}
  NullableTypesFrameM, {$IFDEF USE_GENERICS}NullableTypesFrameE, NullableTypesFrameG,{$ENDIF}
  CreateEntityFrameM, {$IFDEF USE_GENERICS}CreateEntityFrameE, CreateEntityFrameG, CreateEntityFrameO,{$ENDIF}
  ModifyEntityFrameM, {$IFDEF USE_GENERICS}ModifyEntityFrameE, ModifyEntityFrameG, ModifyEntityFrameO,{$ENDIF}
  DeleteEntityFrameM, {$IFDEF USE_GENERICS}DeleteEntityFrameE, DeleteEntityFrameG, DeleteEntityFrameO,{$ENDIF}
  ModifyCascadeFrameM, {$IFDEF USE_GENERICS}ModifyCascadeFrameE, ModifyCascadeFrameG,{$ENDIF}
  DeleteCascadeFrameM, {$IFDEF USE_GENERICS}DeleteCascadeFrameE, DeleteCascadeFrameG,{$ENDIF}
  SubmitChangesFrameM, {$IFDEF USE_GENERICS}SubmitChangesFrameE, SubmitChangesFrameG,{$ENDIF}
{$IFNDEF EXPRESS}
  DataValidationFrameM, {$IFDEF USE_GENERICS}DataValidationFrameE, DataValidationFrameG,{$ENDIF}
  EntityToDataSetFrame, EntitiesToDataSetFrame,
  EntityDataSetFilterFrame, EdtityDataSetMDFrame, EntityDataSetExpressionFrame,
{$ENDIF}
{$IFDEF VER17P}
  DbxSqlite,
{$ENDIF}
  Demo.MetaData, EntityDacAbout;

{ TEntityDemoForm }

function TEntityDemoForm.ApplicationTitle: string;
begin
  Result := 'EntityDAC Demo';
end;

function TEntityDemoForm.GetConnection: TEntityConnection;
begin
  if FConnection = nil then begin
    FConnection := TEntityConnection.Create;
    FConnection.BeforeConnect := BeforeConnect;
    FConnection.Provider := GetProvider;
  end;

  Result := inherited GetConnection;
end;

procedure TEntityDemoForm.cbDialectChange(Sender: TObject);
begin
  if cbDialect.Enabled then begin
    Disconnect;

    GetConnection.DialectName := cbDialect.Text;
  end;
end;

procedure TEntityDemoForm.cbProviderChange(Sender: TObject);
var
  Connection: TEntityConnection;
begin
  Disconnect;

  Connection := GetConnection;
  Connection.Provider := GetProvider;
  cbDialect.Enabled := (Connection.Provider <> nil) and Connection.Provider.MultiDialect;
  cbDialect.ItemIndex := cbDialect.Items.IndexOf(Connection.DialectName);
end;

procedure TEntityDemoForm.Connect(Connection: TEntityConnection);
begin
  Connection.Connect;

  SaveConnectionSettings;
end;

procedure TEntityDemoForm.Disconnect;
begin
  if Assigned(FConnection) then
    FConnection.Disconnect;

  inherited;
end;

procedure TEntityDemoForm.PopulateProvidersList;
var
  i: integer;
begin
  cbProvider.Items.Clear;
  cbDialect.Items.Clear;

  for i := 0 to DataProviderManager.Count - 1 do
    cbProvider.Items.AddObject(DataProviderManager[i].ProviderName, DataProviderManager[i]);

  for i := 0 to SQLDialectManager.Count - 1 do
    cbDialect.Items.Add(SQLDialectManager[i].Name);

  if cbProvider.Items.Count > 0 then
    cbProvider.ItemIndex := 0;
  cbProviderChange(nil);
end;

function TEntityDemoForm.LoadConnectionSettings: boolean;
var
  IniFile: TIniFile;
  Value: string;
  i: integer;
begin
  if FIniFileName = '' then
    FIniFileName := ChangeFileExt(ParamStr(0), '.ini');

  Result := False;
  if not FileExists(FIniFileName) then
    Exit;

  IniFile := TIniFile.Create(FIniFileName);
  try
    Value := IniFile.ReadString('Connection', 'ProviderName', '');
    if Value = '' then
      Exit;
    cbProvider.ItemIndex := cbProvider.Items.IndexOf(Value);
    cbProviderChange(cbProvider);

    Value := IniFile.ReadString('Connection', 'DialectName', '');
    if Value = '' then
      Exit;
    cbDialect.ItemIndex := cbDialect.Items.IndexOf(Value);
    cbDialectChange(cbDialect);

    edConnectionString.Items.Clear;
    for i := 0 to IniFile.ReadInteger('History', 'Count', 0) - 1 do
      edConnectionString.Items.Add(IniFile.ReadString('History', 'Item' + IntToStr(i + 1), ''));

    Value := IniFile.ReadString('Connection', 'ConnectionString', '');
    if Value = '' then
      Exit;
    edConnectionString.Text :=  Value;

    Result := True;
  finally
    IniFile.Free;
  end;
end;

procedure TEntityDemoForm.SaveConnectionSettings;
var
  IniFile: TIniFile;
  Index, Cnt, i: integer;
  ConnectionString: string;
begin
  IniFile := TIniFile.Create(FIniFileName);
  try
    IniFile.WriteString('Connection', 'ProviderName', cbProvider.Text);
    IniFile.WriteString('Connection', 'DialectName', cbDialect.Text);

    IniFile.EraseSection('History');

    ConnectionString := edConnectionString.Text;
    Index := edConnectionString.Items.IndexOf(ConnectionString);
    if Index < 0 then
      edConnectionString.Items.Insert(0, ConnectionString)
    else begin
      edConnectionString.Items.Move(Index, 0);
      edConnectionString.Text := ConnectionString;
    end;

    Cnt := edConnectionString.Items.Count;
    if Cnt > 10 then
      Cnt := 10;
    IniFile.WriteInteger('History', 'Count', Cnt);
    for i := 0 to Cnt - 1 do
      IniFile.WriteString('History', 'Item' + IntToStr(i + 1), edConnectionString.Items[i]);

    IniFile.WriteString('Connection', 'ConnectionString', edConnectionString.Text);
  finally
    IniFile.Free;
  end;
end;

procedure TEntityDemoForm.SetDefaultConnectionSettings;
begin
  if DataProviderManager.FindDataProviderByName('UniDAC') <> nil then begin
    cbProvider.ItemIndex := cbProvider.Items.IndexOf('UniDAC');
    cbProviderChange(cbProvider);
    cbDialect.ItemIndex := cbDialect.Items.IndexOf('SQLite');
    cbDialectChange(cbDialect);
    edConnectionString.Text := 'ProviderName=SQLite;Database=DB\SQLite\demo.db3;LoginPrompt=False';
  end
  else if DataProviderManager.FindDataProviderByName('LiteDAC') <> nil then begin
    cbProvider.ItemIndex := cbProvider.Items.IndexOf('LiteDAC');
    cbProviderChange(cbProvider);
    cbDialect.ItemIndex := cbDialect.Items.IndexOf('SQLite');
    cbDialectChange(cbDialect);
    edConnectionString.Text := 'Database=DB\SQLite\demo.db3;LoginPrompt=False';
{$IFDEF VER17P}
  end
  else begin
    cbProvider.ItemIndex := cbProvider.Items.IndexOf('dbExpress');
    cbProviderChange(cbProvider);
    cbDialect.ItemIndex := cbDialect.Items.IndexOf('SQLite');
    cbDialectChange(cbDialect);
    edConnectionString.Text := 'DriverName=Sqlite;Database=DB\SQLite\demo.db3;LoginPrompt=False';
{$ENDIF}
  end;

  SaveConnectionSettings;
end;

function TEntityDemoForm.GetProvider: TDataProvider;
begin
  if cbProvider.ItemIndex >= 0 then
    Result := TDataProvider(cbProvider.Items.Objects[cbProvider.ItemIndex])
  else
    Result := nil;
end;

procedure TEntityDemoForm.lbAboutClick(Sender: TObject);
begin
  inherited;
  EntityDacAboutForm.ShowModal;
end;

procedure TEntityDemoForm.BeforeConnect(Sender: TObject);
begin
  TEntityConnection(Sender).ConnectionString := edConnectionString.Text;
end;

procedure TEntityDemoForm.DoExecScript(const FileName: string);
var
  Script: TStrings;
  SQL: string;
  i: integer;
begin
  if FileExists(FileName) then begin
    Script := TStringList.Create;
    try
      try
        GetConnection.StartTransaction;
        Script.LoadFromFile(FileName);
        SQL := '';
        for i := 0 to Script.Count - 1 do begin
          if (i < Script.Count) and (Script[i] <> '') and (copy(Script[i], 1, 2) <> '--') then
            SQL := SQL + Script[i]
          else begin
            GetConnection.ExecuteSQL(SQL);
            SQL := '';
          end;
        end;
        if SQL <> '' then
          GetConnection.ExecuteSQL(SQL);
        GetConnection.CommitTransaction;
      except
        on e: Exception do begin
          if GetConnection.InTransaction then
            GetConnection.RollbackTransaction;
          raise;
        end;
      end;
    finally
      Script.Free;
    end;
  end
  else
    raise Exception.Create('The script does not exist');
end;

function TEntityDemoForm.ProductColor: TColor;
begin
  Result := clGray;
end;

procedure TEntityDemoForm.ExecCreateScript;
begin
  DoExecScript('Scripts\' + GetConnection.DialectName + '\create.sql');

  ShowMessage('Database created');
end;

procedure TEntityDemoForm.ExecDropScript;
begin
  DoExecScript('Scripts\' + GetConnection.DialectName + '\drop.sql');

  ShowMessage('Database dropped');
end;

function TEntityDemoForm.ProductName: string;
begin
  Result := 'EntityDAC Demo';
end;

procedure TEntityDemoForm.RegisterDemos;
begin
  Demos.RegisterCategory('EntityDAC Demo', 'EntityDAC Demo');
  Demos.RegisterCategory('Working with DataContext', 'Working with DataContext');

  Demos.RegisterCategory('Working with Code-Mapped Entities', 'Working with Code-Mapped Entities', 'Working with DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterCategory('Working with generics', 'Working with generics', 'Working with DataContext');
  Demos.RegisterCategory('Working with Attribute-Mapped Entities', 'Working with Attribute-Mapped Entities', 'Working with DataContext');
  Demos.RegisterCategory('Working with Attribute-Mapped Objects', 'Working with Attribute-Mapped Objects', 'Working with DataContext');
{$ENDIF}

  Demos.RegisterCategory('Working with EntityDataSet', 'Working with EntityDataSet');

  Demos.RegisterCategory('Working with LINQ queries', 'Working with LINQ queries');
  Demos.RegisterCategory('Projection Operators', 'Projection Operators', 'Working with LINQ queries');
  Demos.RegisterCategory('Restriction Operators', 'Restriction Operators', 'Working with LINQ queries');
  Demos.RegisterCategory('Ordering Operators', 'Ordering Operators', 'Working with LINQ queries');
  Demos.RegisterCategory('Grouping Operators', 'Grouping Operators', 'Working with LINQ queries');
  Demos.RegisterCategory('Join Operators', 'Join Operators', 'Working with LINQ queries');
  Demos.RegisterCategory('Set Operators', 'Set Operators', 'Working with LINQ queries');
  Demos.RegisterCategory('Partitioning Operators', 'Partitioning Operators', 'Working with LINQ queries');
  Demos.RegisterCategory('Aggregate Operators', 'Aggregate Operators', 'Working with LINQ queries');
  Demos.RegisterCategory('Quantifiers', 'Quantifiers', 'Working with LINQ queries');

  Demos.RegisterDemo('Nullable Types', 'Nullable Types', '', 'Working with Code-Mapped Entities', TNullableTypesFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Nullable Types', 'Nullable Types as Generic', '', 'Working with generics', TNullableTypesFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Nullable Types', 'Nullable Types', '', 'Working with Attribute-Mapped Entities', TNullableTypesFrameE, 0, 1, '', 'DataContext');
{$ENDIF}

  Demos.RegisterDemo('Get Single Entity', 'Get Single Entity', '', 'Working with Code-Mapped Entities', TGetEntityFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Get Single Entity', 'Get Single Entity as Generic', '', 'Working with generics', TGetEntityFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Get Single Entity', 'Get Single Entity', '', 'Working with Attribute-Mapped Objects', TGetEntityFrameO, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Get Single Entity', 'Get Single Entity', '', 'Working with Attribute-Mapped Entities', TGetEntityFrameE, 0, 1, '', 'DataContext');
{$ENDIF}
  Demos.RegisterDemo('Get Entity Collection', 'Get Entity Collection', '', 'Working with Code-Mapped Entities', TGetEntitiesFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Get Entity Collection', 'Get Entity Collection as Generic', '', 'Working with generics', TGetEntitiesFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Get Entity Collection', 'Get Entity Collection', '', 'Working with Attribute-Mapped Objects', TGetEntitiesFrameO, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Get Entity Collection', 'Get Entity Collection', '', 'Working with Attribute-Mapped Entities', TGetEntitiesFrameE, 0, 1, '', 'DataContext');
{$ENDIF}
  Demos.RegisterDemo('Get Entity By Reference', 'Get Entity By Reference', '', 'Working with Code-Mapped Entities', TGetEntityReferenceFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Get Entity By Reference', 'Get Entity By Reference as Generic', '', 'Working with generics', TGetEntityReferenceFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Get Entity By Reference', 'Get Entity By Reference', '', 'Working with Attribute-Mapped Objects', TGetEntityReferenceFrameO, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Get Entity By Reference', 'Get Entity By Reference', '', 'Working with Attribute-Mapped Entities', TGetEntityReferenceFrameE, 0, 1, '', 'DataContext');
{$ENDIF}
  Demos.RegisterDemo('Get Entities By Collection', 'Get Entity By Collection', '', 'Working with Code-Mapped Entities', TGetEntityCollectionFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Get Entities By Collection', 'Get Entity By Collection as Generic', '', 'Working with generics', TGetEntityCollectionFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Get Entities By Collection', 'Get Entity By Collection', '', 'Working with Attribute-Mapped Objects', TGetEntityCollectionFrameO, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Get Entities By Collection', 'Get Entity By Collection', '', 'Working with Attribute-Mapped Entities', TGetEntityCollectionFrameE, 0, 1, '', 'DataContext');
{$ENDIF}
  Demos.RegisterDemo('Create New Entity', 'Create New Entity', '', 'Working with Code-Mapped Entities', TCreateEntityFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Create New Entity', 'Create New Entity as Generic', '', 'Working with generics', TCreateEntityFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Create New Entity', 'Create New Entity', '', 'Working with Attribute-Mapped Objects', TCreateEntityFrameO, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Create New Entity', 'Create New Entity', '', 'Working with Attribute-Mapped Entities', TCreateEntityFrameE, 0, 1, '', 'DataContext');
{$ENDIF}
  Demos.RegisterDemo('Modify Entity', 'Modify Entity', '', 'Working with Code-Mapped Entities', TModifyEntityFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Modify Entity', 'Modify Entity as Generic', '', 'Working with generics', TModifyEntityFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Modify Entity', 'Modify Entity', '', 'Working with Attribute-Mapped Objects', TModifyEntityFrameO, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Modify Entity', 'Modify Entity', '', 'Working with Attribute-Mapped Entities', TModifyEntityFrameE, 0, 1, '', 'DataContext');
{$ENDIF}
  Demos.RegisterDemo('Delete Entity', 'Delete Entity', '', 'Working with Code-Mapped Entities', TDeleteEntityFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Delete Entity', 'Delete Entity as Generic', '', 'Working with generics', TDeleteEntityFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Delete Entity', 'Delete Entity', '', 'Working with Attribute-Mapped Objects', TDeleteEntityFrameO, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Delete Entity', 'Delete Entity', '', 'Working with Attribute-Mapped Entities', TDeleteEntityFrameE, 0, 1, '', 'DataContext');
{$ENDIF}
  Demos.RegisterDemo('Modify Entity Cascade', 'Modify Entity Cascade', '', 'Working with Code-Mapped Entities', TModifyCascadeFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Modify Entity Cascade', 'Modify Entity Cascade as Generic', '', 'Working with generics', TModifyCascadeFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Modify Entity Cascade', 'Modify Entity Cascade', '', 'Working with Attribute-Mapped Entities', TModifyCascadeFrameE, 0, 1, '', 'DataContext');
{$ENDIF}
  Demos.RegisterDemo('Delete Entity Cascade', 'Delete Entity Cascade', '', 'Working with Code-Mapped Entities', TDeleteCascadeFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Delete Entity Cascade', 'Delete Entity Cascade as Generic', '', 'Working with generics', TDeleteCascadeFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Delete Entity Cascade', 'Delete Entity Cascade', '', 'Working with Attribute-Mapped Entities', TDeleteCascadeFrameE, 0, 1, '', 'DataContext');
{$ENDIF}
  Demos.RegisterDemo('Submit All Changes', 'Submit All Changes', '', 'Working with Code-Mapped Entities', TSubmitChangesFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Submit All Changes', 'Submit All Changes as Generic', '', 'Working with generics', TSubmitChangesFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Submit All Changes', 'Submit All Changes', '', 'Working with Attribute-Mapped Entities', TSubmitChangesFrameE, 0, 1, '', 'DataContext');
{$ENDIF}
{$IFNDEF EXPRESS}
  Demos.RegisterDemo('Data Validation', 'Data Validation', '', 'Working with Code-Mapped Entities', TDataValidationFrameM, 0, 1, '', 'DataContext');
{$IFDEF USE_GENERICS}
  Demos.RegisterDemo('Data Validation', 'Data Validation as Generic', '', 'Working with generics', TDataValidationFrameG, 0, 1, '', 'DataContext');
  Demos.RegisterDemo('Data Validation', 'Data Validation', '', 'Working with Attribute-Mapped Entities', TDataValidationFrameE, 0, 1, '', 'DataContext');
{$ENDIF}
{$ENDIF}

{$IFNDEF EXPRESS}
  Demos.RegisterDemo('Single Entity to DataSet', 'Single entity to DataSet', '', 'Working with EntityDataSet', TEntityToDataSetFrame, 0, 1, '', 'EntityDataSet');
  Demos.RegisterDemo('Entity Collection to DataSet', 'Entity collection to DataSet', '', 'Working with EntityDataSet', TEntitiesToDataSetFrame, 0, 1, '', 'EntityDataSet');
  Demos.RegisterDemo('Apply Filter to EntityDatSet', 'Apply Filter to EntityDatSet', '', 'Working with EntityDataSet', TEntityDataSetFilterFrame, 0, 1, '', 'EntityDataSet');
  Demos.RegisterDemo('Master-Detail in EntityDataSet', 'Master-Detail in EntityDataSet', '', 'Working with EntityDataSet', TEdtityDataSetMDFrame, 0, 1, '', 'EntityDataSet');
  Demos.RegisterDemo('Expressions in EntityDataSet', 'Expressions in EntityDataSet', '', 'Working with EntityDataSet', TEntityDataSetExpressionFrame, 0, 1, '', 'EntityDataSet');
{$ENDIF}

  Demos.RegisterDemo('Select - Simple', 'Select - Simple', '', 'Projection Operators', TFrameLINQSelect, 0, 1, '', 'LINQ');
  Demos.RegisterDemo('Select - Complex', 'Select - Complex', '', 'Projection Operators', TFrameLINQSelect, 1, 1, '', 'LINQ');
  Demos.RegisterDemo('Select - Filtered', 'Select - Filtered', '', 'Projection Operators', TFrameLINQSelect, 2, 1, '', 'LINQ');
  Demos.RegisterDemo('Select - Many', 'Select - Many', '', 'Projection Operators', TFrameLINQSelect, 3, 1, '', 'LINQ');
  Demos.RegisterDemo('Where - Simple', 'Where - Simple', '', 'Restriction Operators', TFrameLINQWhere, 0, 1, '', 'LINQ');
  Demos.RegisterDemo('Where - Complex', 'Where - Complex', '', 'Restriction Operators', TFrameLINQWhere, 1, 1, '', 'LINQ');
  Demos.RegisterDemo('OrderBy - Simple', 'OrderBy - Simple', '', 'Ordering Operators', TFrameLINQOrder, 0, 1, '', 'LINQ');
  Demos.RegisterDemo('OrderBy - Descending', 'OrderBy - Descending', '', 'Ordering Operators', TFrameLINQOrder, 1, 1, '', 'LINQ');
  Demos.RegisterDemo('OrderBy - Multiple', 'OrderBy - Multiple', '', 'Ordering Operators', TFrameLINQOrder, 2, 1, '', 'LINQ');
  Demos.RegisterDemo('ThenBy - Simple', 'ThenBy - Simple', '', 'Ordering Operators', TFrameLINQOrder, 3, 1, '', 'LINQ');
  Demos.RegisterDemo('ThenBy - Descending', 'ThenBy - Descending', '', 'Ordering Operators', TFrameLINQOrder, 4, 1, '', 'LINQ');
  Demos.RegisterDemo('GroupBy - Simple', 'GroupBy - Simple', '', 'Grouping Operators', TFrameLINQGroup, 0, 1, '', 'LINQ');
  Demos.RegisterDemo('Join - Inner', 'Join - Inner', '', 'Join Operators', TFrameLINQJoin, 0, 1, '', 'LINQ');
  Demos.RegisterDemo('Join - Left', 'Join - Left', '', 'Join Operators', TFrameLINQJoin, 1, 1, '', 'LINQ');
  Demos.RegisterDemo('Join - Right', 'Join - Right', '', 'Join Operators', TFrameLINQJoin, 2, 1, '', 'LINQ');
  Demos.RegisterDemo('Join - Cross', 'Join - Cross', '', 'Join Operators', TFrameLINQJoin, 3, 1, '', 'LINQ');
  Demos.RegisterDemo('Join - Full', 'Join - Full', '', 'Join Operators', TFrameLINQJoin, 4, 1, '', 'LINQ');
  Demos.RegisterDemo('Join - Self', 'Join - Self', '', 'Join Operators', TFrameLINQJoin, 5, 1, '', 'LINQ');
  Demos.RegisterDemo('Set - Distinct', 'Set - Distinct', '', 'Set Operators', TFrameLINQSet, 0, 1, '', 'LINQ');
  Demos.RegisterDemo('Set - Union', 'Set - Union', '', 'Set Operators', TFrameLINQSet, 1, 1, '', 'LINQ');
  Demos.RegisterDemo('Set - Concat', 'Set - Concat', '', 'Set Operators', TFrameLINQSet, 2, 1, '', 'LINQ');
  Demos.RegisterDemo('Set - Except', 'Set - Except', '', 'Set Operators', TFrameLINQSet, 3, 1, '', 'LINQ');
  Demos.RegisterDemo('Set - Intersect', 'Set - Intersect', '', 'Set Operators', TFrameLINQSet, 4, 1, '', 'LINQ');
  Demos.RegisterDemo('Partitioning - Take', 'Partitioning - Take', '', 'Partitioning Operators', TFrameLINQPart, 0, 1, '', 'LINQ');
  Demos.RegisterDemo('Partitioning - Skip', 'Partitioning - Skip', '', 'Partitioning Operators', TFrameLINQPart, 1, 1, '', 'LINQ');
  Demos.RegisterDemo('Partitioning - Take & Skip', 'Partitioning - Take & Skip', '', 'Partitioning Operators', TFrameLINQPart, 2, 1, '', 'LINQ');
  Demos.RegisterDemo('Partitioning - First', 'Partitioning - First', '', 'Partitioning Operators', TFrameLINQPart, 3, 1, '', 'LINQ');
  Demos.RegisterDemo('Partitioning - ElementAt', 'Partitioning - ElementAt', '', 'Partitioning Operators', TFrameLINQPart, 4, 1, '', 'LINQ');
  Demos.RegisterDemo('Count - Results', 'Count - Results', '', 'Aggregate Operators', TFrameLINQAgg, 0, 1, '', 'LINQ');
  Demos.RegisterDemo('Count - Select', 'Count - Select', '', 'Aggregate Operators', TFrameLINQAgg, 1, 1, '', 'LINQ');
  Demos.RegisterDemo('Count - Where', 'Count - Where', '', 'Aggregate Operators', TFrameLINQAgg, 2, 1, '', 'LINQ');
  Demos.RegisterDemo('Sum - Results', 'Sum - Results', '', 'Aggregate Operators', TFrameLINQAgg, 3, 1, '', 'LINQ');
  Demos.RegisterDemo('Sum - Select', 'Sum - Select', '', 'Aggregate Operators', TFrameLINQAgg, 4, 1, '', 'LINQ');
  Demos.RegisterDemo('Any - Results', 'Any - Results', '', 'Quantifiers', TFrameLINQQuant, 0, 1, '', 'LINQ');
  Demos.RegisterDemo('Any - Condition', 'Any - Condition', '', 'Quantifiers', TFrameLINQQuant, 1, 1, '', 'LINQ');
  Demos.RegisterDemo('Any - Select', 'Any - Select', '', 'Quantifiers', TFrameLINQQuant, 2, 1, '', 'LINQ');
  Demos.RegisterDemo('All - Results', 'All - Results', '', 'Quantifiers', TFrameLINQQuant, 3, 1, '', 'LINQ');
  Demos.RegisterDemo('All - Select', 'All - Select', '', 'Quantifiers', TFrameLINQQuant, 4, 1, '', 'LINQ');
end;

initialization
  AutoRegisterProviders := True;

end.
