
//////////////////////////////////////////////////
//  MySQL Data Access Components
//  Copyright  1998-2025 Devart. All right reserved.
//  MyConnection Editor
//////////////////////////////////////////////////

{$IFNDEF CLR}

{$I MyDac.inc}

unit MyConnectionEditor;
{$ENDIF}
interface
uses
{$IFDEF MSWINDOWS}
  Windows, Messages,
{$ENDIF}
  SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, StdCtrls, ExtCtrls, Buttons, 
  {$IFDEF FPC}MaskEdit,{$ELSE}Mask,{$ENDIF}
{$IFDEF DBTOOLS}
  DBToolsClient,
{$ENDIF}
{$IFDEF FPC}
  LResources,
{$ENDIF}
  CRTypes, DBAccess, MyAccess,
  TypInfo, CREditor, DAConnectionEditor, CRFrame;

type
  TMyConnectionEditorForm = class(TDAConnectionEditorForm)
    lbDatabase: TLabel;
    edDatabase: TComboBox;
    lbPort: TLabel;
    edPort: TEdit;
    cbDirect: TCheckBox;
    cbEmbedded: TCheckBox;
    shEmbParams: TTabSheet;
    Label4: TLabel;
    edBaseDir: TEdit;
    btBaseDir: TBitBtn;
    btDataDir: TBitBtn;
    edDataDir: TEdit;
    Label7: TLabel;
    Label8: TLabel;
    edTempDir: TEdit;
    btTempDir: TBitBtn;
    btCharsetsDir: TBitBtn;
    edCharsetsDir: TEdit;
    Label9: TLabel;
    Label10: TLabel;
    edLanguage: TEdit;
    btLanguage: TBitBtn;
    btBinaryLogIndex: TBitBtn;
    edBinaryLogIndex: TEdit;
    Label12: TLabel;
    btBinaryLog: TBitBtn;
    edBinaryLog: TEdit;
    Label11: TLabel;
    cbInnoDBSafeBinLog: TCheckBox;
    cbFlush: TCheckBox;
    cbSkipInnoDB: TCheckBox;
    cbSkipGrantTables: TCheckBox;
    cbLogShortFormat: TCheckBox;
    btAdvanced: TButton;
    cbExisting: TComboBox;
    lbExisting: TLabel;
    procedure edPortExit(Sender: TObject);
    procedure edDatabaseDropDown(Sender: TObject);
    procedure cbDirectClick(Sender: TObject);
    procedure cbEmbeddedClick(Sender: TObject);
    procedure btAdvancedClick(Sender: TObject);
    procedure edPortChange(Sender: TObject);
    procedure edDatabaseExit(Sender: TObject);
    procedure cbExistingChange(Sender: TObject);
    procedure PageControlChanging(Sender: TObject;
      var AllowChange: Boolean);
  protected
  {$IFDEF DBTOOLS}
    function GetExistingConnectionComboBox: TComboBox; override;
    function GetConnectionCondition: string; override;
  {$ENDIF}

    function GetLocalConnection: TCustomMyConnection;

    procedure DoInit; override;
    procedure DoActivate; override;
    procedure DoSaveConnection; override;
    procedure FillInfo(InfoConnection: TCustomDAConnection); override;

    function GetFrameByInitProp: TCRFrame; override;

    procedure ConnToControls; override;
    procedure ShowState(Yellow: boolean = False); override;

    //procedure TrimParams(Params: TStrings);
    procedure ParamsToControls(Params: TStrings);
    procedure ControlsToParams(Params: TStrings);

    procedure DoPageControlChange(Sender: TObject); override;
  public
  {$IFNDEF CLR}
  {$IFNDEF FPC}
    procedure ReplaceEdit(var Edit: TWinControl);
  {$ENDIF}
    procedure SpinEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
  {$ENDIF}
    property LocalConnection: TCustomMyConnection read GetLocalConnection;
  end;

implementation

{$IFNDEF FPC}
{$IFDEF IDE}
{$R *.dfm}
{$ELSE}
{$R MyConnectionEditor.dfm}
{$ENDIF}
{$ELSE}
{$R *.lfm}
{$ENDIF}

uses
  CRFunctions, DAConsts,
  {$IFNDEF STD}MyEmbConnection, MyEmbConParamsEditor,{$ENDIF}
  DacVcl;

{ TMyConnectionEditorForm }

procedure TMyConnectionEditorForm.DoInit;
{$IFDEF MSWINDOWS}
{$IFNDEF CLR}
{$IFNDEF FPC}
var
  WinControl: TWinControl;
{$ENDIF}
{$ENDIF}
{$ENDIF}
begin

{$IFDEF MSWINDOWS}
{$IFNDEF CLR}
{$IFNDEF FPC}
  WinControl := edPort;
  ReplaceEdit(WinControl);
{$ENDIF}
{$ENDIF}
{$ELSE}
  meInfo.Font.Name := 'adobe-courier';
  meInfo.Font.Pitch := fpFixed;
  Label4.Height := 18;
  Label7.Height := 18;
  Label8.Height := 18;
  Label9.Height := 18;
  Label10.Height := 18;
  Label11.Height := 18;
  Label12.Height := 18;
  cbInnoDBSafeBinLog.Width := 217;
{$ENDIF}

  inherited;
  lbVersion.Caption := MyDACVersion + ' ';
  UpdateVersionPosition;

{$IFDEF BETA}
  lbEdition.Caption := 'Beta';
  lbEdition.Font.Color := clRed;
{$ELSE}
{$IFDEF RC}
  lbEdition.Caption := 'Release Candidate';
  lbEdition.Font.Color := clGreen;
{$ELSE}
{$IFDEF TRIAL}
  lbEdition.Caption := 'Trial Edition';
{$ELSE} //ORIGINAL
{$IFDEF STD}
  lbEdition.Caption := 'Standard Edition';
{$ELSE}
  lbEdition.Caption := 'Professional Edition';
{$ENDIF}
{$ENDIF}
{$ENDIF}
{$ENDIF}

{$IFNDEF HAVE_DIRECT}
  cbDirect.Enabled := False;
{$ENDIF}
{$IFNDEF STD}
  shEmbParams.TabVisible := LocalConnection is TMyEmbConnection;
  if LocalConnection is TMyEmbConnection then
    ParamsToControls(TMyEmbConnection(LocalConnection).Params);
{$ELSE}
  shEmbParams.TabVisible := False;
{$ENDIF}

{$IFDEF MSWINDOWS}
{$IFNDEF FPC}
  if not DADesignUtilsClass.DBToolsAvailable and cbExisting.Visible then
{$ENDIF}
{$ENDIF}
  begin
    lbExisting.Visible := False;
    cbExisting.Visible := False;
    
  {$IFDEF UNIX}
//    if LocalConnection is TMyEmbConnection then
//      Height := Height - 20
//    else
  {$ENDIF}
//    Height := Height - 33;

//  {$IFDEF MSWINDOWS}
    Panel.Height := Panel.Height + 33;
//  {$ENDIF}
    Panel.Top := cbExisting.Top;

//    cbLoginPrompt.Top := cbLoginPrompt.Top - 33;
//    cbDirect.Top := cbDirect.Top - 33;
//    cbEmbedded.Top := cbEmbedded.Top - 33;
  end;
end;

procedure TMyConnectionEditorForm.DoActivate;
begin
  inherited;

{$IFNDEF STD}
  if InitialProperty = 'Params' then
    PageControl.ActivePage := shEmbParams;
{$ENDIF}
end;

function TMyConnectionEditorForm.GetFrameByInitProp: TCRFrame;
begin
{$IFNDEF STD}
  if InitialProperty = 'Params' then
    Result := nil
  else
{$ENDIF}
    Result := inherited GetFrameByInitProp;
end;

procedure TMyConnectionEditorForm.DoSaveConnection;
begin
  inherited;

  edDatabaseExit(nil);

  TCustomMyConnection(Connection).Database := LocalConnection.Database;
  if LocalConnection is TMyConnection then begin
    TMyConnection(Connection).Port := TMyConnection(LocalConnection).Port;
    TMyConnection(Connection).Options.Embedded := TMyConnection(LocalConnection).Options.Embedded;
  {$IFDEF HAVE_DIRECT}
    TMyConnection(Connection).Options.Direct := TMyConnection(LocalConnection).Options.Direct;
  {$ENDIF}
  end;

{$IFNDEF STD}
  if FConnection is TMyEmbConnection then
    ControlsToParams(TMyEmbConnection(FConnection).Params);
  // Connection.Assign(FConnection);
{$ENDIF}
end;

procedure TMyConnectionEditorForm.FillInfo(InfoConnection: TCustomDAConnection);

function CompleteWithSpaces(s: string; ResultLength: word): string;
  var
    n, i: integer;
  begin
    result := s;
    n := ResultLength - Length(s);
    if n > 0 then
      for i := 1 to n do
        result := result + ' ';
  end;

var
  MyInfoConnection: TCustomMyConnection;
  MyQuery: TMyQuery;
  OldConnectionTimeout: integer;
begin
{$IFNDEF STD}
  if FConnection is TMyEmbConnection then
    MyInfoConnection := TMyEmbConnection(InfoConnection)
  else
{$ENDIF}
    MyInfoConnection := TMyConnection(InfoConnection);

  OldConnectionTimeout := MyInfoConnection.ConnectionTimeout;
  try
    MyInfoConnection.ConnectionTimeout := 1;
    meInfo.Lines.BeginUpdate;
    meInfo.Lines.Clear;

    if not IsConnected and
      not ((MyInfoConnection.Username = '') and (MyInfoConnection.Password = '') and
      (MyInfoConnection.Server = '') and (MyInfoConnection.Database = '')) then
      inherited;

    if MyInfoConnection.Connected then begin
      meInfo.Lines.Text := 'MySQL server version: ' + MyInfoConnection.ServerVersion + DALineSeparator;

    {$IFDEF HAVE_DIRECT}
      /// CR 3443
      if not (MyInfoConnection is TMyConnection) or not TMyConnection(MyInfoConnection).Options.Direct then
        meInfo.Lines.Text := meInfo.Lines.Text + 'MySQL client version: ' + MyInfoConnection.ClientVersion
      else
        if (MyInfoConnection is TMyConnection) and TMyConnection(MyInfoConnection).Options.Direct then
          meInfo.Lines.Text := meInfo.Lines.Text + 'MySQL client version: Direct';
    {$ELSE}
      meInfo.Lines.Text := meInfo.Lines.Text + 'MySQL client version: ' + MyInfoConnection.ClientVersion;
    {$ENDIF}

      meInfo.Lines.Text := meInfo.Lines.Text + DALineSeparator + DALineSeparator;

      MyQuery := TMyQuery.Create(nil);
      try
        MyQuery.Connection := MyInfoConnection;
        MyQuery.SQL.Text := 'SHOW variables';
        MyQuery.FetchAll := True;
        MyQuery.Open;
        while not MyQuery.Eof do begin
          meInfo.Lines.Add(MyQuery.Fields[0].AsString + ' = ' + MyQuery.Fields[1].AsString);
          MyQuery.Next;
        end;
        meInfo.SelStart := 0;
      finally
        MyQuery.Free;
      end;
    end;

  finally
    meInfo.Lines.EndUpdate;
    MyInfoConnection.ConnectionTimeout := OldConnectionTimeout;
  end;
end;

procedure TMyConnectionEditorForm.ConnToControls;
begin
  inherited;

  edDatabase.Text := LocalConnection.Database;
  if LocalConnection is TMyConnection then begin
    edPort.Text := IntToStr(TMyConnection(LocalConnection).Port);
    cbEmbedded.Checked := TMyConnection(LocalConnection).Options.Embedded;
  {$IFDEF HAVE_DIRECT}
    cbDirect.Checked := TMyConnection(LocalConnection).Options.Direct;
  {$ENDIF}
  end
  else begin
  {$IFNDEF STD}
    Assert(LocalConnection is TMyEmbConnection, LocalConnection.ClassName);
    edServer.Text := '';
    edPort.Text := '';
    cbEmbedded.Enabled := False;
    cbEmbedded.Checked := True;

//    ParamsToControls(TMyEmbConnection(LocalConnection).Params);
  {$ELSE}
    Assert(False);
  {$ENDIF}
  end;
end;

procedure TMyConnectionEditorForm.ShowState(Yellow: boolean = False);
begin
  edServer.Enabled := not cbEmbedded.Checked;
  edPort.Enabled := not cbEmbedded.Checked;
  cbDirect.Enabled := not cbEmbedded.Checked;

  lbServer.Enabled := edServer.Enabled;
  lbPort.Enabled := edPort.Enabled;

  inherited;
end;

procedure TMyConnectionEditorForm.edPortExit(Sender: TObject);
begin
  if FInDoInit then
    Exit;

{$IFDEF DBTOOLS}
  ChooseExistingConnection;
{$ENDIF}
  try
    Assert(LocalConnection is TMyConnection);
    try
      if edPort.Text = '' then begin
        TMyConnection(LocalConnection).Port := 3306;
        edPort.Text := '3306';
      end
      else
        TMyConnection(LocalConnection).Port := StrToInt(edPort.Text);
    except
      PageControl.ActivePage := shConnect;
      edPort.SetFocus;
      edPort.SelectAll;
      raise;
    end;
  finally
    ShowState;
  end;
end;

procedure TMyConnectionEditorForm.edPortChange(Sender: TObject);
var
  Port: integer;
begin
  if FInDoInit or (edPort.Text = '') then
    Exit;

  try
    Assert(LocalConnection is TMyConnection);
    try
      Port := StrToInt(edPort.Text);
      if TMyConnection(LocalConnection).Port <> Port then begin
        PerformDisconnect;
        TMyConnection(LocalConnection).Port := Port;
        Modified := True;
      end;
    except
      PageControl.ActivePage := shConnect;
      edPort.SetFocus;
      edPort.SelectAll;
      raise;
    end;
  finally
    ShowState;
  end;
end;

procedure TMyConnectionEditorForm.edDatabaseDropDown(Sender: TObject);
var
  OldConnected: boolean;
  List: TStringList;
begin
{$IFDEF UNIX}
  (Sender as TComboBox).OnGetItems := nil;
  try
{$ENDIF}
  ShowState(True);
  StartWait;
  try
    OldConnected := LocalConnection.Connected;
    (LocalConnection as TCustomMyConnection).Database := '';
    LocalConnection.PerformConnect;
    if LocalConnection.Connected then begin
      List := TStringList.Create;
      try
      {$IFDEF DARWIN}
        try
      {$ENDIF}
          LocalConnection.GetDatabaseNames(List);
      {$IFDEF DARWIN}
        except
          on E:Exception do
            ShowMessage(E.Message);
        end;
      {$ENDIF}
        AssignStrings(List, edDatabase.Items);
      finally
        LocalConnection.Connected := OldConnected;
        List.Free;
      end;
      if edDatabase.Items.Count < 20 then
        edDatabase.DropDownCount := edDatabase.Items.Count
      else
        edDatabase.DropDownCount := 20;
    end;
  finally
    StopWait;
    ShowState;
  end;
{$IFDEF UNIX}
  finally
    (Sender as TComboBox).OnGetItems := edDatabaseDropDown;
  end;
{$ENDIF}
end;

procedure TMyConnectionEditorForm.cbEmbeddedClick(Sender: TObject);
begin
  inherited;

  if not cbEmbedded.Enabled then
    Exit;

  try
    Assert(LocalConnection is TMyConnection);
    btDisconnectClick(nil);
    TMyConnection(LocalConnection).Options.Embedded := cbEmbedded.Checked;
  {$IFDEF DBTOOLS}
    ChooseExistingConnection;
  {$ENDIF}
  finally
    ShowState(False);
  end;
end;

procedure TMyConnectionEditorForm.cbDirectClick(Sender: TObject);
begin
  if FInDoInit or not cbDirect.Enabled then
    Exit;

  try
    Assert(LocalConnection is TMyConnection);
    btDisconnectClick(nil);
    TMyConnection(LocalConnection).Options.Direct := cbDirect.Checked;
  {$IFDEF DBTOOLS}
    ChooseExistingConnection;
  {$ENDIF}
  finally
    ShowState;
  end;
end;

{procedure TMyConnectionEditorForm.TrimParams(Params: TStrings);
var
  i: integer;
begin
  for i := 0 to Params.Count - 1 do
    Params[i] := Trim(Params[i]);
end;}

procedure TMyConnectionEditorForm.ParamsToControls(Params: TStrings);
begin
  edBaseDir.Text := Params.Values['--basedir'];
  edDataDir.Text := Params.Values['--datadir'];
  edTempDir.Text := Params.Values['--tempdir'];
  edCharsetsDir.Text := Params.Values['--character-sets-dir'];
  edLanguage.Text := Params.Values['--language'];
  edBinaryLog.Text := Params.Values['--log-bin'];
  edBinaryLogIndex.Text := Params.Values['--log-bin-index'];

  cbInnoDBSafeBinLog.Checked := Params.IndexOf('--innodb-safe-binlog') <> -1;
  cbFlush.Checked := Params.IndexOf('--flush') <> -1;
  cbSkipInnoDB.Checked := Params.IndexOf('--skip-innodb') <> -1;
  cbSkipGrantTables.Checked := Params.IndexOf('--skip-grant-tables') <> -1;
  cbLogShortFormat.Checked := Params.IndexOf('--log-short-format') <> -1;
end;

procedure TMyConnectionEditorForm.ControlsToParams(Params: TStrings);
  procedure cbToParams(CheckBox: TCheckBox; const ParamName: string);
  var
    Idx: integer;
  begin
    Idx := Params.IndexOf(ParamName);
    if CheckBox.Checked then begin
      if Idx = -1 then
        Params.Add(ParamName);
    end
    else
      if Idx <> -1 then
        Params.Delete(Idx);
  end;

begin
  Params.Values['--basedir'] := Trim(edBaseDir.Text);
  Params.Values['--datadir'] := Trim(edDataDir.Text);
  Params.Values['--tempdir'] := Trim(edTempDir.Text);
  Params.Values['--character-sets-dir'] := Trim(edCharsetsDir.Text);
  Params.Values['--language'] := Trim(edLanguage.Text);
  Params.Values['--log-bin'] := Trim(edBinaryLog.Text);
  Params.Values['--log-bin-index'] := Trim(edBinaryLogIndex.Text);

  cbToParams(cbInnoDBSafeBinLog, '--innodb-safe-binlog');
  cbToParams(cbFlush, '--flush');
  cbToParams(cbSkipInnoDB, '--skip-innodb');
  cbToParams(cbSkipGrantTables, '--skip-grant-tables');
  cbToParams(cbLogShortFormat, '--log-short-format');
end;

procedure TMyConnectionEditorForm.btAdvancedClick(Sender: TObject);
begin
{$IFNDEF STD}
  Assert(LocalConnection is TMyEmbConnection, LocalConnection.ClassName);
  ControlsToParams(TMyEmbConnection(LocalConnection).Params);

  with TMyEmbConParamsEditorForm.Create(nil, DADesignUtilsClass) do
    try
      Connection := LocalConnection as TMyEmbConnection;
      if ShowModal = mrOK then
        ParamsToControls(TMyEmbConnection(Connection).Params);
    finally
      Free;
    end;
{$ELSE}
  Assert(False)
{$ENDIF}
end;

{$IFNDEF CLR}
{$IFNDEF FPC}
procedure TMyConnectionEditorForm.ReplaceEdit(var Edit: TWinControl);
type
  TSetProc = procedure (Self: TObject; Ptr: pointer);
var
  EditClass: string;
  NewEdit: TCustomControl;
  OldName: string;
  TypeInfo: PTypeInfo;
begin
  if GetClass('TSpinEdit') <> nil then
    EditClass := 'TSpinEdit'
  else
{$IFDEF BCB}
  if GetClass('TCSpinEdit') <> nil then
    EditClass := 'TCSpinEdit'
  else
{$ENDIF}
    EditClass := '';

  if EditClass <> '' then begin
    NewEdit := TCustomControl(GetClass(EditClass).NewInstance);
    NewEdit.Create(Edit.Owner);

    with NewEdit do begin
      Parent := Edit.Parent;
      Left := Edit.Left;
      Top := Edit.Top;
      Width := Edit.Width;
      Height := Edit.Height;
      Align := Edit.Align;
      TabOrder := Edit.TabOrder;
      Anchors := Edit.Anchors;
      //Constraints := Edit.Constraints;
      TypeInfo := GetClass(EditClass).ClassInfo;
      HelpContext := Edit.HelpContext;

      if Edit is TEdit then begin
        SetReadOnly(NewEdit, TEdit(Edit).ReadOnly);
        SetOrdProp(NewEdit, 'Color', integer(TEdit(Edit).Color));
      end;

      OnKeyDown := SpinEditKeyDown;
      if GetPropInfo(Edit.ClassInfo, 'OnChange') <> nil then
        SetMethodProp(NewEdit, GetPropInfo(TypeInfo, 'OnChange'),
          GetMethodProp(Edit, GetPropInfo(Edit.ClassInfo, 'OnChange')));
      SetMethodProp(NewEdit, GetPropInfo(TypeInfo, 'OnExit'),
        GetMethodProp(Edit, GetPropInfo(Edit.ClassInfo, 'OnExit')));
      SetMethodProp(NewEdit, GetPropInfo(TypeInfo, 'OnKeyDown'),
        GetMethodProp(Edit, GetPropInfo(Edit.ClassInfo, 'OnKeyDown')));
      SetMethodProp(NewEdit, GetPropInfo(TypeInfo, 'OnKeyPress'),
        GetMethodProp(Edit, GetPropInfo(Edit.ClassInfo, 'OnKeyPress')));
    end;

    if (Edit.Owner <> nil) and (TForm(Edit.Owner).ActiveControl = Edit) then
      TForm(Edit.Owner).ActiveControl := NewEdit;

    OldName := Edit.Name;
    Edit.Free;
    Edit := TEdit(NewEdit);
    NewEdit.Name := OldName;

    if (EditClass = 'TSpinEdit') {$IFDEF BCB} or (EditClass = 'TCSpinEdit' ){$ENDIF} then begin
      SetOrdProp(NewEdit, 'MaxValue', 65535);
      SetOrdProp(NewEdit, 'MinValue', 0);
    end;
  end;
end;
{$ENDIF}

procedure TMyConnectionEditorForm.SpinEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Key = 13 then
    btConnectClick(self);
end;
{$ENDIF}

procedure TMyConnectionEditorForm.edDatabaseExit(Sender: TObject);
begin
{$IFDEF DBTOOLS}
  ChooseExistingConnection;
{$ENDIF}
  try
    if LocalConnection.Database <> edDatabase.Text then begin
      LocalConnection.Database := edDatabase.Text;
      Modified := True;
    end;
  finally
    ShowState;
  end;
end;

{$IFDEF DBTOOLS}
function TMyConnectionEditorForm.GetExistingConnectionComboBox: TComboBox;
begin
  Result := cbExisting;
end;

function TMyConnectionEditorForm.GetConnectionCondition: string;
begin
  if LocalConnection is TMyConnection then
    Result := ''
  else
    Result := 'Embedded=True'
end;
{$ENDIF}

procedure TMyConnectionEditorForm.cbExistingChange(Sender: TObject);
{$IFDEF DBTOOLS}
var
  ConnectionStrList: TStrings;
begin
  if FInDoInit or (cbExisting.ItemIndex < 0) then
    Exit;
  ConnectionStrList := GetDBToolsService(DADesignUtilsClass).GetConnectionStrList(cbExisting.Text);
  try
    FInExistingChange := True;
    with TCustomMyConnection(LocalConnection) do begin
      if LocalConnection is TMyConnection then begin
        Server := ConnectionStrList.Values['Host'];
        TMyConnection(LocalConnection).Port := StrToInt(ConnectionStrList.Values['Port']);
        with TMyConnection(LocalConnection).SSLOptions do begin
          CACert := ConnectionStrList.Values['SSL CA Cert'];
          Cert := ConnectionStrList.Values['SSL Cert'];
          Key := ConnectionStrList.Values['SSL Key'];
          CipherList := ConnectionStrList.Values['SSL Cipher List'];
        end;
        with TMyConnection(LocalConnection).Options do begin
        {$IFDEF HAVE_DIRECT}
          Direct := StrToBool(ConnectionStrList.Values['Direct']);
        {$ENDIF}
          Embedded := StrToBool(ConnectionStrList.Values['Embedded']);
        end;
      end;
    {$IFNDEF STD}
      if LocalConnection is TMyEmbConnection then begin
        TMyEmbConnection(LocalConnection).Params.Text :=
          StringReplace(ConnectionStrList.Values['Server Parameters'], ';', DALineSeparator, [rfReplaceAll]);
        ParamsToControls(TMyEmbConnection(LocalConnection).Params);
      end;
    {$ENDIF}
      Database := ConnectionStrList.Values['Database'];
      Username := ConnectionStrList.Values['User Id'];
      Password := ConnectionStrList.Values['Password'];
      Options.UseUnicode := StrToBool(ConnectionStrList.Values['Unicode']);
    end;
  finally
    ConnToControls;
    FInExistingChange := False;
  end;
end;
{$ELSE}
begin
end;
{$ENDIF}

procedure TMyConnectionEditorForm.DoPageControlChange(Sender: TObject);
begin
  inherited;

  if TPageControl(Sender).ActivePage = shEmbParams then
    imgLogo.Parent := shEmbParams;
end;

procedure TMyConnectionEditorForm.PageControlChanging(Sender: TObject;
  var AllowChange: Boolean);
begin
{$IFNDEF STD}
  if (PageControl.ActivePage = shEmbParams) and (LocalConnection is TMyEmbConnection) then
    ControlsToParams(TMyEmbConnection(LocalConnection).Params);
{$IFDEF DBTOOLS}
  ChooseExistingConnection;
{$ENDIF}
{$ENDIF}
end;

function TMyConnectionEditorForm.GetLocalConnection: TCustomMyConnection;
begin
  Result := FLocalConnection as TCustomMyConnection;
end;

end.
