unit Threads;

interface

uses
{$IFDEF FPC}
  LResources,
{$ENDIF}
{$IFDEF MSWINDOWS}
  Windows, ActiveX,
{$ELSE}
  cThreads,
{$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Db, DBCtrls,
  Grids, DBGrids, DBAccess, MSAccess, SdacVcl,
  Buttons, ThreadsData, DemoFrame;

const
  WM_ENDTHREAD     = $500;
  WM_EXCEPTTHREAD  = $501;
  WM_ENDEXECUTE    = $502;

type
  TThreadsFrame = class(TDemoFrame)
    Timer: TTimer;
    MSDataSource: TDataSource;
    scDeleteAll: TMSSQL;
    Panel3: TPanel;
    Splitter2: TSplitter;
    Panel4: TPanel;
    meSQL: TMemo;
    meLog: TMemo;
    DBGrid1: TDBGrid;
    Panel2: TPanel;
    Panel5: TPanel;
    btStart: TSpeedButton;
    btStop: TSpeedButton;
    btRun: TSpeedButton;
    btRunMax: TSpeedButton;
    btClearLog: TSpeedButton;
    Panel6: TPanel;
    btOpen: TSpeedButton;
    btDeleteAll: TSpeedButton;
    DBNavigator1: TDBNavigator;
    Panel11: TPanel;
    Panel10: TPanel;
    Label1: TLabel;
    Label2: TLabel;
    lbThreadCount: TLabel;
    lbExceptCount: TLabel;
    Panel12: TPanel;
    lbInterval: TLabel;
    lbTime: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    Panel13: TPanel;
    Panel14: TPanel;
    Label3: TLabel;
    edCount: TEdit;
    Panel9: TPanel;
    Label4: TLabel;
    rbInsert: TRadioButton;
    rbSelect: TRadioButton;
    meErrorLog: TMemo;
    MSQuery: TMSQuery;
    procedure btRunClick(Sender: TObject);
    procedure TimerTimer(Sender: TObject);
    procedure btStartClick(Sender: TObject);
    procedure btStopClick(Sender: TObject);
    procedure edCountChange(Sender: TObject);
    procedure btRunMaxClick(Sender: TObject);
    procedure btClearLogClick(Sender: TObject);
    procedure btOpenClick(Sender: TObject);
    procedure rgModeClick(Sender: TObject);
    procedure btDeleteAllClick(Sender: TObject);

  private
    ThreadCount: integer;
    ThreadNum: integer;
    hCountSec: TRTLCriticalSection;
    BegTime: TDateTime;
    MaxThread: integer;
    Interval: integer;
    ExceptCount: integer;
    EventLog, ExceptLog: TStringList;

    procedure ShowStatus;
    procedure TestConnected;
  public
    destructor Destroy; override;
    procedure Initialize; override;
    procedure SetDebug(Value: boolean); override;
  end;

  TDemoThread = class(TThread)
  private
    FFrame: TThreadsFrame;

  protected
    procedure Execute; override;

  public
    constructor Create(Frame: TThreadsFrame);
end;

implementation

uses
 {$IFNDEF VER130}Variants,{$ENDIF}
  SdacDemoForm;

{$IFNDEF FPC}
{$IFDEF CLR}
{$R *.nfm}
{$ELSE}
{$R *.dfm}
{$ENDIF}
{$ENDIF}

constructor TDemoThread.Create(Frame: TThreadsFrame);
begin
  inherited Create(True);
  
  FFrame := Frame;
  FreeOnTerminate := True;
  Suspended := False;
end;

procedure TDemoThread.Execute;
var
  Data: TdmThreadsData;
  ThreadNum: integer;
  i: integer;
begin
 {$IFDEF MSWINDOWS}
  i := CoInitializeEx(nil, COINIT_APARTMENTTHREADED);
  if i <> S_OK then
    raise Exception.Create('err - ' + IntToStr(i));
 {$ENDIF}

  EnterCriticalSection(FFrame.hCountSec);
    Inc(FFrame.ThreadCount);
    Inc(FFrame.ThreadNum);
    ThreadNum := FFrame.ThreadNum;
  LeaveCriticalSection(FFrame.hCountSec);
  Synchronize(FFrame.ShowStatus);

  Data := TdmThreadsData.Create(nil);
  try
  try
    EnterCriticalSection(FFrame.hCountSec);
      FFrame.EventLog.Add(IntToStr(ThreadNum) + ' Connecting...');
    LeaveCriticalSection(FFrame.hCountSec);
    Synchronize(FFrame.ShowStatus);

    FFrame.AssignConnectionTo(Data.MSConnection);
    Data.MSConnection.loginPrompt := False;
    Data.MSConnection.Connect;

    EnterCriticalSection(FFrame.hCountSec);
      FFrame.EventLog.Add(IntToStr(ThreadNum) + ' Connected');
    LeaveCriticalSection(FFrame.hCountSec);
    Synchronize(FFrame.ShowStatus);

    with data do begin
    if FFrame.rbInsert.Checked then begin
      // INSERT
        MSSQL.ParamByName('ID').AsInteger := Random(10000);
        MSSQL.Execute;
//        MSConnection.Commit;
      EnterCriticalSection(FFrame.hCountSec);
        FFrame.EventLog.Add(IntToStr(ThreadNum) + ' Executed');
      LeaveCriticalSection(FFrame.hCountSec);
      Synchronize(FFrame.ShowStatus);
      end
    else
      if FFrame.rbSelect.Checked then begin
      // SELECT
        MSQuery.Open;
        i := 0;
        while not MSQuery.Eof do begin
          MSQuery.Next;
          Inc(i);
        end;
        EnterCriticalSection(FFrame.hCountSec);
          FFrame.EventLog.Add(IntToStr(ThreadNum) + ' Fetched ' + IntToStr(i) + ' rows');
        LeaveCriticalSection(FFrame.hCountSec);
        Synchronize(FFrame.ShowStatus);
        MSQuery.Close;
      end;

      MSConnection.Disconnect;
      EnterCriticalSection(FFrame.hCountSec);
      FFrame.EventLog.Add(IntToStr(ThreadNum) + ' Disconnected');
      LeaveCriticalSection(FFrame.hCountSec);
      Synchronize(FFrame.ShowStatus);
    end;
  except
    on E:Exception do begin
     {$IFDEF MSWINDOWS}
      MessageBeep(1000);
     {$ENDIF}
      EnterCriticalSection(FFrame.hCountSec);
        FFrame.EventLog.Add(IntToStr(ThreadNum) + ' ' + IntToStr(FFrame.ThreadCount) + ' Exception ' + E.Message);
        FFrame.ExceptLog.Add(IntToStr(ThreadNum) + ' ' + IntToStr(FFrame.ThreadCount) + ' Exception ' + E.Message);
        Inc(FFrame.ExceptCount);
      LeaveCriticalSection(FFrame.hCountSec);
      Synchronize(FFrame.ShowStatus);
    end;
  end;
  finally
    Data.Free;
   {$IFDEF MSWINDOWS}
    CoUninitialize;
   {$ENDIF}
  end;

  EnterCriticalSection(FFrame.hCountSec);
    Dec(FFrame.ThreadCount);
  LeaveCriticalSection(FFrame.hCountSec);
  Synchronize(FFrame.ShowStatus);
 {$IFDEF MSWINDOWS}
  MessageBeep(1000);
 {$ENDIF}
end;

procedure TThreadsFrame.ShowStatus;
begin
  lbThreadCount.Caption := IntToStr(ThreadCount);
  lbExceptCount.Caption := IntToStr(ExceptCount);

  if meLog.Lines.Count > 1000 then
    meLog.Lines.Clear;
  meLog.Lines.AddStrings(EventLog);
  EventLog.Clear;

  meErrorLog.Lines.AddStrings(ExceptLog);
  ExceptLog.Clear;
end;

const
  Delay = 1000;

destructor TThreadsFrame.Destroy;
begin
 {$IFDEF MSWINDOWS}
  DeleteCriticalSection(hCountSec);
 {$ELSE}
  DoneCriticalSection(hCountSec);
 {$ENDIF}
  EventLog.Free;
  ExceptLog.Free;

  inherited;
end;

procedure TThreadsFrame.Initialize;
begin
  inherited;

  MSQuery.Connection := Connection as TMSConnection;
  scDeleteAll.Connection := Connection as TMSConnection;

  MaxThread := 40;
  Interval := 2000;
  ExceptCount := 0;
  EventLog := TStringList.Create;
  ExceptLog := TStringList.Create;
 {$IFDEF MSWINDOWS}
  InitializeCriticalSection(hCountSec);
 {$ELSE}
  InitCriticalSection(hCountSec);
 {$ENDIF}
  Randomize;

  dmThreadsData := TdmThreadsData.Create(Self);
  edCount.Text := IntToStr(MaxThread);
  rbSelect.Checked := True;
end;

procedure TThreadsFrame.btRunClick(Sender: TObject);
begin
  TestConnected;
  TDemoThread.Create(Self);
end;

procedure TThreadsFrame.btRunMaxClick(Sender: TObject);
var
  i: integer;
begin
  TestConnected;
  for i := 1 to MaxThread do
    TDemoThread.Create(Self);
end;

procedure TThreadsFrame.btStartClick(Sender: TObject);
begin
  TestConnected;
  edCount.Text := IntToStr(MaxThread);
  BegTime := Time;
  TimerTimer(nil);
end;

procedure TThreadsFrame.btStopClick(Sender: TObject);
begin
  Timer.Enabled := False;
end;

procedure TThreadsFrame.TimerTimer(Sender: TObject);
begin
  if ThreadCount < MaxThread then begin
    btRunClick(nil);
    if ThreadCount < (MaxThread div 10) * 9 then
      Dec(Interval, Interval div 10);
  end
  else
    Inc(Interval, Interval div 10);

  lbInterval.Caption := IntToStr(Interval);
  lbExceptCount.Caption := IntToStr(ExceptCount);
  Timer.Interval := Random(Interval - 1) + 1;
  lbTime.Caption := TimeToStr(Time - BegTime);

  lbInterval.Caption := lbInterval.Caption + ' / ' + IntToStr(Timer.Interval);

  Timer.Enabled := True;
end;

procedure TThreadsFrame.edCountChange(Sender: TObject);
begin
  MaxThread := StrToInt(edCount.Text);
end;

procedure TThreadsFrame.btClearLogClick(Sender: TObject);
begin
  meLog.Lines.Clear;
  meErrorLog.Lines.Clear;
  ThreadNum := 0;
  ExceptCount := 0;
  lbExceptCount.Caption := IntToStr(ExceptCount);
end;

procedure TThreadsFrame.btOpenClick(Sender: TObject);
begin
  MSQuery.Close;
  MSQuery.Open;
end;

procedure TThreadsFrame.rgModeClick(Sender: TObject);
begin
  if rbInsert.Checked then
    meSQL.Lines.Assign(dmThreadsData.MSSQL.SQL)
  else
    meSQL.Lines.Assign(dmThreadsData.MSQuery.SQL)
end;

procedure TThreadsFrame.btDeleteAllClick(Sender: TObject);
begin
  scDeleteAll.Execute;
//  MSConnection.Commit;
end;


procedure TThreadsFrame.TestConnected;
begin
  SdacForm.MSConnection.Connect;
  if not SdacForm.MSConnection.Connected then
    Abort;
end;

procedure TThreadsFrame.SetDebug(Value: boolean);
begin
  MSQuery.Debug := Value;
end;

{$IFDEF FPC}
initialization
  {$i Threads.lrs}
{$ENDIF}
end.
