
//////////////////////////////////////////////////
//  DB Access Components
//  Copyright  1998-2025 Devart. All right reserved.
//////////////////////////////////////////////////

{$IFNDEF CLR}

{$I Dac.inc}

unit DADesignUtils;
{$ENDIF}

interface

uses
  Classes, SysUtils,
{$IFDEF DBTOOLS}
  DBToolsIntf,
  CRAccess,
{$ENDIF}
  CRTypes, CRDesignUtils, CRDataTypeMap, DB, DBAccess, DASQLGenerator;

type
//{$IFDEF DBTOOLS}
  TNeedToCheckDbTools = (ncNone, ncExpired, ncNoAddin, ncIncompatible);
//{$ENDIF}

  TDADesignUtils = class(TCRDesignUtils)
  public
    class function GetProjectName: string; override; // Returns ProjectName = ('DataEditor', 'ODAC', 'SDAC', 'MyDAC', ...)

  { Component }
    class function GetDesignCreate(Obj: TComponent): boolean; override;
    class procedure SetDesignCreate(Obj: TComponent; Value: boolean); override;
  {$IFNDEF FPC}
    class function GetConnectionList: TObject; virtual;   //avoid circular link error
  {$ENDIF}

  { Connection support }
    class function HasConnection(Obj: TComponent): boolean; virtual;
    class function GetConnection(Obj: TComponent): TCustomDAConnection; virtual;
    class procedure SetConnection(Obj: TComponent; Value: TCustomDAConnection); virtual;
    class function UsedConnection(Obj: TComponent): TCustomDAConnection; virtual;

  { SQL support }
    class function GetSQL(Obj: TComponent; StatementType: TStatementType = stQuery): TStrings; virtual;
    class procedure SetSQL(Obj: TComponent; Value: TStrings; StatementType: TStatementType = stQuery); overload; virtual;
    class procedure SetSQL(Obj: TComponent; Value: string; StatementType: TStatementType = stQuery); overload; virtual;

    class function GetSQLPropName(Obj: TComponent; StatementType: TStatementType): string; virtual;

    class function GetParams(Obj: TComponent): TDAParams; virtual;
    class procedure SetParams(Obj: TComponent; Value: TDAParams); virtual;
    class function GetMacros(Obj: TComponent): TMacros;
    class procedure SetMacros(Obj: TComponent; Value: TMacros);
    class function GetConditions(Obj: TComponent): TDAConditions;
    class procedure SetConditions(Obj: TComponent; Value: TDAConditions);
    class procedure Execute(Obj: TComponent);
    class function GetAfterExecute(Obj: TComponent): TAfterExecuteEvent;
    class procedure SetAfterExecute(Obj: TComponent; Value: TAfterExecuteEvent);
    class procedure Close(Obj: TComponent);

  { DataSet support}
    class function GetStatementTypes: TStatementTypes; virtual; // allowable StatementTypes for GetSQL and SetSQL

  { TDATable support }
    class function GetTableName(Obj: TCustomDADAtaSet): string; virtual;
    class procedure SetTableName(Obj: TCustomDADAtaSet; const Value: string); virtual;
    class function GetOrderFields(Obj: TCustomDADAtaSet): string; virtual;
    class procedure SetOrderFields(Obj: TCustomDADAtaSet; const Value: string); virtual;
    class procedure PrepareSQL(Obj: TCustomDADAtaSet); virtual;

  { TDAStoredProc support}
    class function GetStoredProcName(Obj: TCustomDADataSet): string; virtual;
    class procedure SetStoredProcName(Obj: TCustomDADataSet; const Value: string); virtual;

    class function GetConverterManagerClass: TConverterManagerClass; virtual;

  {$IFDEF USE_SYNEDIT}
    class function SQLDialect: integer ; override; // SynHighlighterSQL TSQLDialect = (sqlStandard, sqlInterbase6, sqlMSSQL7, sqlMySQL, sqlOracle, sqlSybase, sqlIngres, sqlMSSQL2K);
  {$ENDIF}

    class function DBToolsAvailable: boolean; override;

//  {$IFDEF DBTOOLS}
    class function DBToolsService: TObject; virtual; //avoid circular link error
    class function NeedToCheckDbTools: TNeedToCheckDbTools; virtual;
    class function GetDBToolsServiceVersion: int64; virtual;
    class function GetDBToolsServiceVersionStr: string;
    class function GetDBToolsMenuCaption: string; virtual;
    class function GetFullName(Obj: TComponent): string; virtual;
    class function GetObjectType(Obj: TComponent): string; virtual;
    class procedure SetDBToolsDownloadParams(VerbCheck: boolean; Incompatible: boolean); virtual;
    class function HasParams(Obj: TComponent): boolean;
    class function IsStoredProc(Obj: TComponent): boolean; virtual;
    class procedure GetDBToolsConnectionList(Connection: TCustomDAConnection); virtual;
    class function AreDBToolsUsed: boolean; virtual;
    class procedure CheckComponent(Component: TComponent); virtual;
//  {$ENDIF}
  end;

  TDADesignUtilsClass = class of TDADesignUtils;

implementation

uses
{$IFDEF MSWINDOWS}
  Windows, ComObj,
{$ENDIF}
  Forms,
{$IFDEF CLR}
  System.Runtime.InteropServices,
{$ELSE}
  CLRClasses,
{$ENDIF}
{$IFDEF DBTOOLS}
  DBToolsClient,
{$ENDIF}
  DAConsts, DALoader, DADump, DAScript, DAAlerter;

{ TDADesignUtils }

class function TDADesignUtils.GetProjectName: string;
begin
 Result := 'DAC';
end;

class function TDADesignUtils.GetDesignCreate(Obj: TComponent): boolean;
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetDesignCreate', 'Obj', nil]))
  else
  if Obj is TDATransaction then
    Result := TDBAccessUtils.GetDesignCreate(TDATransaction(Obj))
  else
  if Obj is TCustomDASQL then
    Result := TDBAccessUtils.GetDesignCreate(TCustomDASQL(Obj))
  else
  if Obj is TCustomDADataSet then
    Result := TDBAccessUtils.GetDesignCreate(TCustomDADataSet(Obj))
  else
  if Obj is TDALoader then
    Result := TDALoaderUtils.GetDesignCreate(TDALoader(Obj))
  else
  if Obj is TDADump then
    Result := TDADumpUtils.GetDesignCreate(TDADump(Obj))
  else
  if Obj is TDAScript then
    Result := TDAScriptUtils.GetDesignCreate(TDAScript(Obj))
  else
  if Obj is TDAAlerter then
    Result := TDAAlerterUtils.GetDesignCreate(TDAAlerter(Obj))
  else
  if Obj is TCustomDAUpdateSQL then
    Result := TDBAccessUtils.GetDesignCreate(TCustomDAUpdateSQL(Obj))
  else
  if Obj is TDAMetaData then
    Result := TDBAccessUtils.GetDesignCreate(TDAMetaData(Obj))
  else
  if Obj is TCRDataSource then
    Result := TDBAccessUtils.GetDesignCreate(TCRDataSource(Obj))
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetDesignCreate', 'Obj', Obj.ClassName]));
end;

class procedure TDADesignUtils.SetDesignCreate(Obj: TComponent; Value: boolean);
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetDesignCreate', 'Obj', nil]))
  else
  if Obj is TDATransaction then
    TDBAccessUtils.SetDesignCreate(TDATransaction(Obj), Value)
  else
  if Obj is TCustomDASQL then
    TDBAccessUtils.SetDesignCreate(TCustomDASQL(Obj), Value)
  else
  if Obj is TCustomDADataSet then
    TDBAccessUtils.SetDesignCreate(TCustomDADataSet(Obj), Value)
  else
  if Obj is TDALoader then
    TDALoaderUtils.SetDesignCreate(TDALoader(Obj), Value)
  else
  if Obj is TDADump then
    TDADumpUtils.SetDesignCreate(TDADump(Obj), Value)
  else
  if Obj is TDAScript then
    TDAScriptUtils.SetDesignCreate(TDAScript(Obj), Value)
  else
  if Obj is TDAAlerter then
    TDAAlerterUtils.SetDesignCreate(TDAAlerter(Obj), Value)
  else
  if Obj is TCustomDAUpdateSQL then
    TDBAccessUtils.SetDesignCreate(TCustomDAUpdateSQL(Obj), Value)
  else
  if Obj is TDAMetaData then
    TDBAccessUtils.SetDesignCreate(TDAMetaData(Obj), Value)
  else
  if Obj is TCRDataSource then
    TDBAccessUtils.SetDesignCreate(TCRDataSource(Obj), Value)
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetDesignCreate', 'Obj', Obj.ClassName]));
end;

{$IFNDEF FPC}
class function TDADesignUtils.GetConnectionList: TObject;
begin
  raise Exception.Create(Format(SFunctionMustBeOverriden, ['TDADesignUtils.GetConnectionList']));
end;
{$ENDIF}

class function TDADesignUtils.HasConnection(Obj: TComponent): boolean;
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.HasConnection', 'Obj', nil]))
  else
  if Obj is TDATransaction then
    Result := True
  else
  if Obj is TCustomDASQL then
    Result := True
  else
  if Obj is TCustomDADataSet then
    Result := True
  else
  if Obj is TDAScript then
    Result := True
  else
  if Obj is TDALoader then
    Result := True
  else
  if Obj is TDADump then
    Result := True
  else
  if Obj is TDAAlerter then
    Result := True
  else
  if Obj is TDAMetaData then
    Result := True
  else
    Result := False;
end;

class function TDADesignUtils.GetConnection(Obj: TComponent): TCustomDAConnection;
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetConnection', 'Obj', nil]))
  else
  if Obj is TDATransaction then
    Result := TDATransaction(Obj).DefaultConnection
  else
  if Obj is TCustomDASQL then
    Result := TCustomDASQL(Obj).Connection
  else
  if Obj is TCustomDADataSet then
    Result := TCustomDADataSet(Obj).Connection
  else
  if Obj is TDAScript then
    Result := TDAScript(Obj).Connection
  else
  if Obj is TDALoader then
    Result := TDALoader(Obj).Connection
  else
  if Obj is TDADump then
    Result := TDADump(Obj).Connection
  else
  if Obj is TDAAlerter then
    Result := TDAAlerter(Obj).Connection
  else
  if Obj is TDAMetaData then
    Result := TDAMetaData(Obj).Connection
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetConnection', 'Obj', Obj.ClassName]));
end;

class procedure TDADesignUtils.SetConnection(Obj: TComponent; Value: TCustomDAConnection);
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetConnection', 'Obj', nil]))
  else
  if Obj is TDATransaction then
    TDATransaction(Obj).DefaultConnection := Value
  else
  if Obj is TCustomDASQL then
    TCustomDASQL(Obj).Connection := Value
  else
  if Obj is TCustomDADataSet then
    TCustomDADataSet(Obj).Connection := Value
  else
  if Obj is TDAScript then
    TDAScript(Obj).Connection := Value
  else
  if Obj is TDALoader then
    TDALoader(Obj).Connection := Value
  else
  if Obj is TDADump then
    TDADump(Obj).Connection := Value
  else
  if Obj is TDAAlerter then
    TDAAlerter(Obj).Connection := Value
  else
  if Obj is TDAMetaData then
    TDAMetaData(Obj).Connection := Value
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetConnection', 'Obj', Obj.ClassName]));
end;

class function TDADesignUtils.UsedConnection(Obj: TComponent): TCustomDAConnection;
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.UsedConnection', 'Obj', nil]))
  else
  if Obj is TCustomDASQL then
    Result := TDBAccessUtils.UsedConnection(TCustomDASQL(Obj))
  else
  if Obj is TCustomDADataSet then
    Result := TDBAccessUtils.UsedConnection(TCustomDADataSet(Obj))
  else
  if Obj is TDAScript then
    Result := TDAScriptUtils.UsedConnection(TDAScript(Obj))
  else
  if Obj is TDALoader then
    Result := TDALoaderUtils.UsedConnection(TDALoader(Obj))
  else
  if Obj is TCustomDAUpdateSQL then begin
    if TCustomDAUpdateSQL(Obj).DataSet <> nil then
      Result := TDBAccessUtils.UsedConnection(TCustomDAUpdateSQL(Obj).DataSet)
    else
      Result := nil;
  end
  else
  if Obj is TDADump then
    Result := TDADump(Obj).Connection
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.UsedConnection', 'Obj', Obj.ClassName]));
end;

class function TDADesignUtils.GetStatementTypes: TStatementTypes;
begin
  Result := [stQuery, stInsert, stUpdate, stDelete, stRefresh, stLock, stRecCount];
end;

class function TDADesignUtils.GetSQL(Obj: TComponent; StatementType: TStatementType = stQuery): TStrings;
begin
{$IFNDEF FPC}
  Assert(Obj <> nil);
{$ELSE}
  if Obj = nil then
    Exit;
{$ENDIF}

  Assert((StatementType = stQuery) or (Obj is TCustomDADataSet) or (Obj is TCustomDAUpdateSQL));
  if Obj is TCustomDASQL then
    Result := TCustomDASQL(Obj).SQL
  else
  if (Obj is TDADump) then
    Result := TDADump(Obj).SQL
  else
  if (Obj is TDAScript) then
    Result := TDAScript(Obj).SQL
  else
  if Obj is TCustomDADataSet then begin
    case StatementType of
      stQuery:
        Result := TCustomDADataSet(Obj).SQL;
      stInsert:
        Result := TCustomDADataSet(Obj).SQLInsert;
      stUpdate:
        Result := TCustomDADataSet(Obj).SQLUpdate;
      stDelete:
        Result := TCustomDADataSet(Obj).SQLDelete;
      stRefresh:
        Result := TCustomDADataSet(Obj).SQLRefresh;
      stLock:
        Result := TCustomDADataSet(Obj).SQLLock;
      stRecCount:
        Result := TCustomDADataSet(Obj).SQLRecCount;
      else
        raise Exception.Create(Format(SInternalErrorInvalidValue,
          ['TDADesignUtils.GetSQL', 'StatementType', IntToStr(Integer(StatementType))]));
    end;
  end
  else
  if Obj is TCustomDAUpdateSQL then begin
    case StatementType of
      stInsert:
        Result := TCustomDAUpdateSQL(Obj).InsertSQL;
      stUpdate:
        Result := TCustomDAUpdateSQL(Obj).ModifySQL;
      stDelete:
        Result := TCustomDAUpdateSQL(Obj).DeleteSQL;
      stRefresh:
        Result := TCustomDAUpdateSQL(Obj).RefreshSQL;
      stLock:
        Result := TCustomDAUpdateSQL(Obj).LockSQL;
      else
        raise Exception.Create(Format(SInternalErrorInvalidValue,
          ['TDADesignUtils.GetSQL', 'StatementType', IntToStr(Integer(StatementType))]));
    end;
  end
  else
  if Obj is TDAScript then
    Result := TDAScript(Obj).SQL
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetSQL', 'Obj', Obj.ClassName]));
end;

class function TDADesignUtils.GetSQLPropName(Obj: TComponent; StatementType: TStatementType): string;
begin
  Result := '';
  Assert(Obj <> nil);
  Assert((StatementType = stQuery) or (Obj is TCustomDADataSet) or (Obj is TCustomDAUpdateSQL));
  if (Obj is TCustomDASQL) or (Obj is TDADump) or (Obj is TDAScript) then
    Result := 'SQL'
  else
  if Obj is TCustomDADataSet then
    case StatementType of
      stQuery:
        Result := 'SQL';
      stInsert:
        Result := 'SQLInsert';
      stUpdate:
        Result := 'SQLUpdate';
      stDelete:
        Result := 'SQLDelete';
      stRefresh:
        Result := 'SQLRefresh';
      stLock:
        Result := 'SQLLock';
      stRecCount:
        Result := 'SQLRecCount';
      else
        raise Exception.Create(Format(SInternalErrorInvalidValue,
          ['TDADesignUtils.GetSQLPropName', 'StatementType', IntToStr(Integer(StatementType))]));
    end
  else
  if Obj is TCustomDAUpdateSQL then
    case StatementType of
      stInsert:
        Result := 'InsertSQL';
      stUpdate:
        Result := 'ModifySQL';
      stDelete:
        Result := 'DeleteSQL';
      stRefresh:
        Result := 'RefreshSQL';
      stLock:
        Result := 'LockSQL';
      else
        raise Exception.Create(Format(SInternalErrorInvalidValue,
          ['TDADesignUtils.GetSQLPropName', 'StatementType', IntToStr(Integer(StatementType))]));
    end;
end;

class procedure TDADesignUtils.SetSQL(Obj: TComponent; Value: TStrings; StatementType: TStatementType = stQuery);
begin
  Assert(Obj <> nil);
  Assert((StatementType = stQuery) or (Obj is TCustomDADataSet) or (Obj is TCustomDAUpdateSQL));
  if Obj is TCustomDASQL then
    TCustomDASQL(Obj).SQL := Value
  else
  if (Obj is TDADump) then
    TDADump(Obj).SQL := Value
  else
  if (Obj is TDAScript) then
    TDAScript(Obj).SQL := Value
  else
  if Obj is TCustomDADataSet then begin
    case StatementType of
      stQuery:
        TCustomDADataSet(Obj).SQL := Value;
      stInsert:
        TCustomDADataSet(Obj).SQLInsert := Value;
      stUpdate:
        TCustomDADataSet(Obj).SQLUpdate := Value;
      stDelete:
        TCustomDADataSet(Obj).SQLDelete := Value;
      stRefresh:
        TCustomDADataSet(Obj).SQLRefresh := Value;
      stLock:
        TCustomDADataSet(Obj).SQLLock := Value;
      stRecCount:
        TCustomDADataSet(Obj).SQLRecCount := Value;
      else
        raise Exception.Create(Format(SInternalErrorInvalidValue,
          ['TDADesignUtils.SetSQL', 'StatementType', IntToStr(Integer(StatementType))]));
    end;
  end
  else
  if Obj is TCustomDAUpdateSQL then begin
    case StatementType of
      stInsert:
        TCustomDAUpdateSQL(Obj).InsertSQL := Value;
      stUpdate:
        TCustomDAUpdateSQL(Obj).ModifySQL := Value;
      stDelete:
        TCustomDAUpdateSQL(Obj).DeleteSQL := Value;
      stRefresh:
        TCustomDAUpdateSQL(Obj).RefreshSQL := Value;
      stLock:
        TCustomDAUpdateSQL(Obj).LockSQL := Value;
      else
        raise Exception.Create(Format(SInternalErrorInvalidValue,
          ['TDADesignUtils.SetSQL', 'StatementType', IntToStr(Integer(StatementType))]));
    end;
  end
  else
  if Obj is TDAScript then
    TDAScript(Obj).SQL := Value
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetSQL', 'Obj', Obj.ClassName]));
end;

class procedure TDADesignUtils.SetSQL(Obj: TComponent; Value: string; StatementType: TStatementType = stQuery);
var
  List: TStringList;
begin
  List := TStringList.Create;
  try
    List.Text := Value;
    SetSQL(Obj, List, StatementType);
  finally
    List.Free;
  end;
end;

class function TDADesignUtils.GetParams(Obj: TComponent): TDAParams;
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetParams', 'Obj', nil]))
  else
  if Obj is TCustomDASQL then
    Result := TCustomDASQL(Obj).Params
  else
  if Obj is TCustomDADataSet then
    Result := TCustomDADataSet(Obj).Params
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetParams', 'Obj', Obj.ClassName]));
end;

class procedure TDADesignUtils.SetParams(Obj: TComponent; Value: TDAParams);
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetParams', 'Obj', nil]))
  else
  if Obj is TCustomDASQL then
    TCustomDASQL(Obj).Params := Value
  else
  if Obj is TCustomDADataSet then
    TCustomDADataSet(Obj).Params := Value
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetParams', 'Obj', Obj.ClassName]));
end;

class function TDADesignUtils.GetMacros(Obj: TComponent): TMacros;
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetMacros', 'Obj', nil]))
  else
  if Obj is TCustomDASQL then
    Result := TCustomDASQL(Obj).Macros
  else
  if Obj is TCustomDADataSet then
    Result := TCustomDADataSet(Obj).Macros
  else
  if Obj is TDAScript then
    Result := TDAScript(Obj).Macros
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetMacros', 'Obj', Obj.ClassName]));
end;

class procedure TDADesignUtils.SetMacros(Obj: TComponent; Value: TMacros);
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetMacros', 'Obj', nil]))
  else
  if Obj is TCustomDASQL then
    TCustomDASQL(Obj).Macros := Value
  else
  if Obj is TCustomDADataSet then
    TCustomDADataSet(Obj).Macros := Value
  else
  if Obj is TDAScript then
    TDAScript(Obj).Macros := Value
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetMacros', 'Obj', Obj.ClassName]));
end;

class function TDADesignUtils.GetConditions(Obj: TComponent): TDAConditions;
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetConditions', 'Obj', nil]))
  else
  if Obj is TCustomDADataSet then
    Result := TCustomDADataSet(Obj).Conditions
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetConditions', 'Obj', Obj.ClassName]));
end;

class procedure TDADesignUtils.SetConditions(Obj: TComponent; Value: TDAConditions);
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetConditions', 'Obj', nil]))
  else
  if Obj is TCustomDADataSet then
    TCustomDADataSet(Obj).Conditions := Value
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetConditions', 'Obj', Obj.ClassName]));
end;

class procedure TDADesignUtils.Execute(Obj: TComponent);
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.Execute', 'Obj', nil]))
  else
  if Obj is TCustomDASQL then
    TCustomDASQL(Obj).Execute
  else
  if Obj is TCustomDADataSet then
    TCustomDADataSet(Obj).Execute
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.Execute', 'Obj', Obj.ClassName]));
end;

class function TDADesignUtils.GetAfterExecute(Obj: TComponent): DBAccess.TAfterExecuteEvent;
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetAfterExecute', 'Obj', nil]))
  else
  if Obj is TCustomDASQL then
    Result := TCustomDASQL(Obj).AfterExecute
  else
  if Obj is TCustomDADataSet then
    Result := TCustomDADataSet(Obj).AfterExecute
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.GetAfterExecute', 'Obj', Obj.ClassName]));
end;

class procedure TDADesignUtils.SetAfterExecute(Obj: TComponent; Value: DBAccess.TAfterExecuteEvent);
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetAfterExecute', 'Obj', nil]))
  else
  if Obj is TCustomDASQL then
    TCustomDASQL(Obj).AfterExecute := Value
  else
  if Obj is TCustomDADataSet then
    TCustomDADataSet(Obj).AfterExecute := Value
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.SetAfterExecute', 'Obj', Obj.ClassName]));
end;

class procedure TDADesignUtils.Close(Obj: TComponent);
begin
  if Obj = nil then
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.Close', 'Obj', nil]))
  else
  if Obj is TCustomDASQL then
  else
  if Obj is TCustomDADataSet then
    TCustomDADataSet(Obj).Close
  else
    raise Exception.Create(Format(SInternalErrorInvalidValue,
      ['TDADesignUtils.Close', 'Obj', Obj.ClassName]));
end;

class function TDADesignUtils.GetTableName(Obj: TCustomDADAtaSet): string;
begin
  raise Exception.Create(Format(SFunctionMustBeOverriden, ['TDADesignUtils.GetTableName']));
{$IFDEF FPC}
  Result := '';
{$ENDIF}
end;

class procedure TDADesignUtils.SetTableName(Obj: TCustomDADAtaSet; const Value: string);
begin
  raise Exception.Create(Format(SProcedureMustBeOverriden, ['TDADesignUtils.SetTableName']));
end;

class function TDADesignUtils.GetOrderFields(Obj: TCustomDADAtaSet): string;
begin
  raise Exception.Create(Format(SFunctionMustBeOverriden, ['TDADesignUtils.GetOrderFields']));
{$IFDEF FPC}
  Result := '';
{$ENDIF}
end;

class procedure TDADesignUtils.SetOrderFields(Obj: TCustomDADAtaSet; const Value: string);
begin
  raise Exception.Create(Format(SProcedureMustBeOverriden, ['TDADesignUtils.SetOrderFields']));
end;

class procedure TDADesignUtils.PrepareSQL(Obj: TCustomDADAtaSet);
begin
  raise Exception.Create(Format(SProcedureMustBeOverriden, ['TDADesignUtils.PrepareSQL']));
end;

class function TDADesignUtils.GetStoredProcName(Obj: TCustomDADataSet): string;
begin
  raise Exception.Create(Format(SFunctionMustBeOverriden, ['TDADesignUtils.GetStoredProcName']));
{$IFDEF FPC}
  Result := '';
{$ENDIF}
end;

class procedure TDADesignUtils.SetStoredProcName(Obj: TCustomDADataSet; const Value: string);
begin
  raise Exception.Create(Format(SProcedureMustBeOverriden, ['TDADesignUtils.SetStoredProcName']));
end;

class function TDADesignUtils.GetConverterManagerClass: TConverterManagerClass;
begin
  raise Exception.Create(Format(SFunctionMustBeOverriden, ['TDADesignUtils.GetConverterManagerClass']));
{$IFDEF FPC}
  Result := nil;
{$ENDIF}
end;

{$IFDEF USE_SYNEDIT}
class function TDADesignUtils.SQLDialect: integer;
begin
  Result := 0; // sqlStandard
end;
{$ENDIF}

class function TDADesignUtils.DBToolsAvailable: boolean;
begin
  Result := {$IFDEF DBTOOLS}DBToolsService <> nil{$ELSE}False{$ENDIF};
end;

//{$IFDEF DBTOOLS}

class function TDADesignUtils.DBToolsService: TObject;
begin
  Result := nil;
end;

class function TDADesignUtils.NeedToCheckDbTools: TNeedToCheckDbTools;
begin
  raise Exception.Create(Format(SFunctionMustBeOverriden, ['TDADesignUtils.NeedToCheckDbTools']));
{$IFDEF FPC}
  Result := ncNone;
{$ENDIF}
end;

class function TDADesignUtils.GetDBToolsServiceVersion: int64;
begin
  raise Exception.Create(Format(SFunctionMustBeOverriden, ['TDADesignUtils.GetDBToolsServiceVersion']));
{$IFDEF FPC}
  Result := 0;
{$ENDIF}
end;

class function TDADesignUtils.GetDBToolsServiceVersionStr: string;
var
  n: int64;
  i: integer;
begin
  n := GetDBToolsServiceVersion;
  Result := '';
  for i := 0 to 3 do begin
    Result := IntToStr(n and $ffff) + Result;
    if i < 3 then begin
      Result := '.' + Result;
      n := n shr 16;
    end;
  end;
end;

class function TDADesignUtils.GetDBToolsMenuCaption: string;
begin
  Result := '';
  Assert(False, 'Must be overriden');
end;

class function TDADesignUtils.GetFullName(Obj: TComponent): string;
begin
  Result := '';
  if not (Obj is TCustomDAConnection) then
    Assert(False, Obj.ClassName);
end;

class function TDADesignUtils.GetObjectType(Obj: TComponent): string;
begin
  Result := '';
  if not (Obj is TCustomDAConnection) then
    Assert(False, Obj.ClassName);
end;

class procedure TDADesignUtils.SetDBToolsDownloadParams(VerbCheck: boolean; Incompatible: boolean);
begin
  raise Exception.Create(Format(SProcedureMustBeOverriden, ['TDADesignUtils.SetDBToolsDownloadParams']));
end;

class function TDADesignUtils.HasParams(Obj: TComponent): boolean;
begin
  Result := (Obj is TCustomDASQL) or (Obj is TCustomDADataSet);
end;

class procedure TDADesignUtils.GetDBToolsConnectionList(Connection: TCustomDAConnection);
begin
  raise Exception.Create(Format(SProcedureMustBeOverriden, ['TDADesignUtils.GetDBToolsConnectionList']));
end;

class function TDADesignUtils.IsStoredProc(Obj: TComponent): boolean;
begin
  raise Exception.Create(Format(SFunctionMustBeOverriden, ['TCRColFrame.IsStoredProc']));
{$IFDEF FPC}
  Result := False;
{$ENDIF}
end;

class function TDADesignUtils.AreDBToolsUsed: boolean;
begin
  raise Exception.Create(Format(SFunctionMustBeOverriden, ['TCRColFrame.AreDBToolsUsed']));
{$IFDEF FPC}
  Result := False;
{$ENDIF}
end;

class procedure TDADesignUtils.CheckComponent(Component: TComponent);
begin
end;
//{$ENDIF}

end.
