//////////////////////////////////////////////////
//  SQL Server Data Access Components
//  Copyright  1998-2025 Devart. All right reserved.
//  MSConnection Editor
//////////////////////////////////////////////////

{$IFNDEF CLR}

{$I Sdac.inc}

unit MSConnectionEditor;
{$ENDIF}

interface

uses
{$IFDEF FPC}
  LResources, LCLType,
{$ENDIF}
{$IFDEF MSWINDOWS}
  Windows,
{$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms,
  {$IFNDEF FPC}Mask,{$ENDIF} Dialogs, StdCtrls, ExtCtrls, ComCtrls, Buttons,
{$IFDEF DBTOOLS}
  DBToolsClient,
{$ENDIF}
  CRTypes, DBAccess, MSClasses, MSAccess, DAConnectionEditor;

{$IFDEF MSWINDOWS}
const
  WM_SETDATABASETEXT = WM_USER + 1;
{$ENDIF}

type
  TMSConnectionEditorForm = class(TDAConnectionEditorForm)
    lbDatabase: TLabel;
    edDatabase: TComboBox;
    lbPort: TLabel;
    edPort: TEdit;
    lbAuth: TLabel;
    edAuth: TComboBox;
    btQueryAnalyzer: TButton;
    btManagementStudio: TButton;
  {$IFDEF MSWINDOWS}
  {$IFNDEF FPC}    
    cbExisting: TComboBox;
    lbExisting: TLabel;
  {$ENDIF}
  {$ENDIF}
    procedure edAuthChange(Sender: TObject);
    procedure edPortExit(Sender: TObject);
    procedure edDatabaseDropDown(Sender: TObject);
    procedure btQueryAnalyzerClick(Sender: TObject);
    procedure edPortChange(Sender: TObject);
    procedure edDatabaseExit(Sender: TObject);
    procedure edDatabaseKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure edDatabaseChange(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure btManagementStudioClick(Sender: TObject);
    procedure edServerDropDown(Sender: TObject); override;
    procedure cbExistingChange(Sender: TObject);
  private
    FDataBaseText: string;
    FCurrItemIndex: Integer;
    FListGot: boolean;
  {$IFDEF MSWINDOWS}
    procedure WMSetDataBaseText(var Message: TMessage); message WM_SETDATABASETEXT;
  {$ENDIF}
    function GetLocalConnection: TCustomMSConnection;
    procedure AssignDatabase(const Value: string);
  protected
  {$IFDEF DBTOOLS}
    function GetExistingConnectionComboBox: TComboBox; override;
  {$ENDIF}
    procedure Resize; override;
    procedure DoInit; override;
    procedure FillInfo(InfoConnection: TCustomDAConnection); override;
    procedure DoSaveConnection; override;

    procedure ConnToControls; override;

  {$IFDEF MSWINDOWS}
    procedure AddServerToList; override;
    procedure GetDatabaseList(List: TStrings);
  {$ENDIF}
    function IsValidKeyValue(Value: string; Name: string): boolean;
  public
    property LocalConnection: TCustomMSConnection read GetLocalConnection;
  end;

implementation

{$IFNDEF FPC}
{$IFDEF CLR}
{$R MSConnectionEditor.dfm}
{$ELSE}
{$R *.dfm}
{$ENDIF}
{$ELSE}
{$R *.lfm}
{$ENDIF}

uses
  CRFunctions, DacVcl{$IFDEF SDAC}, MSDesign{$ENDIF}, Registry;

{ TMSConnectionEditorForm }

procedure TMSConnectionEditorForm.Resize;
begin
  inherited;

  if btManagementStudio <> nil then begin
    btManagementStudio.Top := BtnPanel.Top + btOk.Top;
    btQueryAnalyzer.Top := btManagementStudio.Top;
  end;
end;

procedure TMSConnectionEditorForm.DoInit;
begin
  inherited;
  
  FInDoInit := True;
  try
    lbVersion.Caption := SDACVersion + ' ';
    UpdateVersionPosition;

  {$IFDEF BETA}
    lbEdition.Caption := 'Beta';
    lbEdition.Font.Color := clRed;
  {$ELSE}
  {$IFDEF RC}
    lbEdition.Caption := 'Release Candidate';
    lbEdition.Font.Color := clGreen;
  {$ELSE}
  {$IFDEF TRIAL}
    lbEdition.Caption := 'Trial Edition';
  {$ELSE} //ORIGINAL
  {$IFDEF STD}
    lbEdition.Caption := 'Standard Edition';
  {$ELSE}
    lbEdition.Caption := 'Professional Edition';
  {$ENDIF}
  {$ENDIF}
  {$ENDIF}
  {$ENDIF}

  {$IFDEF MSWINDOWS}
  {$IFNDEF FPC}
    if not DADesignUtilsClass.DBToolsAvailable and cbExisting.Visible then
    begin
      lbExisting.Visible := False;
      cbExisting.Visible := False;
      Panel.Height := Panel.Height + 33;
      Panel.Top := cbExisting.Top;
    end;
  {$ENDIF}
  {$ENDIF}

    btQueryAnalyzer.Visible := {$IFDEF MSWINDOWS}IsServerToolInstalled(stQueryAnalyser) and (LocalConnection.Options.Provider <> prCompact){$ELSE}False{$ENDIF};
    btManagementStudio.Visible := {$IFDEF MSWINDOWS}IsServerToolInstalled(stManagementStudio) and (LocalConnection.Options.Provider <> prCompact){$ELSE}False{$ENDIF};

    FCurrItemIndex := -1;
    if LocalConnection.Options.Provider = prCompact then begin
      //cbLoginPrompt.Enabled := False;
      lbAuth.Enabled := False;
      edAuth.Enabled := False;
      lbUsername.Enabled := False;
      edUserName.Enabled := False;
      lbServer.Enabled := False;
      edServer.Enabled := False;
      lbPort.Enabled := False;
      edPort.Enabled := False;
    end;
    FListGot := False;
  finally
    FInDoInit := False;
  end;
end;

procedure TMSConnectionEditorForm.DoSaveConnection;
begin
  inherited;

  TCustomMSConnection(Connection).Database := LocalConnection.Database;
  if LocalConnection is TMSConnection then begin
    TMSConnection(Connection).Authentication := TMSConnection(LocalConnection).Authentication;
    TMSConnection(Connection).Port := TMSConnection(LocalConnection).Port;
  end;
end;

procedure TMSConnectionEditorForm.FillInfo(InfoConnection: TCustomDAConnection);
var
  MSInfoConnection: TCustomMSConnection;
  OldLoginPrompt: boolean;
  OldConnectionTimeout: integer;
  MSSQLConnection: TMSSQLConnection;
  St: string;
begin
  MSInfoConnection := TCustomMSConnection(InfoConnection);

  OldLoginPrompt := MSInfoConnection.LoginPrompt;
  OldConnectionTimeout := 1;
  if MSInfoConnection is TMSConnection then
    OldConnectionTimeout := TMSConnection(MSInfoConnection).ConnectionTimeout;
  try
    MSInfoConnection.LoginPrompt := False;
    if not IsConnected and
      not ((MSInfoConnection.Username = '') and (MSInfoConnection.Password = '') and
      (MSInfoConnection.Server = '') and (MSInfoConnection.Database = '')) then
      try
        ShowState(True);
        if MSInfoConnection is TMSConnection then
          TMSConnection(MSInfoConnection).ConnectionTimeout := 1;
        MSInfoConnection.Connect;
      except
        on E: Exception do
        begin
          //Application.ShowException(E); - silent exception. Please see CR MyDAC 3443
        end;
      end;
    meInfo.Lines.Clear;
    MSSQLConnection := TMSSQLConnection(TDBAccessUtils.GetIConnection(MSInfoConnection));
    if MSSQLConnection <> nil then
    begin
      if MSInfoConnection.Connected then begin
        St := MSSQLConnection.GetServerVersionFull;
        if st <> ':' then
          meInfo.Lines.Add(St);
      end;
      if (MSInfoConnection.Options.Provider <> prCompact) and
         (MSInfoConnection.Options.Provider <> prDirect)
      then begin
        St := MSSQLConnection.GetClientVersionFull + ': ' + MSSQLConnection.GetClientVersion;
        if St <> ': ' then
          meInfo.Lines.Add(St);
      end;
    end;
  finally
    MSInfoConnection.LoginPrompt := OldLoginPrompt;
    if MSInfoConnection is TMSConnection then
      TMSConnection(MSInfoConnection).ConnectionTimeout := OldConnectionTimeout;
    ShowState(False);
  end;
end;

procedure TMSConnectionEditorForm.ConnToControls;
begin
  inherited;

  edDatabase.Text := LocalConnection.Database;
  if LocalConnection is TMSConnection then begin
    edAuth.ItemIndex := Ord(TMSConnection(LocalConnection).Authentication);
    edUsername.Enabled := TMSConnection(LocalConnection).Authentication = auServer;
    edPassword.Enabled := edUsername.Enabled;
    lbUsername.Enabled := edUsername.Enabled;
    lbPassword.Enabled := edUsername.Enabled;
    cbLoginPrompt.Enabled := edUsername.Enabled;
    edPort.Text := IntToStr(TMSConnection(LocalConnection).Port);
  end;
end;

procedure TMSConnectionEditorForm.edServerDropDown(Sender: TObject);
begin
  if FListGot then
    Exit;
    
  FListGot := True;
  
  inherited;
end;

procedure TMSConnectionEditorForm.cbExistingChange(Sender: TObject);
{$IFDEF DBTOOLS}
var
  ConnectionStrList: TStrings;
  MSConnection: TMSConnection;
  IntegratedSecurity: string;
begin
  if FInDoInit or (cbExisting.ItemIndex < 0) then
    Exit;
  ConnectionStrList := GetDBToolsService(DADesignUtilsClass).GetConnectionStrList(cbExisting.Text);
  try
    FInExistingChange := True;
    MSConnection := TMSConnection(LocalConnection);
    MSConnection.Server := ConnectionStrList.Values['Data Source'];
    MSConnection.Username := ConnectionStrList.Values['User ID'];
    MSConnection.Password := ConnectionStrList.Values['Password'];
    MSConnection.Database := ConnectionStrList.Values['Initial Catalog'];
    MSConnection.Password := ConnectionStrList.Values['Password'];
    IntegratedSecurity := ConnectionStrList.Values['Integrated Security'];
    if (IntegratedSecurity = 'SSPI') or StrToBool(IntegratedSecurity) then
      MSConnection.Authentication := auWindows
    else
      MSConnection.Authentication := auServer;
  finally
    ConnToControls;
    FInExistingChange := False;
  end;
end;
{$ELSE}
begin
end;
{$ENDIF}

{$IFDEF MSWINDOWS}
procedure TMSConnectionEditorForm.AddServerToList;
var
  ConnectKey: string;
  ValueNames, Values: TStringList;
  i: integer;
  s: string;
begin
  if LocalConnection.Options.Provider = prCompact then begin
    if FRegistry <> nil then begin
      ValueNames := nil;
      Values := nil;
      ConnectKey := string(FRegistry.CurrentPath);
      try
        ValueNames := TStringList.Create;
        Values := TStringList.Create;

        Values.Add(LocalConnection.Database); // Add current database at first position

        FRegistry.CloseKey;
        FRegistry.OpenKey(ConnectKey + '\Everywhere', True);

        FRegistry.GetValueNames(ValueNames);
        ValueNames.Sort;

        for i := 0 to ValueNames.Count - 1 do begin
          s := Trim(FRegistry.ReadString(ValueNames[i]));
          if (s <> '') and (Values.IndexOf(s) = -1) then
            Values.Add(s);
          FRegistry.DeleteValue(ValueNames[i]); // Clear old list
        end;

        // Store updated list in registry
        for i := 0 to Values.Count - 1 do begin
          s := Format('Database %d', [i]);
          FRegistry.WriteString(s, Values[i]);
        end;

      finally
        ValueNames.Free;
        Values.Free;
      end;
    end;
  end
  else
    inherited;
end;

procedure TMSConnectionEditorForm.GetDatabaseList(List: TStrings);
var
  ConnectKey: string;
  ValueNames, Values: TStringList;
  i: integer;
begin
  List.Clear;
  if FRegistry <> nil then begin
    ValueNames := nil;
    Values := nil;
    try
      ValueNames := TStringList.Create;
      Values := TStringList.Create;
      ConnectKey := string(FRegistry.CurrentPath);
      try
        FRegistry.CloseKey;
        if FRegistry.OpenKey(ConnectKey + '\Everywhere', False) then begin
          FRegistry.GetValueNames(ValueNames);
          ValueNames.Sort;
          for i := 0 to ValueNames.Count - 1 do
            if IsValidKeyValue(ValueNames[i], 'Database') then
              List.Add(FRegistry.ReadString(ValueNames[i]));
        end;
      finally
        FRegistry.CloseKey;
        FRegistry.OpenKey(ConnectKey, False);
      end;
    finally
      ValueNames.Free;
      Values.Free;
    end;
  end;
  List.Add('<Browse...>');
end;
{$ENDIF}

function TMSConnectionEditorForm.IsValidKeyValue(Value: string; Name: string): boolean;
var
  p: integer;
begin
  p := Pos(AnsiUpperCase(Name), AnsiUpperCase(Value));
  if p <> 0 then begin
    Inc(p, Length(Name) - 1);
    if p < Length(Value) then
      Inc(p);
    while (Byte(Value[p]) in [$30..$30+9, $20]) and (p <> Length(Value)) do
      Inc(p);
    Result := p = Length(Value);
  end
  else
    Result := False;
end;

procedure TMSConnectionEditorForm.edDatabaseDropDown(Sender: TObject);
var
  List: TStringList;
  OldLoginPrompt: Boolean;
begin
  StartWait;
  try
  {$IFDEF MSWINDOWS}
    if LocalConnection.Options.Provider = prCompact then begin
      List := TStringList.Create;
      try
        GetDatabaseList(List);
        AssignStrings(List, edDatabase.Items);
        if edDatabase.Items.Count < 20 then
          edDatabase.DropDownCount := edDatabase.Items.Count
        else
          edDatabase.DropDownCount := 20;
      finally
        List.Free;
      end;
    end
    else
  {$ENDIF}
    begin
      edDatabase.Items.Clear;
      OldLoginPrompt := LocalConnection.LoginPrompt;
      List := TStringList.Create;
      try
        LocalConnection.LoginPrompt := False;
        MSAccess.GetDatabasesList(LocalConnection, List);
        List.Sort;
        edDatabase.Items.Assign(List);
        if edDatabase.Items.Count < 20 then
          edDatabase.DropDownCount := edDatabase.Items.Count
        else
          edDatabase.DropDownCount := 20;
      finally
        // edDatabase.Text := LocalConnection.Database;
        List.Free;
        LocalConnection.LoginPrompt := OldLoginPrompt;
      end;
    end;
  finally
    StopWait;
  end;
end;

procedure TMSConnectionEditorForm.edAuthChange(Sender: TObject);
begin
  if FInDoInit then
    Exit;

  try
    Assert(LocalConnection is TMSConnection);
    if TMSConnection(LocalConnection).Authentication <> TMSAuthentication(edAuth.ItemIndex) then begin
      TMSConnection(LocalConnection).Authentication := TMSAuthentication(edAuth.ItemIndex);
      Modified := True;
      PerformDisconnect;
      edUsername.Enabled := TMSConnection(LocalConnection).Authentication = auServer;
      edPassword.Enabled := edUsername.Enabled;
      lbUsername.Enabled := edUsername.Enabled;
      lbPassword.Enabled := edUsername.Enabled;
      cbLoginPrompt.Enabled := edUsername.Enabled;
    {$IFDEF DBTOOLS}
      ChooseExistingConnection;
    {$ENDIF}
    end;
  finally
    ShowState;
  end;
end;

procedure TMSConnectionEditorForm.btQueryAnalyzerClick(Sender: TObject);
begin
  SaveControlData;
{$IFDEF MSWINDOWS}
  RunServerTool(stQueryAnalyser, LocalConnection);
{$ENDIF}
end;

procedure TMSConnectionEditorForm.btManagementStudioClick(Sender: TObject);
begin
  SaveControlData;
{$IFDEF MSWINDOWS}
  RunServerTool(stManagementStudio, LocalConnection);
{$ENDIF}
end;

procedure TMSConnectionEditorForm.edPortExit(Sender: TObject);
begin
  if FInDoInit then
    Exit;

  try
    Assert(LocalConnection is TMSConnection);
    try
      if edPort.Text = '' then begin
        TMSConnection(LocalConnection).Port := DefaultSDACPort;
        edPort.Text := IntToStr(DefaultSDACPort);
      end
      else
        TMSConnection(LocalConnection).Port := StrToInt(edPort.Text);
    {$IFDEF DBTOOLS}
      ChooseExistingConnection;
    {$ENDIF}
    except
      PageControl.ActivePage := shConnect;
      edPort.SetFocus;
      edPort.SelectAll;
      raise;
    end;
  finally
    ShowState;
  end;
end;

procedure TMSConnectionEditorForm.edPortChange(Sender: TObject);
var
  Port: integer;
begin
  if FInDoInit or (edPort.Text = '') then
    Exit;

  try
    Assert(LocalConnection is TMSConnection);
    try
      Port := StrToInt(edPort.Text);
      if TMSConnection(LocalConnection).Port <> Port then begin
        PerformDisconnect;
        TMSConnection(LocalConnection).Port := Port;
        Modified := True;
      {$IFDEF DBTOOLS}
        ChooseExistingConnection;
      {$ENDIF}
      end;
    except
      PageControl.ActivePage := shConnect;
      edPort.SetFocus;
      edPort.SelectAll;
      raise;
    end;
  finally
    ShowState;
  end;
end;

procedure TMSConnectionEditorForm.edDatabaseExit(Sender: TObject);
begin
{$IFDEF DBTOOLS}
  ChooseExistingConnection;
{$ENDIF}
end;

procedure TMSConnectionEditorForm.edDatabaseKeyUp(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if Key = VK_RETURN then
    edDatabaseExit(Sender);
end;

procedure TMSConnectionEditorForm.edDatabaseChange(Sender: TObject);
var
  Dialog: TOpenDialog;
  OldConnected: boolean;
begin
  if FInDoInit then
    Exit;

  try
    if LocalConnection.Options.Provider <> prCompact then
      AssignDatabase(edDatabase.Text)
    else begin
      if edDatabase.Text = '<Browse...>' then begin
        Dialog := nil;
        try
          Dialog := TOpenDialog.Create(nil);
        {$IFDEF MSWINDOWS}
          Dialog.Filter := 'SQL Server Database Files (*.sdf)|*.sdf|All Files (*.*)|*.*';
        {$ELSE}
          Dialog.Filter := 'All Files (*)|*';
        {$ENDIF}
          Dialog.Options := Dialog.Options + [ofPathMustExist];
          if Dialog.Execute then begin
            LocalConnection.Connected := False;
            FDataBaseText := Dialog.FileName;
          {$IFDEF MSWINDOWS}
            PostMessage(Handle, WM_SETDATABASETEXT, 0, 0);
          {$ENDIF}
          end
          else
            edDatabase.ItemIndex := FCurrItemIndex;
        finally
          Dialog.Free;
        end;
      end
      else
        FCurrItemIndex := edDatabase.Items.IndexOf(edDatabase.Text);
      OldConnected := LocalConnection.Connected;
      try
        LocalConnection.Connected := False;
        LocalConnection.Database := edDatabase.Text;
      {$IFDEF DBTOOLS}
        ChooseExistingConnection;
      {$ENDIF}
      finally
        LocalConnection.Connected := OldConnected;
      end;
    end;
  finally
    ShowState;
  end;
end;

{$IFDEF MSWINDOWS}
procedure TMSConnectionEditorForm.WMSetDataBaseText(var Message: TMessage);
var
  OldConnected: boolean;
begin
  edDatabase.SetFocus;
  edDatabase.Text := FDataBaseText;
  OldConnected := LocalConnection.Connected;
  try
    LocalConnection.Connected := False;
    LocalConnection.Database := edDatabase.Text;
  finally
    LocalConnection.Connected := OldConnected;
  end;
  edDatabase.SelectAll;
end;
{$ENDIF}

function TMSConnectionEditorForm.GetLocalConnection: TCustomMSConnection;
begin
  Result := FLocalConnection as TCustomMSConnection;
end;

procedure TMSConnectionEditorForm.AssignDatabase(const Value: string);
begin
  if LocalConnection.Database <> Value then begin
    PerformDisconnect;
    LocalConnection.Database := Value;
    Modified := True;
  end;
end;

procedure TMSConnectionEditorForm.FormShow(Sender: TObject);
begin
  inherited;
  TDBAccessUtils.SetLockLoginPrompt(LocalConnection, True);
end;

procedure TMSConnectionEditorForm.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  inherited;
  if CanClose then
    TDBAccessUtils.SetLockLoginPrompt(LocalConnection, False);
end;

{$IFDEF DBTOOLS}
function TMSConnectionEditorForm.GetExistingConnectionComboBox: TComboBox;
begin
  Result := cbExisting;
end;
{$ENDIF}

end.
