//////////////////////////////////////////////////
//  SQL Server Data Access Components
//  Copyright  1998-2025 Devart. All right reserved.
//  SDAC Design
//////////////////////////////////////////////////

{$I Sdac.inc}
unit MSDesign;

interface

uses
{$IFDEF MSWINDOWS}
  Windows, Messages, Registry,
{$ENDIF}
  Graphics, Controls, Forms, Dialogs, StdCtrls,
{$IFDEF FPC}
  PropEdits, ComponentEditors,
{$ELSE}
  DesignIntf, DesignEditors,
  {$IFNDEF BCB}FldLinks, ColnEdit,{$ENDIF}
{$ENDIF}
  SysUtils, Classes, TypInfo, DADesign, CRDesign, CRTypes, DBAccess, MSAccess,
{$IFDEF DBTOOLS}
  DBToolsClient,
{$ENDIF}
{$IFNDEF STD}
{$IFDEF MSWINDOWS}
  MSCompactConnection,
{$ENDIF}
  MSTransaction,
{$ENDIF}
  SdacVcl;

type
{ ------------  SDac property editors ----------- }

  TMSConnectStringPropertyEditor = class(TStringProperty)
  protected
    FForm: TForm;
    FSucceeded: boolean;
    FConnectString: string;
    procedure DoActivate(Sender: TObject);
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure Edit; override;
  end;

  TMSServerNamePropertyEditor = class (TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
    function AutoFill: Boolean; override;
  end;

  TMSDatabaseNamePropertyEditor = class (TStringProperty)
  protected
    procedure GetDialogOptions(Dialog: TOpenDialog); virtual;
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
    function AutoFill: boolean; override;
  end;

{$IFNDEF STD}
  TMSQueuePropertyEditor = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

  TMSServicePropertyEditor = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

  TMSContractPropertyEditor = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

  TMSTargetDatabaseNamePropertyEditor = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

  TMSTableNamesEditor = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure Edit; override;
  end;

  TMSLocaleIdentifierPropertyEditor = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

{$ENDIF}

  TMSConnectDialogPropertyEditor = class(TComponentProperty)
  private
    FCheckProc: TGetStrProc;
    procedure CheckComponent(const Value: string);
  public
    procedure GetValues(Proc: TGetStrProc); override;
  end;

{ ------------  SDac component editors ----------- }

  TMSConnectionEditor = class(TDAConnectionEditor)
  protected
    FQueryAnalyserIndex: integer;
    FManagementStudioIndex: integer;
    procedure InitVerbs; override;
  {$IFNDEF STD}
    procedure Convert;
  {$ENDIF}
  public
    procedure ExecuteVerb(Index: integer); override;
  end;

{$IFNDEF STD}
{$IFDEF MSWINDOWS}
  TMSCompactConnectionEditor = class(TDAConnectionEditor)
  protected
    FManagementStudioIndex: integer;
    procedure InitVerbs; override;
    procedure Convert;
  public
    procedure ExecuteVerb(Index: integer); override;
  end;
{$ENDIF}
{$ENDIF}

  TMSDataSetEditor = class(TDAComponentEditor);

  TMSQueryEditor = class(TMSDataSetEditor)
  protected
    FQueryAnalyserIndex: integer;
    FManagementStudioIndex: integer;
    procedure InitVerbs; override;
  public
    procedure ExecuteVerb(Index: integer); override;
  end;

  TMSSQLEditor = class(TDASQLEditor)
  protected
    procedure InitVerbs; override;
  end;

  TMSTableEditor = class(TMSDataSetEditor)
  protected
    procedure InitVerbs; override;
  end;

  TMSStoredProcEditor = class(TMSDataSetEditor)
  protected
    procedure InitVerbs; override;
  public
    procedure ExecuteVerb(Index: integer); override;
  end;

  TMSUpdateSQLEditor = class(TDAUpdateSQLEditor)
  protected
    procedure InitVerbs; override;
  end;

  TMSScriptEditor = class(TDASQLEditor)
  protected
    procedure InitVerbs; override;
  end;

{$IFNDEF STD} 
  TMSDumpEditor = class(TDAComponentEditor)
  protected
    procedure InitVerbs; override;
  end;
{$ENDIF}

  TMSTableDataEditor = class(TDAComponentEditor)
  protected
    procedure InitVerbs; override;
  end;

  TMSTableTypeNameEditor = class (TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
    function AutoFill: boolean; override;
  end;

{$IFNDEF FPC}
  TMSConnectionList = class (TDAConnectionList)
  protected
    function GetConnectionType: TCustomDAConnectionClass; override;
  end;

  TMSDesignNotification = class(TDADesignNotification)
  public
    procedure ItemInserted(const ADesigner: IDesigner; AItem: TPersistent); override;
    function CreateConnectionList: TDAConnectionList; override;
    procedure SelectionChanged(const ADesigner: IDesigner;
      const ASelection: IDesignerSelections); override;
    function GetConnectionPropertyName: string; override;
  end;
{$ENDIF}

procedure Register;

{$IFDEF MSWINDOWS}
type
  TServerTool = (stQueryAnalyser, stManagementStudio);
  
procedure RunServerTool(ServerTool: TServerTool; Connection: TCustomMSConnection; const SQL: TStrings = nil);
procedure RunServerToolConnection(ServerTool: TServerTool; Connection: TCustomMSConnection; const SQLText: string);
procedure RunServerToolDataSet(ServerTool: TServerTool; DataSet: TCustomMSDataSet);
procedure RunServerToolMSSQL(ServerTool: TServerTool; MSSQL: TMSSQL);

function IsServerToolInstalled(ServerTool: TServerTool): boolean;
{$ENDIF}

implementation

uses
{$IFNDEF FPC}
  ToolsAPI, MSMenu,
{$ENDIF}
{$IFDEF MSWINDOWS}
  ShellAPI, ActiveX, ComObj, OLEDBAccess, OLEDBIntf, OLEDBC, MSConnectionString,
{$ENDIF}
  CRFunctions, DB, DAConsts, MSClasses,
  MSDesignUtils, MSConnectionEditor, MSQueryEditor, MSSQLEditor, MSStoredProcEditor,
  MSTableEditor, DATableEditor, MSUpdateSQLEditor, DAScriptEditor, MSScript
{$IFNDEF STD}
  , MSLoader, MSDump, MSDumpEditor, MSNamesEditor, MSServiceBroker{$IFDEF MSWINDOWS}, MSCompactConnectionEditor{$ENDIF} 
{$ENDIF}
  ;

var
  TmpFiles: TStringList;
  CachedServerList: TStrings = nil;

{$IFDEF MSWINDOWS}
function GetServerToolCommand(ServerTool: TServerTool): string;
type
  TRegKeyString = record
    Root: HKEY;
    Path, KeyName: string;
    AdditionalPath: string;
  end;
  TRegKeyArray = array[0..2] of TRegKeyString;
const
  QAKeyPaths: TRegKeyArray =
    ((Root: HKEY_LOCAL_MACHINE; Path: 'SOFTWARE\Microsoft\Microsoft SQL Server\80\Tools\ClientSetup'; KeyName: 'SQLPath'; AdditionalPath: '\Binn\isqlw'),
     (Root: HKEY_LOCAL_MACHINE; Path: 'SOFTWARE\Microsoft\MSSQLServer\Setup'; KeyName: 'SQLPath'; AdditionalPath: '\Binn\isqlw'),
     (Root: HKEY_CLASSES_ROOT; Path: 'SQLFile\Shell\open\command'; KeyName: ''; AdditionalPath: '\isqlw'));
  MMSKeyPaths: TRegKeyArray =
    ((Root: HKEY_LOCAL_MACHINE; Path: 'SOFTWARE\Microsoft\Microsoft SQL Server\100\Tools\ClientSetup'; KeyName: 'SQLPath'; AdditionalPath: '\Binn\VSShell\Common7\IDE\ssms'),
     (Root: HKEY_LOCAL_MACHINE; Path: 'SOFTWARE\Microsoft\Microsoft SQL Server\90\Tools\ClientSetup'; KeyName: 'SQLPath'; AdditionalPath: '\Binn\VSShell\Common7\IDE\sqlwb'),
     (Root: HKEY_CLASSES_ROOT; Path: 'sqlwb.sql.9.0\Shell\Open\Command'; KeyName: ''; AdditionalPath: '\sqlwb'));
var
  Reg: TRegistry;
  i: integer;
  KeyPaths: TRegKeyArray;
begin
  case ServerTool of
    stQueryAnalyser: begin
      Result := 'isqlw';
      KeyPaths := QAKeyPaths;
    end;
    stManagementStudio: begin
      Result := 'sqlwb';
      KeyPaths := MMSKeyPaths;
    end
    else
      Assert(False);
  end;

  Reg := TRegistry.Create(KEY_READ);
  try
    for i := Low(KeyPaths) to High(KeyPaths) do begin
      Reg.RootKey := KeyPaths[i].Root;
      if Reg.OpenKeyReadOnly(KeyPaths[i].Path) then begin
        Result := Reg.ReadString(KeyPaths[i].KeyName);
        Reg.CloseKey;
        if Result <> '' then begin
          Result := Result + KeyPaths[i].AdditionalPath;
          Break;
        end;
      end;
    end;
  finally
    Reg.Free;
  end;
end;

function IsServerToolInstalled(ServerTool: TServerTool): boolean;
var
  Cmd: string;
begin
  Cmd := GetServerToolCommand(ServerTool);
  Result := FileExists(Cmd + '.exe');
end;

procedure RunServerTool(ServerTool: TServerTool; Connection: TCustomMSConnection; const SQL: TStrings = nil);
var
  Cmd, CmdParam: string;
  TmpPath, TmpFileName: array[0..MAX_PATH] of Char;
  SqlFileHandle: integer;
  SqlFileName: string;
  Code: integer;
begin
  if Connection = nil then
    DatabaseError(SConnectionNotDefined);

  CmdParam := ''; //'-1';

  if Connection.Server <> '' then
    CmdParam := CmdParam + ' -S ' + Connection.Server;

  if Connection.Database <> '' then
    CmdParam := CmdParam + ' -d ' + Connection.Database;

  if (Connection is TMSConnection) and (TMSConnection(Connection).Authentication = auWindows) then
    CmdParam := CmdParam + ' -E'
  else begin
    if Connection.Username <> '' then
      CmdParam := CmdParam + ' -U ' + Connection.Username;
    CmdParam := CmdParam + ' -P ' + Connection.Password;
  end;

  if (SQL <> nil) and (SQL.Count > 0) then begin
    Assert(GetTempPath(MAX_PATH, TmpPath) <> 0, 'Error in call GetTempPath');
    Assert(GetTempFileName(TmpPath, 'sql'#0, 0, TmpFileName) <> 0, 'Error in call GetTempFileName');

    if ServerTool = stManagementStudio then begin
      SqlFileName := ChangeFileExt(TmpFileName, '.sql');
      SqlFileHandle := FileCreate(SqlFileName);
      Assert(SqlFileHandle > 0);
      FileClose(SqlFileHandle);
      TmpFiles.Add(SqlFileName);
      SQl.SaveToFile(SqlFileName);
    end
    else
      SQl.SaveToFile(TmpFileName);

    TmpFiles.Add(TmpFileName);
    case ServerTool of
      stQueryAnalyser:
        CmdParam := CmdParam + ' -f ' + StrPas(TmpFileName);
      stManagementStudio:
        CmdParam := '"' + SqlFileName + '" ' + CmdParam;
      else
        Assert(False);
    end;
  end;

  Cmd := GetServerToolCommand(ServerTool);

  Code := ShellExecute(0, nil, @Cmd[1], @CmdParam[1], nil, SW_SHOWNORMAL);
  if Code <= 32 then
    raise Exception.CreateFmt('Error executing "%s %s". Code = %d', [Cmd, CmdParam, Code]);
end;

procedure RunServerToolConnection(ServerTool: TServerTool; Connection: TCustomMSConnection; const SQLText: string);
var
  SQL: TStringList;
begin
  SQL := TStringList.Create;
  try
    SQL.Add(SQLtext);
    RunServerTool(ServerTool, Connection, SQL);
  finally
    SQL.Free;
  end;
end;

procedure RunServerToolDataSet(ServerTool: TServerTool; DataSet: TCustomMSDataSet);
begin
  RunServerToolConnection(ServerTool, DataSet.Connection, TMSAccessUtils.GetCommandSQL(DataSet));
end;

procedure RunServerToolMSSQL(ServerTool: TServerTool; MSSQL: TMSSQL);
begin
  RunServerToolConnection(ServerTool, MSSQL.Connection, TMSAccessUtils.GetCommandSQL(MSSQL));
end;
{$ENDIF}

{ TMSConnectStringProperty }

function TMSConnectStringPropertyEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;

procedure TMSConnectStringPropertyEditor.DoActivate(Sender: TObject);
{$IFDEF MSWINDOWS}
var
  DataInit: IDataInitialize;
  DBPrompt: IDBPromptInitialize;
  DataSource: IUnknown;
  InitStr: PWideChar;
  InitialString: WideString;
begin
  FSucceeded := False;
  DataInit := CreateComObject(CLSID_DataLinks) as IDataInitialize;
  InitialString := WideString(FConnectString);

  if Pos('Provider=', FConnectString) = 0 then begin
    if InitialString <> '' then
      InitialString := InitialString + ';';
    InitialString := InitialString + 'Provider=SQLOLEDB.1';
  end;

  DataInit.GetDataSource(nil, CLSCTX_INPROC_SERVER, PWideChar(InitialString), IID_IUnknown, DataSource);
  DBPrompt := CreateComObject(CLSID_DataLinks) as IDBPromptInitialize;

  if Succeeded(DBPrompt.PromptDataSource(nil, FForm.Handle,
    DBPROMPTOPTIONS_PROPERTYSHEET + DBPROMPTOPTIONS_DISABLE_PROVIDER_SELECTION, 0, nil, nil, IID_IUnknown, DataSource)) then
  begin
    InitStr := nil;
    DataInit.GetInitializationString(DataSource, True, InitStr );
    FConnectString := InitStr;
    FSucceeded := True;
  end;

  PostMessage(FForm.Handle, WM_CLOSE, 0, 0);
{$ELSE}
begin
{$ENDIF}
end;

procedure TMSConnectStringPropertyEditor.Edit;
var
  Connection: TCustomMSConnection;
{$IFDEF MSWINDOWS}
  OLEStringBuilder: TOLEDBConnectionStringBuilder;
{$ENDIF}
begin
  Connection := TCustomMSConnection(GetComponent(0));

{$IFDEF MSWINDOWS}
  OLEStringBuilder := TOLEDBConnectionStringBuilder.GetInstance(Connection);
  try
    OLEStringBuilder.ConnectionString := Connection.ConnectString;
    OLEStringBuilder.Convert;

    FConnectString := OLEStringBuilder.ConnectionString;
{$ELSE}
  FConnectString := Connection.ConnectString;
{$ENDIF}

    FForm := TForm.Create(nil);
    try
      FForm.BorderStyle := bsNone;
      FForm.Position := poScreenCenter;
      FForm.Width := 10;
      FForm.Height := 10;
      FForm.OnActivate := DoActivate;
      FForm.ShowModal;
    finally
      FForm.Free;
    end;
    if FSucceeded then begin {Cannot move to DoActivate}
    {$IFDEF MSWINDOWS}
      OLEStringBuilder.ConnectionString := FConnectString;
      OLEStringBuilder.WithUnknownParams := True;
      OLEStringBuilder.Revert;
      Connection.ConnectString := OLEStringBuilder.ConnectionString;
    {$ELSE}
      Connection.ConnectString := FConnectString;
    {$ENDIF}
      Modified;
    end;
{$IFDEF MSWINDOWS}
  finally
    OLEStringBuilder.Free;
  end;
{$ENDIF}
end;

{ TMSServerNamePropertyEditor }

function TMSServerNamePropertyEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList];
end;

function TMSServerNamePropertyEditor.AutoFill: Boolean;
begin
  Result := False;
end;

procedure TMSServerNamePropertyEditor.GetValues(Proc: TGetStrProc);
var
  List: TStringList;
  i: integer;
  OldCursor: TCursor;
  Connection: TMSConnection;
  IsEverywhere: boolean;
begin
  List := TStringList.Create;
  OldCursor := Screen.Cursor;
  Screen.Cursor := crSQLWait;
  try
    Connection := nil;
    if GetComponent(0) is TMSConnection then
      Connection := GetComponent(0) as TMSConnection;

    IsEverywhere := (Connection <> nil) and (Connection.Options.Provider = prCompact);
    if not IsEverywhere then begin
      if CachedServerList = nil then
        CachedServerList := TStringList.Create;

      if CachedServerList.Count = 0 then begin
        GetServerList(List);
        CachedServerList.Assign(List);
      end else
        List.Assign(CachedServerList);
    end;

    for i := 0 to List.Count - 1 do
      Proc(List[i]);
  finally
    List.Free;
    Screen.Cursor := OldCursor;
  end;
end;

{ TMSDatabaseNamePropertyEditor }

procedure TMSDatabaseNamePropertyEditor.GetDialogOptions(Dialog: TOpenDialog);
begin
{$IFDEF MSWINDOWS}
  Dialog.Filter := 'SQL Server Database Files (*.sdf)|*.sdf|All Files (*.*)|*.*';
{$ELSE}
  Dialog.Filter := 'All Files (*)|*';
{$ENDIF}
  Dialog.Options := Dialog.Options + [ofPathMustExist];
end;

function TMSDatabaseNamePropertyEditor.GetAttributes: TPropertyAttributes;
var
  Connection: TCustomMSConnection;
begin
  Connection := nil;
  if GetComponent(0) is TCustomMSConnection then
    Connection := GetComponent(0) as TCustomMSConnection
  else
    if GetComponent(0) is TCustomMSDataset then
      Connection := TCustomMSDataset(GetComponent(0)).Connection as TCustomMSConnection;
  if Connection = nil then
    Exit;

  if Connection.Options.Provider <> prCompact then
    Result := [paValueList]
  else
    Result := [paRevertable, paDialog, paMultiSelect];
end;

function TMSDatabaseNamePropertyEditor.AutoFill: boolean;
begin
  Result := False;
end;

procedure TMSDatabaseNamePropertyEditor.Edit;
var
  OpenDialog: TOpenDialog;
begin
  OpenDialog := TOpenDialog.Create(nil);
  GetDialogOptions(OpenDialog);
  if OpenDialog.Execute then
    SetValue(string(OpenDialog.FileName));
  OpenDialog.Free;
end;

procedure TMSDatabaseNamePropertyEditor.GetValues(Proc: TGetStrProc);
var
  List: TStringList;
  Connection: TCustomMSConnection;
  i: integer;
  OldConnected: boolean;
begin
  Connection := nil;
  if GetComponent(0) is TCustomMSConnection then
    Connection := GetComponent(0) as TCustomMSConnection
  else
    if GetComponent(0) is TCustomMSDataset then
      Connection := TCustomMSDataset(GetComponent(0)).Connection as TCustomMSConnection;
  if Connection = nil then
    Exit;

  OldConnected := Connection.Connected;
  List := TStringList.Create;
  try
    try
      GetDatabasesList(Connection, List);
    except
      {$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF}.Modified;
      raise;
    end;

    List.Sort;
    for i := 0 to List.Count - 1 do
      Proc(List[i]);
  finally
    List.Free;

    if (OldConnected <> Connection.Connected) and
      ({$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF} <> nil) then
      {$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF}.Modified;
  end;
end;

{$IFNDEF STD}
{ TMSQueuePropertyEditor }

function TMSQueuePropertyEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList];
end;

procedure TMSQueuePropertyEditor.GetValues(Proc: TGetStrProc);
var
  List: TStringList;
  Connection: TCustomMSConnection;
  ServiceBroker: TMSServiceBroker;
  i: integer;
  OldConnected: boolean;
begin
  ServiceBroker := nil;
  Connection := nil;
  if GetComponent(0) is TMSServiceBroker then begin
    ServiceBroker := TMSServiceBroker(GetComponent(0));
    Connection := ServiceBroker.Connection;
  end;
  if Connection = nil then
    Exit;

  OldConnected := Connection.Connected;
  List := TStringList.Create;
  try
    try
      ServiceBroker.GetQueueNames(List);
    except
      {$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF}.Modified;
      raise;
    end;
    List.Sort;

    for i := 0 to List.Count - 1 do
      Proc(List[i]);
  finally
    List.Free;

    if (OldConnected <> Connection.Connected) and
      ({$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF} <> nil) then
      {$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF}.Modified;
  end;
end;

{ TMSServicePropertyEditor }

function TMSServicePropertyEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList, paSortList, paMultiSelect];
end;

procedure TMSServicePropertyEditor.GetValues(Proc: TGetStrProc);
var
  List: TStringList;
  Connection: TCustomMSConnection;
  ServiceBroker: TMSServiceBroker;
  i: integer;
  OldConnected: boolean;
begin
  ServiceBroker := nil;
  Connection := nil;
  if GetComponent(0) is TMSServiceBroker then begin
    ServiceBroker := TMSServiceBroker(GetComponent(0));
    Connection := ServiceBroker.Connection;
  end;
  if Connection = nil then
    Exit;

  OldConnected := Connection.Connected;
  List := TStringList.Create;
  try
    try
      ServiceBroker.GetServiceNames(List);
    except
      {$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF}.Modified;
      raise;
    end;

    for i := 0 to List.Count - 1 do
      Proc(List[i]);
  finally
    List.Free;

    if (OldConnected <> Connection.Connected) and
      ({$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF} <> nil) then
      {$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF}.Modified;
  end;
end;

{ TMSContractPropertyEditor }

function TMSContractPropertyEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList];
end;

procedure TMSContractPropertyEditor.GetValues(Proc: TGetStrProc);
var
  List: TStringList;
  Connection: TCustomMSConnection;
  ServiceBroker: TMSServiceBroker;
  i: integer;
  OldConnected: boolean;
begin
  ServiceBroker := nil;
  Connection := nil;
  if GetComponent(0) is TMSServiceBroker then begin
    ServiceBroker := TMSServiceBroker(GetComponent(0));
    Connection := ServiceBroker.Connection;
  end;
  if Connection = nil then
    Exit;

  OldConnected := Connection.Connected;
  List := TStringList.Create;
  try
    try
      ServiceBroker.GetContractNames(List);
    except
      {$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF}.Modified;
      raise;
    end;
    List.Sort;

    for i := 0 to List.Count - 1 do
      Proc(List[i]);
  finally
    List.Free;

    if (OldConnected <> Connection.Connected) and
      ({$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF} <> nil) then
      {$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF}.Modified;
  end;
end;

{ TMSTargetDatabaseNamePropertyEditor }

function TMSTargetDatabaseNamePropertyEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList];
end;

procedure TMSTargetDatabaseNamePropertyEditor.GetValues(Proc: TGetStrProc);
var
  List: TStringList;
  Connection: TCustomMSConnection;
  i: integer;
  OldConnected: boolean;
begin
  Connection := nil;
  if GetComponent(0) is TMSServiceBroker then
    Connection := TMSServiceBroker(GetComponent(0)).Connection as TCustomMSConnection;
  if Connection = nil then
    Exit;

  OldConnected := Connection.Connected;
  List := TStringList.Create;
  try
    try
      GetDatabasesList(Connection, List);
    except
      {$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF}.Modified;
      raise;
    end;
    List.Sort;

    for i := 0 to List.Count - 1 do
      Proc(List[i]);
  finally
    List.Free;

    if (OldConnected <> Connection.Connected) and
      ({$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF} <> nil) then
      {$IFDEF FPC}FindRootDesigner(GetComponent(0) as TComponent){$ELSE}Designer{$ENDIF}.Modified;
  end;
end;

{ TMSTableNamesEditor }

function TMSTableNamesEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;

procedure TMSTableNamesEditor.Edit;
var
  Comp: TComponent;
  Conn: TCustomDAConnection;

begin
  Comp := TComponent(GetComponent(0));
  Conn := TMSDesignUtils.GetConnection(Comp);
  if Conn = nil then
    Exit;

  with TMSNamesEditorForm.Create(nil, TMSDesignUtils) do
    try
      Connection := Conn as TMSConnection;
      if Comp is TMSDump then
        Names := TMSDump(Comp).TableNames
      else
        Assert(False);

      ShowModal;
      if ModalResult = mrOk then begin
        if Comp is TMSDump then
          TMSDump(Comp).TableNames := Names
        else
          Assert(False);
      end;
  finally
    Free;
  end;
end;

{ TMSLocaleIdentifierPropertyEditor }

function TMSLocaleIdentifierPropertyEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList];
end;

procedure TMSLocaleIdentifierPropertyEditor.GetValues(Proc: TGetStrProc);
{$IFDEF MSWINDOWS}
var
  List: TStringList;
  i: integer;
begin
  List := TStringList.Create;
  try
    GetLocaleIdentifierList(List);
    for i := 0 to List.Count - 1 do
      Proc(List[i]);
  finally
    List.Free;
  end;
{$ELSE}
begin
{$ENDIF}
end;
{$ENDIF}

{ TMSConnectDialogPropertyEditor }

procedure TMSConnectDialogPropertyEditor.CheckComponent(const Value: string);
var
  Component: TComponent;
begin
  Component := {$IFDEF FPC}PropertyHook{$ELSE}Designer{$ENDIF}.GetComponent(Value);
  if Component <> nil then begin
    if not ({$IFDEF VER17P}(GetIsClassByName(Component, 'TMSConnectDialogFmx')) or {$ENDIF}(Component is TMSConnectDialog)) then
      Exit;
  end;
  FCheckProc(Value);
end;

procedure TMSConnectDialogPropertyEditor.GetValues(Proc: TGetStrProc);
begin
  FCheckProc := Proc;
  inherited GetValues(CheckComponent);
end;

{ TMSConnectionEditor }

procedure TMSConnectionEditor.ExecuteVerb(Index: integer);
begin
{$IFDEF MSWINDOWS}
  if (Index = FQueryAnalyserIndex) and (FQueryAnalyserIndex <> -1) then
    RunServerTool(stQueryAnalyser, Component as TCustomMSConnection)
  else
    if (Index = FManagementStudioIndex) and (FManagementStudioIndex <> -1) then
      RunServerTool(stManagementStudio, Component as TCustomMSConnection)
    else
{$ENDIF}
      inherited ExecuteVerb(Index);
end;

{$IFNDEF STD}
procedure TMSConnectionEditor.Convert;
begin
{$IFNDEF FPC}
  if Designer <> nil then begin
    Assert(Component is TMSConnection);
    TMSConnection(Component).Options.Provider := prCompact; 
    ConvertToClass(Self.Designer, Component, TMSCompactConnection);
  end;
{$ENDIF}
end;
{$ENDIF}

procedure TMSConnectionEditor.InitVerbs;
{$IFDEF MSWINDOWS}
var
  ServerToolMethod: TVerbMethod;
  Connection: TCustomMSConnection;
{$ENDIF}
begin
  AddVerb('Connection Editor...', TMSConnectionEditorForm, TMSDesignUtils);
{$IFNDEF STD}
{$IFNDEF FPC}
  AddVerb('Convert to TMSCompactConnection', Convert);
{$ENDIF}
{$ENDIF}
{$IFDEF DBTOOLS}
  AddDBToolsVerbs([dbtFindInDatabaseExplorer]);
{$ENDIF}

{$IFDEF MSWINDOWS}
  Connection := Component as TCustomMSConnection;
  ServerToolMethod := nil;
  FQueryAnalyserIndex := -1;
  FManagementStudioIndex := -1;
  if (Connection <> nil) and (Connection.Options.Provider <> prCompact) then begin
    if IsServerToolInstalled(stQueryAnalyser) then
      FQueryAnalyserIndex := AddVerb('Query Analyzer...', ServerToolMethod);
    if IsServerToolInstalled(stManagementStudio) then
      FManagementStudioIndex := AddVerb('Management Studio...', ServerToolMethod);
  end;
{$ENDIF}
end;

{$IFNDEF STD}
{$IFDEF MSWINDOWS}
{ TMSCompactConnectionEditor }

procedure TMSCompactConnectionEditor.ExecuteVerb(Index: integer);
begin
{$IFDEF MSWINDOWS}
  if (Index = FManagementStudioIndex) and (FManagementStudioIndex <> -1) then
    RunServerTool(stManagementStudio, Component as TCustomMSConnection)
  else
{$ENDIF}
    inherited ExecuteVerb(Index);
end;

procedure TMSCompactConnectionEditor.Convert;
begin
{$IFNDEF FPC}
  if Designer <> nil then begin
    Assert(Component is TMSCompactConnection);
    ConvertToClass(Self.Designer, Component, TMSConnection);
  end;
{$ENDIF}
end;

procedure TMSCompactConnectionEditor.InitVerbs;
var
  ServerToolMethod: TVerbMethod;
  Connection: TCustomMSConnection;
begin
  AddVerb('Connection Editor...', TMSCompactConnectionEditorForm, TMSDesignUtils);
{$IFNDEF FPC}
  AddVerb('Convert to TMSConnection', Convert);
{$ENDIF}

  Connection := Component as TCustomMSConnection;
  ServerToolMethod := nil;
  FManagementStudioIndex := -1;
  if (Connection <> nil) then
    if IsServerToolInstalled(stManagementStudio) then
      FManagementStudioIndex := AddVerb('Management Studio...', ServerToolMethod);
end;
{$ENDIF}
{$ENDIF}

{ TMSQueryEditor }

procedure TMSQueryEditor.ExecuteVerb(Index: integer);
begin
{$IFDEF MSWINDOWS}
  if (Index = FQueryAnalyserIndex) and (FQueryAnalyserIndex <> -1) then
    RunServerToolDataSet(stQueryAnalyser, Component as TCustomMSDataSet)
  else
    if (Index = FManagementStudioIndex) and (FManagementStudioIndex <> -1) then
      RunServerToolDataSet(stManagementStudio, Component as TCustomMSDataSet)
    else
{$ENDIF}
      inherited ExecuteVerb(Index);
end;

procedure TMSQueryEditor.InitVerbs;
{$IFDEF MSWINDOWS}
var
  Connection: TCustomMSConnection;
  ServerToolMethod: TVerbMethod;
{$ENDIF}
begin
  AddVerb('Fields &Editor...', ShowFieldsEditor);
  AddVerb('MSQuery E&ditor...', TMSQueryEditorForm, TMSDesignUtils);
  AddVerb('Data Editor...', ShowDataEditor);
{$IFDEF DBTOOLS}
  AddDBToolsVerbs([dbtEditSelectSql, dbtEditInsertSql, dbtEditUpdateSql, dbtEditDeleteSql, dbtEditLockSql, dbtEditRefreshSql, dbtQueryBuilder, dbtDebugSql, dbtRetrieveData, dbtEditRecCountSQL]);
{$ENDIF}

{$IFDEF MSWINDOWS}
  Connection := (Component as TCustomMSDataSet).Connection;
  ServerToolMethod := nil;
  FQueryAnalyserIndex := -1;
  FManagementStudioIndex := -1;
  if (Connection <> nil) and (Connection.Options.Provider <> prCompact) then begin
    if IsServerToolInstalled(stQueryAnalyser) then
      FQueryAnalyserIndex := AddVerb('Query Analyzer...', ServerToolMethod);
    if IsServerToolInstalled(stManagementStudio) then
      FManagementStudioIndex := AddVerb('Management Studio...', ServerToolMethod);
  end;
{$ENDIF}

  inherited;
end;

{ TMSSQLEditor }

procedure TMSSQLEditor.InitVerbs;
begin
  AddVerb('MSSQL E&ditor...', TMSSQLEditorForm, TMSDesignUtils);
{$IFDEF DBTOOLS}
  AddDBToolsVerbs([dbtEditSql, dbtDebugSql]);
{$ENDIF}
end;

{ TMSStoredProcEditor }

procedure TMSStoredProcEditor.ExecuteVerb(Index: integer);
begin
{$IFNDEF FPC}
  if Index = GetVerbCount - 1 then
    ConvertToClass(Designer, Component, TMSQuery)
  else
{$ENDIF}
    inherited ExecuteVerb(Index);
end;

procedure TMSStoredProcEditor.InitVerbs;
begin
  AddVerb('Fields &Editor...', ShowFieldsEditor);
  AddVerb('MSStoredProc E&ditor...', TMSStoredProcEditorForm, TMSDesignUtils);
  AddVerb('Data Editor...', ShowDataEditor);
{$IFDEF DBTOOLS}
  AddDBToolsVerbs([dbtFindInDatabaseExplorer, dbtEditInsertSql, dbtEditUpdateSql, dbtEditDeleteSql, dbtEditLockSql, dbtEditRefreshSql, dbtDebugSql{, dbtCompile, dbtCompileDebug}, dbtEditRecCountSQL]);
{$ENDIF}
{$IFNDEF FPC}
  AddVerb('Convert to TMSQuery', ShowDataEditor);
{$ENDIF}

  inherited;
end;

{ TMSTableEditor }

procedure TMSTableEditor.InitVerbs;
begin
  AddVerb('Fields &Editor...', ShowFieldsEditor);
  AddVerb('MSTable E&ditor...', TMSTableEditorForm, TMSDesignUtils);
  AddVerb('Data Editor...', ShowDataEditor);
{$IFDEF DBTOOLS}
  AddDBToolsVerbs([dbtEditDatabaseObject, dbtFindInDatabaseExplorer]);
{$ENDIF}

  inherited;
end;

{ TMSUpdateSQLEditor }

procedure TMSUpdateSQLEditor.InitVerbs;
begin
  inherited;
  AddVerb('MSUpdateSQL E&ditor...', TMSUpdateSQLEditorForm, TMSDesignUtils);
{$IFDEF DBTOOLS}
  AddDBToolsVerbs([dbtEditInsertSql, dbtEditUpdateSql, dbtEditDeleteSql, dbtEditLockSql, dbtEditRefreshSql]);
{$ENDIF}
end;

{ TMSScriptEditor }

procedure TMSScriptEditor.InitVerbs;
begin
  inherited;
  AddVerb('MSScript E&ditor...', TDAScriptEditorForm, TMSDesignUtils);
{$IFDEF DBTOOLS}
  AddDBToolsVerbs([dbtEditSql, dbtDebugSql]);
{$ENDIF}
end;

{$IFNDEF STD}
{ TMSDumpEditor }

procedure TMSDumpEditor.InitVerbs;
begin
  inherited;
  AddVerb('MSDump E&ditor...', TMSDumpEditorForm, TMSDesignUtils);
end;
{$ENDIF}

procedure TMSTableDataEditor.InitVerbs;
begin
  AddVerb('Fields &Editor...', ShowFieldsEditor);

  inherited;
end;

{ TMSTableTypeNameEditor }

function TMSTableTypeNameEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList];
end;

function TMSTableTypeNameEditor.AutoFill: boolean;
begin
  Result := False;
end;

procedure TMSTableTypeNameEditor.GetValues(Proc: TGetStrProc);
{$IFDEF MSWINDOWS}
var
  List: TStringList;
  i: integer;
  Component: TComponent;
  UsedConnection: TCustomMSConnection;
begin
  Assert(PropCount > 0, 'PropCount = 0');
  Component := GetComponent(0) as TComponent;
  Assert(Component is TMSTableData, Component.ClassName);

  UsedConnection := TMSTableData(Component).Connection;
  if UsedConnection = nil then
    Exit;

  List := TStringList.Create;
  try
    UsedConnection.GetTableTypeNames(List);
    for i := 0 to List.Count - 1 do
      Proc(List[i]);
  finally
    List.Free;
  end;
{$ELSE}
begin
{$ENDIF}
end;

{$IFNDEF FPC}
{ TMSConnectionList }

function TMSConnectionList.GetConnectionType: TCustomDAConnectionClass;
begin
  Result := TCustomMSConnection;
end;

{ TMSDesignNotification }

function TMSDesignNotification.CreateConnectionList: TDAConnectionList;
begin
  Result := TMSConnectionList.Create;
end;

function TMSDesignNotification.GetConnectionPropertyName: string;
begin
  Result := 'Connection';
end;

procedure TMSDesignNotification.ItemInserted(const ADesigner: IDesigner; AItem: TPersistent);
begin
  if (AItem <> nil) and ((AItem is TCustomMSDataSet) or (AItem is TMSSQL) or (AItem is TMSScript) or 
  {$IFNDEF STD}
    (AItem is TMSLoader) or (AItem is TMSDump) or (AItem is TMSServiceBroker) or
    (AItem is TMSChangeNotification) or (AItem is TMSTransaction) or
  {$ENDIF}
    (AItem is TMSTableData) or
    (AItem is TMSDataSource)) then
    FItem := AItem;
end;

procedure TMSDesignNotification.SelectionChanged(const ADesigner: IDesigner;
  const ASelection: IDesignerSelections);
var
  ModuleServices: IOTAModuleServices;
  CurrentModule: IOTAModule;
  Project: IOTAProject;
  ProjectOptions: IOTAProjectOptions;
  DelphiPath: string;
  s: string;
begin
  CurrentProjectOutputDir := '';
  ModuleServices := BorlandIDEServices as IOTAModuleServices;
  CurrentModule := ModuleServices.CurrentModule;

  if CurrentModule.OwnerCount = 0 then
    Exit;

  Project := CurrentModule.Owners[0];
  ProjectOptions := Project.ProjectOptions;
  CurrentProjectOutputDir := Trim(ProjectOptions.Values['OutputDir']);

  if (CurrentProjectOutputDir <> '') then begin
    if (CurrentProjectOutputDir[1] = '.') then begin // relative path
      s := Trim(ExtractFilePath(Project.FileName));
      if s = '' then
        CurrentProjectOutputDir := ''
      else
        CurrentProjectOutputDir := IncludeTrailingBackslash(s) + CurrentProjectOutputDir;
    end
    else
    if Pos('$(DELPHI)', UpperCase(CurrentProjectOutputDir)) > 0 then begin
      DelphiPath := GetEnvironmentVariable('DELPHI');
      CurrentProjectOutputDir := StringReplace(CurrentProjectOutputDir, '$(DELPHI)', DelphiPath, [rfReplaceAll, rfIgnoreCase]);
    end;
  end
  else
    CurrentProjectOutputDir := Trim(ExtractFilePath(Project.FileName));
end;
{$ENDIF}

procedure Register;
begin
  // Register property editors
  RegisterPropertyEditor(TypeInfo(string), TMSConnection, 'ConnectString', TMSConnectStringPropertyEditor);
  RegisterPropertyEditor(TypeInfo(string), TMSConnection, 'Server', TMSServerNamePropertyEditor);
  RegisterPropertyEditor(TypeInfo(string), TMSConnection, 'Database', TMSDatabaseNamePropertyEditor);
  RegisterPropertyEditor(TypeInfo(TCustomConnectDialog), TMSConnection, 'ConnectDialog', TMSConnectDialogPropertyEditor);

{$IFNDEF STD}
{$IFDEF MSWINDOWS}
  RegisterPropertyEditor(TypeInfo(string), TMSCompactConnection, 'Database', TMSDatabaseNamePropertyEditor);
  RegisterPropertyEditor(TypeInfo(TCustomConnectDialog), TMSCompactConnection, 'ConnectDialog', TMSConnectDialogPropertyEditor);
  RegisterPropertyEditor(TypeInfo(string), TMSCompactConnectionOptions, 'LocaleIdentifier', TMSLocaleIdentifierPropertyEditor);
{$ENDIF}
{$ENDIF}

  RegisterPropertyEditor(TypeInfo(TStrings), TMSQuery, 'SQL', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSQuery, 'SQLDelete', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSQuery, 'SQLInsert', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSQuery, 'SQLLock', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSQuery, 'SQLRecCount', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSQuery, 'SQLRefresh', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSQuery, 'SQLUpdate', TDAPropertyEditor);

  RegisterPropertyEditor(TypeInfo(TStrings), TMSSQL, 'SQL', TDAPropertyEditor);

  RegisterPropertyEditor(TypeInfo(TStrings), TMSStoredProc, 'SQL', nil);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSStoredProc, 'SQLDelete', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSStoredProc, 'SQLInsert', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSStoredProc, 'SQLLock', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSStoredProc, 'SQLRecCount', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSStoredProc, 'SQLRefresh', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSStoredProc, 'SQLUpdate', TDAPropertyEditor);

  RegisterPropertyEditor(TypeInfo(TStrings), TMSUpdateSQL, 'InsertSQL', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSUpdateSQL, 'ModifySQL', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSUpdateSQL, 'DeleteSQL', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSUpdateSQL, 'RefreshSQL', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TMSUpdateSQL, 'LockSQL', TDAPropertyEditor);
  RegisterPropertyEditor(TypeInfo(Boolean), TMSDataSetOptions, 'AllFieldsEditable', nil);

  RegisterPropertyEditor(TypeInfo(string), TMSMetadata, 'DatabaseName', TMSDatabaseNamePropertyEditor);


  RegisterPropertyEditor(TypeInfo(TStrings), TMSScript, 'SQL', TDAPropertyEditor);

{$IFNDEF STD}
  RegisterPropertyEditor(TypeInfo(string), TMSServiceBroker, 'Service', TMSServicePropertyEditor);
  RegisterPropertyEditor(TypeInfo(string), TMSDump, 'TableNames', TMSTableNamesEditor);
{$ENDIF}
  RegisterPropertyEditor(TypeInfo(string), TMSTableData, 'TableTypeName', TMSTableTypeNameEditor);

  // Register component editors
  DARegisterComponentEditor(TMSConnection, TMSConnectionEditor, TMSConnectionEditorForm, TMSDesignUtils);
{$IFNDEF STD}
{$IFDEF MSWINDOWS}
  DARegisterComponentEditor(TMSCompactConnection, TMSCompactConnectionEditor, TMSCompactConnectionEditorForm, TMSDesignUtils);
{$ENDIF}
{$ENDIF}
  DARegisterComponentEditor(TMSQuery, TMSQueryEditor, TMSQueryEditorForm, TMSDesignUtils);
  DARegisterComponentEditor(TMSSQL, TMSSQLEditor, TMSSQLEditorForm, TMSDesignUtils);
  DARegisterComponentEditor(TMSTable, TMSTableEditor, TMSTableEditorForm, TMSDesignUtils);
  DARegisterComponentEditor(TMSStoredProc, TMSStoredProcEditor, TMSStoredProcEditorForm, TMSDesignUtils);
  DARegisterComponentEditor(TMSUpdateSQL, TMSUpdateSQLEditor, TMSUpdateSQLEditorForm, TMSDesignUtils);
  DARegisterComponentEditor(TMSScript, TMSScriptEditor, TDAScriptEditorForm, TMSDesignUtils);
{$IFNDEF STD}
  DARegisterComponentEditor(TMSLoader, TDALoaderEditor, nil, TMSDesignUtils);
  DARegisterComponentEditor(TMSDump, TMSDumpEditor, TMSDumpEditorForm, TMSDesignUtils);

  DARegisterComponentEditor(TMSServiceBroker, TDAComponentEditor, nil, TMSDesignUtils);
  DARegisterComponentEditor(TMSMetaData, TDAComponentEditor, nil, TMSDesignUtils);
{$ENDIF}
  DARegisterComponentEditor(TMSTableData, TMSTableDataEditor, nil, TMSDesignUtils);

{$IFNDEF FPC}
  RegisterComponentEditor(TMSDataSource, TCRDataSourceEditor);

  Menu.AddItems(SysInit.HInstance);
{$ENDIF}
end;

procedure ClearTmpFiles;
var
  i: integer;
begin
  if TmpFiles = nil then
    exit;

  for i := 0 to TmpFiles.Count - 1 do
    DeleteFile(TmpFiles[i]);
end;

{$IFNDEF FPC}
var
  Notificator: IDesignNotification;
{$ENDIF}

initialization
  TmpFiles := TStringList.Create;

{$IFDEF DBTOOLS}
  if not Assigned(DBTools) then
    DBTools := TDBTools.Create;
{$ENDIF}

{$IFNDEF FPC}
  Notificator := TMSDesignNotification.Create;
  RegisterDesignNotification(Notificator);
{$ENDIF}

finalization
  ClearTmpFiles;
  CachedServerList.Free;

{$IFNDEF FPC}
  UnRegisterDesignNotification(Notificator);
  Notificator := nil;
{$ENDIF}

end.
