
//////////////////////////////////////////////////
//  SQL Server Data Access Components
//  Copyright  1998-2025 Devart. All right reserved.
//////////////////////////////////////////////////

{$IFNDEF CLR}

{$I Sdac.inc}

unit MSDesignUtils;
{$ENDIF}

interface

uses
{$IFDEF DBTOOLS}
  DBToolsClient, DBToolsIntf, DB, DesignIntf,
{$ENDIF}
  Classes, SysUtils, CRTypes, CRDataTypeMap, DBAccess, DADesignUtils;

type

{$IFDEF DBTOOLS}
  TMSIntegrationTool = (sitDBForgeStudio, sitNone);
{$ENDIF}

  TMSDesignUtils = class(TDADesignUtils)
    class function GetProjectName: string; override;

    class function GetDesignCreate(Obj: TComponent): boolean; override;
    class procedure SetDesignCreate(Obj: TComponent; Value: boolean); override;
  {$IFNDEF FPC}
    class function GetConnectionList: TObject; override;
  {$ENDIF}

  { Connection support }
    class function HasConnection(Obj: TComponent): boolean; override;
    class function GetConnection(Obj: TComponent): TCustomDAConnection; override;
    class procedure SetConnection(Obj: TComponent; Value: TCustomDAConnection); override;
    class function UsedConnection(Obj: TComponent): TCustomDAConnection; override;

  { TDATable support }
    class function GetTableName(Obj: TCustomDADAtaSet): string; override;
    class procedure SetTableName(Obj: TCustomDADAtaSet; const Value: string); override;
    class function GetOrderFields(Obj: TCustomDADAtaSet): string; override;
    class procedure SetOrderFields(Obj: TCustomDADAtaSet; const Value: string); override;
    class procedure PrepareSQL(Obj: TCustomDADAtaSet); override;
    class function GetStoredProcName(Obj: TCustomDADataSet): string; override;
    class procedure SetStoredProcName(Obj: TCustomDADataSet; const Value: string); override;

  { Data Type Mapping }
    class function GetConverterManagerClass: TConverterManagerClass; override;

  {$IFDEF USE_SYNEDIT}
    class function SQLDialect: integer ; override; // SynHighlighterSQL TSQLDialect = (sqlStandard, sqlInterbase6, sqlMSSQL7, sqlMySQL, sqlOracle, sqlSybase, sqlIngres, sqlMSSQL2K);
  {$ENDIF}
  {$IFDEF DBTOOLS}
    class function DBToolsService: TObject; override;
    class function NeedToCheckDbTools: TNeedToCheckDbTools; override;
    class function GetDBToolsServiceVersion: int64; override;
    class function GetDBToolsMenuCaption: string; override;
    class function GetFullName(Obj: TComponent): string; override;
    class function GetObjectType(Obj: TComponent): string; override;
    class function IsStoredProc(Obj: TComponent): boolean; override;

    class procedure GetDBToolsConnectionList(Connection: TCustomDAConnection); override;
    class procedure SetDBToolsDownloadParams(VerbCheck: boolean; Incompatible: boolean); override;

    class procedure LoadIntegrationTool;
    class procedure SaveIntegrationTool(IntegrationTool: TMSIntegrationTool);
    class function GetCurrentIntegrationTool: TMSIntegrationTool;
    class procedure UpdateIntegrationTool;
    class function AreDBToolsUsed: boolean; override;
    class procedure CheckComponent(Component: TComponent); override;
  {$ENDIF}
  end;

  TMSDesignUtilsClass = class of TMSDesignUtils;

{$IFDEF DBTOOLS}
{$I SdacVer.inc}
  sSDACVersion = 'SDAC ' + SDACVersion;

  sMSIntegrationTool = 'MSIntegrationTool';
  sDBForgeStudioTool = 'dbForge Studio for SQL Server';
  sNoTool = 'No integration';
  sDBToolsNames: array[TMSIntegrationTool] of string = (sDBForgeStudioTool, '');

  sDevToolsDownloadCaption = 'SDAC Information';
  sDevAskIncompatible = 'DevAskIncompatible';
  sDevAskNoAddin = 'DevAskNoAddin';
  sProductAskIncompatible = 'SDACAskIncompatible';
{$ENDIF}

implementation

uses
{$IFNDEF STD}
  MSServiceBroker,
{$ENDIF}
{$IFDEF DBTOOLS}
  DBForgeStudioClientImp, Download, Registry, Windows,
{$ENDIF}
  MSDataTypeMap, MSAccess{$IFDEF SDAC}, MSDesign{$ENDIF}, MSClasses;

{ TMSDesignUtils }

class function TMSDesignUtils.GetProjectName: string;
begin
  Result := 'SDAC';
end;

class function TMSDesignUtils.GetDesignCreate(Obj: TComponent): boolean;
begin
  Assert(Obj <> nil);
{$IFNDEF STD}
  if Obj is TMSServiceBroker then
    Result := True
  else
{$ENDIF}
  if Obj is TMSTableData then
    Result := True
  else
    Result := inherited GetDesignCreate(Obj);
end;

class procedure TMSDesignUtils.SetDesignCreate(Obj: TComponent; Value: boolean);
begin
  Assert(Obj <> nil);
{$IFNDEF STD}
  if Obj is TMSServiceBroker then
  else
{$ENDIF}
  if Obj is TMSTableData then
  else
    inherited;
end;

{$IFNDEF FPC}
class function TMSDesignUtils.GetConnectionList: TObject;
begin
{$IFDEF SDAC}
  Result := TMSConnectionList.Create;
{$ELSE}
  Result := nil;
{$ENDIF}
end;
{$ENDIF}

class function TMSDesignUtils.HasConnection(Obj: TComponent): boolean;
begin
{$IFNDEF STD}
  if Obj is TMSServiceBroker then
    Result := True
  else
{$ENDIF}
  if Obj is TMSTableData then
    Result := True
  else
    Result := inherited HasConnection(Obj);
end;

class function TMSDesignUtils.GetConnection(Obj: TComponent): TCustomDAConnection;
begin
  Assert(Obj <> nil);
{$IFNDEF STD}
  if Obj is TMSServiceBroker then
    Result := TMSServiceBroker(Obj).Connection
  else
{$ENDIF}
  if Obj is TMSTableData then
    Result := TMSTableData(Obj).Connection
  else
    Result := inherited GetConnection(Obj);
end;

class procedure TMSDesignUtils.SetConnection(Obj: TComponent; Value: TCustomDAConnection);
begin
  Assert(Obj <> nil);
{$IFNDEF STD}
  if Obj is TMSServiceBroker then
    TMSServiceBroker(Obj).Connection := Value as TMSConnection
  else
{$ENDIF}
  if Obj is TMSTableData then
    TMSTableData(Obj).Connection := Value as TMSConnection
  else
    inherited;
end;

class function TMSDesignUtils.UsedConnection(Obj: TComponent): TCustomDAConnection;
begin
  Assert(Obj <> nil);
  if Obj is TMSTableData then
    Result := TMSTableData(Obj).Connection
  else
    Result := inherited UsedConnection(Obj);
end;

class function TMSDesignUtils.GetTableName(Obj: TCustomDADAtaSet): string;
begin
  Assert(Obj is TCustomMSTable, Obj.ClassName);
  Result := TCustomMSTable(Obj).TableName;
end;

class procedure TMSDesignUtils.SetTableName(Obj: TCustomDADAtaSet;
  const Value: string);
begin
  Assert(Obj is TCustomMSTable, Obj.ClassName);
  TCustomMSTable(Obj).TableName := Value;
end;

class procedure TMSDesignUtils.PrepareSQL(Obj: TCustomDADAtaSet);
begin
  Assert(Obj is TCustomMSTable, Obj.ClassName);
  TCustomMSTable(Obj).PrepareSQL;
end;

class function TMSDesignUtils.GetOrderFields(Obj: TCustomDADAtaSet): string;
begin
  Assert(Obj is TCustomMSTable, Obj.ClassName);
  Result := TCustomMSTable(Obj).OrderFields;
end;

class procedure TMSDesignUtils.SetOrderFields(Obj: TCustomDADAtaSet;
  const Value: string);
begin
  Assert(Obj is TCustomMSTable, Obj.ClassName);
  TCustomMSTable(Obj).OrderFields := Value;
end;

class function TMSDesignUtils.GetStoredProcName(Obj: TCustomDADataSet): string;
begin
  Assert(Obj is TMSStoredProc, Obj.ClassName);
  Result := TMSStoredProc(Obj).StoredProcName;
end;

class procedure TMSDesignUtils.SetStoredProcName(Obj: TCustomDADataSet; const Value: string);
begin
  Assert(Obj is TMSStoredProc, Obj.ClassName);
  TMSStoredProc(Obj).StoredProcName := Value;
end;

class function TMSDesignUtils.GetConverterManagerClass: TConverterManagerClass;
begin
  Result := TConverterManagerClass(TMSMapRules.GetConverterManager.ClassType);
end;

{$IFDEF USE_SYNEDIT}
class function TMSDesignUtils.SQLDialect: integer;
begin
  Result := 7; // sqlMSSQL2K
end;
{$ENDIF}

{$IFDEF DBTOOLS}
const
  MSSQLServiceCLSID: TGUID = '{30E566DC-94A3-46B1-A91F-615AEE9AD8E0}';
  MSProviderKey = 'dbForge Studio for SQL Server';

var
  MSSQLService: TObject;
  MSSQLServiceFault: boolean;
  MSSQLServiceNeedToCheck: TNeedToCheckDbTools;
  MSSQLServiceVersion: int64;
  MSSQLIntegrationTool: TMSIntegrationTool;
  MSSQLServiceGUID: TGUID;

class function TMSDesignUtils.DBToolsService: TObject;
const
  dbtBigInt = 0;
  dbtBinary = 1;
  dbtBit = 2;
  dbtChar = 3;
  dbtDateTime = 4;
  dbtDecimal = 5;
  dbtFloat = 6;
  dbtImage = 7;
  dbtInt = 8;
  dbtMoney = 9;
  dbtNChar = 10;
  dbtNText = 11;
  dbtNVarChar = 12;
  dbtReal = 13;
  dbtUniqueIdentifier = 14;
  dbtSmallDateTime = 15;
  dbtSmallInt = 16;
  dbtSmallMoney = 17;
  dbtText = 18;
  dbtTimestamp = 19;
  dbtTinyInt = 20;
  dbtVarBinary = 21;
  dbtVarChar = 22;
  dbtVariant = 23;
  dbtXml = 25;
  dbtUdt = 29;
  dbtStructured = 30;
  dbtDate = 31;
  dbtTime = 32;
  dbtDateTime2 = 33;
  dbtDateTimeOffset = 34;
var
  DBToolsServiceClass: TCustomDBToolsServiceClass;
begin
  if (MSSQLIntegrationTool <> sitNone) and (MSSQLService = nil) and not MSSQLServiceFault then begin
    MSSQLServiceFault := true;
    if MSSQLIntegrationTool = sitDBForgeStudio then
      DBToolsServiceClass := TDBForgeStudioService
    else
      DBToolsServiceClass := nil;
    if DBToolsServiceClass <> nil then begin
      MSSQLService := DBTools.CreateDBToolsService(DBToolsServiceClass, TMSDesignUtils,
        MSSQLServiceGUID, 'Data Source=;Initial Catalog=;Integrated Security=;'
        + 'User ID=;Password=;',
        MSProviderKey, MSSQLServiceVersion, MSSQLServiceNeedToCheck);
      MSSQLServiceFault := MSSQLService = nil;
      if not MSSQLServiceFault then
        with TCustomDBToolsService(MSSQLService) do begin
          AddParamTypeMap(ftLargeint, dbtBigInt);
          AddParamTypeMap(ftBytes, dbtBinary);
          AddParamTypeMap(ftBoolean, dbtBit);
          AddParamTypeMap(ftFixedChar, dbtChar);
          AddParamTypeMap(ftDateTime, dbtDateTime);
          AddParamTypeMap(ftBCD, dbtDecimal);
          AddParamTypeMap(ftFloat, dbtFloat);
          AddParamTypeMap(ftBlob, dbtImage);
          AddParamTypeMap(ftInteger, dbtInt);
          AddParamTypeMap(ftCurrency, dbtMoney);
        {$IFDEF VER10P}
          AddParamTypeMap(ftFixedWideChar, dbtNChar);
          AddParamTypeMap(ftWideMemo, dbtNText);
          AddParamTypeMap(ftWideMemo, dbtXml);
        {$ELSE}
          AddParamTypeMap(ftFixedChar, dbtNChar);
          AddParamTypeMap(ftMemo, dbtNText);
          AddParamTypeMap(ftMemo, dbtXml);
        {$ENDIF}
          AddParamTypeMap(ftWideString, dbtNVarChar);
          AddParamTypeMap(ftFloat, dbtReal);
          AddParamTypeMap(ftGuid, dbtUniqueIdentifier);
          AddParamTypeMap(ftDateTime, dbtSmallDateTime);
          AddParamTypeMap(ftSmallint, dbtSmallInt);
          AddParamTypeMap(ftCurrency, dbtSmallMoney);
          AddParamTypeMap(ftMemo, dbtText);
          AddParamTypeMap(ftTimeStamp, dbtTimestamp);
        {$IFDEF VER12P}
          AddParamTypeMap(ftByte, dbtTinyInt);
        {$ELSE}
          AddParamTypeMap(ftSmallint, dbtTinyInt);
        {$ENDIF}
          AddParamTypeMap(ftVarBytes, dbtVarBinary);
          AddParamTypeMap(ftString, dbtVarChar);
          AddParamTypeMap(ftVariant, dbtVariant);
          AddParamTypeMap(ftBlob, dbtUdt);
          AddParamTypeMap(ftDate, dbtDate);
          AddParamTypeMap(ftTime, dbtTime);
          AddParamTypeMap(ftDateTime, dbtDateTime2);
          AddParamTypeMap(ftDateTime, dbtDateTimeOffset);
        end;
    end;
  end;
  Result := MSSQLService;
end;

class function TMSDesignUtils.NeedToCheckDbTools: TNeedToCheckDbTools;
begin
  DBToolsAvailable;
  Result := MSSQLServiceNeedToCheck;
end;

class function TMSDesignUtils.GetDBToolsServiceVersion: int64;
begin
  DBToolsAvailable;
  Result := MSSQLServiceVersion;
end;

class function TMSDesignUtils.GetDBToolsMenuCaption: string;
begin
  Result := sDBToolsNames[TMSDesignUtils.GetCurrentIntegrationTool];
end;

class procedure TMSDesignUtils.GetDBToolsConnectionList(Connection: TCustomDAConnection);
var
  MSConnection: TMSConnection;
  Service: TCustomDBToolsService;
begin
  Assert(MSSQLService <> nil);
  Service := TCustomDBToolsService(MSSQLService);
  if Connection <> nil then begin
    Assert(Connection is TMSConnection);
    MSConnection := TMSConnection(Connection);
    Service.PutConnectionParam(MSConnection.Server); // Host
    Service.PutConnectionParam(MSConnection.Database); // Database
    Service.PutConnectionParam(BoolToStr(MSConnection.Authentication = auWindows, True)); // Authentication
    Service.PutConnectionParam(MSConnection.Username, cfCaseSensitive); //User ID
    Service.PutConnectionParam(MSConnection.Password, cfNone); // Password
  end;
end;

class function TMSDesignUtils.GetFullName(Obj: TComponent): string;
var
  DataBase: string;
  Owner: string;
  ObjName: string;
begin
  if Obj is TCustomMSTable then
    Result := GetTableName(TCustomDADataSet(Obj))
  else
  if Obj is TCustomMSStoredProc then
    Result := GetStoredProcName(TCustomDADataSet(Obj))
  else
    Result := inherited GetFullName(Obj);

  MSSQLInfo.SplitObjectName(Result, DataBase, Owner, ObjName);
  if Owner = '' then
    Result := 'dbo.' + Result;
end;

class function TMSDesignUtils.GetObjectType(Obj: TComponent): string;
begin
  if Obj is TCustomMSTable then
    Result := 'Table,View'
  else
  if Obj is TCustomMSStoredProc then
    Result := 'Procedure,Function'
  else
    Result := inherited GetObjectType(Obj);
end;

class function TMSDesignUtils.IsStoredProc(Obj: TComponent): boolean;
begin
  Result := Obj is TCustomMSStoredProc;
end;

class procedure TMSDesignUtils.SetDBToolsDownloadParams(VerbCheck: boolean; Incompatible: boolean);
var
  AskStr, AtomName, UrlStr, UrlExeStr, ToolsNameStr: string;
begin
  if MSSQLIntegrationTool = sitDBForgeStudio then begin
    if VerbCheck then begin
      AskStr := '_Verb';
      AtomName := '';
    end
    else begin
      AskStr := '';
      AtomName := 'CheckForDBForgeMySQL atom';
    end;
    UrlStr := 'www.devart.com/dbforge/sql/studio';
    UrlExeStr := 'dbforgesql.exe';
    ToolsNameStr := sDBForgeStudioTool;
    SetToolsCheckingParams(sDevToolsDownloadCaption, ToolsNameStr, sSDACVersion,
      sDevAskIncompatible + AskStr, sDevAskNoAddin + AskStr,
      'Software\Devart\sdac', 'Sdac', 'dbForge Studio',
      UrlStr, UrlExeStr, AtomName);
  end;
end;

class procedure TMSDesignUtils.LoadIntegrationTool;
var
  reg: TRegistry;
  ts: string;
begin
  MSSQLIntegrationTool := sitDBForgeStudio;
  reg := TRegistry.Create(KEY_READ OR KEY_WRITE);
  try
    reg.RootKey := HKEY_CURRENT_USER;
    if reg.OpenKey('\SOFTWARE\Devart\' + TMSDesignUtils.GetProjectName, False) then begin
      ts := reg.ReadString(sMSIntegrationTool);
      if AnsiSameText(ts, sDBForgeStudioTool) then
        MSSQLIntegrationTool := sitDBForgeStudio
      else
      if AnsiSameText(ts, sNoTool) then
        MSSQLIntegrationTool := sitNone;
    end;
  finally
    reg.Free;
  end;
end;

class procedure TMSDesignUtils.SaveIntegrationTool(IntegrationTool: TMSIntegrationTool);
var
  reg: TRegistry;
  ts: string;
begin
  if IntegrationTool = MSSQLIntegrationTool then exit;
  reg := TRegistry.Create(KEY_READ OR KEY_WRITE);
  try
    if IntegrationTool = sitDBForgeStudio then
      ts := sDBForgeStudioTool
    else
      ts := sNoTool;
    reg.RootKey := HKEY_CURRENT_USER;
    reg.OpenKey('\SOFTWARE\Devart\' + TMSDesignUtils.GetProjectName, True);
    reg.WriteString(sMSIntegrationTool, ts);
    MSSQLIntegrationTool := IntegrationTool;

    FreeAndNil(MSSQLService);
    MSSQLServiceFault := False;
    MSSQLServiceNeedToCheck := ncNone;
    MSSQLServiceVersion := 0;
    UnregisterDesignNotification(DBTools.DesignNotification);
    DBTools.DesignNotification := nil;
    TMSDesignUtils.UpdateIntegrationTool;
  finally
    reg.Free;
  end;
end;

class function TMSDesignUtils.GetCurrentIntegrationTool: TMSIntegrationTool;
begin
  Result := MSSQLIntegrationTool;
end;

class procedure TMSDesignUtils.UpdateIntegrationTool;
begin
  if MSSQLIntegrationTool = sitDBForgeStudio then
    MSSQLServiceGUID := MSSQLServiceCLSID;
  if MSSQLIntegrationTool <> sitNone then
    DBTools.CheckDevTools(MSSQLServiceGUID, MSProviderKey, TMSDesignUtils, MSSQLServiceVersion);
end;

class function TMSDesignUtils.AreDBToolsUsed: boolean;
begin
  Result := MSSQLIntegrationTool <> sitNone;
end;

class procedure TMSDesignUtils.CheckComponent(Component: TComponent);
begin
  if (Component <> nil) then begin
    if (Component is TMSTable) and (TMSTable(Component).TableName = '') then
      DatabaseError('Table name is not defined')
    else
    if (Component is TMSStoredProc) and (TMSStoredProc(Component).StoredProcName = '') then
      DatabaseError('Stored procedure name is not defined')
    else
    if (Component is TMSUpdateSQL) and (TMSUpdateSQL(Component).DataSet = nil) then
      DatabaseError('There is no linked dataset');
  end;
end;
{$ENDIF}

initialization
{$IFDEF DBTOOLS}
  TMSDesignUtils.LoadIntegrationTool;
  TMSDesignUtils.UpdateIntegrationTool;
{$ENDIF}

finalization
{$IFDEF DBTOOLS}
  FreeAndNil(MSSQLService);
{$ENDIF}

end.
