
//////////////////////////////////////////////////
//  SQL Server Data Access Components
//  Copyright  1998-2025 Devart. All right reserved.
//  Connect Form
//////////////////////////////////////////////////

unit MSConnectForm;

interface
{$IFNDEF UNIX}
{$IFNDEF POSIX}
  {$DEFINE MSWINDOWS}
{$ENDIF}
{$ENDIF}

uses
{$IFDEF FPC}
  LResources,
{$ELSE}
  Windows, Messages,
{$ENDIF}
  SysUtils, Classes, Graphics, Controls, StdCtrls, ExtCtrls, Forms,
  CRTypes, CRFunctions, DBAccess, MSAccess,
{$IFDEF MSWINDOWS}
  OLEDBAccess, OLEDBC,
{$ENDIF}
  MSClasses;

type
  TMSConnectForm = class(TForm)
    Panel: TPanel;
    pUserName: TPanel;
    lbUsername: TLabel;
    edUsername: TEdit;
    pPassword: TPanel;
    pServer: TPanel;
    pPort: TPanel;
    pDatabase: TPanel;
    edPassword: TEdit;
    lbPassword: TLabel;
    lbServer: TLabel;
    edServer: TComboBox;
    lbPort: TLabel;
    edPort: TEdit;
    lbDatabase: TLabel;
    edDatabase: TComboBox;
    pButtons: TPanel;
    btCancel: TButton;
    btConnect: TButton;
    pAuthentication: TPanel;
    lbAuthentication: TLabel;
    cbAuthentication: TComboBox;
    procedure btConnectClick(Sender: TObject);
    procedure edServerDropDown(Sender: TObject);
    procedure edDatabaseDropDown(Sender: TObject);
    procedure edExit(Sender: TObject);
    procedure cbAuthenticationChange(Sender: TObject);

  private
    FConnectDialog: TCustomConnectDialog;
    FRetries: integer;
    FRetry: boolean;

    FListGot: boolean;

    procedure SetConnectDialog(Value: TCustomConnectDialog);

  protected
    procedure DoInit; virtual;
    procedure DoConnect; virtual;

  public

  published
    property ConnectDialog: TCustomConnectDialog read FConnectDialog write SetConnectDialog;
  end;

implementation

{$IFNDEF FPC}
{$IFDEF CLR}
{$R MSConnectForm.dfm}
{$ELSE}
{$R *.dfm}
{$ENDIF}
{$ELSE}
{$R *.lfm}
{$ENDIF}

uses
  DAConsts;

procedure TMSConnectForm.DoInit;
var
  CurTop, i: integer;
  ConnectDialogOptions: TConnectDialogOptionArray;
  CurPanel: TPanel;
  CurControl: TWinControl;
  NeedSetActiveControl, IsCompact: Boolean;
begin
  FRetry := False;
  FRetries := FConnectDialog.Retries;
  Caption := FConnectDialog.Caption;
  FListGot := False;

  IsCompact := (FConnectDialog.Connection is TCustomMSConnection) and
    (TCustomMSConnection(FConnectDialog.Connection).Options.Provider = prCompact);

  FConnectDialog.GetOptions(ConnectDialogOptions);
  CurTop := 0;
  NeedSetActiveControl := True;
  for i := Low(ConnectDialogOptions) to High(ConnectDialogOptions) do begin
    case ConnectDialogOptions[i].Kind of
      okServer: begin
        pServer.Visible := ConnectDialogOptions[i].Visible and not IsCompact;
        CurPanel := pServer;
        CurControl := edServer;
        lbServer.Caption := ConnectDialogOptions[i].Caption;
      end;
      okUserName: begin
        pUserName.Visible := ConnectDialogOptions[i].Visible and not IsCompact;
        CurPanel := pUserName;
        CurControl := edUsername;
        lbUsername.Caption := ConnectDialogOptions[i].Caption;
      end;
      okPassword: begin
        pPassword.Visible := ConnectDialogOptions[i].Visible;
        CurPanel := pPassword;
        CurControl := edPassword;
        lbPassword.Caption := ConnectDialogOptions[i].Caption;
      end;
      okDatabase: begin
        pDatabase.Visible := ConnectDialogOptions[i].Visible;
        CurPanel := pDatabase;
        CurControl := edDatabase;
        lbDatabase.Caption := ConnectDialogOptions[i].Caption;
      end;
      okPort: begin
        pPort.Visible := ConnectDialogOptions[i].Visible and not IsCompact;
        CurPanel := pPort;
        CurControl := edPort;
        lbPort.Caption := ConnectDialogOptions[i].Caption;
      end;
      okAuthentication: begin
        pAuthentication.Visible := ConnectDialogOptions[i].Visible and not IsCompact;
        CurPanel := pAuthentication;
        CurControl := cbAuthentication;
        lbAuthentication.Caption := ConnectDialogOptions[i].Caption;
      end;
    else
      raise Exception.Create(Format(SInternalErrorInvalidValue,
        ['TMSConnectForm.DoInit', 'ConnectDialogOptions[i].Kind',
        IntToStr(Integer(ConnectDialogOptions[i].Kind))]));
    end;
    CurPanel.TabOrder := i;
    if CurPanel.Visible then begin
      if i = 0 then
        CurPanel.Top := CurTop + 2
      else
        CurPanel.Top := CurTop;
      CurTop := CurTop + CurPanel.Height;
    end;
    if NeedSetActiveControl and CurPanel.Visible then begin
      ActiveControl := CurControl;
      NeedSetActiveControl := False;
    end;
  end;
  Panel.Height := CurTop + 2;
  pButtons.Top := Panel.Height + 8;
  ClientHeight := pButtons.Top + pButtons.Height + 2;

  btConnect.Caption := FConnectDialog.ConnectButton;
  btCancel.Caption := FConnectDialog.CancelButton;

  edUsername.Text := FConnectDialog.Connection.Username;
  edPassword.Text := FConnectDialog.Connection.Password;
  edServer.Text := FConnectDialog.Connection.Server;

  if FConnectDialog.Connection is TCustomMSConnection then
    edDatabase.Text := TCustomMSConnection(FConnectDialog.Connection).Database;
  if FConnectDialog.Connection is TMSConnection then begin
    edPort.Text := IntToStr(TMSConnection(FConnectDialog.Connection).Port);
    cbAuthentication.ItemIndex := Ord(TMSConnection(FConnectDialog.Connection).Authentication);
    cbAuthenticationChange(cbAuthentication);
  end;

  if (edUsername.Text <> '') and (edPassword.Text = '') and
    pPassword.Visible and edPassword.Enabled then
      ActiveControl := edPassword;
end;

procedure TMSConnectForm.cbAuthenticationChange(Sender: TObject);
var
  IsWindowsAuth: boolean;
begin
  IsWindowsAuth := TMSAuthentication(cbAuthentication.ItemIndex) = auWindows;
  if pUserName.Visible then begin
    lbUsername.Enabled := not IsWindowsAuth;
    edUsername.Enabled := not IsWindowsAuth;
  end;
  if pPassword.Visible then begin
    lbPassword.Enabled := not IsWindowsAuth;
    edPassword.Enabled := not IsWindowsAuth;
  end;
end;

procedure TMSConnectForm.DoConnect;
const
  TDS_ERROR_AUTH_FAILED = $4818;
begin
  try
    edExit(nil);
    FConnectDialog.Connection.PerformConnect(FRetry);
    ModalResult := mrOk;
  except
    on E: Exception do // EDAError can't be used in Trial Edition
      if IsClass(E, EDAError) then begin
        Dec(FRetries);
        FRetry := True;
        if FRetries = 0 then
          ModalResult := mrCancel;

      {$IFDEF MSWINDOWS}
        if IsClass(E, EMSError) then begin
          if EMSError(E).MSSQLErrorCode <= NE_MAX_NETERROR then
            ActiveControl := edServer
          else
          if EMSError(E).OLEDBErrorCode = DB_SEC_E_AUTH_FAILED then
            if ActiveControl <> edUsername then
              ActiveControl := edPassword;
        end
        else
      {$ENDIF}
        begin
          if EDAError(E).ErrorCode <= TDS_ERROR_AUTH_FAILED then
            if ActiveControl <> edUsername then
              ActiveControl := edPassword;
        end;

        raise;
      end
      else
        raise;
  end;
end;

procedure TMSConnectForm.SetConnectDialog(Value: TCustomConnectDialog);
begin
  FConnectDialog := Value;
  DoInit;
end;

procedure TMSConnectForm.btConnectClick(Sender: TObject);
begin
  DoConnect;
end;

procedure TMSConnectForm.edServerDropDown(Sender: TObject);
var
  OldCursor: TCursor;
  List: TStringList;
begin
  if FListGot then
    Exit;

  FListGot := True;
  OldCursor := Screen.Cursor;
  Screen.Cursor := crSQLWait;
  try
    List := TStringList.Create;
    try
      FConnectDialog.GetServerList(List);
      AssignStrings(List, edServer.Items);
    finally
      List.Free;
    end;
  finally
    Screen.Cursor := OldCursor;
  end;
end;

procedure TMSConnectForm.edDatabaseDropDown(Sender: TObject);
var
  OldCursor: TCursor;
  OldLoginPrompt: boolean;
  OldConnected: boolean;
  List: TStringList;
begin
  if not (FConnectDialog.Connection is TMSConnection) then
    Exit;

  edDatabase.Items.Clear;
  OldLoginPrompt := FConnectDialog.Connection.LoginPrompt;
  OldCursor := Screen.Cursor;
  OldConnected := FConnectDialog.Connection.Connected;
  Screen.Cursor := crSQLWait;

  FConnectDialog.Connection.Password := edPassword.Text;
  FConnectDialog.Connection.Server := edServer.Text;
  FConnectDialog.Connection.UserName := edUsername.Text;

  List := TStringList.Create;
  try
    FConnectDialog.Connection.LoginPrompt := False;
    TDBAccessUtils.SetLockLoginPrompt(FConnectDialog.Connection, True);
    GetDatabasesList(TMSConnection(FConnectDialog.Connection), List);
    AssignStrings(List, edDatabase.Items);
  finally
    List.Free;
    FConnectDialog.Connection.Connected := OldConnected;
    FConnectDialog.Connection.LoginPrompt := OldLoginPrompt;
    TDBAccessUtils.SetLockLoginPrompt(FConnectDialog.Connection, False);
    Screen.Cursor := OldCursor;
  end;
end;

procedure TMSConnectForm.edExit(Sender: TObject);
begin
  try
    if pPassword.Visible then
      FConnectDialog.Connection.Password := edPassword.Text;

    if pServer.Visible then
      FConnectDialog.Connection.Server := edServer.Text;

    if pUsername.Visible then
      FConnectDialog.Connection.UserName := edUsername.Text;

    if (FConnectDialog.Connection is TCustomMSConnection) and pDatabase.Visible then
      TCustomMSConnection(FConnectDialog.Connection).Database := edDatabase.Text;

    if (FConnectDialog.Connection is TMSConnection) then begin
      if pPort.Visible then
        TMSConnection(FConnectDialog.Connection).Port := StrToInt(edPort.Text);
      if pAuthentication.Visible then
        TMSConnection(FConnectDialog.Connection).Authentication := TMSAuthentication(cbAuthentication.ItemIndex);
    end;
  except
    ActiveControl := Sender as TWinControl;
    raise;
  end;
end;

initialization
{  if GetClass('TMSConnectForm') = nil then
    Classes.RegisterClass(TMSConnectForm);}
end.

